/*----------------------------------------------------------------------------*\
|
| Copyright (C) 1997, 2000  James A. Cureington
|                           tonyc@acm.org
| 
| This program is free software; you can redistribute it and/or
| modify it under the terms of the GNU General Public License
| as published by the Free Software Foundation; either version 2
| of the License, or any later version.
| 
| This program is distributed in the hope that it will be useful,
| but WITHOUT ANY WARRANTY; without even the implied warranty of
| MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
| GNU General Public License for more details.
| 
| You should have received a copy of the GNU General Public License
| along with this program; if not, write to the Free Software
| Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
|
|
| $Id: fsm_wish.C,v 1.1 2001/03/04 00:08:31 tonycu Exp $
|
| $Log: fsm_wish.C,v $
| Revision 1.1  2001/03/04 00:08:31  tonycu
| initial check-in at sourceforge...
|
| Revision 1.18  2001/02/18 20:19:19  tony
| added casting of types for Tcl_SetVar2
|
| Revision 1.17  2000/05/24 04:46:40  tony
| Made changes to support FSKC_PRINT_COMMAND environment variable.  Also
| changed FSK_* env vars to FSKC_*.
|
| Revision 1.16  2000/05/24 03:19:08  tony
| Updated all copyright dates.
|
| Revision 1.15  1999/09/13 01:21:05  tony
|
| Updated copyright notice date
|
 * Revision 1.14  1997/03/31  00:50:55  tony
 *      made commenting changes
 *
 * Revision 1.13  1997/03/26  07:04:06  tony
 *      Fixed defects uncovered from test plan
 *
 * Revision 1.12  1997/03/22  17:29:26  tony
 *      1) changed FSM_BIN_DIR to FSKC_BIN_DIR
 *      2) upcased state and event name in find window, so user can
 *         enter the names in upper or lower case
 *      3) move find from file to edit menu
 *
 * Revision 1.11  1997/03/08  03:47:48  tony
 *      1) remove new-lines (using trimright) in event, state, and action-block
 *         edit windows
 *      2) the nextState value was not being incremented in the action-blocks
 *         when a state was inserted before it if the nextState value was 0
 *      3) the copy buffer was not being update when states were deleted
 *      4) if the function called in the action-block is vAbortFsk, don't
 *         enclose it in a return(), i.e. return(vAbortFsk)
 *      5) remove leading and trailing spaces from event and state names
 *      6) change no_op function to take a parameter and return it; also,
 *         rename it uiNoOp
 *      7) change vPrintFskHistory to take a parameter and return it; also,
 *         rename it uiPrintFskHistory
 *      8) add check for duplicate event and state names
 *      9) rename "recursive" variables to "iterative"
 *
 * Revision 1.10  1997/03/02  19:49:13  tony
 *      - Added code to allow user defined error handler
 *      - Added call to infinite state check
 *
 * Revision 1.9  1997/03/01  23:23:53  tony
 *      Made the following changes:
 *           - Add version to FSK file for validation when reading in
 *           - Check for " in parameter list
 *           - Add CVS Id to source and header file
 *           - Add copyright notice to header files
 *           - Add EDIT menu item and associated entries
 *           - Make mouse button 3 cause the commnad pop-up to be displayed
 *             instead of mouse button 2
 *           - Add box to insert header files into
 *           - Changed pchz to pch
 *           - Allow macros to be used in parameter list
 *           - Add function and associated calls to check for infinite state
 *             condition; this is a warning, not an error condition
 *           - Add over-ride abort function capability to GUI
 *           - Changed "modified save pop-up" window to be displayed before
 *             any window is displayed to read and RSK, or create a new one
 *
 * Revision 1.8  1997/02/10  04:00:39  tony
 *      1) added code to generate C source code
 *      2) added trace history
 *
 * Revision 1.7  1997/02/08  22:50:11  tony
 *      1) added info window
 *      2) changed text to fixed width
 *      3) validate fsk before running
 *      4) added fsk filename to top of main window
 *      5) all path names were not being updated on a "save as"
 *      6) added code to print various reports...
 *
 * Revision 1.6  1997/01/26  04:52:23  tony
 * Made changes to:
 *      - clear, copy, paste, and undo action blocks
 *      - comment out Edit menu option
 *      - add ".fsk" to files as needed
 *      - insert user defined default function next state when a state or
 *        event is inserted
 *      - check for invalid number of states and events in newFsk
 *      - add "save if modified" to quit menu item
 *      - add "processing" and "not selectable" cursors
 *      - add procs to position state and event rows, positionState &
 *        positionEvent
 *      - add find state, event, and function commands
 *      - add generate "C" code command
 *      - add "print" Fsk command
 *      - add run Fsk command
 *
 * Revision 1.5  1997/01/15  01:02:26  tony
 * added code to:
 *        - Add undo to TK and lower-level routines for all FSK
 *           edit commands; vSetUndoCommand and iUndoLastCommand
 *        - Added "undo" to state/event mouse menu bar.
 *
 * Revision 1.3  1997/01/13  08:21:25  tony
 * added code to:
 *        - delete states
 *        - insert states
 *
 * Revision 1.2  1997/01/11  22:29:43  tony
 *   Added code to edit state, event, and action blocks.
 *
 * Revision 1.1  1997/01/06  06:32:25  tony
 * Initial check-in of PRODUCTION code.
 *
|  Revision 1.1  1996/05/01  03:07:15  tony
|  this prototype function was derived from tkAppInit.c.  command was added to
|  load state, event, and action block text....
| 
|
\*----------------------------------------------------------------------------*/
#include <common.h>
#include <fsk.h>
#include <fsk.C>

STATIC CHAR *pchFsmWishId = "@(#) $Id: fsm_wish.C,v 1.1 2001/03/04 00:08:31 tonycu Exp $";

#define NOTHING_TO_UNDO             1
#define INIT_UNDO_COMMAND           2

#define CUT_STATE                   3
#define PASTE_STATE                 4
#define INSERT_STATE                5
#define DELETE_STATE                6
#define UPDATE_STATE_BLOCK          7

#define CUT_EVENT                   8
#define PASTE_EVENT                 9
#define INSERT_EVENT                10
#define DELETE_EVENT                11
#define UPDATE_EVENT_BLOCK          12

#define PASTE_ACTION_BLOCK          13
#define CLEAR_ACTION_BLOCK          14
#define UPDATE_ACTION_BLOCK         15


#define NO_COPY_BUFFER              1
#define STATE_COPY_BUFFER           2
#define EVENT_COPY_BUFFER           3
#define ACTION_BLOCK_COPY_BUFFER    4

                         // undo command structure
struct UndoCommand {
   INT  iCommandType;
   INT  iDirection;
   UINT uiFskRow;
   UINT uiFskColumn;
                         // for single state, event, and action-block undo's
   FskActionBlock cFskActionBlock;
   FskState       cFskState;
   FskEvent       cFskEvent;
                         // for complete state or event undo (complete row or
                         // column)
   Fsk* pcFsk;
};

                         // copy buffer structure
struct CopyBuffer {
   INT            iBufferType;
   FskActionBlock cFskActionBlock;
   Fsk*           pcFsk;
};


INT   iCurrentFskModifiedFlag;
CHAR  achTkArrayIndex[100];
CHAR  achTkVariableValue[200];
CHAR* pchResult;

                         // the FSK object displayed to the user
STATIC Fsk         cFsk;

STATIC UndoCommand strUndoCommand;
STATIC CopyBuffer  strCopyBuffer;


#define CHECK_TCL_SETVAR(pchTclResult)                                     \
   if (pchTclResult == NULL)                                               \
   {                                                                       \
      printf("Internal Tcl/Tk error detected, aborting\n");                \
      LOG_ERROR("error Tcl_SetVar returned NULL");                         \
      abort();                                                             \
   }



/*----------------------------------------------------------------------------*\
|                                                                              |
| this file was derived from "tkAppInit.c"                                     |
|                                                                              |
\*----------------------------------------------------------------------------*/

extern "C" {
#include "tk.h"
}

#ifdef SUN
   /*
    * The following variable is a special hack that is needed in order for
    * Sun shared libraries to be used for Tcl.
    */

   extern int matherr();
   int *tclDummyMathPtr = (int *) matherr;
#endif


/*----------------------------------------------------------------------------*\
|                                                                              |
| Function:  Convert a string to an unsigned int.                              |
|                                                                              |
| Returns:   INT                                                               |
|            0 == success                                                      |
|           -1 == failure                                                      |
|                                                                              |
\*----------------------------------------------------------------------------*/
INT
iStringToUnsInt(UINT* puiValue,    // OU  unsigned integer representation of
                                   //     the string.                       
                CHAR* pchValue     // IN  pointer to string to convert to 
                                   //     unsigned int   
               )
{

   INT   iBase = 10;
   ULONG ulNumber;
   CHAR* pchStrToLError = NULL;


   if (pchValue == NULL)
   {
      return(-1);
   }

   ulNumber = (ULONG) strtol(pchValue, &pchStrToLError, iBase);


   if ((pchStrToLError == NULL) || !(*pchStrToLError == '\0'))
   {
      return(-1);
   }

                         // make sure we don't overflow the uint...
   if (ulNumber > UINT_MAX)
   {
      return(-1);
   }

   *puiValue = (UINT) ulNumber;

   return(0);
}



/*----------------------------------------------------------------------------*\
|                                                                              |
| Function:  Determine if the GUI FSK modified flag needs to be updated.       |
|            If it does, update it.                                            |
|                                                                              |
| Returns:   none                                                              |
|            On failure, abort is called.                                      |
|                                                                              |
\*----------------------------------------------------------------------------*/
VOID 
vCheckModifiedFlag(INT*  piEventArrayCount,
                                   // OU  Number of events in the array
                   Tcl_Interp* pstrInterp
                                   // OU  pointer to Tcl interp struct
                  )
{
                          // see if the fsk modified flag needs to be set
   if (*cFsk.piModified != iCurrentFskModifiedFlag)
   {
      iCurrentFskModifiedFlag = *cFsk.piModified;
      sprintf(achTkArrayIndex, "eventArray,%d", *piEventArrayCount);
      (*piEventArrayCount)++;
      if (iCurrentFskModifiedFlag == TRUE)
      {
         sprintf(achTkVariableValue, "%s", "FSK_MODIFIED");
      }
      else
      {
         sprintf(achTkVariableValue, "%s", "FSK_NOT_MODIFIED");
      }

      pchResult = Tcl_SetVar2(pstrInterp, "fsk", achTkArrayIndex,
                              achTkVariableValue, TCL_GLOBAL_ONLY);
      CHECK_TCL_SETVAR(pchResult);
   }

   return;
}                                                    



/*----------------------------------------------------------------------------*\
|                                                                              |
| Function:  Load the FSK file                                                 |
|                                                                              |
| Returns:   none                                                              |
|            On failure, abort is called.                                      |
|                                                                              |
\*----------------------------------------------------------------------------*/
VOID 
vLoadDisplayMessage(CHAR* pchMessage,  
                                   // IN  Message to return to GUI to be
                                   //     displayed to the user         
                    INT*  piEventArrayCount,
                                   // OU  Number of events in the array
                    Tcl_Interp* pstrInterp
                                   // OU  pointer to Tcl interp struct
                   )
{

   pchResult = Tcl_SetVar2(pstrInterp, "fsk", "message",
                               pchMessage, TCL_GLOBAL_ONLY);
   CHECK_TCL_SETVAR(pchResult);
                           

   sprintf(achTkArrayIndex, "eventArray,%d", *piEventArrayCount);
   (*piEventArrayCount)++;

   pchResult = Tcl_SetVar2(pstrInterp, "fsk", achTkArrayIndex,
                               "DISPLAY_MESSAGE", TCL_GLOBAL_ONLY);
   CHECK_TCL_SETVAR(pchResult);


   sprintf(achTkVariableValue, "%d", *piEventArrayCount);

   pchResult = Tcl_SetVar2(pstrInterp, "fsk", "eventArrayLength",
                               achTkVariableValue, TCL_GLOBAL_ONLY);
   CHECK_TCL_SETVAR(pchResult);

   return;
}                                                    



/*----------------------------------------------------------------------------*\
|                                                                              |
| Function:  Find the given state, event, or function.  For states and events, |
|            the associated number is loaded.  If a function is requested,     |
|            the associated row/column of each action-block the function       |
|            is found in is loaded into a scroll-able window.                  |
|                                                                              |
| Returns:   INT                                                               |
|            On failure, abort is called.                                      |
|                                                                              |
\*----------------------------------------------------------------------------*/
INT
iFindFskItem(ClientData    strClientData, 
                                   // IN  
             Tcl_Interp*   pstrInterp,
                                   // OU  pointer to Tcl interp struct
             INT           iArgCount, 
                                   // IN  argument count
             CHAR**        ppchArgVector
                                   // IN  argument vector containing 
                                   //     (1) Indicates to find a STATE,
                                   //         EVENT, or FUNCTION
                                   //     (2) The state, event, or function
                                   //         name to find in the FSK
        )
{

   INT   iResult;
   INT   iEventArrayCount = 0;
   INT   iFound = 0;
   INT   iStrLength;
   INT   iAllocatedMem = 200;
   UINT  uiNumStates;
   UINT  uiNumEvents;
   UINT  uiCurrentState;
   UINT  uiCurrentEvent;
   UINT  uiLoop;
   UINT  uiTemp;
   CHAR* pchScrollList;
   FskState       cFskState;
   FskEvent       cFskEvent;
   FskActionBlock cFskActionBlock;


   if (iArgCount != 3)
   {
      LOG_ERROR("wrong number of args");
      abort();
   }

   iStrLength = strlen(ppchArgVector[2]);

   if (strcmp(ppchArgVector[1], "STATE") == 0)
   {
                          // search for state name
      uiNumStates = cFsk.uiGetNumStates();
      for (uiLoop=0; uiLoop<uiNumStates; uiLoop++)
      {
         if (cFsk.iGetStateData(uiLoop, &cFskState))
         {
                          // internal error...
            LOG_ERROR(achAppErrorString);
            vLoadDisplayMessage(achAppErrorString, 
                                &iEventArrayCount, pstrInterp);
            return(TCL_OK);
         }

         if (strncmp(cFskState.pchName, ppchArgVector[2], iStrLength) == 0)
         {
            iFound = 1;
            sprintf(achTkArrayIndex, "eventArray,%d", iEventArrayCount);
            iEventArrayCount++;
            pchResult = Tcl_SetVar2(pstrInterp, "fsk", achTkArrayIndex,
                              "POSITION_STATE_BLOCK", TCL_GLOBAL_ONLY);
            CHECK_TCL_SETVAR(pchResult);

            sprintf(achTkVariableValue, "%u", uiLoop);
            pchResult = Tcl_SetVar2(pstrInterp, "fsk", "stateToPosition",
                                    achTkVariableValue, TCL_GLOBAL_ONLY);
            CHECK_TCL_SETVAR(pchResult);
            break;
         }
      }

      if (!iFound)
      {
         sprintf(achErrorString, "%s state not found\n", ppchArgVector[1]);
         vLoadDisplayMessage(achErrorString, &iEventArrayCount, pstrInterp);
      }
   }
   else if (strcmp(ppchArgVector[1], "EVENT") == 0)
   {
                          // search for event name
      uiNumEvents = cFsk.uiGetNumEvents();
      for (uiLoop=0; uiLoop<uiNumEvents; uiLoop++)
      {
         if (cFsk.iGetEventData(uiLoop, &cFskEvent))
         {
                          // internal error...
            LOG_ERROR(achAppErrorString);
            vLoadDisplayMessage(achAppErrorString, 
                                &iEventArrayCount, pstrInterp);
            return(TCL_OK);
         }

         if (strncmp(cFskEvent.pchName, ppchArgVector[2], iStrLength) == 0)
         {
            iFound = 1;
            sprintf(achTkArrayIndex, "eventArray,%d", iEventArrayCount);
            iEventArrayCount++;
            pchResult = Tcl_SetVar2(pstrInterp, "fsk", achTkArrayIndex,
                              "POSITION_EVENT_BLOCK", TCL_GLOBAL_ONLY);
            CHECK_TCL_SETVAR(pchResult);

            sprintf(achTkVariableValue, "%u", uiLoop);
            pchResult = Tcl_SetVar2(pstrInterp, "fsk", "eventToPosition",
                                    achTkVariableValue, TCL_GLOBAL_ONLY);
            CHECK_TCL_SETVAR(pchResult);
            break;
         }
      }

      if (!iFound)
      {
         sprintf(achErrorString, "event %s not found\n", ppchArgVector[1]);
         vLoadDisplayMessage(achErrorString, &iEventArrayCount, pstrInterp);
      }
   }
   else if (strcmp(ppchArgVector[1], "FUNCTION") == 0)
   {
                          // find ALL the action-blocks that contain
                          // the given function

      pchScrollList = (CHAR*) malloc(iAllocatedMem);
      CHECK_MALLOC(pchScrollList);
      sprintf(pchScrollList, " %s function found at event,state location :\n",
              ppchArgVector[2]);

      if (cFsk.iGetFirstActionBlock(&cFskActionBlock,
                                    &uiCurrentEvent, &uiCurrentState))
      {
                                     // this is an internal error...
         sprintf(achErrorString, "Could not get action-block data, %u, %u",
                 uiCurrentEvent, uiCurrentState);
         LOG_ERROR(achErrorString);
         vLoadDisplayMessage(achErrorString, &iEventArrayCount, pstrInterp);
         return(TCL_OK);
      }

      do
      {
         if (strncmp(cFskActionBlock.pchFunction, ppchArgVector[2],
                     iStrLength) == 0)
         {
            iFound = 1;
                                     // add event/state location to list
            iAllocatedMem += 30;
            pchScrollList = (CHAR*) realloc(pchScrollList, iAllocatedMem);
            CHECK_MALLOC(pchScrollList);
            sprintf(&pchScrollList[strlen(pchScrollList)], 
                    "   %u,%u\n", uiCurrentEvent, uiCurrentState);
                    
         }

      } while (cFsk.iGetNextActionBlock(&cFskActionBlock,
                                  &uiCurrentEvent, &uiCurrentState) == 0);

      if (iFound)
      {
         pchResult = Tcl_SetVar2(pstrInterp, "fsk", "scrollMessage",
                                 pchScrollList, TCL_GLOBAL_ONLY);
         CHECK_TCL_SETVAR(pchResult);

         sprintf(achTkArrayIndex, "eventArray,%d", iEventArrayCount);
         iEventArrayCount++;
         pchResult = Tcl_SetVar2(pstrInterp, "fsk", achTkArrayIndex,
                           "DISPLAY_SCROLL_MESSAGE", TCL_GLOBAL_ONLY);
         CHECK_TCL_SETVAR(pchResult);
      }
      else
      {
         sprintf(achErrorString, "%s function not found\n", ppchArgVector[1]);
         vLoadDisplayMessage(achErrorString, &iEventArrayCount, pstrInterp);
      }

      free(pchScrollList);
   }
   else
   {
      LOG_ERROR("Internal error");
      abort();
   }

   sprintf(achTkVariableValue, "%d", iEventArrayCount);
   pchResult = Tcl_SetVar2(pstrInterp, "fsk", "eventArrayLength",
                           achTkVariableValue, TCL_GLOBAL_ONLY);
   CHECK_TCL_SETVAR(pchResult);


   return(TCL_OK);
}



/*----------------------------------------------------------------------------*\
|                                                                              |
| Function:  Set the contents of a GUI state block to the contents of the      |
|            given state block object                                          |
|                                                                              |
| Returns:   none                                                              |
|            On failure, abort is called.                                      |
|                                                                              |
\*----------------------------------------------------------------------------*/
VOID 
vSetGuiStateBlockData(UINT uiStateNumber,
                                   // IN  index to GUI state to set
                      FskState*  pcFskState,
                                   // IN  fsk state to init the GUI state with
                      Tcl_Interp* pstrInterp
                                   // OU  pointer to Tcl interp struct
                     )
{

                         // set stateBlock(uiStateNumber,*) to the contents
                         // of the provided FSK state
   sprintf(achTkArrayIndex, "%u,name", uiStateNumber);

   pchResult = Tcl_SetVar2(pstrInterp, "stateBlock", achTkArrayIndex,
                               (CHAR*)pcFskState->pchName, TCL_GLOBAL_ONLY);
   CHECK_TCL_SETVAR(pchResult);


   sprintf(achTkArrayIndex, "%u,description", uiStateNumber);

   pchResult = Tcl_SetVar2(pstrInterp, "stateBlock", achTkArrayIndex,
                               (CHAR*)pcFskState->pchDescription, 
                               TCL_GLOBAL_ONLY);
   CHECK_TCL_SETVAR(pchResult);

   return;
}                                                    



/*----------------------------------------------------------------------------*\
|                                                                              |
| Function:  Set the contents of a GUI event block to the contents of the      |
|            given event block object                                          |
|                                                                              |
| Returns:   none                                                              |
|            On failure, abort is called.                                      |
|                                                                              |
\*----------------------------------------------------------------------------*/
VOID 
vSetGuiEventBlockData(UINT uiEventNumber,
                                   // IN  index to GUI event to set
                      FskEvent*  pcFskEvent,
                                   // IN  fsk event to init the GUI event with
                      Tcl_Interp* pstrInterp
                                   // OU  pointer to Tcl interp struct
                     )
{

                         // set eventBlock(uiEventNumber,*) to the contents
                         // of the provided FSK event


   sprintf(achTkArrayIndex, "%u,name", uiEventNumber);

   pchResult = Tcl_SetVar2(pstrInterp, "eventBlock", achTkArrayIndex,
                               (CHAR*)pcFskEvent->pchName, TCL_GLOBAL_ONLY);
   CHECK_TCL_SETVAR(pchResult);


   sprintf(achTkArrayIndex, "%u,description", uiEventNumber);

   pchResult = Tcl_SetVar2(pstrInterp, "eventBlock", achTkArrayIndex,
                               (CHAR*)pcFskEvent->pchDescription, 
                               TCL_GLOBAL_ONLY);
   CHECK_TCL_SETVAR(pchResult);

   return;
}                                                    



/*----------------------------------------------------------------------------*\
|                                                                              |
| Function:  Set the contents of a GUI action-block to the contents of the     |
|            given FSK action-block object                                     |
|                                                                              |
| Returns:   none                                                              |
|            On failure, abort is called.                                      |
|                                                                              |
\*----------------------------------------------------------------------------*/
VOID 
vSetGuiActionBlockData(UINT uiRowNumber,
                                   // IN  action-block row number
                       UINT uiColumnNumber,
                                   // IN  action-block column number
                       FskActionBlock*  pcFskActionBlock,
                                   // IN  fsk action-block to init the GUI 
                                   //     action-block with
                       Tcl_Interp* pstrInterp
                                   // OU  pointer to Tcl interp struct
                      )
{
   CHAR  achFunctionName[FUNCTION_NAME_LEN];
   CHAR  achFunctionParameters[FUNCTION_NAME_LEN];
   CHAR* pchTemp;
                        

                         // split function name into the name and parm list
   strcpy(achFunctionName, (CHAR*)pcFskActionBlock->pchFunction);
   pchTemp = strchr(achFunctionName, '(');
   if (pchTemp != NULL)
   {
      strcpy(achFunctionParameters, pchTemp);
      *pchTemp = '\0';
   }
   else
   {
                         // if "(" not found, set parameter list to NULL
                         // the validation function will catch this...
      achFunctionParameters[0] = '\0';
   }

                         // set actionBlock(uiRowNumber,uiColumnNumber,*) 
                         // to the contents of the provided action-block
   sprintf(achTkArrayIndex, 
           "%u,%u,functionName", uiRowNumber, uiColumnNumber);

   pchResult = Tcl_SetVar2(pstrInterp, "actionBlock", achTkArrayIndex,
                               achFunctionName, TCL_GLOBAL_ONLY);
   CHECK_TCL_SETVAR(pchResult);


   sprintf(achTkArrayIndex, 
           "%u,%u,functionParameters", uiRowNumber, 
           uiColumnNumber);

   pchResult = Tcl_SetVar2(pstrInterp, "actionBlock", achTkArrayIndex,
                               achFunctionParameters, TCL_GLOBAL_ONLY);
   CHECK_TCL_SETVAR(pchResult);


   sprintf(achTkArrayIndex, 
           "%u,%u,nextState", uiRowNumber, uiColumnNumber);

   if (*pcFskActionBlock->puiNextState < UNASSIGNED_STATE)
   {
      sprintf(achTkVariableValue, "%u", *pcFskActionBlock->puiNextState);
   }
   else
   {
      sprintf(achTkVariableValue, "*"); 
   }

   pchResult = Tcl_SetVar2(pstrInterp, "actionBlock", achTkArrayIndex,
                               achTkVariableValue, TCL_GLOBAL_ONLY);
   CHECK_TCL_SETVAR(pchResult);


   sprintf(achTkArrayIndex, 
           "%u,%u,iterative", uiRowNumber, uiColumnNumber);

   sprintf(achTkVariableValue, "%u", *pcFskActionBlock->piIterative);

   pchResult = Tcl_SetVar2(pstrInterp, "actionBlock", achTkArrayIndex,
                               achTkVariableValue, TCL_GLOBAL_ONLY);
   CHECK_TCL_SETVAR(pchResult);


   sprintf(achTkArrayIndex, 
           "%u,%u,description", uiRowNumber, uiColumnNumber);

   sprintf(achTkVariableValue, "%s", pcFskActionBlock->pchDescription);

   pchResult = Tcl_SetVar2(pstrInterp, "actionBlock", achTkArrayIndex,
                               achTkVariableValue, TCL_GLOBAL_ONLY);
   CHECK_TCL_SETVAR(pchResult);


   return;
}                                                    



/*----------------------------------------------------------------------------*\
|                                                                              |
| Function:  Sets the undo structure up for the given command.                 |
|            This function must first be called with                           |
|            vSetUndoCommand(INIT_UNDO_COMMAND, 0, 0, 0,);                     |
|            before it or the iUndoLastCommand(...) can be called.             |
|                                                                              |
| Returns:   none                                                              |
|            On failure, abort is called.                                      |
|                                                                              |
\*----------------------------------------------------------------------------*/
VOID 
vSetUndoCommand(INT   iCurrentCommand,
                                   // IN  the current command about to be
                                   //     executed, INSERT_STATE, COPY_EVENT,
                                   //     etc.  
                UINT  uiFskRow,
                                   // IN  the row number the command will
                                   //     act on
                UINT  uiFskColumn,
                                   // IN  the column number the command will
                                   //     act on
                INT   iDirection
                                   // IN  the "direction" of the comand, if  
                                   //     needed
               )
{

   STATIC INT iFirstTimeThrough = 1;
   INT    iResult;
   UINT   uiNumberOfStates;
   UINT   uiNumberOfEvents;

                         // just a safty check
   if (iFirstTimeThrough)
   {
      if (iCurrentCommand != INIT_UNDO_COMMAND)
      {
         LOG_ERROR("Internal error");
         abort();
      }

      iFirstTimeThrough = 0;
                         // create a new fsk so we don't have to check
                         // if we should delete it before creating a new one
                         // each time.  this way, we have know we have one to 
                         // delete each time we create one...
      strUndoCommand.pcFsk = new Fsk(1, 1);
      CHECK_NEW(strUndoCommand.pcFsk);
      strUndoCommand.iCommandType = NOTHING_TO_UNDO;
      return;
   }

                         // the actions in each case statement should do
                         // the opposit of the statement
   switch (iCurrentCommand)
   {
      case NOTHING_TO_UNDO: 
      {
         strUndoCommand.iCommandType = NOTHING_TO_UNDO;
         break;
      }

      case UPDATE_STATE_BLOCK  :
      {
                         // save the state block for later undoing
         strUndoCommand.iCommandType = UPDATE_STATE_BLOCK;
         strUndoCommand.uiFskColumn  = uiFskColumn;
         iResult = cFsk.iGetStateData(uiFskColumn, &strUndoCommand.cFskState);
         if (iResult)
         {
            LOG_ERROR("Internal error");
            abort();
         }
         break;
      }

      case UPDATE_EVENT_BLOCK :
      {
                         // save the event block for later undoing
         strUndoCommand.iCommandType = UPDATE_EVENT_BLOCK;
         strUndoCommand.uiFskRow  = uiFskRow;
         iResult = cFsk.iGetEventData(uiFskRow, &strUndoCommand.cFskEvent);
         if (iResult)
         {
            LOG_ERROR("Internal error");
            abort();
         }
         break;
      }


      case PASTE_STATE   :
      case INSERT_STATE  :
      {
                         // delete the inserted or pasted state
         strUndoCommand.iCommandType = DELETE_STATE;
         strUndoCommand.uiFskColumn  = uiFskColumn;

                         // if insert or paste is after, we must increment
                         // the column number
         if (iDirection == AFTER)
         {
            strUndoCommand.uiFskColumn++;
         }

         break;
      }

      case INSERT_EVENT :
      case PASTE_EVENT  :
      {
                         // delete the inserted or pasted event
         strUndoCommand.iCommandType = DELETE_EVENT;
         strUndoCommand.uiFskRow     = uiFskRow;

                         // if insert or paste is after, we must increment
                         // the row number
         if (iDirection == AFTER)
         {
            strUndoCommand.uiFskRow++;
         }

         break;
      }

      case CUT_STATE :
      case DELETE_STATE :
      {
                         // paste the cut or deleted state
         strUndoCommand.iCommandType = PASTE_STATE;
         strUndoCommand.uiFskColumn  = uiFskColumn;

                         // always set to BEFORE to paste at the current state
                         // unless we are deleting the last state
         if (strUndoCommand.uiFskColumn != (cFsk.uiGetNumStates()-1))
         {
            strUndoCommand.iDirection = BEFORE;
         }
         else
         {
                         // paste after the next to the last state....
            strUndoCommand.uiFskColumn--;
            strUndoCommand.iDirection = AFTER;
         }

         delete (strUndoCommand.pcFsk);
         uiNumberOfEvents = cFsk.uiGetNumEvents();
         strUndoCommand.pcFsk = new Fsk(uiNumberOfEvents, 1);
         CHECK_NEW(strUndoCommand.pcFsk);

         iResult = cFsk.iCopyState(uiFskColumn, strUndoCommand.pcFsk, 0);
         if (iResult)
         {
            LOG_ERROR("Internal error");
            abort();
         }

         break;
      }

      case CUT_EVENT :
      case DELETE_EVENT :
      {
                         // paste the cut or deleted event
         strUndoCommand.iCommandType = PASTE_EVENT;
         strUndoCommand.uiFskRow     = uiFskRow;

                         // always set to BEFORE, to paste at the current event
                         // unless we are deleting the last event
         if (strUndoCommand.uiFskRow != (cFsk.uiGetNumEvents()-1))
         {
            strUndoCommand.iDirection = BEFORE;
         }
         else
         {
                         // paste after the next to the last event....
            strUndoCommand.uiFskRow--;
            strUndoCommand.iDirection = AFTER;
         }


         delete (strUndoCommand.pcFsk);
         uiNumberOfStates = cFsk.uiGetNumStates();
         strUndoCommand.pcFsk = new Fsk(1, uiNumberOfStates);
         CHECK_NEW(strUndoCommand.pcFsk);

         iResult = cFsk.iCopyEvent(uiFskRow, strUndoCommand.pcFsk, 0);
         if (iResult)
         {
            LOG_ERROR("Internal error");
            abort();
         }

         break;
      }

      case UPDATE_ACTION_BLOCK :
      case PASTE_ACTION_BLOCK  :
      case CLEAR_ACTION_BLOCK  :
      {
                         // save the action block for pasting over the
                         // cleared or pasted data
         strUndoCommand.iCommandType = UPDATE_ACTION_BLOCK;
         strUndoCommand.uiFskRow     = uiFskRow;
         strUndoCommand.uiFskColumn  = uiFskColumn;
         iResult = cFsk.iGetActionBlockData(uiFskRow, uiFskColumn, 
                                            &strUndoCommand.cFskActionBlock);
         if (iResult)
         {
            LOG_ERROR("Internal error");
            abort();
         }

         break;
      }

      case INIT_UNDO_COMMAND: 
      {
                         // should only init the first time through, the
                         // init was already handled above...
         LOG_ERROR("Internal error");
         abort();
      }

      default : 
      {
         LOG_ERROR("Internal error");
         abort();
      }


   } /* switch (iCurrentCommand) */

   return;
}                                                    



/*----------------------------------------------------------------------------*\
|                                                                              |
| Function:  Undoes the last edit command; cut, copy, paste, clear...          |
|            This functions companion function is vSetUndoCommand.             |
|                                                                              |
| Returns:   INT                                                               |
|            TCL_OK = success                                                  |
|            abort is called on non-recoverable errors                         |
|                                                                              |
\*----------------------------------------------------------------------------*/
INT 
iUndoLastCommand(ClientData    strClientData, 
                                   // IN  
                 Tcl_Interp*   pstrInterp,
                                   // OU  pointer to Tcl interp struct
                 INT           iArgCount, 
                                   // IN  argument count
                 CHAR**        ppchArgVector
                                   // IN  none
                )
{
   INT   iResult;
   INT   iUpdateStates            = 0;
   INT   iUpdateEvents            = 0;
   INT   iUpdateAllActionBlocks   = 0;
   INT   iEventArrayCount         = 0;
   UINT  uiFskColumn;
   UINT  uiFskRow;
   UINT  uiNumStates;
   UINT  uiNumEvents;
   UINT  uiCurrentState;
   UINT  uiCurrentEvent;
   UINT  uiLoop;
   UINT  uiTemp;
   FskActionBlock cFskActionBlock;
   FskState       cFskState;
   FskEvent       cFskEvent;

                         // determine what to undo
   switch (strUndoCommand.iCommandType)
   {
      case NOTHING_TO_UNDO: 
      {
         vLoadDisplayMessage("Nothing to Undo", 
                             &iEventArrayCount, pstrInterp);
         return(TCL_OK);
      }

      case PASTE_STATE :
      {

                         // paste the state
         iResult = cFsk.iInsertState(strUndoCommand.uiFskColumn,
                                     strUndoCommand.iDirection);
         if (iResult)
         {
            LOG_ERROR("Internal error");
            LOG_ERROR(achAppErrorString);
            abort();
         }


         uiFskColumn = strUndoCommand.uiFskColumn;

                        // if we inserted after the current state, 
                        // increment the state so we are at the new
                        // inserted state
         sprintf(achTkArrayIndex, "eventArray,%d", iEventArrayCount);
         iEventArrayCount++;
         if (strUndoCommand.iDirection == AFTER)
         {
            strUndoCommand.uiFskColumn++;
            pchResult = Tcl_SetVar2(pstrInterp, "fsk", achTkArrayIndex,
                                    "PASTE_STATE_AFTER", TCL_GLOBAL_ONLY);
         }
         else
         {
            pchResult = Tcl_SetVar2(pstrInterp, "fsk", achTkArrayIndex,
                                    "PASTE_STATE_BEFORE", TCL_GLOBAL_ONLY);
         }
         CHECK_TCL_SETVAR(pchResult);


                        // search all action-blocks checking for the
                        // nextState value set to LAST_SET_TO_UNASSIGNED_STATE.
                        // all next states set to this value should be changed
                        // to the column/state we just undeleted because these
                        // nextState values referenced this state before it was
                        // deleted.
         if (cFsk.iGetFirstActionBlock(&cFskActionBlock,
                                       &uiCurrentEvent, &uiCurrentState))
         {
                                     // this is an internal error...
            sprintf(achErrorString, "Could not get action-block data, %u, %u",
                    uiCurrentEvent, uiCurrentState);
            LOG_ERROR(achErrorString);
            abort();
         }

         do
         {
            uiTemp = *(cFskActionBlock.puiNextState);
            if (uiTemp == LAST_SET_TO_UNASSIGNED_STATE)
            {
               cFskActionBlock.vSetNextState(strUndoCommand.uiFskColumn);
               if (cFsk.iSetActionBlockData(uiCurrentEvent, uiCurrentState, 
                                            &cFskActionBlock))
               {
                  LOG_ERROR("Internal error");
                  abort();
               }
            }

         } while (cFsk.iGetNextActionBlock(&cFskActionBlock,
                                     &uiCurrentEvent, &uiCurrentState) == 0);



                        // copy undo command state 0 to the specified FSK state
         iResult = strUndoCommand.pcFsk->iCopyState(0, &cFsk, 
                                              strUndoCommand.uiFskColumn);
         if (iResult)
         {
            LOG_ERROR("Internal error");
            LOG_ERROR(achAppErrorString);
            abort();
         }

         /* ********************************************************** */
         uiNumStates = cFsk.uiGetNumStates();
         uiNumEvents = cFsk.uiGetNumEvents();

                        // any action-blocks in the state copy buffer
                        // that the nextState is greater than or equal
                        // to the location we inserted, should be
                        // incremented by one to preserve the nextState 
                        // location
         if ((strCopyBuffer.iBufferType == STATE_COPY_BUFFER) &&
             (strCopyBuffer.pcFsk->uiGetNumEvents() == uiNumEvents))
         {
            for (uiLoop=0; uiLoop<uiNumEvents; uiLoop++)
            {
               if (strCopyBuffer.pcFsk->iGetActionBlockData(uiLoop, 0, 
                                                            &cFskActionBlock))
               {
                  LOG_ERROR("Internal error");
                  abort();
               }

               uiTemp = *(cFskActionBlock.puiNextState);
               if ((uiTemp >= uiFskColumn) && (uiTemp < MAX_FSK_STATES))
               {
                  uiTemp++;
                  cFskActionBlock.vSetNextState(uiTemp);
                  if (strCopyBuffer.pcFsk->iSetActionBlockData(uiLoop, 
                                                         0, &cFskActionBlock))
                  {
                     LOG_ERROR("Internal error");
                     abort();
                  }
               }
               else if (uiTemp == LAST_SET_TO_UNASSIGNED_STATE)
               {
                  cFskActionBlock.vSetNextState(uiFskColumn);
                  if (strCopyBuffer.pcFsk->iSetActionBlockData(uiLoop, 
                                                         0, &cFskActionBlock))
                  {
                     LOG_ERROR("Internal error");
                     abort();
                  }
               }
            }
         }
         else if (strCopyBuffer.iBufferType == ACTION_BLOCK_COPY_BUFFER)
         {
                        // update the action-block copy buffer next states
            uiTemp = *(strCopyBuffer.cFskActionBlock.puiNextState);
            if ((uiTemp >= uiFskColumn) && (uiTemp < MAX_FSK_STATES))
            {
               uiTemp++;
               strCopyBuffer.cFskActionBlock.vSetNextState(uiTemp);
            }
            else if (uiTemp == LAST_SET_TO_UNASSIGNED_STATE)
            {
               strCopyBuffer.cFskActionBlock.vSetNextState(uiFskColumn);
            }
         }
         /* ********************************************************** */

                         // NOTE: we must wait til we have undone the 
                         //       command before setting the new undo command
                         //       since we use the struct above...
         vSetUndoCommand(PASTE_STATE, 0, uiFskColumn,
                         strUndoCommand.iDirection);

         iUpdateStates          = 1;
         iUpdateAllActionBlocks = 1;
         break;
      }

      case PASTE_EVENT :
      {
                         // paste the event
         iResult = cFsk.iInsertEvent(strUndoCommand.uiFskRow,
                                     strUndoCommand.iDirection);
         if (iResult)
         {
            LOG_ERROR("Internal error");
            LOG_ERROR(achAppErrorString);
            abort();
         }

         uiFskRow = strUndoCommand.uiFskRow;

                        // if we inserted after the current event, 
                        // increment the event so we are at the new
                        // inserted event
         sprintf(achTkArrayIndex, "eventArray,%d", iEventArrayCount);
         iEventArrayCount++;
         if (strUndoCommand.iDirection == AFTER)
         {
            strUndoCommand.uiFskRow++;
            pchResult = Tcl_SetVar2(pstrInterp, "fsk", achTkArrayIndex,
                                    "PASTE_EVENT_AFTER", TCL_GLOBAL_ONLY);
         }
         else
         {
            pchResult = Tcl_SetVar2(pstrInterp, "fsk", achTkArrayIndex,
                                    "PASTE_EVENT_BEFORE", TCL_GLOBAL_ONLY);
         }
         CHECK_TCL_SETVAR(pchResult);

                        // copy undo command event 0 to the specified FSK event
         iResult = strUndoCommand.pcFsk->iCopyEvent(0, &cFsk, 
                                                    strUndoCommand.uiFskRow);
         if (iResult)
         {
            LOG_ERROR("Internal error");
            LOG_ERROR(achAppErrorString);
            abort();
         }

                         // NOTE: we must wait til we have undone the 
                         //       command before setting the new undo command
                         //       since we use the struct above...
         vSetUndoCommand(PASTE_EVENT, uiFskRow, 0,
                         strUndoCommand.iDirection);

         iUpdateEvents          = 1;
         iUpdateAllActionBlocks = 1;
         break;
      }

      case DELETE_STATE :
      {
                         // save copy of column since vSetUndoCommand will
                         // modify it
         uiFskColumn = strUndoCommand.uiFskColumn;

         vSetUndoCommand(DELETE_STATE, 0, uiFskColumn, 0);

                         // delete the state
         iResult = cFsk.iDeleteState(uiFskColumn);
         if (iResult)
         {
            LOG_ERROR("Internal error");
            abort();
         }
         sprintf(achTkArrayIndex, "eventArray,%d", iEventArrayCount);
         iEventArrayCount++;
         pchResult = Tcl_SetVar2(pstrInterp, "fsk", achTkArrayIndex,
                                 "DELETE_STATE", TCL_GLOBAL_ONLY);
         CHECK_TCL_SETVAR(pchResult);


         uiNumStates = cFsk.uiGetNumStates();
         uiNumEvents = cFsk.uiGetNumEvents();

                        // any action-blocks in the state copy buffer
                        // that the nextState is greater than or equal
                        // to the location we deleted, should be
                        // decremented by one to preserve the nextState 
                        // location
         if ((strCopyBuffer.iBufferType == STATE_COPY_BUFFER) &&
             (strCopyBuffer.pcFsk->uiGetNumEvents() == uiNumEvents))
         {
            for (uiLoop=0; uiLoop<uiNumEvents; uiLoop++)
            {
               if (strCopyBuffer.pcFsk->iGetActionBlockData(uiLoop, 0, 
                                                            &cFskActionBlock))
               {
                  LOG_ERROR("Internal error");
                  abort();
               }

               uiTemp = *(cFskActionBlock.puiNextState);
               if ((uiTemp > uiFskColumn) && (uiTemp < MAX_FSK_STATES))
               {
                  uiTemp--;
                  cFskActionBlock.vSetNextState(uiTemp);
                  if (strCopyBuffer.pcFsk->iSetActionBlockData(uiLoop, 
                                                         0, &cFskActionBlock))
                  {
                     LOG_ERROR("Internal error");
                     abort();
                  }
               }
               else if (uiTemp == uiFskColumn)
               {
                  cFskActionBlock.vSetNextState(LAST_SET_TO_UNASSIGNED_STATE);
                  if (strCopyBuffer.pcFsk->iSetActionBlockData(uiLoop, 
                                                         0, &cFskActionBlock))
                  {
                     LOG_ERROR("Internal error");
                     abort();
                  }
               }
               else if (uiTemp == LAST_SET_TO_UNASSIGNED_STATE)
               {
                  cFskActionBlock.vSetNextState(UNASSIGNED_STATE);
                  if (strCopyBuffer.pcFsk->iSetActionBlockData(uiLoop, 
                                                         0, &cFskActionBlock))
                  {
                     LOG_ERROR("Internal error");
                     abort();
                  }
               }
            }
         }
         else if (strCopyBuffer.iBufferType == ACTION_BLOCK_COPY_BUFFER)
         {
                        // update the action-block copy buffer next states
            uiTemp = *(strCopyBuffer.cFskActionBlock.puiNextState);
            if ((uiTemp > uiFskColumn) && (uiTemp < MAX_FSK_STATES))
            {
               uiTemp--;
               strCopyBuffer.cFskActionBlock.vSetNextState(uiTemp);
            }
            else if (uiTemp == uiFskColumn)
            {
               strCopyBuffer.cFskActionBlock.vSetNextState(
                                             LAST_SET_TO_UNASSIGNED_STATE);
            }
            else if (uiTemp == LAST_SET_TO_UNASSIGNED_STATE)
            {
               strCopyBuffer.cFskActionBlock.vSetNextState(UNASSIGNED_STATE);
            }
         }


         iUpdateStates          = 1;
         iUpdateAllActionBlocks = 1;

         break;
      }

      case DELETE_EVENT :
      {
                         // save copy of row since vSetUndoCommand will
                         // modify it
         uiFskRow = strUndoCommand.uiFskRow;

         vSetUndoCommand(DELETE_EVENT, uiFskRow, 0, 0);

                         // delete the event
         iResult = cFsk.iDeleteEvent(uiFskRow);
         if (iResult)
         {
            LOG_ERROR("Internal error");
            abort();
         }

         sprintf(achTkArrayIndex, "eventArray,%d", iEventArrayCount);
         iEventArrayCount++;
         pchResult = Tcl_SetVar2(pstrInterp, "fsk", achTkArrayIndex,
                                 "DELETE_EVENT", TCL_GLOBAL_ONLY);
         CHECK_TCL_SETVAR(pchResult);

         iUpdateEvents          = 1;
         iUpdateAllActionBlocks = 1;
         break;
      }

      case UPDATE_ACTION_BLOCK :
      {
                         // save the info action-block before calling 
                         // vSetUndoCommand...

         uiFskRow        = strUndoCommand.uiFskRow;
         uiFskColumn     = strUndoCommand.uiFskColumn;
         cFskActionBlock = strUndoCommand.cFskActionBlock;

         vSetUndoCommand(UPDATE_ACTION_BLOCK, uiFskRow, uiFskColumn, 0);

         iResult = cFsk.iSetActionBlockData(uiFskRow, uiFskColumn, 
                                            &cFskActionBlock);
         if (iResult)
         {
            LOG_ERROR("Internal error");
            abort();
         }

         vSetGuiActionBlockData(uiFskRow, uiFskColumn, 
                                &cFskActionBlock, pstrInterp);

         sprintf(achTkVariableValue, "%u", uiFskRow);
         pchResult = Tcl_SetVar2(pstrInterp, "actionBlock", "rowToUpdate",
                                 achTkVariableValue, TCL_GLOBAL_ONLY);
         CHECK_TCL_SETVAR(pchResult);

         sprintf(achTkVariableValue, "%u", uiFskColumn);
         pchResult = Tcl_SetVar2(pstrInterp, "actionBlock", "columnToUpdate",
                                 achTkVariableValue, TCL_GLOBAL_ONLY);
         CHECK_TCL_SETVAR(pchResult);

         sprintf(achTkArrayIndex, "eventArray,%d", iEventArrayCount);
         iEventArrayCount++;
         pchResult = Tcl_SetVar2(pstrInterp, "fsk", achTkArrayIndex,
                                 "UPDATE_ACTION_BLOCK", TCL_GLOBAL_ONLY);
         CHECK_TCL_SETVAR(pchResult);

         break;
      }

      case UPDATE_STATE_BLOCK  :
      {
                         // save off the undo state block
         uiFskColumn = strUndoCommand.uiFskColumn;
         cFskState   = strUndoCommand.cFskState;

         vSetUndoCommand(UPDATE_STATE_BLOCK, 0, uiFskColumn, 0);

         iResult = cFsk.iSetStateData(uiFskColumn, &cFskState);
         if (iResult)
         {
            LOG_ERROR("Internal error");
            abort();
         }

         vSetGuiStateBlockData(uiFskColumn, &cFskState, pstrInterp);

         sprintf(achTkVariableValue, "%u", uiFskColumn);
         pchResult = Tcl_SetVar2(pstrInterp, "fsk", "stateToUpdate",
                                 achTkVariableValue, TCL_GLOBAL_ONLY);
         CHECK_TCL_SETVAR(pchResult);

         sprintf(achTkArrayIndex, "eventArray,%d", iEventArrayCount);
         iEventArrayCount++;
         pchResult = Tcl_SetVar2(pstrInterp, "fsk", achTkArrayIndex,
                                 "UPDATE_STATE_BLOCK", TCL_GLOBAL_ONLY);
         CHECK_TCL_SETVAR(pchResult);

         break;
      }

      case UPDATE_EVENT_BLOCK  :
      {
                         // save off the undo event block
         uiFskRow = strUndoCommand.uiFskRow;
         cFskEvent   = strUndoCommand.cFskEvent;

         vSetUndoCommand(UPDATE_EVENT_BLOCK, uiFskRow, 0, 0);

         iResult = cFsk.iSetEventData(uiFskRow, &cFskEvent);
         if (iResult)
         {
            LOG_ERROR("Internal error");
            abort();
         }

         vSetGuiEventBlockData(uiFskRow, &cFskEvent, pstrInterp);

         sprintf(achTkVariableValue, "%u", uiFskRow);
         pchResult = Tcl_SetVar2(pstrInterp, "fsk", "eventToUpdate",
                                 achTkVariableValue, TCL_GLOBAL_ONLY);
         CHECK_TCL_SETVAR(pchResult);

         sprintf(achTkArrayIndex, "eventArray,%d", iEventArrayCount);
         iEventArrayCount++;
         pchResult = Tcl_SetVar2(pstrInterp, "fsk", achTkArrayIndex,
                                 "UPDATE_EVENT_BLOCK", TCL_GLOBAL_ONLY);
         CHECK_TCL_SETVAR(pchResult);

         break;
      }


      default : 
      {
         LOG_ERROR("Internal error");
         abort();
      }

   } /* switch (strUndoCommand.iCommandType) */



   uiNumStates = cFsk.uiGetNumStates();
   uiNumEvents = cFsk.uiGetNumEvents();

   if (iUpdateStates)
   {
                          // update the state blocks
      for (uiLoop=0; uiLoop<uiNumStates; uiLoop++)
      {
         if (cFsk.iGetStateData(uiLoop, &cFskState))
         {
                          // internal error...
            LOG_ERROR("Internal error");
            abort();
         }
         vSetGuiStateBlockData(uiLoop, &cFskState, pstrInterp);
      }
   }
   else if (iUpdateEvents)
   {
                          // update the event blocks
      for (uiLoop=0; uiLoop<uiNumEvents; uiLoop++)
      {
         if (cFsk.iGetEventData(uiLoop, &cFskEvent))
         {
                          // internal error...
            LOG_ERROR("Internal error");
            abort();
         }
         vSetGuiEventBlockData(uiLoop, &cFskEvent, pstrInterp);
      }
   }

   if (iUpdateAllActionBlocks)
   {
                          // update all action-blocks
      if (cFsk.iGetFirstActionBlock(&cFskActionBlock,
                                    &uiCurrentEvent, &uiCurrentState))
      {
                                     // this is an internal error...
         sprintf(achErrorString, "Could not get action-block data, %u, %u",
                 uiCurrentEvent, uiCurrentState);
         LOG_ERROR(achErrorString);
         abort();
      }

      do
      {

         vSetGuiActionBlockData(uiCurrentEvent, uiCurrentState, 
                                &cFskActionBlock, pstrInterp);

      } while (cFsk.iGetNextActionBlock(&cFskActionBlock,
                                     &uiCurrentEvent, &uiCurrentState) == 0);
   }

   sprintf(achTkVariableValue, "%u", uiNumStates);

   pchResult = Tcl_SetVar2(pstrInterp, "fsk", "numStates",
                           achTkVariableValue, TCL_GLOBAL_ONLY);
   CHECK_TCL_SETVAR(pchResult);


   sprintf(achTkVariableValue, "%u", uiNumEvents);

   pchResult = Tcl_SetVar2(pstrInterp, "fsk", "numEvents",
                           achTkVariableValue, TCL_GLOBAL_ONLY);
   CHECK_TCL_SETVAR(pchResult);

   vCheckModifiedFlag(&iEventArrayCount, pstrInterp);

   sprintf(achTkVariableValue, "%d", iEventArrayCount);
   pchResult = Tcl_SetVar2(pstrInterp, "fsk", "eventArrayLength",
                           achTkVariableValue, TCL_GLOBAL_ONLY);
   CHECK_TCL_SETVAR(pchResult);

   return(TCL_OK);
}                                                    



/*----------------------------------------------------------------------------*\
|                                                                              |
| Function:  Copy FSK objects to GUI member structures.  This function is      |
|            called when a new fsk is created or read in.                      |
|                                                                              |
| Returns:   INT                                                               |
|            On failure, abort is called.                                      |
|                                                                              |
\*----------------------------------------------------------------------------*/
INT
iCopyFskToGui(Fsk*          pcFsk,
                                   // IN  FSK containing data to set GUI 
                                   //     structures to
              CHAR*         pchFskFileName,
                                   // IN  name to assign to FSK filename
              INT*          piEventArrayCount,
                                   // OU  number of events on GUI array
              Tcl_Interp*   pstrInterp 
                                   // OU  pointer to Tcl interp struct
             )
{

   INT   iResult;
   UINT  uiNumEvents;
   UINT  uiNumStates;
   UINT  uiLoop;
   UINT  uiLoop2;
   CHAR  achFskFileName[300];
   CHAR* pchTemp;

   FskState cFskState;
   FskEvent cFskEvent;
   FskActionBlock cFskActionBlock;


   strCopyBuffer.iBufferType = NO_COPY_BUFFER;

   if (strCopyBuffer.pcFsk != NULL)
   {
      delete (strCopyBuffer.pcFsk);
      strCopyBuffer.pcFsk = NULL;
   }

   vSetUndoCommand(NOTHING_TO_UNDO, 0, 0, 0);

   uiNumEvents = pcFsk->uiGetNumEvents();
   uiNumStates = pcFsk->uiGetNumStates();


                         // initialize all *GUI* state, event and
                         // action-blocks

   for (uiLoop=0; uiLoop<uiNumStates; uiLoop++)
   {
                         // set the GUI state block data
      if (pcFsk->iGetStateData(uiLoop, &cFskState))
      {
         LOG_ERROR("could not get state data");
         LOG_ERROR(achAppErrorString);
         abort();
      }

      vSetGuiStateBlockData(uiLoop, &cFskState, pstrInterp);

      for (uiLoop2=0; uiLoop2<uiNumEvents; uiLoop2++)
      {
                         // set all action-blocks associated with
                         // this state
         if (pcFsk->iGetActionBlockData(uiLoop2, uiLoop, &cFskActionBlock))
         {
            LOG_ERROR("could not get action-block data");
            LOG_ERROR(achAppErrorString);
            abort();
         }

         vSetGuiActionBlockData(uiLoop2, uiLoop, &cFskActionBlock,
                                pstrInterp);
      }
   }


                         // set the GUI event block data
   for (uiLoop=0; uiLoop<uiNumEvents; uiLoop++)
   {
      if (pcFsk->iGetEventData(uiLoop, &cFskEvent))
      {
         LOG_ERROR("could not get event data");
         LOG_ERROR(achAppErrorString);
         abort();
      }

      vSetGuiEventBlockData(uiLoop, &cFskEvent, pstrInterp);
   }
                           


                         // set other GUI data members
   pchResult = Tcl_SetVar2(pstrInterp, "fsk", "description",
                          (CHAR*)pcFsk->pchDescription, TCL_GLOBAL_ONLY);
   CHECK_TCL_SETVAR(pchResult);

   pchResult = Tcl_SetVar2(pstrInterp, "fsk", "defaultFunction",
                           (CHAR*)pcFsk->pchDefaultFunctionName, 
                           TCL_GLOBAL_ONLY);
   CHECK_TCL_SETVAR(pchResult);


   pchResult = Tcl_SetVar2(pstrInterp, "fsk", "returnType",
                          (CHAR*)pcFsk->pchReturnType, TCL_GLOBAL_ONLY);
   CHECK_TCL_SETVAR(pchResult);

   pchResult = Tcl_SetVar2(pstrInterp, "fsk", "headerFiles",
                          (CHAR*)pcFsk->pchHeaderFiles, TCL_GLOBAL_ONLY);
   CHECK_TCL_SETVAR(pchResult);

   pchResult = Tcl_SetVar2(pstrInterp, "fsk", "userDefinedErrorHandler",
                          (CHAR*)
                            (((*(pcFsk->piUserDefinedErrorHandler) == TRUE)
                            ?"1":"0")), 
                          TCL_GLOBAL_ONLY);
   CHECK_TCL_SETVAR(pchResult);

   pchTemp = strrchr(pchFskFileName, '/');
   if (pchTemp != NULL)
   {
      strcpy(achFskFileName, ++pchTemp);
   }
   else
   {
                         // fsk name does not contain a "/"
      strcpy(achFskFileName, pchFskFileName);
   }

   pchResult = Tcl_SetVar2(pstrInterp, "fsk", "fileName",
                           achFskFileName, TCL_GLOBAL_ONLY);
   CHECK_TCL_SETVAR(pchResult);


                         // remove the ".fsk" from the fsk filename
   pchTemp = strrchr(achFskFileName, '.');
   if (pchTemp == NULL)
   {
      LOG_ERROR("Internal Error");
      abort();
   }

   *pchTemp = '\0';
   strcat(achFskFileName, ".c");
   pchResult = Tcl_SetVar2(pstrInterp, "fsk", "sourceFileName",
                           achFskFileName, TCL_GLOBAL_ONLY);
   CHECK_TCL_SETVAR(pchResult);


   *pchTemp = '\0';
   strcat(achFskFileName, "_table.txt");
   pchResult = Tcl_SetVar2(pstrInterp, "fsk", "printTableName",
                           achFskFileName, TCL_GLOBAL_ONLY);
   CHECK_TCL_SETVAR(pchResult);


   *pchTemp = '\0';
   strcat(achFskFileName, "_state.txt");
   pchResult = Tcl_SetVar2(pstrInterp, "fsk", "printStateFileName",
                           achFskFileName, TCL_GLOBAL_ONLY);
   CHECK_TCL_SETVAR(pchResult);


   *pchTemp = '\0';
   strcat(achFskFileName, "_event.txt");
   pchResult = Tcl_SetVar2(pstrInterp, "fsk", "printEventFileName",
                           achFskFileName, TCL_GLOBAL_ONLY);
   CHECK_TCL_SETVAR(pchResult);


   pchResult = Tcl_SetVar2(pstrInterp, "fsk", "dateModified",
                           (CHAR*)pcFsk->pchLastModifiedDate, TCL_GLOBAL_ONLY);
   CHECK_TCL_SETVAR(pchResult);


   if (*pcFsk->puiDefaultNextState == 0)
   {
      sprintf(achTkVariableValue, "%s", "FIRST_STATE");
   }
   else if (*pcFsk->puiDefaultNextState == UNASSIGNED_STATE)
   {
      sprintf(achTkVariableValue, "%s", "UNASSIGNED_STATE");
   }
   else 
   {
      sprintf(achTkVariableValue, "%s", "LAST_STATE");
   }

   pchResult = Tcl_SetVar2(pstrInterp, "fsk", "defaultNextState",
                           achTkVariableValue, TCL_GLOBAL_ONLY);
   CHECK_TCL_SETVAR(pchResult);


   sprintf(achTkVariableValue, "%u", uiNumStates);

   pchResult = Tcl_SetVar2(pstrInterp, "fsk", "numStates",
                           achTkVariableValue, TCL_GLOBAL_ONLY);
   CHECK_TCL_SETVAR(pchResult);


   sprintf(achTkVariableValue, "%u", uiNumEvents);

   pchResult = Tcl_SetVar2(pstrInterp, "fsk", "numEvents",
                           achTkVariableValue, TCL_GLOBAL_ONLY);
   CHECK_TCL_SETVAR(pchResult);


   sprintf(achTkVariableValue, "%d", *piEventArrayCount);

   pchResult = Tcl_SetVar2(pstrInterp, "fsk", "eventArrayLength",
                           achTkVariableValue, TCL_GLOBAL_ONLY);
   CHECK_TCL_SETVAR(pchResult);



   return(TCL_OK);

}



/*----------------------------------------------------------------------------*\
|                                                                              |
| Function:  Set the default function and nextState values in the C code for   |
|            the action-blocks from the GUI variables.                         |
|                                                                              |
| Returns:   none                                                              |
|            0 == success                                                      |
|           !0 == failure                                                      |
|                                                                              |
\*----------------------------------------------------------------------------*/
INT 
iSetFskDefaults(ClientData    strClientData, 
                                   // IN  
                Tcl_Interp*   pstrInterp,
                                   // OU  pointer to Tcl interp struct
                INT           iArgCount, 
                                   // IN  argument count
                CHAR**        ppchArgVector
                                   // IN  none
               )
{
   INT   iResult;
   INT   iEventArrayCount = 0;

                         // get the default function and nextState values
   pchResult = Tcl_GetVar2(pstrInterp, "fsk", "defaultFunction", 0);
   if (pchResult != NULL)
   {
      if (cFsk.iSetDefaultFunction(pchResult))
      {
         LOG_ERROR(achAppErrorString);
         vLoadDisplayMessage(achAppErrorString, 
                             &iEventArrayCount, pstrInterp);
         return(TCL_OK);
      }
   }
   else
   {
      cFsk.iSetDefaultFunction("vAbortFsk();");
      {
         LOG_ERROR(achAppErrorString);
         vLoadDisplayMessage(achAppErrorString, 
                             &iEventArrayCount, pstrInterp);
         return(TCL_OK);
      }

      pchResult = Tcl_SetVar2(pstrInterp, "fsk", "defaultFunction",
                              "vAbortFsk();", TCL_GLOBAL_ONLY);
      CHECK_TCL_SETVAR(pchResult);

   }

   pchResult = Tcl_GetVar2(pstrInterp, "fsk", "defaultNextState", 0);
   if (pchResult != NULL)
   {
      if (strcmp(pchResult, "FIRST_STATE") == 0)
      {
         cFsk.vSetDefaultNextState(0);
      }
      else if (strcmp(pchResult, "LAST_STATE") == 0)
      {
         cFsk.vSetDefaultNextState(cFsk.uiGetNumStates()-1);
      }
      else 
      {
         cFsk.vSetDefaultNextState(UNASSIGNED_STATE);
         pchResult = Tcl_SetVar2(pstrInterp, "fsk", "defaultNextState",
                                 "UNASSIGNED_STATE", TCL_GLOBAL_ONLY);
         CHECK_TCL_SETVAR(pchResult);
      }
   }
   else
   {
      cFsk.vSetDefaultNextState(UNASSIGNED_STATE);
      pchResult = Tcl_SetVar2(pstrInterp, "fsk", "defaultNextState",
                              "UNASSIGNED_STATE", TCL_GLOBAL_ONLY);
      CHECK_TCL_SETVAR(pchResult);
   }

   vSetUndoCommand(NOTHING_TO_UNDO, 0, 0, 0);


   vCheckModifiedFlag(&iEventArrayCount, pstrInterp);

   sprintf(achTkVariableValue, "%d", iEventArrayCount);
   pchResult = Tcl_SetVar2(pstrInterp, "fsk", "eventArrayLength",
                           achTkVariableValue, TCL_GLOBAL_ONLY);
   CHECK_TCL_SETVAR(pchResult);

   return(TCL_OK);
}



/*----------------------------------------------------------------------------*\
|                                                                              |
| Function:  Set the generated source program return type and header files.    |
|                                                                              |
| Returns:   none                                                              |
|            0 == success                                                      |
|           !0 == failure                                                      |
|                                                                              |
\*----------------------------------------------------------------------------*/
INT 
iSetFskSourceSettings(ClientData    strClientData, 
                                   // IN  
                      Tcl_Interp*   pstrInterp,
                                   // OU  pointer to Tcl interp struct
                      INT           iArgCount, 
                                   // IN  argument count
                      CHAR**        ppchArgVector
                                   // IN  none
                     )
{

   INT   iResult;
   INT   iEventArrayCount = 0;

                         // get the fsk return type
   pchResult = Tcl_GetVar2(pstrInterp, "fsk", "returnType", 0);
   if (pchResult != NULL)
   {
      if (cFsk.iSetReturnType(pchResult))
      {
         vLoadDisplayMessage(achAppErrorString, &iEventArrayCount, pstrInterp);
         return(TCL_OK);
      }
   }
   else
   {
      if (cFsk.iSetReturnType(""))
      {
         LOG_ERROR("Internal error");
         vLoadDisplayMessage(achAppErrorString, &iEventArrayCount, pstrInterp);
         return(TCL_OK);
      }

      pchResult = Tcl_SetVar2(pstrInterp, "fsk", "returnType",
                              "", TCL_GLOBAL_ONLY);
      CHECK_TCL_SETVAR(pchResult);

   }

                         // get the fsk header files
   pchResult = Tcl_GetVar2(pstrInterp, "fsk", "headerFiles", 0);
   if (pchResult != NULL)
   {
      if (cFsk.iSetHeaderFilesString(pchResult))
      {
         LOG_DEBUG("Internal error");
         vLoadDisplayMessage(achAppErrorString, &iEventArrayCount, pstrInterp);
         return(TCL_OK);
      }
   }
   else
   {
      if (cFsk.iSetHeaderFilesString(""))
      {
         LOG_DEBUG("Internal error");
         vLoadDisplayMessage(achAppErrorString, &iEventArrayCount, pstrInterp);
         return(TCL_OK);
      }

      pchResult = Tcl_SetVar2(pstrInterp, "fsk", "headerFiles",
                              "", TCL_GLOBAL_ONLY);
      CHECK_TCL_SETVAR(pchResult);

   }

                         // get the user defined error handler flag
   pchResult = Tcl_GetVar2(pstrInterp, "fsk", "userDefinedErrorHandler", 0);
   if (pchResult != NULL)
   {
      cFsk.vSetUserDefinedErrorHandlerFlag(atoi(pchResult));
   }
   else
   {
      pchResult = Tcl_SetVar2(pstrInterp, "fsk", "userDefinedErrorHandler",
                             (CHAR*)
                               (((*(cFsk.piUserDefinedErrorHandler) == TRUE)
                               ?"1":"0")),
                             TCL_GLOBAL_ONLY);
      CHECK_TCL_SETVAR(pchResult);

   }

   vSetUndoCommand(NOTHING_TO_UNDO, 0, 0, 0);


   vCheckModifiedFlag(&iEventArrayCount, pstrInterp);

   sprintf(achTkVariableValue, "%d", iEventArrayCount);
   pchResult = Tcl_SetVar2(pstrInterp, "fsk", "eventArrayLength",
                           achTkVariableValue, TCL_GLOBAL_ONLY);
   CHECK_TCL_SETVAR(pchResult);

   return(TCL_OK);
}



/*----------------------------------------------------------------------------*\
|                                                                              |
| Function:  Set FSK description to the one provided.                          |
|                                                                              |
| Returns:   INT                                                               |
|            On failure, abort is called.                                      |
|                                                                              |
\*----------------------------------------------------------------------------*/
INT
iSetFskDescription(ClientData    strClientData, 
                                   // IN  
                   Tcl_Interp*   pstrInterp,
                                   // OU  pointer to Tcl interp struct
                   INT           iArgCount, 
                                   // IN  argument count
                   CHAR**        ppchArgVector
                                   // IN  argument vector containing 
                                   //     (1) FSK description
        )
{
   INT   iResult;
   INT   iEventArrayCount = 0;
   CHAR* pchFskDescription;


   if (iArgCount != 2)
   {
      LOG_ERROR("wrong number of args");
      abort();
   }

   vSetUndoCommand(NOTHING_TO_UNDO, 0, 0, 0);

   pchFskDescription = ppchArgVector[1];

   iResult = cFsk.iSetDescription(pchFskDescription);
   if (iResult)
   {
      vLoadDisplayMessage(achAppErrorString, &iEventArrayCount, pstrInterp);
      return(TCL_OK);
   }

   vSetUndoCommand(NOTHING_TO_UNDO, 0, 0, 0);


   vCheckModifiedFlag(&iEventArrayCount, pstrInterp);

   sprintf(achTkVariableValue, "%d", iEventArrayCount);
   pchResult = Tcl_SetVar2(pstrInterp, "fsk", "eventArrayLength",
                           achTkVariableValue, TCL_GLOBAL_ONLY);
   CHECK_TCL_SETVAR(pchResult);

   return(TCL_OK);
}



/*----------------------------------------------------------------------------*\
|                                                                              |
| Function:  Set the FSK binaries directory to the env var FSKC_BIN_DIR        |
|                                                                              |
| Returns:   INT                                                               |
|            TCL_OK on success                                                 |
|            TCL_ERROR on failure                                              |
|                                                                              |
\*----------------------------------------------------------------------------*/
INT
iSetFskBinDir(ClientData    strClientData, 
                                   // IN  
              Tcl_Interp*   pstrInterp,
                                   // OU  pointer to Tcl interp struct
              INT           iArgCount, 
                                   // IN  argument count
              CHAR**        ppchArgVector
                                   // IN  none
        )
{

   CHAR* pchEnvVar;
   CHAR* pchBinDir;


                         // load binary directory
   if ((pchEnvVar = getenv("FSKC_BIN_DIR")) == NULL)
   {
      printf("\n");
      printf("ERROR: FSKC_BIN_DIR envirionment variable not set.\n");
      printf("       This envirionment variable must be set to the\n");
      printf("       directory containing the FSKC CASE Tool\n");
      printf("       binary/executable file and Tcl/Tk scripts.\n");
      printf("\n");
      return(TCL_ERROR);
   }

   pchBinDir = (CHAR*) malloc(strlen(pchEnvVar)+1);
   CHECK_MALLOC(pchBinDir);

   strcpy(pchBinDir, pchEnvVar);

                         // remove the last "/" if it exists
   if (pchBinDir[strlen(pchBinDir)] == '/')
   {
      pchBinDir[strlen(pchBinDir)] = '\0';
   }

   pchResult = Tcl_SetVar2(pstrInterp, "fsk", "binDirectory",
                           pchBinDir, TCL_GLOBAL_ONLY);
   CHECK_TCL_SETVAR(pchResult);

   free(pchBinDir);

   return(TCL_OK);
}

/*----------------------------------------------------------------------------*\
|                                                                              |
| Function:  Load an initial FSK into the GUI, this is only called when the    |
|            GUI is first started                                              |
|                                                                              |
| Returns:   INT                                                               |
|            On failure, abort is called or TCL_ERROR is returned              |
|                                                                              |
\*----------------------------------------------------------------------------*/
INT
iInitFsk(ClientData    strClientData, 
                                   // IN  
         Tcl_Interp*   pstrInterp,
                                   // OU  pointer to Tcl interp struct
         INT           iArgCount, 
                                   // IN  argument count
         CHAR**        ppchArgVector
                                   // IN  none
        )
{

   STATIC INT   iFirstTimeThrough = 1;
   INT   iResult;
   INT   iEventArrayCount = 0;
   CHAR* pchDot = ".";
   CHAR* pchDotSlash = "./";
   CHAR* pchEnvVar;
   CHAR* pchBinDir;

   if (!iFirstTimeThrough)
   {
      LOG_ERROR("Internal error: iInitFsk called twice");
      abort();
   }

   iFirstTimeThrough = 0;

   strCopyBuffer.pcFsk   = NULL;

   vSetUndoCommand(INIT_UNDO_COMMAND, 0, 0, 0);

                         // set the main window title
   sprintf(achTkVariableValue, "Finite State Kernel Creator %s", 
           pchFskVersion);
   pchResult = Tcl_SetVar2(pstrInterp, "fsk", "FSK_CASE_NAME",
                           achTkVariableValue, TCL_GLOBAL_ONLY);
   CHECK_TCL_SETVAR(pchResult);

                         // set the default function and next state
   if (iSetFskDefaults(strClientData, pstrInterp,
                       iArgCount, ppchArgVector) != 0)
   {
      LOG_ERROR("Internal error: could not set default values");
      abort();
   }

                         // set the source file settings
   if (iSetFskSourceSettings(strClientData, pstrInterp,
                             iArgCount, ppchArgVector) != 0)
   {
      LOG_ERROR("Internal error: could not set source settings");
      abort();
   }

                         // initialize the fsk
   iResult = cFsk.iInitFsk(3,4);
   if (iResult)
   {
      LOG_ERROR(achAppErrorString);
      vLoadDisplayMessage(achAppErrorString, &iEventArrayCount, pstrInterp);
      return(TCL_OK);
   }


   iResult = iCopyFskToGui(&cFsk, "scratch.fsk", &iEventArrayCount, 
                           pstrInterp);
 
   if (iResult)
   {
      LOG_ERROR("could not load data to Tcl/Tk");
      abort();
   }


   pchResult = Tcl_SetVar2(pstrInterp, "fsk", "completeFileName",
                           "./scratch.fsk", TCL_GLOBAL_ONLY);
   CHECK_TCL_SETVAR(pchResult);


                         // load the binDirectory var
   if (iSetFskBinDir(strClientData, pstrInterp, iArgCount, ppchArgVector) != 
       TCL_OK)
   {
      return(TCL_ERROR);
   }

                         // load environment vars
   if ((pchEnvVar = getenv("FSKC_PATH")) == NULL)
   {
      pchEnvVar = pchDot;
   }

   pchResult = Tcl_SetVar2(pstrInterp, "fsk", "fileToOpenPath",
                           pchEnvVar, TCL_GLOBAL_ONLY);
   CHECK_TCL_SETVAR(pchResult);

   pchResult = Tcl_SetVar2(pstrInterp, "fsk", "fileToSaveAsPath",
                           pchEnvVar, TCL_GLOBAL_ONLY);
   CHECK_TCL_SETVAR(pchResult);


   if ((pchEnvVar = getenv("FSKC_SOURCE_FILE_PATH")) == NULL)
   {
      pchEnvVar = pchDotSlash;
   }

   pchResult = Tcl_SetVar2(pstrInterp, "fsk", "sourceFilePath",
                           pchEnvVar, TCL_GLOBAL_ONLY);
   CHECK_TCL_SETVAR(pchResult);


   if ((pchEnvVar = getenv("FSKC_STATE_TABLE_PRINT_PATH")) == NULL)
   {
      pchEnvVar = pchDotSlash;
   }

   pchResult = Tcl_SetVar2(pstrInterp, "fsk", "stateTablePrintPath",
                           pchEnvVar, TCL_GLOBAL_ONLY);
   CHECK_TCL_SETVAR(pchResult);


   if ((pchEnvVar = getenv("FSKC_EVENT_DESCRIPTION_PATH")) == NULL)
   {
      pchEnvVar = pchDotSlash;
   }

   pchResult = Tcl_SetVar2(pstrInterp, "fsk", "eventDescriptionPath",
                           pchEnvVar, TCL_GLOBAL_ONLY);
   CHECK_TCL_SETVAR(pchResult);


   if ((pchEnvVar = getenv("FSKC_STATE_DESCRIPTION_PATH")) == NULL)
   {
      pchEnvVar = pchDotSlash;
   }

   pchResult = Tcl_SetVar2(pstrInterp, "fsk", "stateDescriptionPath",
                           pchEnvVar, TCL_GLOBAL_ONLY);
   CHECK_TCL_SETVAR(pchResult);

                         // force the flag to be updated
   iCurrentFskModifiedFlag = (*cFsk.piModified) + 1;
   vCheckModifiedFlag(&iEventArrayCount, pstrInterp);

   sprintf(achTkVariableValue, "%d", iEventArrayCount);
   pchResult = Tcl_SetVar2(pstrInterp, "fsk", "eventArrayLength",
                           achTkVariableValue, TCL_GLOBAL_ONLY);
   CHECK_TCL_SETVAR(pchResult);



   return(TCL_OK);
}



/*----------------------------------------------------------------------------*\
|                                                                              |
| Function:  Setup a new FSK.                                                  |
|                                                                              |
| Returns:   INT                                                               |
|            On failure, abort is called.                                      |
|                                                                              |
\*----------------------------------------------------------------------------*/
INT
iSetupNewFsk(ClientData    strClientData, 
                                   // IN  
             Tcl_Interp*   pstrInterp,
                                   // OU  pointer to Tcl interp struct
             INT           iArgCount, 
                                   // IN  argument count
             CHAR**        ppchArgVector
                                   // IN  argument vector containing 
                                   //     (1) Number of events
                                   //     (2) Number of states
                                   //     (3) FSK return type
                                   //     (4) FSK description
        )
{

   INT   iResult;
   INT   iEventArrayCount = 0;
   UINT  uiNumStates;
   UINT  uiNumEvents;
   CHAR* pchFskReturnType;
   CHAR* pchFskDescription;


   if (iArgCount != 5)
   {
      LOG_ERROR("wrong number of args");
      abort();
   }

   vSetUndoCommand(NOTHING_TO_UNDO, 0, 0, 0);

   iResult = iStringToUnsInt(&uiNumEvents, ppchArgVector[1]);
   if (iResult)
   {
      vLoadDisplayMessage("Bad number of events given", 
                          &iEventArrayCount, pstrInterp);
      return(TCL_OK);
   }

   iResult = iStringToUnsInt(&uiNumStates, ppchArgVector[2]);
   if (iResult)
   {
      vLoadDisplayMessage("Bad number of states given", 
                          &iEventArrayCount, pstrInterp);
      return(TCL_OK);
   }

   pchFskReturnType = ppchArgVector[3];
   pchFskDescription = ppchArgVector[4];

                         // initialize the new fsk
   iResult = cFsk.iInitFsk(uiNumEvents, uiNumStates);
   if (iResult)
   {
      LOG_ERROR(achAppErrorString);
      vLoadDisplayMessage(achAppErrorString, &iEventArrayCount, pstrInterp);
      return(TCL_OK);
   }

   iResult = cFsk.iSetReturnType(pchFskReturnType);
   if (iResult)
   {
      vLoadDisplayMessage(achAppErrorString, &iEventArrayCount, pstrInterp);
      return(TCL_OK);
   }

   iResult = cFsk.iSetDescription(pchFskDescription);
   if (iResult)
   {
      vLoadDisplayMessage(achAppErrorString, &iEventArrayCount, pstrInterp);
      return(TCL_OK);
   }


   iResult = iCopyFskToGui(&cFsk, "scratch.fsk", &iEventArrayCount, 
                           pstrInterp);
 
   if (iResult)
   {
      LOG_ERROR("could not copy data to Tcl/Tk");
      abort();
   }


   pchResult = Tcl_SetVar2(pstrInterp, "fsk", "completeFileName",
                           "./scratch.fsk", TCL_GLOBAL_ONLY);
   CHECK_TCL_SETVAR(pchResult);

   sprintf(achTkArrayIndex, "eventArray,%d", iEventArrayCount);
   iEventArrayCount++;
   pchResult = Tcl_SetVar2(pstrInterp, "fsk", achTkArrayIndex,
                           "NEW_FSK", TCL_GLOBAL_ONLY);
   CHECK_TCL_SETVAR(pchResult);

   sprintf(achTkArrayIndex, "eventArray,%d", iEventArrayCount);
   iEventArrayCount++;
   pchResult = Tcl_SetVar2(pstrInterp, "fsk", achTkArrayIndex,
                           "UPDATE_FSK_NAME", TCL_GLOBAL_ONLY);
   CHECK_TCL_SETVAR(pchResult);

   vCheckModifiedFlag(&iEventArrayCount, pstrInterp);

   sprintf(achTkVariableValue, "%d", iEventArrayCount);
   pchResult = Tcl_SetVar2(pstrInterp, "fsk", "eventArrayLength",
                           achTkVariableValue, TCL_GLOBAL_ONLY);
   CHECK_TCL_SETVAR(pchResult);


   return(TCL_OK);
}



/*----------------------------------------------------------------------------*\
|                                                                              |
| Function:  Change the state name and description of the given state.         |
|                                                                              |
| Returns:   INT                                                               |
|            On failure, abort is called.                                      |
|                                                                              |
\*----------------------------------------------------------------------------*/
INT
iUpdateStateData(ClientData    strClientData, 
                                   // IN  
                 Tcl_Interp*   pstrInterp,
                                   // OU  pointer to Tcl interp struct
                 INT           iArgCount, 
                                   // IN  argument count
                 CHAR**        ppchArgVector
                                   // IN  argument vector containing 
                                   //     (1) New state name
                                   //     (2) New state description
                                   //     (3) State number
        )
{

   INT   iResult;
   INT   iEventArrayCount = 0;
   UINT  uiStateNumber;
   CHAR* pchStateName;
   CHAR* pchStateDescription;
   FskState cState;


   if (iArgCount != 4)
   {
      LOG_ERROR("wrong number of args");
      abort();
   }

   pchStateName = ppchArgVector[1];
   pchStateDescription = ppchArgVector[2];

   iResult = iStringToUnsInt(&uiStateNumber, ppchArgVector[3]);
   if (iResult)
   {
      LOG_ERROR("Bad state number");
      abort();
   }

   vSetUndoCommand(UPDATE_STATE_BLOCK, 0, uiStateNumber, 0);

   iResult = cState.iSetData(pchStateName, pchStateDescription);
   if (iResult)
   {
      vLoadDisplayMessage(achAppErrorString, &iEventArrayCount, pstrInterp);
      return(TCL_OK);
   }

   iResult = cFsk.iSetStateData(uiStateNumber, &cState);
   if (iResult)
   {
      vLoadDisplayMessage(achAppErrorString, &iEventArrayCount, pstrInterp);
      return(TCL_OK);
   }

   vSetGuiStateBlockData(uiStateNumber, &cState, pstrInterp);

   pchResult = Tcl_SetVar2(pstrInterp, "fsk", "stateToUpdate",
                           ppchArgVector[3], TCL_GLOBAL_ONLY);
   CHECK_TCL_SETVAR(pchResult);



   sprintf(achTkArrayIndex, "eventArray,%d", iEventArrayCount);
   iEventArrayCount++;
   pchResult = Tcl_SetVar2(pstrInterp, "fsk", achTkArrayIndex,
                           "UPDATE_STATE_BLOCK", TCL_GLOBAL_ONLY);
   CHECK_TCL_SETVAR(pchResult);

   vCheckModifiedFlag(&iEventArrayCount, pstrInterp);

   sprintf(achTkVariableValue, "%d", iEventArrayCount);
   pchResult = Tcl_SetVar2(pstrInterp, "fsk", "eventArrayLength",
                           achTkVariableValue, TCL_GLOBAL_ONLY);
   CHECK_TCL_SETVAR(pchResult);


   return(TCL_OK);
}



/*----------------------------------------------------------------------------*\
|                                                                              |
| Function:  Change the event name and description of the given event.         |
|                                                                              |
| Returns:   INT                                                               |
|            On failure, abort is called.                                      |
|                                                                              |
\*----------------------------------------------------------------------------*/
INT
iUpdateEventData(ClientData    strClientData, 
                                   // IN  
                 Tcl_Interp*   pstrInterp,
                                   // OU  pointer to Tcl interp struct
                 INT           iArgCount, 
                                   // IN  argument count
                 CHAR**        ppchArgVector
                                   // IN  argument vector containing 
                                   //     (1) New event name
                                   //     (2) New event description
                                   //     (3) Event number
        )
{

   INT   iResult;
   INT   iEventArrayCount = 0;
   UINT  uiEventNumber;
   CHAR* pchEventName;
   CHAR* pchEventDescription;
   FskEvent cEvent;


   if (iArgCount != 4)
   {
      LOG_ERROR("wrong number of args");
      abort();
   }

   pchEventName = ppchArgVector[1];
   pchEventDescription = ppchArgVector[2];

   iResult = iStringToUnsInt(&uiEventNumber, ppchArgVector[3]);
   if (iResult)
   {
      LOG_ERROR("Bad event number");
      abort();
   }

   vSetUndoCommand(UPDATE_EVENT_BLOCK, uiEventNumber, 0, 0);

   iResult = cEvent.iSetData(pchEventName, pchEventDescription);
   if (iResult)
   {
      vLoadDisplayMessage(achAppErrorString, &iEventArrayCount, pstrInterp);
      return(TCL_OK);
   }

   iResult = cFsk.iSetEventData(uiEventNumber, &cEvent);
   if (iResult)
   {
      vLoadDisplayMessage(achAppErrorString, &iEventArrayCount, pstrInterp);
      return(TCL_OK);
   }

   vSetGuiEventBlockData(uiEventNumber, &cEvent, pstrInterp);

   pchResult = Tcl_SetVar2(pstrInterp, "fsk", "eventToUpdate",
                           ppchArgVector[3], TCL_GLOBAL_ONLY);
   CHECK_TCL_SETVAR(pchResult);



   sprintf(achTkArrayIndex, "eventArray,%d", iEventArrayCount);
   iEventArrayCount++;
   pchResult = Tcl_SetVar2(pstrInterp, "fsk", achTkArrayIndex,
                           "UPDATE_EVENT_BLOCK", TCL_GLOBAL_ONLY);
   CHECK_TCL_SETVAR(pchResult);

   vCheckModifiedFlag(&iEventArrayCount, pstrInterp);

   sprintf(achTkVariableValue, "%d", iEventArrayCount);
   pchResult = Tcl_SetVar2(pstrInterp, "fsk", "eventArrayLength",
                           achTkVariableValue, TCL_GLOBAL_ONLY);
   CHECK_TCL_SETVAR(pchResult);


   return(TCL_OK);
}



/*----------------------------------------------------------------------------*\
|                                                                              |
| Function:  Update the specified action-block with the given data.            |
|                                                                              |
| Returns:   INT                                                               |
|            On failure, abort is called.                                      |
|                                                                              |
\*----------------------------------------------------------------------------*/
INT
iUpdateActionBlockData(ClientData    strClientData, 
                                   // IN  
                       Tcl_Interp*   pstrInterp,
                                   // OU  pointer to Tcl interp struct
                       INT           iArgCount, 
                                   // IN  argument count
                       CHAR**        ppchArgVector
                                   // IN  argument vector containing 
                                   //     (1) New function name
                                   //     (2) New next state
                                   //     (3) New iterative flag
                                   //     (4) New description
                                   //     (5) Row number of action-block
                                   //     (6) Column number of action-block
        )
{

   INT   iResult;
   INT   iEventArrayCount = 0;
   INT   iIterativeFlag;
   UINT  uiRowNumber;
   UINT  uiColumnNumber;
   UINT  uiNextState;
   CHAR* pchFunctionName;
   CHAR* pchFunctionDescription;
   FskActionBlock cActionBlock;


   if (iArgCount != 7)
   {
      LOG_ERROR("wrong number of args");
      abort();
   }

   pchFunctionName = ppchArgVector[1];


   if (strchr(ppchArgVector[2], '-') != NULL)
   {
      sprintf(achErrorString, "Transition state must be positive \"%s\"\n", 
                               ppchArgVector[2]);
      vLoadDisplayMessage(achErrorString, &iEventArrayCount, pstrInterp);
      return(TCL_OK);
   }

   if (*(ppchArgVector[2]) == '\0')
   {
      uiNextState = UNASSIGNED_STATE;
   }
   else
   {
      iResult = iStringToUnsInt(&uiNextState, ppchArgVector[2]);
      if (iResult)
      {
         sprintf(achErrorString, "Bad transition state given \"%s\"\n", 
                                  ppchArgVector[2]);
         vLoadDisplayMessage(achErrorString, &iEventArrayCount, pstrInterp);
         return(TCL_OK);
      }

      if (uiNextState >= cFsk.uiGetNumStates())
      {
         sprintf(achErrorString, "State number \"%s\" does not exist\n", 
                                  ppchArgVector[2]);
         vLoadDisplayMessage(achErrorString, &iEventArrayCount, pstrInterp);
         return(TCL_OK);
      }
   }

   iIterativeFlag = atoi(ppchArgVector[3]);
   pchFunctionDescription = ppchArgVector[4];

   iResult = iStringToUnsInt(&uiRowNumber, ppchArgVector[5]);
   if (iResult)
   {
      LOG_ERROR("Bad event number");
      abort();
   }

   iResult = iStringToUnsInt(&uiColumnNumber, ppchArgVector[6]);
   if (iResult)
   {
      LOG_ERROR("Bad state number");
      abort();
   }

   vSetUndoCommand(UPDATE_ACTION_BLOCK, uiRowNumber, uiColumnNumber, 0);

   iResult = cActionBlock.iSetData(uiNextState, iIterativeFlag, 
                                   pchFunctionName, pchFunctionDescription);
   if (iResult)
   {
      vLoadDisplayMessage(achAppErrorString, &iEventArrayCount, pstrInterp);
      return(TCL_OK);
   }


   iResult = cFsk.iSetActionBlockData(uiRowNumber, uiColumnNumber, 
                                      &cActionBlock);
   if (iResult)
   {
      vLoadDisplayMessage(achAppErrorString, &iEventArrayCount, pstrInterp);
      return(TCL_OK);
   }

   vSetGuiActionBlockData(uiRowNumber, uiColumnNumber, 
                          &cActionBlock, pstrInterp);

   pchResult = Tcl_SetVar2(pstrInterp, "actionBlock", "rowToUpdate",
                           ppchArgVector[5], TCL_GLOBAL_ONLY);
   CHECK_TCL_SETVAR(pchResult);

   pchResult = Tcl_SetVar2(pstrInterp, "actionBlock", "columnToUpdate",
                           ppchArgVector[6], TCL_GLOBAL_ONLY);
   CHECK_TCL_SETVAR(pchResult);



   sprintf(achTkArrayIndex, "eventArray,%d", iEventArrayCount);
   iEventArrayCount++;
   pchResult = Tcl_SetVar2(pstrInterp, "fsk", achTkArrayIndex,
                           "UPDATE_ACTION_BLOCK", TCL_GLOBAL_ONLY);
   CHECK_TCL_SETVAR(pchResult);

   vCheckModifiedFlag(&iEventArrayCount, pstrInterp);

   sprintf(achTkVariableValue, "%d", iEventArrayCount);
   pchResult = Tcl_SetVar2(pstrInterp, "fsk", "eventArrayLength",
                           achTkVariableValue, TCL_GLOBAL_ONLY);
   CHECK_TCL_SETVAR(pchResult);


   return(TCL_OK);
}



/*----------------------------------------------------------------------------*\
|                                                                              |
| Function:  Clear the contents of the specified action-block.                 |
|                                                                              |
| Returns:   INT                                                               |
|            On failure, abort is called.                                      |
|                                                                              |
\*----------------------------------------------------------------------------*/
INT
iClearActionBlockData(ClientData    strClientData, 
                                   // IN  
                      Tcl_Interp*   pstrInterp,
                                   // OU  pointer to Tcl interp struct
                      INT           iArgCount, 
                                   // IN  argument count
                      CHAR**        ppchArgVector
                                   // IN  argument vector containing 
                                   //     (1) Row number of action-block
                                   //     (2) Column number of action-block
        )
{

   INT   iResult;
   INT   iEventArrayCount = 0;
   UINT  uiRowNumber;
   UINT  uiColumnNumber;
   FskActionBlock cActionBlock;


   if (iArgCount != 3)
   {
      LOG_ERROR("wrong number of args");
      abort();
   }


   iResult = iStringToUnsInt(&uiRowNumber, ppchArgVector[1]);
   if (iResult)
   {
      LOG_ERROR("Bad event number");
      abort();
   }

   iResult = iStringToUnsInt(&uiColumnNumber, ppchArgVector[2]);
   if (iResult)
   {
      LOG_ERROR("Bad state number");
      abort();
   }

   vSetUndoCommand(CLEAR_ACTION_BLOCK, uiRowNumber, uiColumnNumber, 0);

   iResult = cActionBlock.iSetData(UNASSIGNED_STATE, FALSE, "", "");
   if (iResult)
   {
      LOG_ERROR(achAppErrorString);
      vLoadDisplayMessage(achAppErrorString, &iEventArrayCount, pstrInterp);
      return(TCL_OK);
   }


   iResult = cFsk.iSetActionBlockData(uiRowNumber, uiColumnNumber, 
                                      &cActionBlock);
   if (iResult)
   {
      vLoadDisplayMessage(achAppErrorString, &iEventArrayCount, pstrInterp);
      return(TCL_OK);
   }

   vSetGuiActionBlockData(uiRowNumber, uiColumnNumber, 
                          &cActionBlock, pstrInterp);

   pchResult = Tcl_SetVar2(pstrInterp, "actionBlock", "rowToUpdate",
                           ppchArgVector[1], TCL_GLOBAL_ONLY);
   CHECK_TCL_SETVAR(pchResult);

   pchResult = Tcl_SetVar2(pstrInterp, "actionBlock", "columnToUpdate",
                           ppchArgVector[2], TCL_GLOBAL_ONLY);
   CHECK_TCL_SETVAR(pchResult);


   sprintf(achTkArrayIndex, "eventArray,%d", iEventArrayCount);
   iEventArrayCount++;
   pchResult = Tcl_SetVar2(pstrInterp, "fsk", achTkArrayIndex,
                           "UPDATE_ACTION_BLOCK", TCL_GLOBAL_ONLY);
   CHECK_TCL_SETVAR(pchResult);

   vCheckModifiedFlag(&iEventArrayCount, pstrInterp);

   sprintf(achTkVariableValue, "%d", iEventArrayCount);
   pchResult = Tcl_SetVar2(pstrInterp, "fsk", "eventArrayLength",
                           achTkVariableValue, TCL_GLOBAL_ONLY);
   CHECK_TCL_SETVAR(pchResult);


   return(TCL_OK);
}



/*----------------------------------------------------------------------------*\
|                                                                              |
| Function:  Copies the contents of the specified action-block to the          |
|            copy buffer.                                                      |
|                                                                              |
| Returns:   INT                                                               |
|            On failure, abort is called.                                      |
|                                                                              |
\*----------------------------------------------------------------------------*/
INT
iCopyActionBlockData(ClientData    strClientData, 
                                   // IN  
                      Tcl_Interp*   pstrInterp,
                                   // OU  pointer to Tcl interp struct
                      INT           iArgCount, 
                                   // IN  argument count
                      CHAR**        ppchArgVector
                                   // IN  argument vector containing 
                                   //     (1) Row number of action-block
                                   //     (2) Column number of action-block
        )
{

   INT   iResult;
   INT   iEventArrayCount = 0;
   UINT  uiRowNumber;
   UINT  uiColumnNumber;
   FskActionBlock cActionBlock;


   if (iArgCount != 3)
   {
      LOG_ERROR("wrong number of args");
      abort();
   }


   iResult = iStringToUnsInt(&uiRowNumber, ppchArgVector[1]);
   if (iResult)
   {
      LOG_ERROR("Bad event number");
      abort();
   }

   iResult = iStringToUnsInt(&uiColumnNumber, ppchArgVector[2]);
   if (iResult)
   {
      LOG_ERROR("Bad state number");
      abort();
   }

   iResult = cFsk.iGetActionBlockData(uiRowNumber, uiColumnNumber, 
                                      &cActionBlock);
   if (iResult)
   {
      LOG_ERROR(achAppErrorString);
      vLoadDisplayMessage(achAppErrorString, &iEventArrayCount, pstrInterp);
      return(TCL_OK);
   }

   iResult = strCopyBuffer.cFskActionBlock.iSetData(
                               *cActionBlock.puiNextState,
                               *cActionBlock.piIterative,
                                (CHAR*) cActionBlock.pchFunction,
                                (CHAR*) cActionBlock.pchDescription);
   if (iResult)
   {
      LOG_ERROR(achAppErrorString);
      vLoadDisplayMessage(achAppErrorString, &iEventArrayCount, pstrInterp);
      return(TCL_OK);
   }

   strCopyBuffer.iBufferType = ACTION_BLOCK_COPY_BUFFER;

   vCheckModifiedFlag(&iEventArrayCount, pstrInterp);

   sprintf(achTkVariableValue, "%d", iEventArrayCount);
   pchResult = Tcl_SetVar2(pstrInterp, "fsk", "eventArrayLength",
                           achTkVariableValue, TCL_GLOBAL_ONLY);
   CHECK_TCL_SETVAR(pchResult);


   return(TCL_OK);
}



/*----------------------------------------------------------------------------*\
|                                                                              |
| Function:  Pastes the contents of the copy buffer action-block to the        |
|            specified action-block.                                           |
|                                                                              |
| Returns:   INT                                                               |
|            On failure, abort is called.                                      |
|                                                                              |
\*----------------------------------------------------------------------------*/
INT
iPasteActionBlockData(ClientData    strClientData, 
                                   // IN  
                      Tcl_Interp*   pstrInterp,
                                   // OU  pointer to Tcl interp struct
                      INT           iArgCount, 
                                   // IN  argument count
                      CHAR**        ppchArgVector
                                   // IN  argument vector containing 
                                   //     (1) Row number of action-block
                                   //     (2) Column number of action-block
        )
{

   INT   iResult;
   INT   iEventArrayCount = 0;
   UINT  uiRowNumber;
   UINT  uiColumnNumber;


   if (iArgCount != 3)
   {
      LOG_ERROR("wrong number of args");
      abort();
   }


   iResult = iStringToUnsInt(&uiRowNumber, ppchArgVector[1]);
   if (iResult)
   {
      LOG_ERROR("Bad event number");
      abort();
   }

   iResult = iStringToUnsInt(&uiColumnNumber, ppchArgVector[2]);
   if (iResult)
   {
      LOG_ERROR("Bad state number");
      abort();
   }

   if (strCopyBuffer.iBufferType != ACTION_BLOCK_COPY_BUFFER)
   {
      vLoadDisplayMessage("Copy buffer does not contain an ACTION-BLOCK", 
                           &iEventArrayCount, pstrInterp);
      return(TCL_OK);
   }

   vSetUndoCommand(PASTE_ACTION_BLOCK, uiRowNumber, uiColumnNumber, 0);

   iResult = cFsk.iSetActionBlockData(uiRowNumber, uiColumnNumber, 
                                      &strCopyBuffer.cFskActionBlock);
   if (iResult)
   {
      LOG_ERROR(achAppErrorString);
      vLoadDisplayMessage(achAppErrorString, &iEventArrayCount, pstrInterp);
      return(TCL_OK);
   }


   vSetGuiActionBlockData(uiRowNumber, uiColumnNumber, 
                          &strCopyBuffer.cFskActionBlock, pstrInterp);

   sprintf(achTkVariableValue, "%u", uiRowNumber);
   pchResult = Tcl_SetVar2(pstrInterp, "actionBlock", "rowToUpdate",
                           achTkVariableValue, TCL_GLOBAL_ONLY);
   CHECK_TCL_SETVAR(pchResult);

   sprintf(achTkVariableValue, "%u", uiColumnNumber);
   pchResult = Tcl_SetVar2(pstrInterp, "actionBlock", "columnToUpdate",
                           achTkVariableValue, TCL_GLOBAL_ONLY);
   CHECK_TCL_SETVAR(pchResult);

   sprintf(achTkArrayIndex, "eventArray,%d", iEventArrayCount);
   iEventArrayCount++;
   pchResult = Tcl_SetVar2(pstrInterp, "fsk", achTkArrayIndex,
                           "UPDATE_ACTION_BLOCK", TCL_GLOBAL_ONLY);
   CHECK_TCL_SETVAR(pchResult);


   vCheckModifiedFlag(&iEventArrayCount, pstrInterp);

   sprintf(achTkVariableValue, "%d", iEventArrayCount);
   pchResult = Tcl_SetVar2(pstrInterp, "fsk", "eventArrayLength",
                           achTkVariableValue, TCL_GLOBAL_ONLY);
   CHECK_TCL_SETVAR(pchResult);


   return(TCL_OK);
}



/*----------------------------------------------------------------------------*\
|                                                                              |
| Function:  Delete the given state from the fsk.                              |
|                                                                              |
| Returns:   INT                                                               |
|            On failure, abort is called.                                      |
|                                                                              |
\*----------------------------------------------------------------------------*/
INT
iDeleteState(ClientData    strClientData, 
                                   // IN  
             Tcl_Interp*   pstrInterp,
                                   // OU  pointer to Tcl interp struct
             INT           iArgCount, 
                                   // IN  argument count
             CHAR**        ppchArgVector
                                   // IN  argument vector containing 
                                   //     (1) State number to delete
        )
{

   INT   iResult;
   INT   iEventArrayCount = 0;
   UINT  uiStateToDelete;
   UINT  uiNumStates;
   UINT  uiNumEvents;
   UINT  uiCurrentState;
   UINT  uiCurrentEvent;
   UINT  uiLoop;
   UINT  uiTemp;
   FskState cFskState;
   FskActionBlock cFskActionBlock;


   if (iArgCount != 2)
   {
      LOG_ERROR("wrong number of args");
      abort();
   }

   iResult = iStringToUnsInt(&uiStateToDelete, ppchArgVector[1]);
   if (iResult)
   {
      LOG_ERROR("Bad state number");
      abort();
   }

   vSetUndoCommand(DELETE_STATE, 0, uiStateToDelete, 0);

   iResult = cFsk.iDeleteState(uiStateToDelete);
   if (iResult)
   {
      LOG_ERROR(achAppErrorString);
      vLoadDisplayMessage(achAppErrorString, &iEventArrayCount, pstrInterp);
      return(TCL_OK);
   }


   uiNumStates = cFsk.uiGetNumStates();
   uiNumEvents = cFsk.uiGetNumEvents();

                        // any action-blocks in the state copy buffer
                        // that the nextState is greater than or equal
                        // to the location we deleted, should be
                        // decremented by one to preserve the nextState 
                        // location
   if ((strCopyBuffer.iBufferType == STATE_COPY_BUFFER) &&
       (strCopyBuffer.pcFsk->uiGetNumEvents() == uiNumEvents))
   {
      for (uiLoop=0; uiLoop<uiNumEvents; uiLoop++)
      {
         if (strCopyBuffer.pcFsk->iGetActionBlockData(uiLoop, 0, 
                                                      &cFskActionBlock))
         {
            LOG_ERROR("Internal error");
            abort();
         }

         uiTemp = *(cFskActionBlock.puiNextState);
         if ((uiTemp > uiStateToDelete) && (uiTemp < MAX_FSK_STATES))
         {
            uiTemp--;
            cFskActionBlock.vSetNextState(uiTemp);
            if (strCopyBuffer.pcFsk->iSetActionBlockData(uiLoop, 
                                                         0, &cFskActionBlock))
            {
               LOG_ERROR("Internal error");
               abort();
            }
         }
         else if (uiTemp == uiStateToDelete)
         {
            cFskActionBlock.vSetNextState(LAST_SET_TO_UNASSIGNED_STATE);
            if (strCopyBuffer.pcFsk->iSetActionBlockData(uiLoop, 
                                                         0, &cFskActionBlock))
            {
               LOG_ERROR("Internal error");
               abort();
            }
         }
         else if (uiTemp == LAST_SET_TO_UNASSIGNED_STATE)
         {
            cFskActionBlock.vSetNextState(UNASSIGNED_STATE);
            if (strCopyBuffer.pcFsk->iSetActionBlockData(uiLoop, 
                                                         0, &cFskActionBlock))
            {
               LOG_ERROR("Internal error");
               abort();
            }
         }
      }
   }
   else if (strCopyBuffer.iBufferType == ACTION_BLOCK_COPY_BUFFER)
   {
                        // update the action-block copy buffer next states
      uiTemp = *(strCopyBuffer.cFskActionBlock.puiNextState);
      if ((uiTemp > uiStateToDelete) && (uiTemp < MAX_FSK_STATES))
      {
         uiTemp--;
         strCopyBuffer.cFskActionBlock.vSetNextState(uiTemp);
      }
      else if (uiTemp == uiStateToDelete)
      {
         strCopyBuffer.cFskActionBlock.vSetNextState(
                                       LAST_SET_TO_UNASSIGNED_STATE);
      }
      else if (uiTemp == LAST_SET_TO_UNASSIGNED_STATE)
      {
         strCopyBuffer.cFskActionBlock.vSetNextState(UNASSIGNED_STATE);
      }
   }

       
                          // update modified state blocks
   for (uiLoop=uiStateToDelete; uiLoop<uiNumStates; uiLoop++)
   {
      if (cFsk.iGetStateData(uiLoop, &cFskState))
      {
                          // internal error...
         LOG_ERROR("Internal error");
         abort();
      }
      vSetGuiStateBlockData(uiLoop, &cFskState, pstrInterp);
   }

                          // TODO, we could probably speed things up a bit
                          //       by maintaining a list of modified
                          //       action-blocks, this list would
                          //       need to be built by the dynamic matrix  
                          //       class....should investigate this later  

                          // update all action-blocks
   if (cFsk.iGetFirstActionBlock(&cFskActionBlock,
                                 &uiCurrentEvent, &uiCurrentState))
   {
                                     // this is an internal error...
      sprintf(achErrorString, "Could not get action-block data, %u, %u",
              uiCurrentEvent, uiCurrentState);
      LOG_ERROR(achErrorString);
      abort();
   }

   do
   {

      vSetGuiActionBlockData(uiCurrentEvent, uiCurrentState, &cFskActionBlock,
                             pstrInterp);

   } while (cFsk.iGetNextActionBlock(&cFskActionBlock,
                                     &uiCurrentEvent, &uiCurrentState) == 0);




   sprintf(achTkVariableValue, "%u", uiNumStates);

   pchResult = Tcl_SetVar2(pstrInterp, "fsk", "numStates",
                           achTkVariableValue, TCL_GLOBAL_ONLY);
   CHECK_TCL_SETVAR(pchResult);


   sprintf(achTkVariableValue, "%u", uiNumEvents);

   pchResult = Tcl_SetVar2(pstrInterp, "fsk", "numEvents",
                           achTkVariableValue, TCL_GLOBAL_ONLY);
   CHECK_TCL_SETVAR(pchResult);


   sprintf(achTkArrayIndex, "eventArray,%d", iEventArrayCount);
   iEventArrayCount++;
   pchResult = Tcl_SetVar2(pstrInterp, "fsk", achTkArrayIndex,
                           "DELETE_STATE", TCL_GLOBAL_ONLY);
   CHECK_TCL_SETVAR(pchResult);

   vCheckModifiedFlag(&iEventArrayCount, pstrInterp);

   sprintf(achTkVariableValue, "%d", iEventArrayCount);
   pchResult = Tcl_SetVar2(pstrInterp, "fsk", "eventArrayLength",
                           achTkVariableValue, TCL_GLOBAL_ONLY);
   CHECK_TCL_SETVAR(pchResult);


   return(TCL_OK);
}



/*----------------------------------------------------------------------------*\
|                                                                              |
| Function:  Insert a state before or after the provided state number.         |
|                                                                              |
| Returns:   INT                                                               |
|            On failure, abort is called.                                      |
|                                                                              |
\*----------------------------------------------------------------------------*/
INT
iInsertState(ClientData    strClientData, 
                                   // IN  
             Tcl_Interp*   pstrInterp,
                                   // OU  pointer to Tcl interp struct
             INT           iArgCount, 
                                   // IN  argument count
             CHAR**        ppchArgVector
                                   // IN  argument vector containing 
                                   //     (1) Indicates to insert BEFORE or
                                   //         AFTER the given state number  
                                   //     (2) State number to insert before
                                   //         or after
        )
{

   INT   iResult;
   INT   iInsertDirection;
   INT   iEventArrayCount = 0;
   UINT  uiStateToInsert;
   UINT  uiNumStates;
   UINT  uiNumEvents;
   UINT  uiCurrentState;
   UINT  uiCurrentEvent;
   UINT  uiLoop;
   UINT  uiTemp;
   FskState       cFskState;
   FskActionBlock cFskActionBlock;


   if (iArgCount != 3)
   {
      LOG_ERROR("wrong number of args");
      abort();
   }

   if (strcmp(ppchArgVector[1], "BEFORE") == 0)
   {
      iInsertDirection = BEFORE;
   }
   else
   {
      iInsertDirection = AFTER;
   }

   iResult = iStringToUnsInt(&uiStateToInsert, ppchArgVector[2]);
   if (iResult)
   {
      LOG_ERROR("Internal error, bad state number");
      abort();
   }

   vSetUndoCommand(INSERT_STATE, 0, uiStateToInsert, iInsertDirection);

                          // update the default next state if needed
   if ((*cFsk.puiDefaultNextState !=  0) &&
       (*cFsk.puiDefaultNextState !=  UNASSIGNED_STATE))
   {
      uiTemp = cFsk.uiGetNumStates();
      if (((uiTemp-1) == uiStateToInsert) && (iInsertDirection == AFTER))
      {
                          // if inserting after the last state, set the 
                          // default next state to the last state+1
                          // no need to add one since it is 1 based
         cFsk.vSetDefaultNextState(uiTemp);
      }
      else
      {
                          // else set it to the last state
         cFsk.vSetDefaultNextState(uiTemp-1);
      }
   }

   iResult = cFsk.iInsertState(uiStateToInsert, iInsertDirection);
   if (iResult)
   {
      LOG_ERROR(achAppErrorString);
      vLoadDisplayMessage(achAppErrorString, &iEventArrayCount, pstrInterp);
      return(TCL_OK);
   }


   uiNumStates = cFsk.uiGetNumStates();
   uiNumEvents = cFsk.uiGetNumEvents();


                                     // we inserted after the state, we 
                                     // must increment the value for it to 
                                     // point to the new state
                                     // if we inserted before the state, we 
                                     // do not decrement it because all the 
                                     // on the current state...
   if (iInsertDirection == AFTER)
   {
      uiStateToInsert++;
   }

                        // any action-blocks in the state copy buffer
                        // that the nextState is greater than or equal
                        // to the location we are inserting into, should be
                        // incremented by one to preserve the nextState 
                        // location
   if ((strCopyBuffer.iBufferType == STATE_COPY_BUFFER) &&
       (strCopyBuffer.pcFsk->uiGetNumEvents() == uiNumEvents))
   {
      for (uiLoop=0; uiLoop<uiNumEvents; uiLoop++)
      {
         if (strCopyBuffer.pcFsk->iGetActionBlockData(uiLoop, 0, 
                                                      &cFskActionBlock))
         {
            LOG_ERROR("Internal error");
            abort();
         }

         uiTemp = *(cFskActionBlock.puiNextState);
         if ((uiTemp >= uiStateToInsert) && (uiTemp <  (uiNumStates-1)))
         {
            uiTemp++;
            cFskActionBlock.vSetNextState(uiTemp);
            if (strCopyBuffer.pcFsk->iSetActionBlockData(uiLoop, 
                                                      0, &cFskActionBlock))
            {
               LOG_ERROR("Internal error");
               abort();
            }
         }
         else if (uiTemp == LAST_SET_TO_UNASSIGNED_STATE)
         {
            cFskActionBlock.vSetNextState(UNASSIGNED_STATE);
            if (strCopyBuffer.pcFsk->iSetActionBlockData(uiLoop, 
                                                      0, &cFskActionBlock))
            {
               LOG_ERROR("Internal error");
               abort();
            }
         }
      }
   }
   else if (strCopyBuffer.iBufferType == ACTION_BLOCK_COPY_BUFFER)
   {
                        // update action-block copy buffer next states
      uiTemp = *(strCopyBuffer.cFskActionBlock.puiNextState);
      if ((uiTemp >= uiStateToInsert) && (uiTemp <  MAX_FSK_STATES))
      {
         uiTemp++;
         strCopyBuffer.cFskActionBlock.vSetNextState(uiTemp);
      }
      else if (uiTemp == LAST_SET_TO_UNASSIGNED_STATE)
      {
         strCopyBuffer.cFskActionBlock.vSetNextState(UNASSIGNED_STATE);
      }
   }

                          // update modified state blocks
   for (uiLoop=uiStateToInsert; uiLoop<uiNumStates; uiLoop++)
   {
      if (cFsk.iGetStateData(uiLoop, &cFskState))
      {
                          // internal error...
         LOG_ERROR("Internal error");
         abort();
      }
      vSetGuiStateBlockData(uiLoop, &cFskState, pstrInterp);
   }

                          // TODO, we could probably speed things up a bit
                          //       by maintaining a list of modified
                          //       action-blocks, this list would
                          //       need to be built by the dynamic matrix  
                          //       class....should investigate this later  

                          // update all action-blocks
   if (cFsk.iGetFirstActionBlock(&cFskActionBlock,
                                 &uiCurrentEvent, &uiCurrentState))
   {
                                     // this is an internal error...
      sprintf(achErrorString, "Could not get action-block data, %u, %u",
              uiCurrentEvent, uiCurrentState);
      LOG_ERROR(achErrorString);
      abort();
   }

   do
   {

      vSetGuiActionBlockData(uiCurrentEvent, uiCurrentState, &cFskActionBlock,
                             pstrInterp);

   } while (cFsk.iGetNextActionBlock(&cFskActionBlock,
                                     &uiCurrentEvent, &uiCurrentState) == 0);


   sprintf(achTkVariableValue, "%u", uiNumStates);

   pchResult = Tcl_SetVar2(pstrInterp, "fsk", "numStates",
                           achTkVariableValue, TCL_GLOBAL_ONLY);
   CHECK_TCL_SETVAR(pchResult);


   sprintf(achTkVariableValue, "%u", uiNumEvents);

   pchResult = Tcl_SetVar2(pstrInterp, "fsk", "numEvents",
                           achTkVariableValue, TCL_GLOBAL_ONLY);
   CHECK_TCL_SETVAR(pchResult);


   sprintf(achTkArrayIndex, "eventArray,%d", iEventArrayCount);
   iEventArrayCount++;
   if (iInsertDirection == BEFORE)
   {
      pchResult = Tcl_SetVar2(pstrInterp, "fsk", achTkArrayIndex,
                              "INSERT_STATE_BEFORE", TCL_GLOBAL_ONLY);
   }
   else
   {
      pchResult = Tcl_SetVar2(pstrInterp, "fsk", achTkArrayIndex,
                              "INSERT_STATE_AFTER", TCL_GLOBAL_ONLY);
   }
   CHECK_TCL_SETVAR(pchResult);

   vCheckModifiedFlag(&iEventArrayCount, pstrInterp);

   sprintf(achTkVariableValue, "%d", iEventArrayCount);
   pchResult = Tcl_SetVar2(pstrInterp, "fsk", "eventArrayLength",
                           achTkVariableValue, TCL_GLOBAL_ONLY);
   CHECK_TCL_SETVAR(pchResult);


   return(TCL_OK);
}



/*----------------------------------------------------------------------------*\
|                                                                              |
| Function:  Copy a state to a storage buffer for later pasting.               |
|                                                                              |
| Returns:   INT                                                               |
|            On failure, abort is called.                                      |
|                                                                              |
\*----------------------------------------------------------------------------*/
INT
iCopyState(ClientData    strClientData, 
                                   // IN  
           Tcl_Interp*   pstrInterp,
                                   // OU  pointer to Tcl interp struct
           INT           iArgCount, 
                                   // IN  argument count
           CHAR**        ppchArgVector
                                   // IN  argument vector containing 
                                   //     (1) state to copy to storage buffer
        )
{

   INT   iResult;
   INT   iEventArrayCount = 0;
   UINT  uiLoop;
   UINT  uiStateToCopy;
   UINT  uiNumEvents;


   if (iArgCount != 2)
   {
      LOG_ERROR("wrong number of args");
      abort();
   }

   iResult = iStringToUnsInt(&uiStateToCopy, ppchArgVector[1]);
   if (iResult)
   {
      LOG_ERROR("Bad state number");
      abort();
   }
                                     // delete the current copy buffer
   if (strCopyBuffer.pcFsk != NULL)
   {
      delete (strCopyBuffer.pcFsk);
      strCopyBuffer.pcFsk = NULL;
   }

                                     // allocate state copy buffer
   uiNumEvents = cFsk.uiGetNumEvents();

   strCopyBuffer.pcFsk = new Fsk(uiNumEvents, 1);
   CHECK_NEW(strCopyBuffer.pcFsk);
   
   strCopyBuffer.iBufferType = STATE_COPY_BUFFER;

                                     // copy the FSK state to buffer state 0
   iResult = cFsk.iCopyState(uiStateToCopy, strCopyBuffer.pcFsk, 0);
   if (iResult)
   {
      LOG_ERROR(achAppErrorString);
      vLoadDisplayMessage(achAppErrorString, &iEventArrayCount, pstrInterp);
      return(TCL_OK);
   }

   sprintf(achTkVariableValue, "%d", iEventArrayCount);
   pchResult = Tcl_SetVar2(pstrInterp, "fsk", "eventArrayLength",
                           achTkVariableValue, TCL_GLOBAL_ONLY);
   CHECK_TCL_SETVAR(pchResult);

   return(TCL_OK);
}



/*----------------------------------------------------------------------------*\
|                                                                              |
| Function:  Past the state in the storage buffer before or after the          |
|            specified state.                                                  |
|                                                                              |
| Returns:   INT                                                               |
|            On failure, abort is called.                                      |
|                                                                              |
\*----------------------------------------------------------------------------*/
INT
iPasteState(ClientData    strClientData, 
                                   // IN  
            Tcl_Interp*   pstrInterp,
                                   // OU  pointer to Tcl interp struct
            INT           iArgCount, 
                                   // IN  argument count
            CHAR**        ppchArgVector
                                   // IN  argument vector containing 
                                   //     (1) Indicates to insert BEFORE or
                                   //         AFTER the given state number  
                                   //     (2) State number to paste before
                                   //         or after
        )
{

   INT   iResult;
   INT   iEventArrayCount = 0;
   INT   iInsertDirection;
   UINT  uiPasteToState;
   UINT  uiNumStates;
   UINT  uiNumEvents;
   UINT  uiCurrentEvent;
   UINT  uiCurrentState;
   UINT  uiLoop;
   UINT  uiTemp;
   FskState       cFskState;
   FskActionBlock cFskActionBlock;


   if (iArgCount != 3)
   {
      LOG_ERROR("wrong number of args");
      abort();
   }

   if (strCopyBuffer.iBufferType != STATE_COPY_BUFFER)
   {
      vLoadDisplayMessage("Copy buffer does not contain a STATE", 
                           &iEventArrayCount, pstrInterp);
      return(TCL_OK);
   }

                        // make sure copy buffer has same number of events as
                        // the fsk, we can only paste the state in the copy
                        // buffer if it has the same number of states as the
                        // fsk.
   if (cFsk.uiGetNumEvents() != strCopyBuffer.pcFsk->uiGetNumEvents())
   {
      sprintf(achErrorString, 
             "State in copy buffer does not contain same number of events (%u)",
             strCopyBuffer.pcFsk->uiGetNumEvents());
      sprintf(&achErrorString[strlen(achErrorString)], 
              " as current FSK (%u)", cFsk.uiGetNumEvents());
      vLoadDisplayMessage(achErrorString, &iEventArrayCount, pstrInterp);
      return(TCL_OK);
   }


   if (strcmp(ppchArgVector[1], "BEFORE") == 0)
   {
      iInsertDirection = BEFORE;
   }
   else
   {
      iInsertDirection = AFTER;
   }

   iResult = iStringToUnsInt(&uiPasteToState, ppchArgVector[2]);
   if (iResult)
   {
      LOG_ERROR("Internal error, bad state number");
      abort();
   }

   vSetUndoCommand(PASTE_STATE, 0, uiPasteToState, iInsertDirection);

                        // insert the new state to copy the buffer to
   iResult = cFsk.iInsertState(uiPasteToState, iInsertDirection);
   if (iResult)
   {
      LOG_ERROR(achAppErrorString);
      vLoadDisplayMessage(achAppErrorString, &iEventArrayCount, pstrInterp);
      return(TCL_OK);
   }

   uiNumStates = cFsk.uiGetNumStates();
   uiNumEvents = cFsk.uiGetNumEvents();

                        // if we inserted after the current state, 
                        // increment the state so we are at the new
                        // inserted state
   if (iInsertDirection == AFTER)
   {
      uiPasteToState++;
   }

                        // any action-blocks in the state copy buffer
                        // that the nextState is greater than or equal
                        // to the location we are inserting into, should be
                        // incremented by one to preserve the nextState 
                        // location
   for (uiLoop=0; uiLoop<uiNumEvents; uiLoop++)
   {
      if (strCopyBuffer.pcFsk->iGetActionBlockData(uiLoop, 0, &cFskActionBlock))
      {
         LOG_ERROR("Internal error");
         abort();
      }

      uiTemp = *(cFskActionBlock.puiNextState);
      if ((uiTemp >= uiPasteToState) && (uiTemp <  MAX_FSK_STATES))
      {
         uiTemp++;
         cFskActionBlock.vSetNextState(uiTemp);
         if (strCopyBuffer.pcFsk->iSetActionBlockData(uiLoop, 
                                                      0, &cFskActionBlock))
         {
            LOG_ERROR("Internal error");
            abort();
         }
      }
      else if (uiTemp == LAST_SET_TO_UNASSIGNED_STATE)
      {
         cFskActionBlock.vSetNextState(UNASSIGNED_STATE);
         if (strCopyBuffer.pcFsk->iSetActionBlockData(uiLoop, 
                                                      0, &cFskActionBlock))
         {
            LOG_ERROR("Internal error");
            abort();
         }
      }
   }

                        // copy buffer state 0 to the specified FSK state
   iResult = strCopyBuffer.pcFsk->iCopyState(0, &cFsk, uiPasteToState);
   if (iResult)
   {
      LOG_ERROR(achAppErrorString);
      vLoadDisplayMessage(achAppErrorString, &iEventArrayCount, pstrInterp);
      return(TCL_OK);
   }

                          // update gui state blocks
   for (uiLoop=uiPasteToState; uiLoop<uiNumStates; uiLoop++)
   {
      if (cFsk.iGetStateData(uiLoop, &cFskState))
      {
                          // internal error...
         LOG_ERROR("Internal error");
         abort();
      }
      vSetGuiStateBlockData(uiLoop, &cFskState, pstrInterp);
   }

                          // update all action-blocks
   if (cFsk.iGetFirstActionBlock(&cFskActionBlock,
                                 &uiCurrentEvent, &uiCurrentState))
   {
                                     // this is an internal error...
      sprintf(achErrorString, "Could not get action-block data, %u, %u",
              uiCurrentEvent, uiCurrentState);
      LOG_ERROR(achErrorString);
      abort();
   }


   do
   {

      vSetGuiActionBlockData(uiCurrentEvent, uiCurrentState, &cFskActionBlock,
                             pstrInterp);

   } while (cFsk.iGetNextActionBlock(&cFskActionBlock,
                                     &uiCurrentEvent, &uiCurrentState) == 0);


   sprintf(achTkVariableValue, "%u", uiNumStates);

   pchResult = Tcl_SetVar2(pstrInterp, "fsk", "numStates",
                           achTkVariableValue, TCL_GLOBAL_ONLY);
   CHECK_TCL_SETVAR(pchResult);


   sprintf(achTkVariableValue, "%u", uiNumEvents);

   pchResult = Tcl_SetVar2(pstrInterp, "fsk", "numEvents",
                           achTkVariableValue, TCL_GLOBAL_ONLY);
   CHECK_TCL_SETVAR(pchResult);


   sprintf(achTkArrayIndex, "eventArray,%d", iEventArrayCount);
   iEventArrayCount++;
   if (iInsertDirection == BEFORE)
   {
      pchResult = Tcl_SetVar2(pstrInterp, "fsk", achTkArrayIndex,
                              "PASTE_STATE_BEFORE", TCL_GLOBAL_ONLY);
   }
   else
   {
      pchResult = Tcl_SetVar2(pstrInterp, "fsk", achTkArrayIndex,
                              "PASTE_STATE_AFTER", TCL_GLOBAL_ONLY);
   }
   CHECK_TCL_SETVAR(pchResult);

   vCheckModifiedFlag(&iEventArrayCount, pstrInterp);

   sprintf(achTkVariableValue, "%d", iEventArrayCount);
   pchResult = Tcl_SetVar2(pstrInterp, "fsk", "eventArrayLength",
                           achTkVariableValue, TCL_GLOBAL_ONLY);
   CHECK_TCL_SETVAR(pchResult);

   return(TCL_OK);
}



/*----------------------------------------------------------------------------*\
|                                                                              |
| Function:  Delete the given event from the fsk.                              |
|                                                                              |
| Returns:   INT                                                               |
|            On failure, abort is called.                                      |
|                                                                              |
\*----------------------------------------------------------------------------*/
INT
iDeleteEvent(ClientData    strClientData, 
                                   // IN  
             Tcl_Interp*   pstrInterp,
                                   // OU  pointer to Tcl interp struct
             INT           iArgCount, 
                                   // IN  argument count
             CHAR**        ppchArgVector
                                   // IN  argument vector containing 
                                   //     (1) Event number to delete
        )
{

   INT   iResult;
   INT   iEventArrayCount = 0;
   UINT  uiEventToDelete;
   UINT  uiNumStates;
   UINT  uiNumEvents;
   UINT  uiCurrentState;
   UINT  uiCurrentEvent;
   UINT  uiLoop;
   FskEvent cFskEvent;
   FskActionBlock cFskActionBlock;


   if (iArgCount != 2)
   {
      LOG_ERROR("wrong number of args");
      abort();
   }

   iResult = iStringToUnsInt(&uiEventToDelete, ppchArgVector[1]);
   if (iResult)
   {
      LOG_ERROR("Bad event number");
      abort();
   }

   vSetUndoCommand(DELETE_EVENT, uiEventToDelete, 0, 0);

   iResult = cFsk.iDeleteEvent(uiEventToDelete);
   if (iResult)
   {
      LOG_ERROR(achAppErrorString);
      vLoadDisplayMessage(achAppErrorString, &iEventArrayCount, pstrInterp);
      return(TCL_OK);
   }


   uiNumStates = cFsk.uiGetNumStates();
   uiNumEvents = cFsk.uiGetNumEvents();

       
                          // update modified event blocks
   for (uiLoop=uiEventToDelete; uiLoop<uiNumEvents; uiLoop++)
   {
      if (cFsk.iGetEventData(uiLoop, &cFskEvent))
      {
                          // internal error...
         LOG_ERROR("Internal error");
         abort();
      }
      vSetGuiEventBlockData(uiLoop, &cFskEvent, pstrInterp);
   }

                          // update all action-blocks
   if (cFsk.iGetFirstActionBlock(&cFskActionBlock,
                                 &uiCurrentEvent, &uiCurrentState))
   {
                                     // this is an internal error...
      sprintf(achErrorString, "Could not get action-block data, %u, %u",
              uiCurrentEvent, uiCurrentState);
      LOG_ERROR(achErrorString);
      abort();
   }

   do
   {

      vSetGuiActionBlockData(uiCurrentEvent, uiCurrentState, &cFskActionBlock,
                             pstrInterp);

   } while (cFsk.iGetNextActionBlock(&cFskActionBlock,
                                     &uiCurrentEvent, &uiCurrentState) == 0);




   sprintf(achTkVariableValue, "%u", uiNumStates);

   pchResult = Tcl_SetVar2(pstrInterp, "fsk", "numStates",
                           achTkVariableValue, TCL_GLOBAL_ONLY);
   CHECK_TCL_SETVAR(pchResult);


   sprintf(achTkVariableValue, "%u", uiNumEvents);

   pchResult = Tcl_SetVar2(pstrInterp, "fsk", "numEvents",
                           achTkVariableValue, TCL_GLOBAL_ONLY);
   CHECK_TCL_SETVAR(pchResult);


   sprintf(achTkArrayIndex, "eventArray,%d", iEventArrayCount);
   iEventArrayCount++;
   pchResult = Tcl_SetVar2(pstrInterp, "fsk", achTkArrayIndex,
                           "DELETE_EVENT", TCL_GLOBAL_ONLY);
   CHECK_TCL_SETVAR(pchResult);

   vCheckModifiedFlag(&iEventArrayCount, pstrInterp);

   sprintf(achTkVariableValue, "%d", iEventArrayCount);
   pchResult = Tcl_SetVar2(pstrInterp, "fsk", "eventArrayLength",
                           achTkVariableValue, TCL_GLOBAL_ONLY);
   CHECK_TCL_SETVAR(pchResult);


   return(TCL_OK);
}



/*----------------------------------------------------------------------------*\
|                                                                              |
| Function:  Insert a event before or after the provided event number.         |
|                                                                              |
| Returns:   INT                                                               |
|            On failure, abort is called.                                      |
|                                                                              |
\*----------------------------------------------------------------------------*/
INT
iInsertEvent(ClientData    strClientData, 
                                   // IN  
             Tcl_Interp*   pstrInterp,
                                   // OU  pointer to Tcl interp struct
             INT           iArgCount, 
                                   // IN  argument count
             CHAR**        ppchArgVector
                                   // IN  argument vector containing 
                                   //     (1) Indicates to insert BEFORE or
                                   //         AFTER the given event number  
                                   //     (2) Event number to insert before
                                   //         or after
        )
{

   INT   iResult;
   INT   iInsertDirection;
   INT   iEventArrayCount = 0;
   UINT  uiEventToInsert;
   UINT  uiNumStates;
   UINT  uiNumEvents;
   UINT  uiCurrentState;
   UINT  uiCurrentEvent;
   UINT  uiLoop;
   FskEvent       cFskEvent;
   FskActionBlock cFskActionBlock;


   if (iArgCount != 3)
   {
      LOG_ERROR("wrong number of args");
      abort();
   }

   if (strcmp(ppchArgVector[1], "BEFORE") == 0)
   {
      iInsertDirection = BEFORE;
   }
   else
   {
      iInsertDirection = AFTER;
   }

   iResult = iStringToUnsInt(&uiEventToInsert, ppchArgVector[2]);
   if (iResult)
   {
      LOG_ERROR("Internal error, bad event number");
      abort();
   }

   vSetUndoCommand(INSERT_EVENT, uiEventToInsert, 0, iInsertDirection);

   iResult = cFsk.iInsertEvent(uiEventToInsert, iInsertDirection);
   if (iResult)
   {
      LOG_ERROR(achAppErrorString);
      vLoadDisplayMessage(achAppErrorString, &iEventArrayCount, pstrInterp);
      return(TCL_OK);
   }


   uiNumStates = cFsk.uiGetNumStates();
   uiNumEvents = cFsk.uiGetNumEvents();


                                     // we inserted after the event, we 
                                     // must increment the value for it to 
                                     // point to the new event
                                     // if we inserted before the event, we 
                                     // do not decrement it because we are
                                     // on the current event...
   if (iInsertDirection == AFTER)
   {
      uiEventToInsert++;
   }
                          // update modified event blocks
   for (uiLoop=uiEventToInsert; uiLoop<uiNumEvents; uiLoop++)
   {
      if (cFsk.iGetEventData(uiLoop, &cFskEvent))
      {
                          // internal error...
         LOG_ERROR("Internal error");
         abort();
      }
      vSetGuiEventBlockData(uiLoop, &cFskEvent, pstrInterp);
   }

                          // update all action-blocks
   if (cFsk.iGetFirstActionBlock(&cFskActionBlock,
                                 &uiCurrentEvent, &uiCurrentState))
   {
                                     // this is an internal error...
      sprintf(achErrorString, "Could not get action-block data, %u, %u",
              uiCurrentEvent, uiCurrentState);
      LOG_ERROR(achErrorString);
      abort();
   }

   do
   {

      vSetGuiActionBlockData(uiCurrentEvent, uiCurrentState, &cFskActionBlock,
                             pstrInterp);

   } while (cFsk.iGetNextActionBlock(&cFskActionBlock,
                                     &uiCurrentEvent, &uiCurrentState) == 0);


   sprintf(achTkVariableValue, "%u", uiNumStates);

   pchResult = Tcl_SetVar2(pstrInterp, "fsk", "numStates",
                           achTkVariableValue, TCL_GLOBAL_ONLY);
   CHECK_TCL_SETVAR(pchResult);


   sprintf(achTkVariableValue, "%u", uiNumEvents);

   pchResult = Tcl_SetVar2(pstrInterp, "fsk", "numEvents",
                           achTkVariableValue, TCL_GLOBAL_ONLY);
   CHECK_TCL_SETVAR(pchResult);


   sprintf(achTkArrayIndex, "eventArray,%d", iEventArrayCount);
   iEventArrayCount++;
   if (iInsertDirection == BEFORE)
   {
      pchResult = Tcl_SetVar2(pstrInterp, "fsk", achTkArrayIndex,
                              "INSERT_EVENT_BEFORE", TCL_GLOBAL_ONLY);
   }
   else
   {
      pchResult = Tcl_SetVar2(pstrInterp, "fsk", achTkArrayIndex,
                              "INSERT_EVENT_AFTER", TCL_GLOBAL_ONLY);
   }
   CHECK_TCL_SETVAR(pchResult);

   vCheckModifiedFlag(&iEventArrayCount, pstrInterp);

   sprintf(achTkVariableValue, "%d", iEventArrayCount);
   pchResult = Tcl_SetVar2(pstrInterp, "fsk", "eventArrayLength",
                           achTkVariableValue, TCL_GLOBAL_ONLY);
   CHECK_TCL_SETVAR(pchResult);


   return(TCL_OK);
}



/*----------------------------------------------------------------------------*\
|                                                                              |
| Function:  Copy an event to a storage buffer for later pasting.              |
|                                                                              |
| Returns:   INT                                                               |
|            On failure, abort is called.                                      |
|                                                                              |
\*----------------------------------------------------------------------------*/
INT
iCopyEvent(ClientData    strClientData, 
                                   // IN  
           Tcl_Interp*   pstrInterp,
                                   // OU  pointer to Tcl interp struct
           INT           iArgCount, 
                                   // IN  argument count
           CHAR**        ppchArgVector
                                   // IN  argument vector containing 
                                   //     (1) Event to copy to storage buffer
        )
{

   INT   iResult;
   INT   iEventArrayCount = 0;
   UINT  uiLoop;
   UINT  uiEventToCopy;
   UINT  uiNumStates;


   if (iArgCount != 2)
   {
      LOG_ERROR("wrong number of args");
      abort();
   }

   iResult = iStringToUnsInt(&uiEventToCopy, ppchArgVector[1]);
   if (iResult)
   {
      LOG_ERROR("Bad event number");
      abort();
   }
                                     // delete the current copy buffer
   if (strCopyBuffer.pcFsk != NULL)
   {
      delete (strCopyBuffer.pcFsk);
   }

                                     // allocate event copy buffer
   uiNumStates = cFsk.uiGetNumStates();

   strCopyBuffer.pcFsk = new Fsk(1, uiNumStates);
   CHECK_NEW(strCopyBuffer.pcFsk);
   
   strCopyBuffer.iBufferType = EVENT_COPY_BUFFER;

                                     // copy the FSK event to buffer event 0
   iResult = cFsk.iCopyEvent(uiEventToCopy, strCopyBuffer.pcFsk, 0);
   if (iResult)
   {
      LOG_ERROR(achAppErrorString);
      vLoadDisplayMessage(achAppErrorString, &iEventArrayCount, pstrInterp);
      return(TCL_OK);
   }

   sprintf(achTkVariableValue, "%d", iEventArrayCount);
   pchResult = Tcl_SetVar2(pstrInterp, "fsk", "eventArrayLength",
                           achTkVariableValue, TCL_GLOBAL_ONLY);
   CHECK_TCL_SETVAR(pchResult);

   return(TCL_OK);
}



/*----------------------------------------------------------------------------*\
|                                                                              |
| Function:  Past the event in the storage buffer before or after the          |
|            specified event.                                                  |
|                                                                              |
| Returns:   INT                                                               |
|            On failure, abort is called.                                      |
|                                                                              |
\*----------------------------------------------------------------------------*/
INT
iPasteEvent(ClientData    strClientData, 
                                   // IN  
            Tcl_Interp*   pstrInterp,
                                   // OU  pointer to Tcl interp struct
            INT           iArgCount, 
                                   // IN  argument count
            CHAR**        ppchArgVector
                                   // IN  argument vector containing 
                                   //     (1) Indicates to insert BEFORE or
                                   //         AFTER the given event number  
                                   //     (2) Event number to paste before
                                   //         or after
        )
{

   INT   iResult;
   INT   iEventArrayCount = 0;
   INT   iInsertDirection;
   UINT  uiPasteToEvent;
   UINT  uiNumStates;
   UINT  uiNumEvents;
   UINT  uiCurrentEvent;
   UINT  uiCurrentState;
   UINT  uiLoop;
   FskEvent       cFskEvent;
   FskActionBlock cFskActionBlock;


   if (iArgCount != 3)
   {
      LOG_ERROR("wrong number of args");
      abort();
   }

   if (strCopyBuffer.iBufferType != EVENT_COPY_BUFFER)
   {
      vLoadDisplayMessage("Copy buffer does not contain an EVENT", 
                           &iEventArrayCount, pstrInterp);
      return(TCL_OK);
   }

                        // make sure copy buffer has same number of states as
                        // the fsk, we can only paste the event in the copy
                        // buffer if it has the same number of states as the
                        // fsk.
   if (cFsk.uiGetNumStates() != strCopyBuffer.pcFsk->uiGetNumStates())
   {
      sprintf(achErrorString, 
             "Event in copy buffer does not contain same number of states (%u)",
             strCopyBuffer.pcFsk->uiGetNumStates());
      sprintf(&achErrorString[strlen(achErrorString)], 
              " as current FSK (%u)", cFsk.uiGetNumStates());
      vLoadDisplayMessage(achErrorString, &iEventArrayCount, pstrInterp);
      return(TCL_OK);
   }

   if (strcmp(ppchArgVector[1], "BEFORE") == 0)
   {
      iInsertDirection = BEFORE;
   }
   else
   {
      iInsertDirection = AFTER;
   }

   iResult = iStringToUnsInt(&uiPasteToEvent, ppchArgVector[2]);
   if (iResult)
   {
      LOG_ERROR("Internal error, bad event number");
      abort();
   }

   vSetUndoCommand(PASTE_EVENT, uiPasteToEvent, 0, iInsertDirection);

                        // insert the new state to copy the buffer to
   iResult = cFsk.iInsertEvent(uiPasteToEvent, iInsertDirection);
   if (iResult)
   {
      LOG_ERROR(achAppErrorString);
      vLoadDisplayMessage(achAppErrorString, &iEventArrayCount, pstrInterp);
      return(TCL_OK);
   }


                        // if we inserted after the current event, 
                        // increment the event so we are at the new
                        // inserted event
   if (iInsertDirection == AFTER)
   {
      uiPasteToEvent++;
   }

   uiNumStates = cFsk.uiGetNumStates();
   uiNumEvents = cFsk.uiGetNumEvents();

                        // copy buffer event 0 to the specified FSK event
   iResult = strCopyBuffer.pcFsk->iCopyEvent(0, &cFsk, uiPasteToEvent);
   if (iResult)
   {
      LOG_ERROR(achAppErrorString);
      vLoadDisplayMessage(achAppErrorString, &iEventArrayCount, pstrInterp);
      return(TCL_OK);
   }


                          // update gui event blocks
   for (uiLoop=uiPasteToEvent; uiLoop<uiNumEvents; uiLoop++)
   {
      if (cFsk.iGetEventData(uiLoop, &cFskEvent))
      {
                          // internal error...
         LOG_ERROR("Internal error");
         abort();
      }
      vSetGuiEventBlockData(uiLoop, &cFskEvent, pstrInterp);
   }

                          // update all action-blocks
   if (cFsk.iGetFirstActionBlock(&cFskActionBlock,
                                 &uiCurrentEvent, &uiCurrentState))
   {
                                     // this is an internal error...
      sprintf(achErrorString, "Could not get action-block data, %u, %u",
              uiCurrentEvent, uiCurrentState);
      LOG_ERROR(achErrorString);
      abort();
   }


   do
   {

      vSetGuiActionBlockData(uiCurrentEvent, uiCurrentState, &cFskActionBlock,
                             pstrInterp);

   } while (cFsk.iGetNextActionBlock(&cFskActionBlock,
                                     &uiCurrentEvent, &uiCurrentState) == 0);


   sprintf(achTkVariableValue, "%u", uiNumStates);

   pchResult = Tcl_SetVar2(pstrInterp, "fsk", "numStates",
                           achTkVariableValue, TCL_GLOBAL_ONLY);
   CHECK_TCL_SETVAR(pchResult);


   sprintf(achTkVariableValue, "%u", uiNumEvents);

   pchResult = Tcl_SetVar2(pstrInterp, "fsk", "numEvents",
                           achTkVariableValue, TCL_GLOBAL_ONLY);
   CHECK_TCL_SETVAR(pchResult);


   sprintf(achTkArrayIndex, "eventArray,%d", iEventArrayCount);
   iEventArrayCount++;
   if (iInsertDirection == BEFORE)
   {
      pchResult = Tcl_SetVar2(pstrInterp, "fsk", achTkArrayIndex,
                              "PASTE_EVENT_BEFORE", TCL_GLOBAL_ONLY);
   }
   else
   {
      pchResult = Tcl_SetVar2(pstrInterp, "fsk", achTkArrayIndex,
                              "PASTE_EVENT_AFTER", TCL_GLOBAL_ONLY);
   }
   CHECK_TCL_SETVAR(pchResult);

   vCheckModifiedFlag(&iEventArrayCount, pstrInterp);

   sprintf(achTkVariableValue, "%d", iEventArrayCount);
   pchResult = Tcl_SetVar2(pstrInterp, "fsk", "eventArrayLength",
                           achTkVariableValue, TCL_GLOBAL_ONLY);
   CHECK_TCL_SETVAR(pchResult);

   return(TCL_OK);
}



/*----------------------------------------------------------------------------*\
|                                                                              |
| Function:  Set fsk variables that are not update automatically.              |
|                                                                              |
| Returns:   INT                                                               |
|            On failure, abort is called.                                      |
|                                                                              |
\*----------------------------------------------------------------------------*/
INT
iSetFskInfo(ClientData    strClientData, 
                                   // IN  
                Tcl_Interp*   pstrInterp,
                                   // OU  pointer to Tcl interp struct
                INT           iArgCount, 
                                   // IN  argument count
                CHAR**        ppchArgVector
                                   // IN  argument vector containing 
                                   //     (1) none
        )
{

   INT   iResult;
   INT   iEventArrayCount = 0;

   

   sprintf(achTkVariableValue, "%s", (CHAR*)cFsk.pchLastModifiedDate);
   pchResult = Tcl_SetVar2(pstrInterp, "fsk", "dateModified",
                           achTkVariableValue, TCL_GLOBAL_ONLY);
   CHECK_TCL_SETVAR(pchResult);


   sprintf(achTkVariableValue, "%d", iEventArrayCount);
   pchResult = Tcl_SetVar2(pstrInterp, "fsk", "eventArrayLength",
                           achTkVariableValue, TCL_GLOBAL_ONLY);
   CHECK_TCL_SETVAR(pchResult);


   return(TCL_OK);
}



/*----------------------------------------------------------------------------*\
|                                                                              |
| Function:  Load the FSK file                                                 |
|                                                                              |
| Returns:   INT                                                               |
|            On failure, abort is called.                                      |
|                                                                              |
\*----------------------------------------------------------------------------*/
INT
iReadFsk(ClientData    strClientData, 
         Tcl_Interp*   pstrInterp,
                                   // OU  pointer to Tcl interp struct
         INT           iArgCount, 
         CHAR**        ppchArgVector
                                   // IN  argument vector containing 
                                   //     (1) File name to read the FSK from
                                   //     (2) Flag, if set to 
                                   //         "IGNORE_MODIFIED_FLAG" ignore the
                                   //         iModified flag 
        )
{

   INT   iResult;
   INT   iIgnoreModifiedFlag;
   INT   iEventArrayCount = 0;

   if (iArgCount != 3)
   {
      LOG_ERROR("wrong number of args");
      abort();
   }
                         // if current fsk modified, don't read in
                         // another fsk unless iIgnoreModifiedFag
                         // is non-zero
   if (strcmp(ppchArgVector[2], "IGNORE_MODIFIED_FLAG") == 0)
   {
      iIgnoreModifiedFlag = 1;
   }
   else
   {
      iIgnoreModifiedFlag = 0;
   }

                              /* if the current FSK has been modified,    */
                              /* not saved, and the ignore modified flag  */
                              /* is non-zero, prompt user...              */
   if ((*cFsk.piModified) && !(iIgnoreModifiedFlag))
   {
      sprintf(achTkArrayIndex, "eventArray,%d", iEventArrayCount);
      iEventArrayCount++;

      pchResult = Tcl_SetVar2(pstrInterp, "fsk", achTkArrayIndex,
                              "READ_WITHOUT_SAVE_POPUP", TCL_GLOBAL_ONLY);
      CHECK_TCL_SETVAR(pchResult);


      sprintf(achTkVariableValue, "%d", iEventArrayCount);

      pchResult = Tcl_SetVar2(pstrInterp, "fsk", "eventArrayLength",
                              achTkVariableValue, TCL_GLOBAL_ONLY);
      CHECK_TCL_SETVAR(pchResult);

      return(TCL_OK);
   }


   vSetUndoCommand(NOTHING_TO_UNDO, 0, 0, 0);

                         // read in the fsk, the current fsk is 
                         // deleted by iReadFile
   iResult = cFsk.iReadFile(ppchArgVector[1]);
   if (iResult)
   {
      vLoadDisplayMessage(achAppErrorString, &iEventArrayCount, pstrInterp);
      return(TCL_OK);
   }


   iResult = iCopyFskToGui(&cFsk, ppchArgVector[1], &iEventArrayCount, 
                           pstrInterp);

 
   if (iResult)
   {
      LOG_ERROR("could not load data to Tcl/Tk");
      abort();
   }


   pchResult = Tcl_SetVar2(pstrInterp, "fsk", "completeFileName",
                           ppchArgVector[1], TCL_GLOBAL_ONLY);
   CHECK_TCL_SETVAR(pchResult);


   sprintf(achTkArrayIndex, "eventArray,%d", iEventArrayCount);
   iEventArrayCount++;

   pchResult = Tcl_SetVar2(pstrInterp, "fsk", achTkArrayIndex,
                           "NEW_FSK", TCL_GLOBAL_ONLY);
   CHECK_TCL_SETVAR(pchResult);

   sprintf(achTkArrayIndex, "eventArray,%d", iEventArrayCount);
   iEventArrayCount++;
   pchResult = Tcl_SetVar2(pstrInterp, "fsk", achTkArrayIndex,
                           "UPDATE_FSK_NAME", TCL_GLOBAL_ONLY);
   CHECK_TCL_SETVAR(pchResult);

   vCheckModifiedFlag(&iEventArrayCount, pstrInterp);

   sprintf(achTkVariableValue, "%d", iEventArrayCount);
   pchResult = Tcl_SetVar2(pstrInterp, "fsk", "eventArrayLength",
                           achTkVariableValue, TCL_GLOBAL_ONLY);
   CHECK_TCL_SETVAR(pchResult);


   return(TCL_OK);

}



/*----------------------------------------------------------------------------*\
|                                                                              |
| Function:  Save the FSK to the specified file                                |
|                                                                              |
| Returns:   INT                                                               |
|            On failure, abort is called.                                      |
|                                                                              |
\*----------------------------------------------------------------------------*/
INT
iSaveFsk(ClientData    strClientData, 
         Tcl_Interp*   pstrInterp,
                                   // OU  pointer to Tcl interp struct
         INT           iArgCount, 
         CHAR**        ppchArgVector
                                   // IN  argument vector containing 
                                   //     (1) File name to write the FSK to
                                   //     (2) Flag, if set to "OVERWRITE_OK",
                                   //         it is OK to overwrite the 
                                   //         existing file if it exists
        )
{

   INT   iResult;
   INT   iOverwriteOk;
   INT   iEventArrayCount = 0;

   if (iArgCount != 3)
   {
      LOG_ERROR("wrong number of args");
      abort();
   }
                         // don't overwrite the FSK if it exists, unless
                         // OVERWRITE_OK is passed in
   if (strcmp(ppchArgVector[2], "OVERWRITE_OK") == 0)
   {
      iOverwriteOk = 1;
   }
   else
   {
      iOverwriteOk = 0;
   }

                         // write the file
   iResult = cFsk.iWriteFile(ppchArgVector[1], iOverwriteOk);

   if (iResult)
   {
      if (iResult == FILE_EXISTS)
      {
         sprintf(achTkArrayIndex, "eventArray,%d", iEventArrayCount);
         iEventArrayCount++;

         pchResult = Tcl_SetVar2(pstrInterp, "fsk", achTkArrayIndex,
                                 "OVERWRITE_FSK_POPUP", TCL_GLOBAL_ONLY);
         CHECK_TCL_SETVAR(pchResult);


         sprintf(achTkVariableValue, "%d", iEventArrayCount);

         pchResult = Tcl_SetVar2(pstrInterp, "fsk", "eventArrayLength",
                                 achTkVariableValue, TCL_GLOBAL_ONLY);
         CHECK_TCL_SETVAR(pchResult);
      }
      else
      {
         vLoadDisplayMessage(achAppErrorString, &iEventArrayCount, pstrInterp);
      }
      return(TCL_OK);
   }


   pchResult = Tcl_SetVar2(pstrInterp, "fsk", "completeFileName",
                           ppchArgVector[1], TCL_GLOBAL_ONLY);
   CHECK_TCL_SETVAR(pchResult);

   iResult = iCopyFskToGui(&cFsk, ppchArgVector[1], &iEventArrayCount, 
                           pstrInterp);
   if (iResult)
   {
      LOG_ERROR("could not load data to Tcl/Tk");
      abort();
   }

   sprintf(achTkArrayIndex, "eventArray,%d", iEventArrayCount);
   iEventArrayCount++;
   pchResult = Tcl_SetVar2(pstrInterp, "fsk", achTkArrayIndex,
                           "UPDATE_FSK_NAME", TCL_GLOBAL_ONLY);
   CHECK_TCL_SETVAR(pchResult);


   vCheckModifiedFlag(&iEventArrayCount, pstrInterp);

   sprintf(achTkVariableValue, "%d", iEventArrayCount);

   pchResult = Tcl_SetVar2(pstrInterp, "fsk", "eventArrayLength",
                           achTkVariableValue, TCL_GLOBAL_ONLY);
   CHECK_TCL_SETVAR(pchResult);

   return(TCL_OK);

}



/*----------------------------------------------------------------------------*\
|                                                                              |
| Function:  Generate the FSK source code.                                     |
|                                                                              |
| Returns:   INT                                                               |
|            On failure, abort is called.                                      |
|                                                                              |
\*----------------------------------------------------------------------------*/
INT
iGenerateSourceCode(ClientData    strClientData, 
                    Tcl_Interp*   pstrInterp,
                                   // OU  pointer to Tcl interp struct
                    INT           iArgCount, 
                    CHAR**        ppchArgVector
                                   // IN  argument vector containing 
                                   //     (1) transition history
                                   //     (2) Flag, if set to "OVERWRITE_OK",
                                   //         it is OK to overwrite the 
                                   //         existing file if it exists
                   )
{
   INT   iResult;
   INT   iOverwriteOk     = 0;
   INT   iEventArrayCount = 0;
   UINT  uiTransitionHistory;
   CHAR  achFileName[MAX_FILE_AND_PATH_LEN];
   CHAR  achFskName[MAX_FILE_AND_PATH_LEN];

   if (iArgCount != 3)
   {
      sprintf(achErrorString, "wrong number of args, %d", iArgCount);
      LOG_ERROR(achErrorString);
      abort();
   }
                         // don't overwrite the source file if it exists,
                         // unless OVERWRITE_OK is passed in
   if (strcmp(ppchArgVector[2], "OVERWRITE_OK") == 0)
   {
      iOverwriteOk = 1;
   }


   if (strchr(ppchArgVector[1], '-') != NULL)
   {
      sprintf(achErrorString, "Transition history must be positive \"%s\"\n", 
                               ppchArgVector[1]);
      vLoadDisplayMessage(achErrorString, &iEventArrayCount, pstrInterp);
      return(TCL_OK);
   }



   iResult = iStringToUnsInt(&uiTransitionHistory, ppchArgVector[1]);
   if (iResult)
   {
      sprintf(achErrorString, "Bad transition history \"%s\"\n", 
                               ppchArgVector[1]);
      vLoadDisplayMessage(achErrorString, &iEventArrayCount, pstrInterp);
      return(TCL_OK);
   }


   pchResult = Tcl_GetVar2(pstrInterp, "fsk", "fileName", 0);
   CHECK_TCL_SETVAR(pchResult);

   strncpy(achFskName, pchResult, MAX_FILE_AND_PATH_LEN);
   achFskName[MAX_FILE_AND_PATH_LEN-1] = '\0';

                         // build complete filename from path and filename
   pchResult = Tcl_GetVar2(pstrInterp, "fsk", "sourceFilePath", 0);
   if (pchResult == NULL)
   {
      LOG_ERROR("Internal Error, fsk(sourceFilePath) not set");
      abort();
   }
   strcpy(achFileName,pchResult);

   pchResult = Tcl_GetVar2(pstrInterp, "fsk", "sourceFileName", 0);
   if (pchResult == NULL)
   {
      LOG_ERROR("Internal Error, fsk(sourceFileName) not set");
      abort();
   }

   if (achFileName[strlen(achFileName)-1] != '/')
   {
      strcat(&achFileName[strlen(achFileName)], "/");
   }

   strcat(&achFileName[strlen(achFileName)], pchResult);
   

   if (cFsk.iGenerateCSource(achFileName, achFskName, uiTransitionHistory, 
                             iOverwriteOk) != 0)
   {
      vLoadDisplayMessage(achAppErrorString, &iEventArrayCount, pstrInterp);
      return(TCL_OK);
   }

   return(0);
}



/*----------------------------------------------------------------------------*\
|                                                                              |
| Function:  Print the FSK state table, event description, and/or state        |
|            description.                                                      |
|                                                                              |
| Returns:   INT                                                 $             |
|            On failure, abort is called.                                      |
|                                                                              |
\*----------------------------------------------------------------------------*/
INT
iPrintFsk(ClientData    strClientData, 
          Tcl_Interp*   pstrInterp,
                                   // OU  pointer to Tcl interp struct
          INT           iArgCount, 
          CHAR**        ppchArgVector
                                   // IN  argument vector containing 
                                   //     (1) Flag, if non-zero, print the
                                   //         FSK state table
                                   //     (2) Flag, if non-zero, print the
                                   //         FSK event descriptions
                                   //     (3) Flag, if non-zero, print the
                                   //         FSK state descriptions
                                   //     (4) Flag, if set to "OVERWRITE_OK",
                                   //         it is OK to overwrite the 
                                   //         existing file if it exists
                   )
{
   INT   iResult;
   INT   iOverwriteOk       = 0;
   INT   iPrintStateTable   = 0;
   INT   iPrintEventDesc    = 0;
   INT   iPrintStateDesc    = 0;
   INT   iEventArrayCount   = 0;
   CHAR  achFileName[MAX_FILE_AND_PATH_LEN];
   CHAR  achFskName[MAX_FILE_AND_PATH_LEN];
   CHAR* pchFskPrintCommand;
   CHAR* pchCommand;

   if (iArgCount != 5)
   {
      LOG_ERROR("wrong number of args");
      abort();
   }

   // get the FSK print command env variable if it is set
   pchFskPrintCommand = getenv("FSKC_PRINT_COMMAND");


   if (strcmp(ppchArgVector[1], "0") != 0)
   {
      iPrintStateTable = 1;
   }

   if (strcmp(ppchArgVector[2], "0") != 0)
   {
      iPrintEventDesc = 1;
   }

   if (strcmp(ppchArgVector[3], "0") != 0)
   {
      iPrintStateDesc = 1;
   }

                         // don't overwrite the source file if it exists,
                         // unless OVERWRITE_OK is passed in
   if (strcmp(ppchArgVector[4], "OVERWRITE_OK") == 0)
   {
      iOverwriteOk = 1;
   }

   pchResult = Tcl_GetVar2(pstrInterp, "fsk", "fileName", 0);
   CHECK_TCL_SETVAR(pchResult);

   strncpy(achFskName, pchResult, MAX_FILE_AND_PATH_LEN);
   achFskName[MAX_FILE_AND_PATH_LEN-1] = '\0';

   if (iPrintStateTable)
   {
                         // print the state table
      pchResult = Tcl_GetVar2(pstrInterp, "fsk", "stateTablePrintPath", 0);
      if (pchResult == NULL)
      {
         LOG_ERROR("Internal Error, fsk(stateTablePrintPath) not set");
         abort();
      }
      strcpy(achFileName,pchResult);

      pchResult = Tcl_GetVar2(pstrInterp, "fsk", "printTableName", 0);
      if (pchResult == NULL)
      {
         LOG_ERROR("Internal Error, fsk(printTableName) not set");
         abort();
      }

      if (achFileName[strlen(achFileName)-1] != '/')
      {
         strcat(achFileName, "/");
      }

      strcat(achFileName, pchResult);

      if (cFsk.iPrintStateTable(achFileName, achFskName, iOverwriteOk) != 0)
      {
         vLoadDisplayMessage(achAppErrorString, &iEventArrayCount, pstrInterp);
         return(TCL_OK);
      }
      
      if (pchFskPrintCommand != NULL)
      {
         // add to to allocated memory for space and null
         pchCommand = (CHAR*) malloc(sizeof(pchFskPrintCommand) +
                                     sizeof(achFileName) + 2);
         CHECK_MALLOC(pchCommand);

         sprintf(pchCommand, "%s %s", pchFskPrintCommand, achFileName);
         system(pchCommand);
         free(pchCommand);
      }
   }

   if (iPrintEventDesc)
   {
                         // print the event descriptions
      pchResult = Tcl_GetVar2(pstrInterp, "fsk", "eventDescriptionPath", 0);
      if (pchResult == NULL)
      {
         LOG_ERROR("Internal Error, fsk(eventDescriptionPath) not set");
         abort();
      }
      strcpy(achFileName,pchResult);

      pchResult = Tcl_GetVar2(pstrInterp, "fsk", "printEventFileName", 0);
      if (pchResult == NULL)
      {
         LOG_ERROR("Internal Error, fsk(printEventFileName) not set");
         abort();
      }

      if (achFileName[strlen(achFileName)-1] != '/')
      {
         strcat(achFileName, "/");
      }

      strcat(achFileName, pchResult);

      if (cFsk.iPrintEventDescriptions(achFileName, achFskName,
                                       iOverwriteOk) != 0)
      {
         vLoadDisplayMessage(achAppErrorString, &iEventArrayCount, pstrInterp);
         return(TCL_OK);
      }

      if (pchFskPrintCommand != NULL)
      {
         // add to to allocated memory for space and null
         pchCommand = (CHAR*) malloc(sizeof(pchFskPrintCommand) +
                                     sizeof(achFileName) + 2);
         CHECK_MALLOC(pchCommand);

         sprintf(pchCommand, "%s %s", pchFskPrintCommand, achFileName);
         system(pchCommand);
         free(pchCommand);
      }
   }


   if (iPrintStateDesc)
   {
                         // print the state descriptions
      pchResult = Tcl_GetVar2(pstrInterp, "fsk", "stateDescriptionPath", 0);
      if (pchResult == NULL)
      {
         LOG_ERROR("Internal Error, fsk(stateDescriptionPath) not set");
         abort();
      }
      strcpy(achFileName,pchResult);

      pchResult = Tcl_GetVar2(pstrInterp, "fsk", "printStateFileName", 0);
      if (pchResult == NULL)
      {
         LOG_ERROR("Internal Error, fsk(printStateFileName) not set");
         abort();
      }

      if (achFileName[strlen(achFileName)-1] != '/')
      {
         strcat(achFileName, "/");
      }

      strcat(achFileName, pchResult);

      if (cFsk.iPrintStateDescriptions(achFileName, achFskName,
                                       iOverwriteOk) != 0)
      {
         vLoadDisplayMessage(achAppErrorString, &iEventArrayCount, pstrInterp);
         return(TCL_OK);
      }

      if (pchFskPrintCommand != NULL)
      {
         // add to to allocated memory for space and null
         pchCommand = (CHAR*) malloc(sizeof(pchFskPrintCommand) +
                                     sizeof(achFileName) + 2);
         CHECK_MALLOC(pchCommand);

         sprintf(pchCommand, "%s %s", pchFskPrintCommand, achFileName);
         system(pchCommand);
         free(pchCommand);
      }
   }


   return(0);
}



/*----------------------------------------------------------------------------*\
|                                                                              |
| Function:  Get the next state to transition to given the current event       |
|            NAME and state NUMBER.                                            |
|            fsk(currentStateNumber) is loaded with the state number           |
|            fsk(currentStateName) is loaded with the state name               |
|            fsk(currentEventNumber) is loaded with the event number           |
|            fsk(currentEventName) is loaded with the event name               |
|                                                                              |
| Returns:   INT                                                               |
|            On failure, abort is called.                                      |
|                                                                              |
\*----------------------------------------------------------------------------*/
INT
iGetNextStateByName(ClientData    strClientData, 
                                   // IN  
                    Tcl_Interp*   pstrInterp,
                                   // OU  pointer to Tcl interp struct
                    INT           iArgCount, 
                                   // IN  argument count
                    CHAR**        ppchArgVector
                                   // IN  argument vector containing 
                                   //     (1) Event name 
                                   //     (2) Current state NUMBER
        )
{

   INT   iResult;
   INT   iEventArrayCount = 0;
   INT   iFound = 0;
   UINT  uiNumEvents;
   UINT  uiStateNumber;
   UINT  uiEventNumber;
   UINT  uiNextState;
   UINT  uiLoop;
   UINT  uiTemp;
   FskState       cFskState;
   FskEvent       cFskEvent;
   FskActionBlock cFskActionBlock;


   if (iArgCount != 3)
   {
      LOG_ERROR("wrong number of args");
      abort();
   }

   iResult = iStringToUnsInt(&uiStateNumber, ppchArgVector[2]);
   if (iResult)
   {
      sprintf(achErrorString, "Bad state \"%s\"\n", ppchArgVector[2]);
      vLoadDisplayMessage(achErrorString, &iEventArrayCount, pstrInterp);
      return(TCL_OK);
   }

   if (uiStateNumber > (cFsk.uiGetNumStates()-1))
   {
      sprintf(achErrorString, "Bad state \"%s\"\n", ppchArgVector[2]);
      vLoadDisplayMessage(achErrorString, &iEventArrayCount, pstrInterp);
      return(TCL_OK);
   }

                          // get event number associated with the given name
   uiNumEvents = cFsk.uiGetNumEvents();
   for (uiLoop=0; uiLoop<uiNumEvents; uiLoop++)
   {
      if (cFsk.iGetEventData(uiLoop, &cFskEvent))
      {
                          // internal error...
         LOG_ERROR("Internal error");
         abort();
      }

      if (strcmp(cFskEvent.pchName, ppchArgVector[1]) == 0)
      {
         iFound = 1;
         uiEventNumber = uiLoop;
         break;
      }
   }

   if (!iFound)
   {
      sprintf(achErrorString, "event %s not found\n", ppchArgVector[1]);
      vLoadDisplayMessage(achErrorString, &iEventArrayCount, pstrInterp);
      return(TCL_OK);
   }

                          // get the next state number to transition to
   iResult = cFsk.iGetNextState(uiEventNumber,uiStateNumber, &uiNextState);
   if (iResult)
   {
      vLoadDisplayMessage(achAppErrorString, &iEventArrayCount, pstrInterp);
      return(TCL_OK);
   }

                          // get name of next state to transition to
   if (cFsk.iGetStateData(uiNextState, &cFskState))
   {
                          // internal error...
      LOG_ERROR("Internal error");
      abort();
   }
   

   sprintf(achTkVariableValue, "%u", uiEventNumber);
   pchResult = Tcl_SetVar2(pstrInterp, "fsk", "currentEventNumber",
                           achTkVariableValue, TCL_GLOBAL_ONLY);
   CHECK_TCL_SETVAR(pchResult);

   pchResult = Tcl_SetVar2(pstrInterp, "fsk", "currentEventName",
                           ppchArgVector[1], TCL_GLOBAL_ONLY);
   CHECK_TCL_SETVAR(pchResult);

   sprintf(achTkVariableValue, "%u", uiNextState);
   pchResult = Tcl_SetVar2(pstrInterp, "fsk", "currentStateNumber",
                           achTkVariableValue, TCL_GLOBAL_ONLY);
   CHECK_TCL_SETVAR(pchResult);

   pchResult = Tcl_SetVar2(pstrInterp, "fsk", "currentStateName",
                           (CHAR*)cFskState.pchName, TCL_GLOBAL_ONLY);
   CHECK_TCL_SETVAR(pchResult);


   sprintf(achTkVariableValue, "%d", iEventArrayCount);
   pchResult = Tcl_SetVar2(pstrInterp, "fsk", "eventArrayLength",
                           achTkVariableValue, TCL_GLOBAL_ONLY);
   CHECK_TCL_SETVAR(pchResult);


   return(TCL_OK);
}



/*----------------------------------------------------------------------------*\
|                                                                              |
| Function:  Get the next state to transition to given the current event       |
|            NUMBEFR and state NUMBER.                                         |
|            fsk(currentStateNumber) is loaded with the state number           |
|            fsk(currentStateName) is loaded with the state name               |
|            fsk(currentEventNumber) is loaded with the event number           |
|            fsk(currentEventName) is loaded with the event name               |
|                                                                              |
| Returns:   INT                                                               |
|            On failure, abort is called.                                      |
|                                                                              |
\*----------------------------------------------------------------------------*/
INT
iGetNextStateByNumber(ClientData    strClientData, 
                                   // IN  
                      Tcl_Interp*   pstrInterp,
                                   // OU  pointer to Tcl interp struct
                      INT           iArgCount, 
                                   // IN  argument count
                      CHAR**        ppchArgVector
                                   // IN  argument vector containing 
                                   //     (1) Event NUMBER 
                                   //     (2) Current state NUMBER
        )
{

   INT   iResult;
   INT   iEventArrayCount = 0;
   INT   iFound = 0;
   UINT  uiNumEvents;
   UINT  uiStateNumber;
   UINT  uiEventNumber;
   UINT  uiNextState;
   UINT  uiLoop;
   UINT  uiTemp;
   FskState       cFskState;
   FskEvent       cFskEvent;
   FskActionBlock cFskActionBlock;


   if (iArgCount != 3)
   {
      LOG_ERROR("wrong number of args");
      abort();
   }

   iResult = iStringToUnsInt(&uiEventNumber, ppchArgVector[1]);
   if (iResult)
   {
      sprintf(achErrorString, "Bad event \"%s\"\n", ppchArgVector[1]);
      vLoadDisplayMessage(achErrorString, &iEventArrayCount, pstrInterp);
      return(TCL_OK);
   }

   iResult = iStringToUnsInt(&uiStateNumber, ppchArgVector[2]);
   if (iResult)
   {
      sprintf(achErrorString, "Bad state \"%s\"\n", ppchArgVector[2]);
      vLoadDisplayMessage(achErrorString, &iEventArrayCount, pstrInterp);
      return(TCL_OK);
   }

   if (uiEventNumber >= cFsk.uiGetNumEvents())
   {
      sprintf(achErrorString, "Bad event \"%s\"\n", ppchArgVector[1]);
      vLoadDisplayMessage(achErrorString, &iEventArrayCount, pstrInterp);
      return(TCL_OK);
   }

   if (uiStateNumber >= cFsk.uiGetNumStates())
   {
      sprintf(achErrorString, "Bad state \"%s\"\n", ppchArgVector[2]);
      vLoadDisplayMessage(achErrorString, &iEventArrayCount, pstrInterp);
      return(TCL_OK);
   }

                          // get the next state number to transition to
   iResult = cFsk.iGetNextState(uiEventNumber,uiStateNumber, &uiNextState);
   if (iResult)
   {
      LOG_ERROR(achAppErrorString);
      vLoadDisplayMessage(achAppErrorString, &iEventArrayCount, pstrInterp);
      return(TCL_OK);
   }

                          // get name of next state to transition to
   if (cFsk.iGetStateData(uiNextState, &cFskState))
   {
                          // internal error...
      LOG_ERROR(achAppErrorString);
      vLoadDisplayMessage(achAppErrorString, &iEventArrayCount, pstrInterp);
      return(TCL_OK);
   }

                          // get event name
   if (cFsk.iGetEventData(uiEventNumber, &cFskEvent))
   {
                          // internal error...
      LOG_ERROR(achAppErrorString);
      vLoadDisplayMessage(achAppErrorString, &iEventArrayCount, pstrInterp);
      return(TCL_OK);
   }
   

   sprintf(achTkVariableValue, "%u", uiEventNumber);
   pchResult = Tcl_SetVar2(pstrInterp, "fsk", "currentEventNumber",
                           achTkVariableValue, TCL_GLOBAL_ONLY);
   CHECK_TCL_SETVAR(pchResult);

   pchResult = Tcl_SetVar2(pstrInterp, "fsk", "currentEventName",
                           (CHAR*)cFskEvent.pchName, TCL_GLOBAL_ONLY);
   CHECK_TCL_SETVAR(pchResult);

   sprintf(achTkVariableValue, "%u", uiNextState);
   pchResult = Tcl_SetVar2(pstrInterp, "fsk", "currentStateNumber",
                           achTkVariableValue, TCL_GLOBAL_ONLY);
   CHECK_TCL_SETVAR(pchResult);

   pchResult = Tcl_SetVar2(pstrInterp, "fsk", "currentStateName",
                           (CHAR*)cFskState.pchName, TCL_GLOBAL_ONLY);
   CHECK_TCL_SETVAR(pchResult);


   sprintf(achTkVariableValue, "%d", iEventArrayCount);
   pchResult = Tcl_SetVar2(pstrInterp, "fsk", "eventArrayLength",
                           achTkVariableValue, TCL_GLOBAL_ONLY);
   CHECK_TCL_SETVAR(pchResult);


   return(TCL_OK);
}



/*----------------------------------------------------------------------------*\
|                                                                              |
| Function:  Validate the FSK                                                  |
|                                                                              |
| Returns:   INT                                                               |
|            On failure, abort is called.                                      |
|                                                                              |
\*----------------------------------------------------------------------------*/
INT
iValidateFsk(ClientData    strClientData, 
             Tcl_Interp*   pstrInterp,
                                   // OU  pointer to Tcl interp struct
             INT           iArgCount, 
             CHAR**        ppchArgVector
                                   // IN  argument vector containing 
                                   //     (1) none
                   )
{
   INT   iResult;
   INT   iEventArrayCount = 0;
   CHAR* pchMessage = NULL;

   iResult = cFsk.iValidate();
   if (iResult)
   {
      pchResult = Tcl_SetVar2(pstrInterp, "fsk", "validateFailed",
                              "1", TCL_GLOBAL_ONLY);
      CHECK_TCL_SETVAR(pchResult);

      vLoadDisplayMessage(achAppErrorString, &iEventArrayCount, pstrInterp);
      return(TCL_OK);
   }

   pchResult = Tcl_SetVar2(pstrInterp, "fsk", "validateFailed",
                           "0", TCL_GLOBAL_ONLY);
   CHECK_TCL_SETVAR(pchResult);

   sprintf(achTkVariableValue, "%d", iEventArrayCount);
   pchResult = Tcl_SetVar2(pstrInterp, "fsk", "eventArrayLength",
                           achTkVariableValue, TCL_GLOBAL_ONLY);
   CHECK_TCL_SETVAR(pchResult);

   return(TCL_OK);
}



/*----------------------------------------------------------------------------*\
|                                                                              |
| Function:  Check the FSK for infinite states.  If infinite states are found, |
|            a warning message is loaded and returned containing the states    |
|            that are infinite.                                                |
|                                                                              |
| Returns:   INT                                                               |
|            On failure, abort is called.                                      |
|                                                                              |
\*----------------------------------------------------------------------------*/
INT
iInfiniteStateCheck(ClientData    strClientData, 
                    Tcl_Interp*   pstrInterp,
                                   // OU  pointer to Tcl interp struct
                    INT           iArgCount, 
                    CHAR**        ppchArgVector
                                   // IN  argument vector containing 
                                   //     (1) none
                   )
{
   INT   iResult;
   INT   iEventArrayCount = 0;
   CHAR* pchMessage = NULL;


   iResult = cFsk.iInfiniteStateCheck(&pchMessage);
   if (iResult == ERROR)
   {
      pchResult = Tcl_SetVar2(pstrInterp, "fsk", "validateFailed",
                              "1", TCL_GLOBAL_ONLY);
      CHECK_TCL_SETVAR(pchResult);

      vLoadDisplayMessage(achAppErrorString, &iEventArrayCount, pstrInterp);

      if (pchMessage != NULL)
      {
         free(pchMessage);
      }

      return(TCL_OK);
   }

   if (iResult == WARNING)
   {
      pchResult = Tcl_SetVar2(pstrInterp, "fsk", "scrollMessage",
                              pchMessage, TCL_GLOBAL_ONLY);
      CHECK_TCL_SETVAR(pchResult);

      sprintf(achTkArrayIndex, "eventArray,%d", iEventArrayCount);
      iEventArrayCount++;
      pchResult = Tcl_SetVar2(pstrInterp, "fsk", achTkArrayIndex,
                        "DISPLAY_SCROLL_MESSAGE", TCL_GLOBAL_ONLY);
      CHECK_TCL_SETVAR(pchResult);
   }

   sprintf(achTkVariableValue, "%d", iEventArrayCount);
   pchResult = Tcl_SetVar2(pstrInterp, "fsk", "eventArrayLength",
                           achTkVariableValue, TCL_GLOBAL_ONLY);
   CHECK_TCL_SETVAR(pchResult);

   return(TCL_OK);
}


int
main(int argc, char **argv)
{
    Tk_Main(argc, argv, Tcl_AppInit);
    return 0;                     /* Needed only to prevent compiler warning. */
}


int
Tcl_AppInit(Tcl_Interp *interp)
{
    Tk_Window main;

    if (Tcl_Init(interp) == TCL_ERROR) {
       return TCL_ERROR;
    }
    if (Tk_Init(interp) == TCL_ERROR) {
       return TCL_ERROR;
    }

     Tcl_CreateCommand(interp, "ReadFsk", iReadFsk,
                       (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);

     Tcl_CreateCommand(interp, "SaveFsk", iSaveFsk,
                       (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);

     Tcl_CreateCommand(interp, "InitFsk", iInitFsk,
                       (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);

     Tcl_CreateCommand(interp, "SetupNewFsk", iSetupNewFsk,
                       (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);

     Tcl_CreateCommand(interp, "UpdateStateData", iUpdateStateData,
                       (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);

     Tcl_CreateCommand(interp, "UpdateEventData", iUpdateEventData,
                       (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);

     Tcl_CreateCommand(interp, "UpdateActionBlockData", iUpdateActionBlockData,
                       (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);

     Tcl_CreateCommand(interp, "ClearActionBlockData", iClearActionBlockData,
                       (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);

     Tcl_CreateCommand(interp, "CopyActionBlockData", iCopyActionBlockData,
                       (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);

     Tcl_CreateCommand(interp, "PasteActionBlockData", iPasteActionBlockData,
                       (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);

     Tcl_CreateCommand(interp, "SetFskDefaults", 
                       iSetFskDefaults,
                       (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);

     Tcl_CreateCommand(interp, "SetFskSourceSettings", 
                       iSetFskSourceSettings,
                       (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);

     Tcl_CreateCommand(interp, "SetFskDescription", 
                       iSetFskDescription,
                       (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);

     Tcl_CreateCommand(interp, "DeleteState", iDeleteState,
                       (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);

     Tcl_CreateCommand(interp, "InsertState", iInsertState,
                       (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);

     Tcl_CreateCommand(interp, "CopyState", iCopyState,
                       (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);

     Tcl_CreateCommand(interp, "PasteState", iPasteState,
                       (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);

     Tcl_CreateCommand(interp, "UndoLastCommand", iUndoLastCommand,
                       (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);

     Tcl_CreateCommand(interp, "DeleteEvent", iDeleteEvent,
                       (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);

     Tcl_CreateCommand(interp, "InsertEvent", iInsertEvent,
                       (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);

     Tcl_CreateCommand(interp, "CopyEvent", iCopyEvent,
                       (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);

     Tcl_CreateCommand(interp, "PasteEvent", iPasteEvent,
                       (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);

     Tcl_CreateCommand(interp, "FindFskItem", iFindFskItem,
                       (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);

     Tcl_CreateCommand(interp, "GenerateSourceCode", iGenerateSourceCode,
                       (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);

     Tcl_CreateCommand(interp, "PrintFsk", iPrintFsk,
                       (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);

     Tcl_CreateCommand(interp, "GetNextStateByName", iGetNextStateByName,
                       (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);

     Tcl_CreateCommand(interp, "GetNextStateByNumber", iGetNextStateByNumber,
                       (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);

     Tcl_CreateCommand(interp, "ValidateFsk", iValidateFsk,
                       (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);

     Tcl_CreateCommand(interp, "InfiniteStateCheck", iInfiniteStateCheck,
                       (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);

     Tcl_CreateCommand(interp, "SetFskInfo", iSetFskInfo,
                       (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);

     Tcl_CreateCommand(interp, "SetFskBinDir", iSetFskBinDir,
                       (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);

    return(TCL_OK);
}
