#
################################################################################
# Copyright (C) 1997, 2000  James A. Cureington
#                           tonyc@acm.org
# 
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or any later version.
# 
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
#
#
# $Id: fskc_help.tcl,v 1.1 2001/03/04 00:08:31 tonycu Exp $
#
# $Log: fskc_help.tcl,v $
# Revision 1.1  2001/03/04 00:08:31  tonycu
# initial check-in at sourceforge...
#
# Revision 1.3  2000/05/24 03:19:08  tony
# Updated all copyright dates.
#
# Revision 1.2  1999/09/13 01:21:05  tony
#
# Updated copyright notice date
#
# Revision 1.1  1997/03/26  07:05:16  tony
#      initial check-in
#
# 
#
################################################################################


################################################################################
# NAME    : createTextWindow
#
# PURPOSE : This procedure creates a scrollable text window an inserts the 
#           contents of the given file; it takes the name passed in and 
#           appends ".text" to it.
################################################################################
proc createTextWindow {textWindow fileName} {

   global $textWindow.text
   catch {destroy $textWindow.text}

   text $textWindow.text -relief sunken -bd 2 \
        -yscrollcommand "$textWindow.scroll set"
   scrollbar $textWindow.scroll -command "$textWindow.text yview"
   pack $textWindow.scroll -side right -fill y
   pack $textWindow.text -side top -expand 1 -fill both

   $textWindow.text configure -bg white -fg black

   $textWindow.text delete 1.0 end
   set inFile [open $fileName] 
   while {![eof $inFile]} {
      $textWindow.text insert end [read $inFile 1024] 
   }
   close $inFile

   $textWindow.text configure -state disabled
}



################################################################################
# NAME    : showNextMatch
#
# PURPOSE : This procedure finds the next match of the string searched for in
#           the searchForText procedure.  This function is coupled with       
#           showPreviousMatch.
################################################################################
proc showNextMatch {textWin} {
   global nextMatchWin
   global stringMatch

   set nextMatchWin $textWin

   if {$stringMatch(currentIndex) == $stringMatch(numMatches)} {
      return;
   }

   $textWin see $stringMatch($stringMatch(currentIndex),begin)

   if {$stringMatch(currentIndex) > 0} {
      set prevIndex [expr $stringMatch(currentIndex)-1]
      $textWin tag configure stringMatch($prevIndex,tag) \
               -foreground blue
   }

   $textWin tag configure stringMatch($stringMatch(currentIndex),tag) \
            -foreground red


   set stringMatch(currentIndex) [expr $stringMatch(currentIndex)+1]
}



################################################################################
# NAME    : showPreviousMatch
#
# PURPOSE : This procedure finds the previous match of the string searched for
#           in the searchForText procedure.  This function is coupled with       
#           showNextMatch.
################################################################################
proc showPreviousMatch {textWin} {
   global nextMatchWin
   global stringMatch

   set nextMatchWin $textWin


   if {$stringMatch(currentIndex) < 2} {
      return;
   }

   set stringMatch(currentIndex) [expr $stringMatch(currentIndex)-1]

   $textWin tag configure stringMatch($stringMatch(currentIndex),tag) \
            -foreground blue

   set currentIndex [expr $stringMatch(currentIndex)-1]

   $textWin see $stringMatch($currentIndex,begin)

   $textWin tag configure stringMatch($currentIndex,tag) \
            -foreground red

}



################################################################################
# NAME    : searchForText
#
# PURPOSE : This procedure searches for text in the given text window.  It
#           creates tags for use in showNextMatch and showPreviousMatch
################################################################################
proc searchForText {searchString textWin} {
   global stringMatch


   if {$searchString == ""} {
      showNextMatch $textWin
      return;
   }

   set count 0

   catch {$textWin tag configure stringMatch(tags) -foreground black}
   catch {$textWin tag delete stringMatch(tags)}
   for {set indx 0} {$indx < $stringMatch(numMatches)} {incr indx} {
         catch {$textWin tag delete stringMatch($indx,tag)}
   }

   scan [$textWin index end] %d numLines
   for {set indx 1} {$indx < $numLines} {incr indx} {
      $textWin mark set last $indx.0
      while {[regexp -indices $searchString \
             [$textWin get last "last lineend"] indices]} {
         $textWin mark set first "last + [lindex $indices 0] chars"
         $textWin mark set last "last + 1 chars + [lindex $indices 1] chars"

         set stringMatch($count,begin) [$textWin index first]
         set stringMatch($count,end) [$textWin index last]

         $textWin tag add stringMatch(tags) first last
         $textWin tag add stringMatch($count,tag) first last

         set count [expr $count+1]
      }
   }

   set stringMatch(numMatches) $count


   #for {set indx 0} {$indx < $stringMatch(numMatches)} {incr indx} {
   #   puts "start = $stringMatch($indx,begin) end = $stringMatch($indx,end)"
   #}

   $textWin tag configure stringMatch(tags) -foreground blue
   set stringMatch(currentIndex) 0
   showNextMatch $textWin

}



################################################################################
# NAME    : createTextSearchCommandBar
#
# PURPOSE : This procedure creates a command bar allowing searches to be done
#           on the given text window.
################################################################################
proc createTextSearchCommandBar {theWindow} {

   global theWin
   global searchString

   set theWin $theWindow

   canvas $theWin.cmdbar_canvas -height 1m -width 2m
   pack $theWin.cmdbar_canvas -side bottom -expand yes \
        -fill x 
   

   entry $theWin.cmdbar_canvas.search_text -width 20 -relief sunken \
         -justify left -textvariable searchString
   $theWin.cmdbar_canvas.search_text delete 0 end
   pack $theWin.cmdbar_canvas.search_text -side left \
        -anchor w -padx 6 -pady 4

   # remove the cr
   bind $theWin.cmdbar_canvas.search_text <Return>                     \
          { set strLen [string length $searchString];                  \
            set searchString [string range $searchString 0             \
                              [expr $strLen-1]];                       \
            searchForText $searchString $theWin.text;                  \
            set searchString "";                                       \
          }

   button $theWin.cmdbar_canvas.search_button -text Search          \
          -command {searchForText $searchString $theWin.text;       \
                    set searchString "";                            \
                   }
   pack $theWin.cmdbar_canvas.search_button -side left \
        -anchor w -padx 6 -pady 4


   button $theWin.cmdbar_canvas.next_button -text Next              \
          -command {showNextMatch $theWin.text}
   pack $theWin.cmdbar_canvas.next_button -side left                \
        -anchor w -padx 6 -pady 4

   button $theWin.cmdbar_canvas.prev_button -text Previous          \
          -command {showPreviousMatch $theWin.text}
   pack $theWin.cmdbar_canvas.prev_button -side left                \
        -anchor w -padx 6 -pady 4

   button $theWin.cmdbar_canvas.cancel_button -text Cancel          \
          -command {destroy $theWin}
   pack $theWin.cmdbar_canvas.cancel_button -side right             \
        -anchor w -padx 6 -pady 4
        
   focus $theWin.cmdbar_canvas.search_text

}



################################################################################
# NAME    : fskcAboutBox     
#
# PURPOSE : This procedure displays the About box information 
################################################################################
proc fskcAboutBox {} {

   global fsk

   set awin .about_window
   catch {destroy $awin}
   toplevel $awin
   wm title $awin "ABOUT FSKC"
   wm iconname $awin "ABOUT"

   createTextWindow $awin $fsk(binDirectory)/fskc_about.txt

   set fileName $fsk(binDirectory)/gpl.txt

   $awin.text configure -state normal
   set inFile [open $fileName] 
   while {![eof $inFile]} {
      $awin.text insert end [read $inFile 1024]
   }
   close $inFile
   $awin.text configure -state disabled -font $fsk(Font2)

   global stringMatch
   set stringMatch(numMatches)   0
   set stringMatch(currentIndex) 0
   createTextSearchCommandBar $awin


   positionWindow $awin 150 50

}



################################################################################
# NAME    : fskcCommandHelp
#
# PURPOSE : This procedure displays the FSKC command help window
################################################################################
proc fskcCommandHelp {} {

   global fsk

   set hwin .help_window
   catch {destroy $hwin}
   toplevel $hwin
   wm title $hwin "FSKC COMMAND HELP"
   wm iconname $hwin "FSKC HELP"

   global stringMatch
   set stringMatch(numMatches)   0
   set stringMatch(currentIndex) 0
   createTextWindow $hwin $fsk(binDirectory)/fskc_help.txt
   createTextSearchCommandBar $hwin

   positionWindow $hwin 150 50

}

