/*----------------------------------------------------------------------------*\
|
| Copyright (C) 1997, 2000  James A. Cureington
|                           tonyc@acm.org
| 
| This program is free software; you can redistribute it and/or
| modify it under the terms of the GNU General Public License
| as published by the Free Software Foundation; either version 2
| of the License, or any later version.
| 
| This program is distributed in the hope that it will be useful,
| but WITHOUT ANY WARRANTY; without even the implied warranty of
| MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
| GNU General Public License for more details.
| 
| You should have received a copy of the GNU General Public License
| along with this program; if not, write to the Free Software
| Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
|
|                                                                              
| Tony Cureington                                                              
| January 26, 1997                                                             
|                                                                              
| Module:  Virtual page class implementations.
|                                                                              
| $Id: virtual_page.C,v 1.1 2001/03/04 00:05:14 tonycu Exp $
|                                                                              
| $Log: virtual_page.C,v $
| Revision 1.1  2001/03/04 00:05:14  tonycu
| initial check-in at sourceforge...
|
| Revision 1.6  2000/05/24 03:19:02  tony
| Updated all copyright dates.
|
| Revision 1.5  1999/09/13 01:21:13  tony
|
| Updated copyright notice date
|
 * Revision 1.4  1997/03/26  07:07:16  tony
 *      Fixed defects uncovered from test plan
 *
 * Revision 1.3  1997/03/01  23:24:16  tony
 *      Made the following changes:
 *           - Add version to FSK file for validation when reading in
 *           - Check for " in parameter list
 *           - Add CVS Id to source and header file
 *           - Add copyright notice to header files
 *           - Add EDIT menu item and associated entries
 *           - Make mouse button 3 cause the commnad pop-up to be displayed
 *             instead of mouse button 2
 *           - Add box to insert header files into
 *           - Changed pchz to pch
 *           - Allow macros to be used in parameter list
 *           - Add function and associated calls to check for infinite state
 *             condition; this is a warning, not an error condition
 *           - Add over-ride abort function capability to GUI
 *           - Changed "modified save pop-up" window to be displayed before
 *             any window is displayed to read and RSK, or create a new one
 *
 * Revision 1.2  1997/02/08  22:43:07  tony
 *      1) made changes to return ERROR instead of 1 when an error is detected
 *      2) changed makefile to make unit tests seperate, make UT
 *      3) fixed bug in virtual_page.C
 *      4) added MAX and MIN macros
 *      5) added set last modified date function
 *
 * Revision 1.1  1997/01/29  04:21:07  tony
 * initial check-in of virtual page class
 *
|                                                                              
\*----------------------------------------------------------------------------*/
#include <stdio.h>
#include <errno.h>
#include <string.h>

#include <common.h>

#include <virtual_page.h>

STATIC CHAR *pchVirtualPageId = "@(#) $Id: virtual_page.C,v 1.1 2001/03/04 00:05:14 tonycu Exp $";


extern CHAR achAppErrorString[ERROR_STRING_LEN];

/*----------------------------------------------------------------------------*\
|                                                                              |
| Function:  Constructor to allocate a VirtualPage object of the specified     |
|            number of lines and chars.                                        |
|                                                                              |
| Returns:   none                                                              |
|            On failure, ASSERT is called.                                     |
|                                                                              |
\*----------------------------------------------------------------------------*/
VirtualPage::
VirtualPage(UINT uiLines,            // IN  number of lines on page
            UINT uiCharsPerLine      // IN  number of characters per line
            ) 
{

   UINT uiLoop;

   uiNumLines         = uiLines;
   uiNumCharsPerLine  = uiCharsPerLine;
   puiNumLines        = &uiNumLines;
   puiNumCharsPerLine = &uiNumCharsPerLine;
   uiMaxLinesPerPage  = 60;

                         // allocate lines
   ppchVirtualPage = (CHAR**) malloc(sizeof(CHAR**) * uiNumLines);
   CHECK_MALLOC(ppchVirtualPage);

   for (uiLoop=0; uiLoop<uiNumLines; uiLoop++)
   {
                         // add an extra char at the end for the null, we
                         // never modify the null at the end...
      ppchVirtualPage[uiLoop] = (CHAR*) 
                                malloc(sizeof(CHAR*) * (uiNumCharsPerLine+1));
      CHECK_MALLOC(ppchVirtualPage[uiLoop]);
      memset(ppchVirtualPage[uiLoop], ' ', uiNumCharsPerLine);
      (ppchVirtualPage[uiLoop])[uiNumCharsPerLine] = '\0';
   }
}



/*----------------------------------------------------------------------------*\
|                                                                              |
| Function:  Destructor to deallocate all memory associated with a previousley |
|            instantiated VirtualPage class object.                            |
|                                                                              |
| Returns:   none                                                              |
|            On failure, ASSERT is called                                      |
|                                                                              |
\*----------------------------------------------------------------------------*/
VirtualPage::
~VirtualPage() 
{
   UINT uiLoop;

                         // free each line of chars
   for (uiLoop=0; uiLoop<uiNumLines; uiLoop++)
   {
      free(ppchVirtualPage[uiLoop]);
   }
                         // free the lines
   free(ppchVirtualPage);

   ppchVirtualPage = NULL;

}



/*----------------------------------------------------------------------------*\
|                                                                              |
| Function:  Write the contents of the given string to the specified virtual   |
|            page line.                                                        |
|                                                                              |
| Returns:   INT                                                               |
|            0 = success                                                       |
|           !0 = failure                                                       |
|                                                                              |
\*----------------------------------------------------------------------------*/
INT
VirtualPage::
iWriteLine(UINT uiLineNumber,        // IN  line number to print string to
           CHAR* pchTextString      // IN  text string to print
           )
{ 

   UINT uiStrLen;
                         // error checks
   if (uiLineNumber >= uiNumLines)
   {
      sprintf(achErrorString,
        "Given line number (%u) is greater than number of lines in page (%u)",
        uiLineNumber, uiNumLines-1);
      LOG_ERROR(achErrorString);
      return(ERROR);
   }

   uiStrLen = strlen(pchTextString);

   if (uiStrLen == 0)
   {
                         // nothing to do
      return(0);
   }

   if (uiStrLen > uiNumCharsPerLine)
   {
      sprintf(achErrorString,
        "Length of line (%u) is greater than number of chars in line (%u)",
        uiStrLen, uiNumCharsPerLine-1);
      LOG_ERROR(achErrorString);
      return(ERROR);
   }

                         // ok, update the line
   memset(ppchVirtualPage[uiLineNumber], ' ', 
          (sizeof(CHAR) * uiNumCharsPerLine));
   memmove(ppchVirtualPage[uiLineNumber], pchTextString, 
           (sizeof(CHAR)  * uiStrLen));

   return(0);
}



/*----------------------------------------------------------------------------*\
|                                                                              |
| Function:  Write the contents of the given string to the specified virtual   |
|            page line at the given offset.                                    |
|                                                                              |
| Returns:   INT                                                               |
|            0 = success                                                       |
|           !0 = failure                                                       |
|                                                                              |
\*----------------------------------------------------------------------------*/
INT
VirtualPage::
iWriteLine(UINT uiLineNumber,        // IN  line number to print string to
           UINT uiOffset,            // IN  offset within line to begin   
                                     //     printing string
           CHAR* pchTextString      // IN  text string to print
           )
{ 

   UINT uiStrLen;
                         // error checks
   if (uiLineNumber >= uiNumLines)
   {
      sprintf(achErrorString,
        "Given line number (%u) is greater than number of lines in page (%u)",
        uiLineNumber, uiNumLines-1);
      LOG_ERROR(achErrorString);
      return(ERROR);
   }

   uiStrLen = strlen(pchTextString);

                         // nothing to do
   if (uiStrLen == 0)
   {
      return(0);
   }

   if ((uiOffset+uiStrLen) > uiNumCharsPerLine)
   {
      sprintf(achErrorString,
             "Length of line plus offset (%u) is greater than ", 
              (uiOffset+uiStrLen));
      sprintf(&achErrorString[strlen(achErrorString)],
             "the number of chars in line (%u)", uiNumCharsPerLine-1);
      LOG_ERROR(achErrorString);
      return(ERROR);
   }

                         // ok, update the line at the given offset
   memmove(&(ppchVirtualPage[uiLineNumber])[uiOffset],
           pchTextString, (sizeof(CHAR) * uiStrLen));

   return(0);
}



/*----------------------------------------------------------------------------*\
|                                                                              |
| Function:  Write the contents of the given string in the center of the       |
|            specified virtual page line.                                      |
|                                                                              |
| Returns:   INT                                                               |
|            0 = success                                                       |
|           !0 = failure                                                       |
|                                                                              |
\*----------------------------------------------------------------------------*/
INT
VirtualPage::
iWriteCenteredLine(UINT uiLineNumber,        
                                     // IN  line number to print string to
                   CHAR* pchTextString      
                                     // IN  text string to print
                  )
{ 

   UINT uiStartLocation;
   UINT uiStrLen;
                         // error checks
   if (uiLineNumber >= uiNumLines)
   {
      sprintf(achErrorString,
        "Given line number (%u) is greater than number of lines in page (%u)",
        uiLineNumber, uiNumLines-1);
      LOG_ERROR(achErrorString);
      return(ERROR);
   }

   uiStrLen = strlen(pchTextString);

                         // nothing to do
   if (uiStrLen == 0)
   {
      return(0);
   }


   if (uiStrLen > uiNumCharsPerLine)
   {
      sprintf(achErrorString,
        "Length of line (%u) is greater than number of chars in line (%u)",
        uiStrLen, uiNumCharsPerLine-1);
      LOG_ERROR(achErrorString);
      return(ERROR);
   }

                         // compute the location to place the line
   if (uiNumCharsPerLine == 1)
   {
      uiStartLocation = 0;
   }
   else
   {
      uiStartLocation = (uiNumCharsPerLine/2) - (uiStrLen/2);
   }


                         // print the centered line
   memmove(&(ppchVirtualPage[uiLineNumber])[uiStartLocation], pchTextString, 
           (sizeof(CHAR) * uiStrLen));

   return(0);
}



/*----------------------------------------------------------------------------*\
|                                                                              |
| Function:  Write the contents of the given string on the specified line      |
|            right justified.                                                  |
|                                                                              |
| Returns:   INT                                                               |
|            0 = success                                                       |
|           !0 = failure                                                       |
|                                                                              |
\*----------------------------------------------------------------------------*/
INT
VirtualPage::
iWriteRightJustifiedLine(UINT uiLineNumber,        
                                     // IN  line number to print string to
                         CHAR* pchTextString      
                                     // IN  text string to print
                        )
{ 

   UINT uiStartLocation;
   UINT uiStrLen;
                         // error checks
   if (uiLineNumber >= uiNumLines)
   {
      sprintf(achErrorString,
        "Given line number (%u) is greater than number of lines in page (%u)",
        uiLineNumber, uiNumLines-1);
      LOG_ERROR(achErrorString);
      return(ERROR);
   }

   uiStrLen = strlen(pchTextString);

                         // nothing to do
   if (uiStrLen == 0)
   {
      return(0);
   }


   if (uiStrLen > uiNumCharsPerLine)
   {
      sprintf(achErrorString,
        "Length of line (%u) is greater than number of chars in line (%u)",
        uiStrLen, uiNumCharsPerLine-1);
      LOG_ERROR(achErrorString);
      return(ERROR);
   }

                         // compute the location to place the line
   if (uiNumCharsPerLine == 1)
   {
      uiStartLocation = 0;
   }
   else
   {
      uiStartLocation = uiNumCharsPerLine - uiStrLen;
   }


                         // print the centered line
   memmove(&(ppchVirtualPage[uiLineNumber])[uiStartLocation], pchTextString, 
           (sizeof(CHAR) * uiStrLen));

   return(0);
}



/*----------------------------------------------------------------------------*\
|                                                                              |
| Function:  Write the contents of the given string on the specified line      |
|            left justified.                                                   |
|                                                                              |
| Returns:   INT                                                               |
|            0 = success                                                       |
|           !0 = failure                                                       |
|                                                                              |
\*----------------------------------------------------------------------------*/
INT
VirtualPage::
iWriteLeftJustifiedLine(UINT uiLineNumber,        
                                     // IN  line number to print string to
                        CHAR* pchTextString      
                                     // IN  text string to print
                       )
{ 
   return(iWriteLine(uiLineNumber, 0, pchTextString));
}



/*----------------------------------------------------------------------------*\
|                                                                              |
| Function:  Clear the contents of the virtual page.                           |
|                                                                              |
| Returns:   none                                                              |
|            On failure, ASSERT is called.                                     |
|                                                                              |
\*----------------------------------------------------------------------------*/
VOID
VirtualPage::
vClearPage() 
{

   UINT uiLoop;

   for (uiLoop=0; uiLoop<uiNumLines; uiLoop++)
   {
      memset(ppchVirtualPage[uiLoop], ' ', (sizeof(CHAR) * uiNumCharsPerLine));
   }
}



/*----------------------------------------------------------------------------*\
|                                                                              |
| Function:  Sets the max lines on any given page, used to determine if a      |
|            page feed is needed.  hence, if the iMaxLinesPerPage line is      |
|            blank, we need a page feed                                        | 
|                                                                              |
| Returns:   none                                                              |
|            On failure, ASSERT is called.                                     |
|                                                                              |
\*----------------------------------------------------------------------------*/
VOID
VirtualPage::
vSetMaxLinesPerPage(UINT uiMaxLines)
{
   uiMaxLinesPerPage = uiMaxLines;
   return;
}



/*----------------------------------------------------------------------------*\
|                                                                              |
| Function:  Write the VirtalPage to the specified file.  If the file exists,  |
|            don't over-wite it unless the over-write flag is non-zero.        |
|                                                                              |
| Returns:   INT                                                               |
|            0             If the virtual page  was successfully written or    |
|                          appeneded to the the specified file                 |
|            FILE_EXISTS   If the file exists and the over-write flag is zero  |
|            ERROR         If any other type of error is encountered           |
|                                                                              |
\*----------------------------------------------------------------------------*/
INT
VirtualPage::
iPrintPage(CHAR* pchOutputFile,     // IN  name of file to write page to  
           INT   iAppendOrReplace,   // IN  flag, set to REPLACE or APPEND
           INT   iOverWrite          // IN  flag set to OVERWRITE or  
                                     //     DONT_OVERWRITE, this flag is 
                                     //     ignored if APPENDing
          )
{

   UINT   uiLoop;
   FILE*  phOutputFile;
   struct stat strStatBuf;

   if (iAppendOrReplace != APPEND)
   {
      if (iOverWrite == DONT_OVERWRITE)
      {
                                     // see if file exists
         phOutputFile = fopen(pchOutputFile, "r");
      
         if (phOutputFile != NULL)
         {
            fclose(phOutputFile);
                                     // determine if it is a regular file
            stat(pchOutputFile, &strStatBuf);

            if (strStatBuf.st_mode & S_IFREG)
            {
               return(FILE_EXISTS);
            }
            else
            {
               sprintf(achAppErrorString, 
                       "Can't write \"%s\" - not a regular file",
                       pchOutputFile);
               return(ERROR);
            }
         }
                         // else, we got an error, let the open to write 
                         // determine the error (we may not have read perms, 
                         // but may have write...)

      }
      else if (iOverWrite != OVERWRITE) 
      {
         sprintf(achErrorString, 
                 "Internal Error: Bad iOverWrite flag %d", 
                 iOverWrite);
         LOG_ERROR(achErrorString);
         abort();
      }
   }


   if (iAppendOrReplace == REPLACE)
   {
      phOutputFile = fopen(pchOutputFile, "w");
   }
   else if (iAppendOrReplace == APPEND)
   {
      phOutputFile = fopen(pchOutputFile, "a");
   }
   else
   {
      sprintf(achErrorString, 
              "Internal Error: Bad iAppendOrReplace flag %d", 
              iAppendOrReplace);
      LOG_ERROR(achErrorString);
      abort();
   }

   if (phOutputFile == NULL)
   {
                                     // we can't write to the file for some
                                     // reason...
      if (errno == EACCES)
      {
         sprintf(achAppErrorString, "Could not write to %s, permission denied",
                 pchOutputFile);
      }
      else
      {
         sprintf(achAppErrorString, "Could not write to %s, errno = %d",
                 pchOutputFile, errno);
      }

      return(errno);
   }


                                     // ok, print the page...
   for (uiLoop=0; uiLoop<uiNumLines; uiLoop++)
   {
      fprintf(phOutputFile, "%s\n", ppchVirtualPage[uiLoop]);
   }

                                     // only print a page feed if the last line
                                     // is blank
   for (uiLoop=0; uiLoop<uiNumCharsPerLine; uiLoop++)
   {
      if (ppchVirtualPage[uiNumLines-1][uiLoop] == '\0')
      {
                                     // no char found on last line
         fprintf(phOutputFile, "\f\n");
         break;
      }

      if (ppchVirtualPage[uiNumLines-1][uiLoop] != ' ')
      {
                                     // non-blank char found, no page feed
                                     // needed
         break;
      }

      if (uiLoop+1 == uiNumCharsPerLine)
      {
                                     // no non-blank char found and we are
                                     // at the end of the line
         fprintf(phOutputFile, "\f\n");
         break;
      }
   }

   fclose(phOutputFile);
   return(0);
}




