/*----------------------------------------------------------------------------*\
|
| Copyright (C) 1997, 2000, 2001  James A. Cureington
|                                 tonyc@acm.org
| 
| This program is free software; you can redistribute it and/or
| modify it under the terms of the GNU General Public License
| as published by the Free Software Foundation; either version 2
| of the License, or any later version.
| 
| This program is distributed in the hope that it will be useful,
| but WITHOUT ANY WARRANTY; without even the implied warranty of
| MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
| GNU General Public License for more details.
| 
| You should have received a copy of the GNU General Public License
| along with this program; if not, write to the Free Software
| Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
|
|                                                                              
| Tony Cureington                                                              
| September 21, 1996                                                           
|                                                                              
| Module:  Fsk class definitions (Finite State Kernel)            
|          This file contains definitions for Fsk class objects.
|                                                                              
| $Id: fsk.h,v 1.1 2001/03/04 00:05:14 tonycu Exp $
|                                                                              
| $Log: fsk.h,v $
| Revision 1.1  2001/03/04 00:05:14  tonycu
| initial check-in at sourceforge...
|
| Revision 1.18  2001/02/18 23:11:07  tony
| updated contact info and version number
|
| Revision 1.17  2000/05/26 02:39:21  tony
| Changed version number to 2.0
|
| Revision 1.16  2000/05/24 03:19:02  tony
| Updated all copyright dates.
|
| Revision 1.15  1999/09/13 01:21:13  tony
|
| Updated copyright notice date
|
| Revision 1.14  1999/09/12 23:16:53  tony
|
| Made changes to allow previous compatible FSKC file versions to be
| read by a newer FSKC software version (i.e. FSKC file version of 1.0 can be
| read by FSKC software version 1.1a).
|
| Revision 1.13  1999/09/12 22:03:56  tony
|
| Changed #defines to const and inline functions...
|
 * Revision 1.12  1997/03/26  07:07:16  tony
 *      Fixed defects uncovered from test plan
 *
 * Revision 1.11  1997/03/08  03:48:04  tony
 *      1) remove new-lines (using trimright) in event, state, and action-block
 *         edit windows
 *      2) the nextState value was not being incremented in the action-blocks
 *         when a state was inserted before it if the nextState value was 0
 *      3) the copy buffer was not being update when states were deleted
 *      4) if the function called in the action-block is vAbortFsk, don't
 *         enclose it in a return(), i.e. return(vAbortFsk)
 *      5) remove leading and trailing spaces from event and state names
 *      6) change no_op function to take a parameter and return it; also,
 *         rename it uiNoOp
 *      7) change vPrintFskHistory to take a parameter and return it; also,
 *         rename it uiPrintFskHistory
 *      8) add check for duplicate event and state names
 *      9) rename "recursive" variables to "iterative"
 *
 * Revision 1.10  1997/03/01  23:20:52  tony
 *      added copyright notice
 *
 * Revision 1.9  1997/02/10  04:01:09  tony
 *      1) added code to generate C source code
 *      2) added trace history
 *
 * Revision 1.8  1997/02/08  22:43:07  tony
 *      1) made changes to return ERROR instead of 1 when an error is detected
 *      2) changed makefile to make unit tests seperate, make UT
 *      3) fixed bug in virtual_page.C
 *      4) added MAX and MIN macros
 *      5) added set last modified date function
 *
 * Revision 1.7  1997/01/26  04:57:19  tony
 * Made changes to:
 *      - insert user defined default function next state when a state or
 *        event is inserted
 *      - stub out future functions
 *      - change tabs to spaces
 *
 * Revision 1.6  1997/01/14  06:26:25  tony
 * added code to:
 *        - copy state to internal buffer for later pasting
 *        - paste the internal buffer before or after a given state
 *        - add capability to initialize FSK action-blocks to default values
 *        - Moved vSetModifiedFlag member function to private area of FSK class.
 *           The modified flag will be maintained by the FSK class itself.  When
 *          the FSK is modified, it is set to true.  When the FSK is read
 *          or written, it is set to false.
 *
 * Revision 1.5  1997/01/13  08:22:58  tony
 * added code to:
 *        - delete states
 *        - insert states
 *        - search dynamic matrix diagonally
 *        - iGetMatrixCell, to chain through the matrix cells using the links
 *
 * Revision 1.4  1997/01/06  06:36:13  tony
 * Initial check-in of PRODUCTION code.
 *
 * Revision 1.3  1996/12/13  05:54:43  tony
 * Changed iCopyEvent and iCopyState to use the "this" pointer
 * Stubbed out un-implemented member functions
 *
 * Revision 1.2  1996/12/13  04:16:21  tony
 * 1. Moved public arrays in classes to private and added const pointers to
 *    the private members
 *
 * 2. Added assignment operators to classes
 *
 * 3. Changed the dynamic array class - replaced malloc/free with new/delete
 *    so it could work with all objects
 *
 * Revision 1.1  1996/11/27  06:17:58  tony
 * Initial version of FSK class.
 *
|                                                                              
\*----------------------------------------------------------------------------*/
#ifndef D_FSK_H
#define D_FSK_H

#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <assert.h>

#include <common.h>
#include <d_array.h>
#include <d_matrix.h>
#include <virtual_page.h>

#include <d_array.C>
#include <d_matrix.C>

STATIC CHAR *pchFskHId = "@(#) $Id: fsk.h,v 1.1 2001/03/04 00:05:14 tonycu Exp $";

CONST INT MAX_FILE_AND_PATH_LEN     = 401;

CONST INT HEADER_FILES_STRING_LEN   = 401;
CONST INT FSK_VERSION_STRING_LEN    = 201;

                                     // error string to return text messages
                                     // back to the application in
CHAR achAppErrorString[ERROR_STRING_LEN] = "\0";

CONST INT STATE_DESCRIPTIONS        = 1;
CONST INT EVENT_DESCRIPTIONS        = 2;
                                     // max length of state name
CONST INT STATE_NAME_LEN            = 36;
                                     // max length of event name
CONST INT EVENT_NAME_LEN            = 36;
                                     // max length of function name including
                                     // parms i.e. "light_bulb(bulb_number, on)"
CONST INT FUNCTION_NAME_LEN         = 251;
                                     // max length of description for state,
                                     // event, of action block
CONST INT DESCRIPTION_LEN           = 401;

CONST INT MODIFIED_DATE_LEN         = 51;

CONST INT RETURN_TYPE_LEN           = 21;

CONST INT LINES_PER_PAGE            = 60;
                                     // number of lines on a printed page
CONST INT CHARS_PER_LINE            = 80;
                                     // number of characters per line
CONST INT PRINT_BLOCK_WIDTH         = 12;
                                     // number of chars between state table
                                     // bars, |       |
CONST INT PRINT_BLOCK_HEIGHT        = 4; 
                                     // number of lines in a block

                                     // the width below include the left "|"
                                     // and the two spaces on each side of 
                                     // the text
CONST INT NUMBER_BLOCK_WIDTH        = 9;
                                     // state/event number block width
CONST INT NAME_BLOCK_WIDTH          = 23;
                                     // state/event name block width
CONST INT DESCRIPTION_BLOCK_WIDTH   = 47;
                                     // state/event description name block width

                                     // maximumn number of states and events
                                     // NOTE: we must keep this value less than
                                     //       six digits because we only have  
                                     //       six spaces to print the 
                                     //       nextState value in the state table
                                     //       printout.  this is the only reason
                                     //       we don't raise the limit to
                                     //       0x00FFFFFF.
CONST INT MAX_FSK_STATES            = 0x0000FFFF;
CONST INT MAX_FSK_EVENTS            = 0x0000FFFF;

                                     // initial value of state, unassigned  
                                     // this value must be less than the 
                                     // other defines that are used by the
                                     // FSK CASE tool. hence, it should be
                                     // one more than MAX_FSK_STATES

CONST INT UNASSIGNED_STATE          = (MAX_FSK_STATES+1);

                                     // nextState value assigned to action
                                     // blocks nextState when the associated
                                     // state is deleted.  we need this in  
                                     // case the user does an undo, we can  
                                     // change the next state value back    
                                     // to the state that was un-deleted.   

CONST INT LAST_SET_TO_UNASSIGNED_STATE  = 0xFFFFFFFE;


CONST CHAR* CONST pchFskVersion = "2.1";
                                     // version of this FSKC release

CONST CHAR* CONST pachCompatibleFskVersions[] = {"1.0",
                                                 "2.0",
                                                 (CONST CHAR*)0
                                                };
                                     // earlier versions of FSKC that are 
                                     // compatible with this version so we
                                     // can read files from compatible versions

/*----------------------------------------------------------------------------*\
|                                                                              |
| Fsk cell definitions                                                         |
|                                                                              |
\*----------------------------------------------------------------------------*/
class 
FskActionBlock
{
   public:

      FskActionBlock();

      VOID operator=(FskActionBlock &rcFskActionBlock);

      VOID vSetNextState(UINT uiState);

      VOID vSetIterative(INT iIterativeFlag);

      INT  iSetFunction(CHAR* pchNewActionBlockFunction);

      INT  iSetDescription(CHAR* pchNewActionBlockDescription);



      INT  iSetData(UINT  uiState,
                    INT   iIterativeFlag,
                    CHAR* pchNewActionBlockFunction,
                    CHAR* pchNewActionBlockDescription);

      VOID vPrintData();


                                     // next state to enter after function  
                                     // is called
      const UINT*  puiNextState;
                                     // does this function return from the  
                                     // state machine, or re-enter it       
                                     // recursively, non-zero==recursive
      const INT*   piIterative;
      const CHAR*  pchFunction;
      const CHAR*  pchDescription;



   protected:
   private:

      UINT  uiNextState;
      INT   iIterative;
      CHAR  achFunction[FUNCTION_NAME_LEN];
      CHAR  achDescription[DESCRIPTION_LEN];


};



/*----------------------------------------------------------------------------*\
|                                                                              |
| State class definitions                                                      |
|                                                                              |
\*----------------------------------------------------------------------------*/
class 
FskState
{
   public:

      FskState();

      VOID operator=(FskState &rcFskState);

      INT iSetName(CHAR* pchNewStateName);

      INT iSetDescription(CHAR* pchNewStateDescription);

      INT iSetData(CHAR* pchNewStateName,
                   CHAR* pchNewStateDescription);

      VOID vPrintData();

      const CHAR*  pchName;
      const CHAR*  pchDescription;
      MatrixCell<FskActionBlock> *pcActionBlock;

   protected:
   private:

      CHAR  achName[STATE_NAME_LEN];
      CHAR  achDescription[DESCRIPTION_LEN];


};



/*----------------------------------------------------------------------------*\
|                                                                              |
| Event class definitions                                                      |
|                                                                              |
\*----------------------------------------------------------------------------*/
class 
FskEvent
{
   public:

      FskEvent();

      VOID operator=(FskEvent &rcFskEvent);

      INT  iSetName(CHAR* pchNewEventName);

      INT  iSetDescription(CHAR* pchNewEventDescription);

      INT  iSetData(CHAR* pchNewEventName,
                    CHAR* pchNewEventDescription);


      VOID vPrintData();

      const CHAR*  pchName;
      const CHAR*  pchDescription;
      MatrixCell<FskActionBlock> *pcActionBlock;

   protected:
   private:
      CHAR  achName[EVENT_NAME_LEN];
      CHAR  achDescription[DESCRIPTION_LEN];

};



/*----------------------------------------------------------------------------*\
|                                                                              |
| Fsk class definitions                                                        |
|                                                                              |
\*----------------------------------------------------------------------------*/
class 
Fsk : private DynamicArray<FskEvent>, 
      private DynamicArray<FskState>, 
      private DynamicMatrix<FskActionBlock>
{
   public:

      Fsk(UINT uiNumEvents=1, UINT uiNumStates=1);

      virtual ~Fsk();

                                     // get number of events    
      UINT uiGetNumEvents();

                                     // get number of states    
      UINT uiGetNumStates();

                                     // set the FSK description
      INT  iSetDescription(CHAR* pchNewActionBlockDescription);

                                     // set the return type for the FSK
      INT  iSetReturnType(CHAR* pchNewReturnType);

                                     // set the header file string
      INT  iSetHeaderFilesString(CHAR* pchNewHeaderFilesString);

                                     // update the modified date
      VOID vSetModifiedDate();

                                     // insert a new event in the FSK
      INT iInsertEvent(UINT uiRow, INT iInsertDirection);

                                     // insert a new state in the FSK
      INT iInsertState(UINT uiState, INT iInsertDirection);

                                     // delete an FSK event
      INT iDeleteEvent(UINT uiEvent);

                                     // delete a matrix column
      INT iDeleteState(UINT uiState);

                                     // set the contents of an FSK event
      INT iSetEventData(UINT uiEvent, FskEvent* pcEventData);


                                     // set the contents of an FSK state
      INT iSetStateData(UINT uiState, FskState* pcStateData);

                                     // set the contents of an FSK action block
      INT iSetActionBlockData(UINT uiEvent, UINT uiState, 
                              FskActionBlock* pcFskActionBlock);

                                     // get the contents of an FSK event
      INT iGetEventData(UINT uiEvent, FskEvent* pcEventData);

                                     // get the contents of an FSK event
      INT iGetStateData(UINT uiState, FskState* pcStateData);

                                     // get the contents of an FSK action block
      INT iGetActionBlockData(UINT uiEvent, UINT uiState, 
                              FskActionBlock* pcFskActionBlock);


                                     // copy an FSK event to another event
      INT iCopyEvent(UINT uiFromEvent, UINT uiToEvent);

                                     // copy an FSK state to another state
      INT iCopyState(UINT uiFromState, UINT uiToState);


                                     // copy an fsk event from one FSK to
                                     // an event in another FSK
      INT iCopyEvent(UINT uiFromEvent,
                     Fsk* pToFsk,   UINT uiToEvent);

                                     // copy an fsk state from one FSK to
                                     // a state in another FSK
      INT iCopyState(UINT uiFromState,
                     Fsk* pToFsk,   UINT uiToState);

                                     // write the fsk to the specified file
      INT iWriteFile(CHAR* pchOutputFile, INT iOverWrite);

                                     // read an fsk into memory
      INT iReadFile(CHAR* pchInputFile);


                                     // validate the FSK
      INT iValidate();

                                     // validate the FSK
      INT iInfiniteStateCheck(CHAR** ppchMessage);

                                     // generate the FSK source code
      INT iGenerateCSource(CHAR* pchOutputFile, CHAR* pchFskName,
                           UINT uiTransitionHistory, INT iOverWrite);

                                     // generate the state table
      INT iPrintStateTable(CHAR* pchOutputFile, 
                           CHAR* pchFskName,
                           INT   iOverWrite);

                                     // generate the state descriptions
      INT iPrintStateDescriptions(CHAR* pchOutputFile, 
                                  CHAR* pchFskName,
                                  INT iOverWrite);

                                     // generate the envent descriptions
      INT iPrintEventDescriptions(CHAR* pchOutputFile, 
                                  CHAR* pchFskName,
                                  INT iOverWrite);

                                     // get the next state to transition to
      INT iGetNextState(UINT uiCurrentState, UINT uiCurrentEvent, 
                        UINT* uiNextState);

                                     // get the first action-block
      INT iGetFirstActionBlock(FskActionBlock* pcFskActionBlock,
                               UINT* puiEvent, UINT* puiState);

                                     // get the next action-block
      INT iGetNextActionBlock(FskActionBlock* pcFskActionBlock,
                              UINT* puiEvent, UINT* puiState);

                                     // setup and link memory for an FSK
      INT iInitFsk(UINT uiNumEvents=1, UINT uiNumStates=1);

                                     // set the default function for new 
                                     // action-blocks
      INT iSetDefaultFunction(CHAR* pchDefaultFunction);

                                     // set the nextState value for new 
                                     // action-blocks
      VOID vSetDefaultNextState(UINT uiDefaultNextState);

                                     // set the user defined invalid 
                                     // state/event handler flag
      VOID vSetUserDefinedErrorHandlerFlag(INT iFlag);


      const INT*   piModified;
      const INT*   piUserDefinedErrorHandler;
      const CHAR*  pchLastModifiedDate;
      const CHAR*  pchDescription;
      const CHAR*  pchReturnType;
      const CHAR*  pchHeaderFiles;

      const UINT*  puiDefaultNextState;
      const CHAR*  pchDefaultFunctionName;

   protected:

   private:
      INT   iModified;
      INT   iUserDefinedErrorHandler;
      CHAR  achLastModifiedDate[MODIFIED_DATE_LEN];
      CHAR  achDescription[DESCRIPTION_LEN];
      CHAR  achReturnType[RETURN_TYPE_LEN];
      CHAR  achHeaderFiles[HEADER_FILES_STRING_LEN];

      CHAR  achFskFileVersionString[FSK_VERSION_STRING_LEN];
      CHAR  achFskVersionString[FSK_VERSION_STRING_LEN];

                                     // pointer to first fsk state
      DynamicArray<FskState>* pacFskState;
                                     // pointer to first fsk event
      DynamicArray<FskEvent>* pacFskEvent;
                                     // pointer to fsk action-block matrix
      DynamicMatrix<FskActionBlock>* pcFskActionBlocks;
                                     // default fsk action-block, used to
                                     // set default next state and function
      FskActionBlock  cDefaultFskActionBlock;

                                     // clear the current state table
      INT iCreateBlankPrintStateTable(VirtualPage* pcVirtualPage);

                                     // place the contents of the state-block
                                     // at the specified location within the 
                                     // virtual page
      INT iPrintStateBlock(VirtualPage* pcVirtualPage,
                           UINT uiStateBlockLocation,
                           UINT uiStateNumber,
                           FskState* pcFskState);

                                     // place the contents of the event-block
                                     // at the specified location within the 
                                     // virtual page
      INT iPrintEventBlock(VirtualPage* pcVirtualPage,
                           UINT uiEventBlockLocation,
                           UINT uiEventNumber,
                           FskEvent* pcFskEvent);

                                     // place the contents of the action-block
                                     // at the specified location within the 
                                     // virtual page
      INT iPrintActionBlock(VirtualPage* pcVirtualPage,
                            UINT uiRowNumber,
                            UINT uiColumnNumber,
                            FskActionBlock* pcFskActionBlock);

                                     // print the state/event descriptions
      INT iPrintDescriptions(CHAR* pchOutputFile, 
                             CHAR* pchFskName,
                             INT   iDescriptionType,
                             INT   iOverWrite);

                                     // set the modified flag value
      VOID vSetModifiedFlag(INT iFlag);


};

#endif /* D_FSK_H */
