/*----------------------------------------------------------------------------*\
|
| Copyright (C) 1997, 2000  James A. Cureington
|                           tonyc@acm.org
| 
| This program is free software; you can redistribute it and/or
| modify it under the terms of the GNU General Public License
| as published by the Free Software Foundation; either version 2
| of the License, or any later version.
| 
| This program is distributed in the hope that it will be useful,
| but WITHOUT ANY WARRANTY; without even the implied warranty of
| MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
| GNU General Public License for more details.
| 
| You should have received a copy of the GNU General Public License
| along with this program; if not, write to the Free Software
| Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
|
|                                                                              
| Tony Cureington                                                              
| September 28, 1996                                                           
|                                                                              
| Module:  Fsk class implementations (Finite State Kernel)
|          This file contains implementations for the Fsk class.
|          The Fsk class encapsulates the DynamicArray and DynamicMatrix
|          templates.  All actions that can be performed on a Finite State     
|          Kernel are implemented in this module.                              
|                                                                              
| $Id: fsk.C,v 1.1 2001/03/04 00:05:14 tonycu Exp $
|                                                                              
| $Log: fsk.C,v $
| Revision 1.1  2001/03/04 00:05:14  tonycu
| initial check-in at sourceforge...
|
| Revision 1.21  2001/02/18 23:11:07  tony
| updated contact info and version number
|
| Revision 1.20  2001/02/18 20:20:18  tony
| added return type for vSetModifiedFlag
|
| Revision 1.19  2000/07/23 20:47:26  tony
| Updated copyright string placed in generated source code.
|
| Revision 1.18  2000/05/24 03:19:01  tony
| Updated all copyright dates.
|
| Revision 1.17  1999/09/13 01:21:13  tony
|
| Updated copyright notice date
|
| Revision 1.16  1999/09/12 23:16:53  tony
|
| Made changes to allow previous compatible FSKC file versions to be
| read by a newer FSKC software version (i.e. FSKC file version of 1.0 can be
| read by FSKC software version 1.1a).
|
| Revision 1.15  1999/09/12 22:03:56  tony
|
| Changed #defines to const and inline functions...
|
 * Revision 1.14  1997/03/26  07:07:15  tony
 *      Fixed defects uncovered from test plan
 *
 * Revision 1.13  1997/03/22  17:33:40  tony
 *      1) fixed bug in the transition state, the transition state was not
 *         being properly updated when a state was inserted then undo was
 *         selected
 *      2) when reading in an FSK, any transition states set to "LAST_SET_TO_
 *         UNASIGNED" need to be set to unassigned...
 *
 * Revision 1.12  1997/03/08  03:48:04  tony
 *      1) remove new-lines (using trimright) in event, state, and action-block
 *         edit windows
 *      2) the nextState value was not being incremented in the action-blocks
 *         when a state was inserted before it if the nextState value was 0
 *      3) the copy buffer was not being update when states were deleted
 *      4) if the function called in the action-block is vAbortFsk, don't
 *         enclose it in a return(), i.e. return(vAbortFsk)
 *      5) remove leading and trailing spaces from event and state names
 *      6) change no_op function to take a parameter and return it; also,
 *         rename it uiNoOp
 *      7) change vPrintFskHistory to take a parameter and return it; also,
 *         rename it uiPrintFskHistory
 *      8) add check for duplicate event and state names
 *      9) rename "recursive" variables to "iterative"
 *
 * Revision 1.11  1997/03/02  19:46:24  tony
 *      - modified code to allow user defined error handler
 *
 * Revision 1.10  1997/03/01  23:24:15  tony
 *      Made the following changes:
 *           - Add version to FSK file for validation when reading in
 *           - Check for " in parameter list
 *           - Add CVS Id to source and header file
 *           - Add copyright notice to header files
 *           - Add EDIT menu item and associated entries
 *           - Make mouse button 3 cause the commnad pop-up to be displayed
 *             instead of mouse button 2
 *           - Add box to insert header files into
 *           - Changed pchz to pch
 *           - Allow macros to be used in parameter list
 *           - Add function and associated calls to check for infinite state
 *             condition; this is a warning, not an error condition
 *           - Add over-ride abort function capability to GUI
 *           - Changed "modified save pop-up" window to be displayed before
 *             any window is displayed to read and RSK, or create a new one
 *
 * Revision 1.9  1997/02/10  04:01:09  tony
 *      1) added code to generate C source code
 *      2) added trace history
 *
 * Revision 1.8  1997/02/08  22:43:07  tony
 *      1) made changes to return ERROR instead of 1 when an error is detected
 *      2) changed makefile to make unit tests seperate, make UT
 *      3) fixed bug in virtual_page.C
 *      4) added MAX and MIN macros
 *      5) added set last modified date function
 *
 * Revision 1.7  1997/01/26  04:57:19  tony
 * Made changes to:
 *      - insert user defined default function next state when a state or
 *        event is inserted
 *      - stub out future functions
 *      - change tabs to spaces
 *
 * Revision 1.6  1997/01/14  06:26:25  tony
 * added code to:
 *        - copy state to internal buffer for later pasting
 *        - paste the internal buffer before or after a given state
 *        - add capability to initialize FSK action-blocks to default values
 *        - Moved vSetModifiedFlag member function to private area of FSK class.
 *           The modified flag will be maintained by the FSK class itself.  When
 *          the FSK is modified, it is set to true.  When the FSK is read
 *          or written, it is set to false.
 *
 * Revision 1.5  1997/01/13  08:22:58  tony
 * added code to:
 *        - delete states
 *        - insert states
 *        - search dynamic matrix diagonally
 *        - iGetMatrixCell, to chain through the matrix cells using the links
 *
 * Revision 1.4  1997/01/06  06:36:13  tony
 * Initial check-in of PRODUCTION code.
 *
 * Revision 1.3  1996/12/13  05:54:43  tony
 * Changed iCopyEvent and iCopyState to use the "this" pointer
 * Stubbed out un-implemented member functions
 *
 * Revision 1.2  1996/12/13  04:16:21  tony
 * 1. Moved public arrays in classes to private and added const pointers to
 *    the private members
 *
 * 2. Added assignment operators to classes
 *
 * 3. Changed the dynamic array class - replaced malloc/free with new/delete
 *    so it could work with all objects
 *
 * Revision 1.1  1996/11/27  06:17:34  tony
 * Initial version of FSK class.
 *
|                                                                              
\*----------------------------------------------------------------------------*/
#include <stdio.h>
#include <errno.h>
#include <string.h>

#include <common.h>
#include <fsk.h>

STATIC CHAR *pchFskId = "@(#) $Id: fsk.C,v 1.1 2001/03/04 00:05:14 tonycu Exp $";

#define HEADER_OFFSET   3
#define FOOTER_OFFSET   1

VOID vSetModifiedFlag(INT iFlag);

/*----------------------------------------------------------------------------*\
|                                                                              |
| Function:  Constructor to for FskActionBlock class.                          |
|                                                                              |
| Returns:   none                                                              |
|            On failure, ASSERT is called.                                     |
|                                                                              |
\*----------------------------------------------------------------------------*/
FskActionBlock::
FskActionBlock(
              )
{
   uiNextState = UNASSIGNED_STATE;
   iIterative = 0;

   memset(achFunction, '\0', FUNCTION_NAME_LEN);
   memset(achDescription, '\0', DESCRIPTION_LEN);

   // set-up public constant pointers, makeing attributes read only and
   // write-able by member functions
   puiNextState = &uiNextState;
   piIterative = &iIterative;
   pchFunction = achFunction;
   pchDescription = achDescription;
}



/*----------------------------------------------------------------------------*\
|                                                                              |
| Function:  Assignment operator for FskActionBlock object.                    |
|                                                                              |
| Returns:   none                                                              |
|            On failure, ASSERT is called.                                     |
|                                                                              |
\*----------------------------------------------------------------------------*/
VOID
FskActionBlock::
operator=(FskActionBlock &rcFskActionBlock
                                     // IN  reference to source action block 
                                     //     to assign from
         )
{

   INT iReturnValue;

   if (this == &rcFskActionBlock)
   {
      // don't assign to self...
      return;
   }

   iReturnValue = iSetData(rcFskActionBlock.uiNextState,
                           rcFskActionBlock.iIterative,
                           rcFskActionBlock.achFunction,
                           rcFskActionBlock.achDescription);

   if (iReturnValue)
   {
      sprintf(achAppErrorString, "Could not set data, internal error, %d",
              iReturnValue);
      LOG_ERROR(achAppErrorString);
      ASSERT(0);
   }

}



/*----------------------------------------------------------------------------*\
|                                                                              |
| Function:  Set the iterative member of the FskActionBlock to the specified   |
|            value.  If non-zero, indicates the FSK should be re-entered       |
|            after the function is called instead of returning from the FSK.   |
|                                                                              |
| Returns:   none                                                              |
|                                                                              |
|                                                                              |
\*----------------------------------------------------------------------------*/
VOID
FskActionBlock::
vSetIterative(INT iIterativeFlag     // set the iterative value in the    
                                     // ActionBlock to this value         
             )
{ 
   iIterative = iIterativeFlag;  
} 



/*----------------------------------------------------------------------------*\
|                                                                              |
| Function:  Set the next state to be entered after the function for this      |
|            action block is called.  The "validness" of the state is not      |
|            checked, i.e. the number of states could be N and the next state  |
|            could be set to N+3;                                              |
|                                                                              |
| Returns:   none                                                              |
|                                                                              |
\*----------------------------------------------------------------------------*/
VOID
FskActionBlock::
vSetNextState(UINT uiState           // state to enter when we leave this state
             )
{ 
   uiNextState = uiState;   
} 



/*----------------------------------------------------------------------------*\
|                                                                              |
| Function:  Set the function name of the ActionBlock.  This is the function   |
|            name to call with the action block is entered from within the     |
|            FSK.                                                              |
|                                                                              |
| Returns:   INT                                                               |
|            On failure, non-zero value is returned.                           |
|                                                                              |
\*----------------------------------------------------------------------------*/
INT
FskActionBlock::
iSetFunction(CHAR* pchNewActionBlockFunction
                                     // function, with parameters, to call 
                                     // when this ActionBlock is entered   
            )
{ 

   if (strlen(pchNewActionBlockFunction) >= FUNCTION_NAME_LEN)
   {
      sprintf(achAppErrorString,
              "Action block function name to long, max length %d",
              (FUNCTION_NAME_LEN-1));
      strncpy(achFunction, pchNewActionBlockFunction, FUNCTION_NAME_LEN-1);
      achFunction[FUNCTION_NAME_LEN-1] = '\0';
      return(ERROR);
   }

   strcpy(achFunction, pchNewActionBlockFunction);
   return(0);
}



/*----------------------------------------------------------------------------*\
|                                                                              |
| Function:  Set the description of the ActionBlock to the text string         |
|            provided.                                                         |
|                                                                              |
| Returns:   INT                                                               |
|            On failure, non-zero value is returned.                           |
|                                                                              |
\*----------------------------------------------------------------------------*/
INT
FskActionBlock::
iSetDescription(CHAR* pchNewActionBlockDescription
                                     // the description to place in the   
                                     // ActionBlock,  this is just one long
                                     // continuous string of characters.   
                                     // Formating will be done when it is  
                                     // printed.
               )
{ 
   if (strlen(pchNewActionBlockDescription) >= DESCRIPTION_LEN)
   {
      sprintf(achAppErrorString,
              "Action block description to long, max length %d",
              (DESCRIPTION_LEN-1));
      strncpy(achDescription, pchNewActionBlockDescription, DESCRIPTION_LEN-1);
      achDescription[DESCRIPTION_LEN-1] = '\0';
      return(ERROR);
   }

   strcpy(achDescription, pchNewActionBlockDescription);
   return(0);
}



/*----------------------------------------------------------------------------*\
|                                                                              |
| Function:  Set the members of the FskAction Block to the specified values.   |
|                                                                              |
| Returns:   INT                                                               |
|            On failure, non-zero value is returned.                           |
|                                                                              |
\*----------------------------------------------------------------------------*/
INT
FskActionBlock::
iSetData(UINT  uiState,              // state to enter when we leave this state
         INT   iIterativeFlag,       // set the iterative value in the    
                                     // ActionBlock to this value         
         CHAR* pchNewActionBlockFunction,
                                     // function, with parameters, to call 
                                     // when this ActionBlock is entered   
         CHAR* pchNewActionBlockDescription
                                     // the description to place in the   
                                     // ActionBlock,  this is just one long
                                     // continuous string of characters.   
                                     // Formating will be done when it is  
                                     // printed.
        )
{ 
                                     // NOTE: all error messages are set by
                                     //       the called function...
   vSetIterative(iIterativeFlag);

   vSetNextState(uiState);

   if (iSetFunction(pchNewActionBlockFunction))
   {
      return(ERROR);
   }

   if (iSetDescription(pchNewActionBlockDescription))
   {
      return(ERROR);
   }

   return(0);
}



/*----------------------------------------------------------------------------*\
|                                                                              |
| Function:  Print the data members of the ActionBlock.                        |
|                                                                              |
| Returns:   none                                                              |
|            On failure, non-zero value is returned.                           |
|                                                                              |
\*----------------------------------------------------------------------------*/
VOID
FskActionBlock::
vPrintData(
          )
{
   printf("uiNextState     = %u\n", uiNextState);
   printf("iIterative      = %d\n", iIterative);
   printf("achFunction    = %s\n", achFunction);
   printf("achDescription = %s\n", achDescription);

   printf("puiNextState     = %u\n", *puiNextState);
   printf("piIterative      = %d\n", *piIterative);
   printf("pchFunction     = %s\n", pchFunction);
   printf("pchDescription  = %s\n", pchDescription);
}



/*----------------------------------------------------------------------------*\
|                                                                              |
| Function:  Constructor for the FskState class.                               |
|                                                                              |
| Returns:   none                                                              |
|                                                                              |
\*----------------------------------------------------------------------------*/
FskState::
FskState(
        )
{ 

   memset(achName, '\0', STATE_NAME_LEN);
   memset(achDescription, '\0', DESCRIPTION_LEN);

   // set-up public constant pointers, makeing attributes read only and
   // write-able by member functions
   pchName = achName;
   pchDescription = achDescription;
}



/*----------------------------------------------------------------------------*\
|                                                                              |
| Function:  Assignment operator for FskState object.                          |
|                                                                              |
| Returns:   none                                                              |
|            On failure, ASSERT is called.                                     |
|                                                                              |
\*----------------------------------------------------------------------------*/
VOID
FskState::
operator=(FskState &rcFskState
                                     // IN  reference to source state
                                     //     to assign from
         )
{

   INT iReturnValue;

   if (this == &rcFskState)
   {
      // don't assign to self...
      return;
   }

   iReturnValue = iSetData(rcFskState.achName,
                           rcFskState.achDescription);

   pcActionBlock = rcFskState.pcActionBlock;

   if (iReturnValue)
   {
      sprintf(achAppErrorString, "Could not set data, internal error, %d",
              iReturnValue);
      LOG_ERROR(achAppErrorString);
      ASSERT(0);
   }
}



/*----------------------------------------------------------------------------*\
|                                                                              |
| Function:  Set the name of the state block to the one provided.              |
|                                                                              |
| Returns:   INT                                                               |
|            On failure, non-zero value is returned.                           |
|                                                                              |
\*----------------------------------------------------------------------------*/
INT
FskState::
iSetName(CHAR* pchNewStateName      // name to assign to the state block 
        )
{ 
   CHAR *pchTmp;

   if (strlen(pchNewStateName) >= STATE_NAME_LEN)
   {
      sprintf(achAppErrorString,
              "State name to long, max length %d",
              (STATE_NAME_LEN-1));
      strncpy(achName, pchNewStateName, STATE_NAME_LEN-1);
      achName[STATE_NAME_LEN-1] = '\0';
      return(ERROR);
   }

   strcpy(achName, pchNewStateName);

   pchTmp = achName;

                             // remove leading spaces
   if (*pchTmp == ' ')
   {
      while (*pchTmp == ' ')
      {
         pchTmp++;
      }
      memmove(achName, pchTmp, strlen(pchTmp)+1);
   }

                             // remove trailing spaces
   pchTmp = &achName[strlen(achName)-1];
   if (*pchTmp == ' ')
   {
      while (*pchTmp == ' ')
      {
         pchTmp--;
      }
      *pchTmp = '\0';
   }

   return(0);
}



/*----------------------------------------------------------------------------*\
|                                                                              |
| Function:  Set the description of the state block to the specified string.   |
|                                                                              |
| Returns:   INT                                                               |
|            On failure, non-zero value is returned.                           |
|                                                                              |
\*----------------------------------------------------------------------------*/
INT
FskState::
iSetDescription(CHAR* pchNewStateDescription
                                     // description to assign to the state
                                     // block
               )
{ 
   if (strlen(pchNewStateDescription) >= DESCRIPTION_LEN)
   {
      sprintf(achAppErrorString,
              "State description to long, max length %d",
              (DESCRIPTION_LEN-1));
      strncpy(achDescription, pchNewStateDescription, DESCRIPTION_LEN-1);
      achDescription[DESCRIPTION_LEN-1] = '\0';
      return(ERROR);
   }

   strcpy(achDescription, pchNewStateDescription);
   return(0);
}



/*----------------------------------------------------------------------------*\
|                                                                              |
| Function:  Set the data members of the FskState to the specified values.     |
|                                                                              |
| Returns:   INT                                                               |
|            On failure, non-zero value is returned.                           |
|                                                                              |
\*----------------------------------------------------------------------------*/
INT
FskState::
iSetData(CHAR* pchNewStateName,     // name to assign to the state block 
         CHAR* pchNewStateDescription
                                     // description to assign to the state
                                     // block
        )
{ 
                                     // NOTE: error message is set in called
                                     //       function
   if (iSetName(pchNewStateName))
   {
      return(ERROR);
   }

   if (iSetDescription(pchNewStateDescription))
   {
      return(ERROR);
   }

   return(0);
}



/*----------------------------------------------------------------------------*\
|                                                                              |
| Function:  Print the data members of the state.                              |
|                                                                              |
| Returns:   none                                                              |
|            On failure, non-zero value is returned.                           |
|                                                                              |
\*----------------------------------------------------------------------------*/
VOID
FskState::
vPrintData(
          )
{
   printf("achName        = %s\n", achName);
   printf("achDescription = %s\n", achDescription);
   printf("pchName        = %s\n", pchName);
   printf("pchDescription = %s\n", pchDescription);
}



/*----------------------------------------------------------------------------*\
|                                                                              |
| Function:  Constructor for the FskEvent class.                               |
|                                                                              |
| Returns:   none                                                              |
|            On failure, ASSERT is called.                                     |
|                                                                              |
\*----------------------------------------------------------------------------*/
FskEvent::
FskEvent(
        ) 
{ 

   memset(achName, '\0', EVENT_NAME_LEN);
   memset(achDescription, '\0', DESCRIPTION_LEN);

   // set-up public constant pointers, makeing attributes read only and
   // write-able by member functions
   pchName = achName;
   pchDescription = achDescription;
}



/*----------------------------------------------------------------------------*\
|                                                                              |
| Function:  Assignment operator for FskEvent object.                          |
|                                                                              |
| Returns:   none                                                              |
|            On failure, ASSERT is called.                                     |
|                                                                              |
\*----------------------------------------------------------------------------*/
VOID
FskEvent::
operator=(FskEvent &rcFskEvent
                                     // IN  reference to source state
                                     //     to assign from
         )
{

   INT iReturnValue;

   if (this == &rcFskEvent)
   {
      // don't assign to self...
      return;
   }

   iReturnValue = iSetData(rcFskEvent.achName,
                           rcFskEvent.achDescription);

   pcActionBlock = rcFskEvent.pcActionBlock;

   if (iReturnValue)
   {
      sprintf(achAppErrorString, "Could not set data, internal error, %d",
              iReturnValue);
      LOG_ERROR(achAppErrorString);
      ASSERT(0);
   }
}



/*----------------------------------------------------------------------------*\
|                                                                              |
| Function:  Set the name of the event block to the provided string.           |
|                                                                              |
| Returns:   INT                                                               |
|            On failure, non-zero value is returned.                           |
|                                                                              |
\*----------------------------------------------------------------------------*/
INT
FskEvent::
iSetName(CHAR* pchNewEventName
                                     // name to assign to the event block 
        )
{ 
   CHAR *pchTmp;

   if (strlen(pchNewEventName) >= EVENT_NAME_LEN)
   {
      sprintf(achAppErrorString,
              "Event name to long, max length %d",
              (EVENT_NAME_LEN-1));
      strncpy(achName, pchNewEventName, EVENT_NAME_LEN-1);
      achName[EVENT_NAME_LEN-1] = '\0';
      return(ERROR);
   }

   strcpy(achName, pchNewEventName);

   pchTmp = achName;

                             // remove leading spaces
   if (*pchTmp == ' ')
   {
      while (*pchTmp == ' ')
      {
         pchTmp++;
      }
      memmove(achName, pchTmp, strlen(pchTmp)+1);
   }

                             // remove trailing spaces
   pchTmp = &achName[strlen(achName)-1];
   if (*pchTmp == ' ')
   {
      while (*pchTmp == ' ')
      {
         pchTmp--;
      }
      *pchTmp = '\0';
   }


   return(0);
}



/*----------------------------------------------------------------------------*\
|                                                                              |
| Function:  Set the event block description to the provided string.           |
|                                                                              |
| Returns:   INT                                                               |
|            On failure, non-zero value is returned.                           |
|                                                                              |
\*----------------------------------------------------------------------------*/
INT
FskEvent::
iSetDescription(CHAR* pchNewEventDescription
                                     // string to assign the the event block
                                     // descriptiong, just one long continuous
                                     // string of chars, formating done later 
               )
{ 
   if (strlen(pchNewEventDescription) >= DESCRIPTION_LEN)
   {
      sprintf(achAppErrorString,
              "Event description to long, max length %d",
              (DESCRIPTION_LEN-1));
      strncpy(achDescription, pchNewEventDescription, DESCRIPTION_LEN-1);
      achDescription[DESCRIPTION_LEN-1] = '\0';
      return(ERROR);
   }

   strcpy(achDescription, pchNewEventDescription);
   return(0);
}



/*----------------------------------------------------------------------------*\
|                                                                              |
| Function:  Set the data members of the FskEvent to the specified values.     |
|                                                                              |
| Returns:   INT                                                               |
|            On failure, non-zero value is returned.                           |
|                                                                              |
\*----------------------------------------------------------------------------*/
INT
FskEvent::
iSetData(CHAR* pchNewEventName,     // name to assign to the event block 
         CHAR* pchNewEventDescription
                                     // description to assign to the event
                                     // block
        )
{ 
                                     // NOTE: error message is set in called
                                     //       function
   if (iSetName(pchNewEventName))
   {
      return(ERROR);
   }

   if (iSetDescription(pchNewEventDescription))
   {
      return(ERROR);
   }

   return(0);
}



/*----------------------------------------------------------------------------*\
|                                                                              |
| Function:  Print the data members of the event.                              |
|                                                                              |
| Returns:   none                                                              |
|            On failure, non-zero value is returned.                           |
|                                                                              |
\*----------------------------------------------------------------------------*/
VOID
FskEvent::
vPrintData(
          )
{
   printf("achName        = %s\n", achName);
   printf("achDescription = %s\n", achDescription);
   printf("pchName        = %s\n", pchName);
   printf("pchDescription = %s\n", pchDescription);
}



/*----------------------------------------------------------------------------*\
|                                                                              |
| Function:  Allocate memory and link structures for an FSK.  This is called   |
|            by the construcutor when a new Fsk object in instantiated or      |
|            an Fsk is read from a file.                                       |
|                                                                              |
| Returns:   INT                                                               |
|            On failure, non-zero value returned.                              |
|                                                                              |
\*----------------------------------------------------------------------------*/
INT
Fsk::
iInitFsk(UINT uiNumEvents,           // IN  number of events/rows in fsk
         UINT uiNumStates            // IN  number of states/columns in fsk
        ) 
{

   UINT uiLoop;
   UINT uiLoop2;

   iModified = FALSE;
   iUserDefinedErrorHandler = FALSE;
   memset(achLastModifiedDate, '\0', MODIFIED_DATE_LEN);
   memset(achDescription, '\0', DESCRIPTION_LEN);
   memset(achReturnType, '\0', RETURN_TYPE_LEN);
   memset(achHeaderFiles, '\0', HEADER_FILES_STRING_LEN);

   piModified = &iModified;
   piUserDefinedErrorHandler = 
                     &iUserDefinedErrorHandler;
   pchLastModifiedDate = achLastModifiedDate;
   pchDescription = achDescription;
   pchReturnType = achReturnType;
   pchHeaderFiles = achHeaderFiles;

   strcpy(achReturnType, "unsigned int");

   strncpy(achFskVersionString, pchFskVersion, FSK_VERSION_STRING_LEN);
   achFskVersionString[FSK_VERSION_STRING_LEN-1] = '\0';
   strcpy(achFskFileVersionString, achFskVersionString);

   if (uiNumEvents > MAX_FSK_EVENTS)
   {
      sprintf(achAppErrorString, "Maximum number of FSK events = %u",
              MAX_FSK_EVENTS);
      return(ERROR);
   }

   if (uiNumStates > MAX_FSK_STATES)
   {
      sprintf(achAppErrorString, "Maximum number of FSK states = %u",
              MAX_FSK_STATES);
      return(ERROR);
   }

   if ((uiNumEvents == 0) || (uiNumStates == 0))
   {
      sprintf(achAppErrorString, "FSK must have at least one state and event");
      return(ERROR);
   }

   if (pcFskActionBlocks != NULL)
   {
      delete(pcFskActionBlocks);
      delete(pacFskEvent);
      delete(pacFskState);
   }


   pcFskActionBlocks = new DynamicMatrix<FskActionBlock> 
                           (uiNumEvents, uiNumStates);
   CHECK_NEW(pcFskActionBlocks);

   pacFskEvent = new DynamicArray<FskEvent> (uiNumEvents);
   CHECK_NEW(pacFskEvent);

   pacFskState = new DynamicArray<FskState> (uiNumStates);
   CHECK_NEW(pacFskState);

                                     // link each event to the associated  
                                     // fsk row of elements
   for (uiLoop=0; uiLoop<uiNumEvents; uiLoop++)
   {
      (*pacFskEvent)[uiLoop].pcActionBlock = 
                   pcFskActionBlocks->pxGetRow((ULONG)uiLoop);
   }

                                     // link each state to the associated  
                                     // fsk column of elements
   for (uiLoop=0; uiLoop<uiNumStates; uiLoop++)
   {
      (*pacFskState)[uiLoop].pcActionBlock = 
                   pcFskActionBlocks->pxGetColumn((ULONG)uiLoop);
   }

                                     // set default action-block ptrs
   pchDefaultFunctionName = cDefaultFskActionBlock.pchFunction;
   puiDefaultNextState = cDefaultFskActionBlock.puiNextState;

                         // TODO, investigate adding parameter to class of 
                         //       dynamic matrix.  if we could initialize the
                         //       action blocks with a parameter to the
                         //       action-block constructor, through the matrix
                         //       template, we would not need to set these to
                         //       the default value below.  the action-block 
                         //       class could have a default parameter set to
                         //       null; in the ctor, if the param is null the
                         //       members are inited to hard-coded default   
                         //       values;  if the param is not null, the    
                         //       new action-block is initialized with the 
                         //       values in the param.  the problem is I have
                         //       not found a way to instantiate a template  
                         //       class while passing a parameter list to the
                         //       objectect that are created inside the 
                         //       template.  the question is, how can we pass
                         //       parameters to the ActionBlock object without
                         //       passing them through the matrix class below:
                         //       pMatrix =new DynamicMatrix<ActionBlock> (5,5);
                         //       also, the number of parameters may vary 
                         //       depening on what type of object matrix is 
                         //       being created...

                         // set action-blocks to default values

   for (uiLoop=0; uiLoop<uiNumEvents; uiLoop++)
   {
      for (uiLoop2=0; uiLoop2<uiNumStates; uiLoop2++)
      {
         if (iSetActionBlockData(uiLoop, uiLoop2, &cDefaultFskActionBlock))
         {
            sprintf(achAppErrorString, 
                    "Could not set action-block data, %u, %u",
                    uiNumEvents, uiNumStates);
            LOG_ERROR(achAppErrorString);
            return(ERROR);
         }
      }
   }

                         // this is a HACK, the gui displays a fsk when it
                         // first starts up.  so, the fsk is realy not 
                         // modified yet...
   { 
      STATIC INT iFirstTimeThrough = 1;

      if (iFirstTimeThrough)
      {
         vSetModifiedFlag(TRUE);
         iFirstTimeThrough = 0;
      }
      else
      {
         vSetModifiedFlag(FALSE);
      }
   } 

   return(0);
}



/*----------------------------------------------------------------------------*\
|                                                                              |
| Function:  Constructor to allocate an Fsk class object of the specified      |
|            size.                                                             |
|                                                                              |
| Returns:   none                                                              |
|            On failure, ASSERT is called.                                     |
|                                                                              |
\*----------------------------------------------------------------------------*/
Fsk::
Fsk(UINT uiNumEvents,                // IN  number of events/rows in fsk
    UINT uiNumStates                 // IN  number of states/columns in fsk
   ) 
{

   pcFskActionBlocks = NULL;
   pacFskEvent       = NULL;
   pacFskState       = NULL;

   if (iInitFsk(uiNumEvents, uiNumStates))
   {
      LOG_ERROR("Internal error");
      ASSERT(0);
   }
}



/*----------------------------------------------------------------------------*\
|                                                                              |
| Function:  Destructor to deallocate all memory associated with a previousley |
|            instantiated Fsk class object.                                    |
|                                                                              |
| Returns:   none                                                              |
|            On failure, ASSERT is called                                      |
|                                                                              |
\*----------------------------------------------------------------------------*/
Fsk::
~Fsk() 
{
   if (pcFskActionBlocks != NULL)
   {
      delete pcFskActionBlocks;
      pcFskActionBlocks = NULL;
   }

   if (pacFskEvent != NULL)
   {
      delete pacFskEvent;
      pacFskEvent = NULL;
   }

   if (pacFskState != NULL)
   {
      delete pacFskState;
      pacFskState = NULL;
   }
}



/*----------------------------------------------------------------------------*\
|                                                                              |
| Function:  Set the FSK modified flag.                                        |
|                                                                              |
| Returns:   none                                                              |
|                                                                              |
\*----------------------------------------------------------------------------*/
VOID
Fsk::
vSetModifiedFlag(INT iFlag           // IN  if non-zero, set iModified flag to
                                     //     TRUE, else set it to FALSE
                )
{ 
   if (iFlag)
   {
      iModified = TRUE;
   }
   else
   {
      iModified = FALSE;
   }
   return;
}



/*----------------------------------------------------------------------------*\
|                                                                              |
| Function:  Set the FSK user defined state/event handler flag.                |
|                                                                              |
| Returns:   none                                                              |
|                                                                              |
\*----------------------------------------------------------------------------*/
VOID
Fsk::
vSetUserDefinedErrorHandlerFlag(INT iFlag           
                                     // IN  if non-zero, set flag to
                                     //     TRUE, else set it to FALSE
                               )
{ 
   vSetModifiedFlag(TRUE);

   if (iFlag)
   {
      iUserDefinedErrorHandler = TRUE;
   }
   else
   {
      iUserDefinedErrorHandler = FALSE;
   }
   return;
}



/*----------------------------------------------------------------------------*\
|                                                                              |
| Function:  Set the FSK date modified to the current date.                    |
|                                                                              |
| Returns:   none                                                              |
|                                                                              |
\*----------------------------------------------------------------------------*/
VOID
Fsk::
vSetModifiedDate(
                )
{ 
   struct tm *pstrTime;
   time_t     strLocalTime;

   strLocalTime = time(NULL);
   pstrTime = localtime(&strLocalTime);

   strftime(achLastModifiedDate, MODIFIED_DATE_LEN-1, 
            "%A, %b %d, %Y - %I:%M:%S %p", pstrTime);

   achLastModifiedDate[MODIFIED_DATE_LEN-1] = '\0';

   return;
}



/*----------------------------------------------------------------------------*\
|                                                                              |
| Function:  Set the description of the FSK to the text string provided.       |
|                                                                              |
| Returns:   INT                                                               |
|            On failure, non-zero value is returned.                           |
|                                                                              |
\*----------------------------------------------------------------------------*/
INT
Fsk::
iSetDescription(CHAR* pchNewFskDescription
                                     // IN the FSK description this is just one
                                     //    long continuous string of characters.
                                     //    Formating will be done when it is  
                                     //    printed.
               )
{ 
   vSetModifiedFlag(TRUE);

   if (strlen(pchNewFskDescription) >= DESCRIPTION_LEN)
   {
      sprintf(achAppErrorString, "FSK description to long, max length %d",
              (DESCRIPTION_LEN-1));
      strncpy(achDescription, pchNewFskDescription, DESCRIPTION_LEN-1);
      achDescription[DESCRIPTION_LEN-1] = '\0';
      return(ERROR);
   }

   strcpy(achDescription, pchNewFskDescription);
   return(0);
}



/*----------------------------------------------------------------------------*\
|                                                                              |
| Function:  Set the FSK return type to the one provided.                      |
|                                                                              |
| Returns:   INT                                                               |
|            On failure, non-zero value is returned.                           |
|                                                                              |
\*----------------------------------------------------------------------------*/
INT
Fsk::
iSetReturnType(CHAR* pchNewReturnType
                                     // IN  the return type for the FSK       
               )
{ 
   vSetModifiedFlag(TRUE);

   if (strlen(pchNewReturnType) >= RETURN_TYPE_LEN)
   {
      sprintf(achAppErrorString,
              "FSK return type to long, max length %d",
              (RETURN_TYPE_LEN-1));
      strncpy(achReturnType, pchNewReturnType, RETURN_TYPE_LEN-1);
      achReturnType[RETURN_TYPE_LEN-1] = '\0';
      return(ERROR);
   }

   strcpy(achReturnType, pchNewReturnType);
   return(0);
}



/*----------------------------------------------------------------------------*\
|                                                                              |
| Function:  Set the FSK header file string to the provided string             |
|                                                                              |
| Returns:   INT                                                               |
|            On failure, non-zero value is returned.                           |
|                                                                              |
\*----------------------------------------------------------------------------*/
INT
Fsk::
iSetHeaderFilesString(CHAR* pchNewHeaderFilesString
                                     // IN  string of header files
                    )
{ 
   vSetModifiedFlag(TRUE);

   if (strlen(pchNewHeaderFilesString) >= HEADER_FILES_STRING_LEN)
   {
      sprintf(achAppErrorString,
              "To many header files, max length %d",
              (HEADER_FILES_STRING_LEN-1));
      strncpy(achHeaderFiles, pchNewHeaderFilesString, 
              HEADER_FILES_STRING_LEN-1);
      achHeaderFiles[HEADER_FILES_STRING_LEN-1] = '\0';
      return(ERROR);
   }

   strcpy(achHeaderFiles, pchNewHeaderFilesString);
   return(0);
}



/*----------------------------------------------------------------------------*\
|                                                                              |
| Function:  Return the number of FSK events.                                  |
|                                                                              |
| Returns:   UINT                                                              |
|                                                                              |
\*----------------------------------------------------------------------------*/
UINT
Fsk::
uiGetNumEvents() 
{
   return((UINT)pcFskActionBlocks->ulGetNumRows());
}



/*----------------------------------------------------------------------------*\
|                                                                              |
| Function:  Returns the number of FSK states.                                 |
|                                                                              |
| Returns:   UINT                                                              |
|                                                                              |
\*----------------------------------------------------------------------------*/
UINT
Fsk::
uiGetNumStates() 
{
   return((UINT)pcFskActionBlocks->ulGetNumColumns());
}


/*----------------------------------------------------------------------------*\
|                                                                              |
| Function:  Insert a new event in the FSK.                                    |
|                                                                              |
| Returns:   INT                                                               |
|            On failure, a non-zero value is returned.                         |
|                                                                              |
\*----------------------------------------------------------------------------*/
INT
Fsk::
iInsertEvent(UINT uiEvent,           // IN  event to insert before or after 
             INT  iInsertDirection
                                     // IN  enumeration specifying to insert
                                     //     the new event before or after
                                     //     the specified event             
                )
{
   UINT uiLoop;
   UINT uiNumStates;

   
   vSetModifiedFlag(TRUE);

                                     // insert the new fsk/matrix event/row     
   if (pcFskActionBlocks->iInsertRow(uiEvent, iInsertDirection))
   {
      sprintf(achAppErrorString, "Could not insert FSK event row");
      return(ERROR);
   }

                                     // insert the new event
   if (pacFskEvent->iInsertCell(uiEvent, iInsertDirection))
   {
      sprintf(achAppErrorString, "Internal Error: could not insert");
      LOG_ERROR(achAppErrorString);
      return(ERROR);
   }

                                     // link the event block to the matrix
                                     // row...
                                     // we inserted after the event, we 
                                     // must increment the value for it to 
                                     // point to the new event
                                     // if we inserted before the event, we 
                                     // do not decrement it because all the 
                                     // event were moved down, hence we're 
                                     // on the current event...
   if (iInsertDirection == AFTER)
   {
      uiEvent++;
   }

   (*pacFskEvent)[uiEvent].pcActionBlock = 
      pcFskActionBlocks->pxGetRow(uiEvent);


                                     // update the new action-blocks with
                                     // the default action-block
   uiNumStates = uiGetNumStates();

   for (uiLoop=0; uiLoop<uiNumStates; uiLoop++)
   {
     if (iSetActionBlockData(uiEvent, uiLoop, &cDefaultFskActionBlock))
     {
        sprintf(achAppErrorString, "Could not set action-block data, %u, %u",
                uiLoop, uiEvent);
        LOG_ERROR(achAppErrorString);
        return(ERROR);
     }
   }

   return(0);
}



/*----------------------------------------------------------------------------*\
|                                                                              |
| Function:  Insert a new state in the FSK.                                    |
|                                                                              |
| Returns:   INT                                                               |
|            On failure, a non-zero value is returned.                         |
|                                                                              |
\*----------------------------------------------------------------------------*/
INT
Fsk::
iInsertState(UINT uiStateToInsert,   // IN  state to insert before or after 
             INT  iInsertDirection
                                     // IN  enumeration specifying to insert
                                     //     the new state before or after
                                     //     the specified state             
                )
{
   UINT uiLoop;
   UINT uiCurrentState;
   UINT uiCurrentEvent;
   UINT uiNumEvents;
   UINT uiNumStates;
   FskActionBlock cFskActionBlock;

   vSetModifiedFlag(TRUE);
   
                                     // insert the new fsk/matrix
                                     // state/column     
   if (pcFskActionBlocks->iInsertColumn(uiStateToInsert, iInsertDirection))
   {
      sprintf(achAppErrorString, "Could not insert FSK state row");
      return(ERROR);
   }

                                     // insert the new state
   if (pacFskState->iInsertCell(uiStateToInsert, iInsertDirection))
   {
      sprintf(achAppErrorString, "Internal Error: could not insert");
      LOG_ERROR(achAppErrorString);
      return(ERROR);
   }

                                     // link the event block to the matrix
                                     // row...
                                     // we inserted after the state, we 
                                     // must increment the value for it to 
                                     // point to the new state
                                     // if we inserted before the state, we 
                                     // do not decrement it because all the 
                                     // states were moved down, hence we're 
                                     // on the current state...
   if (iInsertDirection == AFTER)
   {
      uiStateToInsert++;
   }

   (*pacFskState)[uiStateToInsert].pcActionBlock = 
      pcFskActionBlocks->pxGetColumn(uiStateToInsert);


                                     // update the new action-blocks with
                                     // the default action-block
   uiNumEvents = uiGetNumEvents();

   for (uiLoop=0; uiLoop<uiNumEvents; uiLoop++)
   {
     if (iSetActionBlockData(uiLoop, uiStateToInsert, &cDefaultFskActionBlock))
     {
        sprintf(achAppErrorString, "Could not set action-block data, %u, %u",
                uiLoop, uiStateToInsert);
        LOG_ERROR(achAppErrorString);
        return(ERROR);
     }
   }

   uiNumStates = uiGetNumStates();
                                     // increment the next state values that 
                                     // are greater than or equal to the state
                                     // we just inserted
   if (pcFskActionBlocks->iGetMatrixCell(FIRST_CELL_IN_MATRIX, cFskActionBlock,
                                         &uiCurrentEvent, &uiCurrentState))
   {
                                     // this is an internal error...
      sprintf(achAppErrorString, "Could not get action-block data, %u, %u",
              uiCurrentEvent, uiCurrentState);
      LOG_ERROR(achAppErrorString);
      return(ERROR);
   }


   do
   {

     if ((*cFskActionBlock.puiNextState >= uiStateToInsert) &&
         (*cFskActionBlock.puiNextState < (uiNumStates-1)))
     {
                                     // increment the nextState value
        cFskActionBlock.vSetNextState((*cFskActionBlock.puiNextState)+1);
     }

     if (iSetActionBlockData(uiCurrentEvent, uiCurrentState, &cFskActionBlock))
     {
        sprintf(achAppErrorString, "Could not set action-block data, %u, %u",
                uiCurrentEvent, uiCurrentState);
        LOG_ERROR(achAppErrorString);
        return(ERROR);
     }

   } while (pcFskActionBlocks->iGetMatrixCell(
                           NEXT_CELL_IN_MATRIX, cFskActionBlock,
                           &uiCurrentEvent, &uiCurrentState) == 0);

   return(0);
}



/*----------------------------------------------------------------------------*\
|                                                                              |
| Function:  Delete an FSK event                                               |
|                                                                              |
| Returns:   INT                                                               |
|            On failure, a non-zero value is returned.                         |
|                                                                              |
\*----------------------------------------------------------------------------*/
INT
Fsk::
iDeleteEvent(UINT uiEvent            // IN  event to delete from the FSK
            )
{

   vSetModifiedFlag(TRUE);

   if (uiGetNumEvents() == 1) 
   {
      sprintf(achAppErrorString, 
              "FSK must have at least one event, can't delete last event");
      return(ERROR);
   }

                                     // delete the fsk event action blocks
   if (pcFskActionBlocks->iDeleteRow(uiEvent))
   {
      sprintf(achAppErrorString, "Could delete FSK event row");
      return(ERROR);
   }

                                     // delete the event block
   if (pacFskEvent->iDeleteCell(uiEvent))
   {
                                     // if we fail, we can't back out the 
                                     // deletion of the fsk event row,
                                     // so drop core
                                     // TODO, add the capability to back
                                     //       out a deletion of a row.
      sprintf(achAppErrorString, "Could delete FSK event");
      return(ERROR);
   }

   return(0);
}



/*----------------------------------------------------------------------------*\
|                                                                              |
| Function:  Copy the contents/members of an FSK event from the one provided   |
|                                                                              |
| Returns:   INT                                                               |
|            On failure, a non-zero value is returned.                         |
|                                                                              |
\*----------------------------------------------------------------------------*/
INT
Fsk::
iSetEventData(UINT uiEvent,          // IN  FSK event to set data members in
              FskEvent* pcEventData  // IN  event containing data to set FSK
                                     //     data members to
             )
{

   vSetModifiedFlag(TRUE);

   if (uiEvent >= uiGetNumEvents()) 
   {
      sprintf(achAppErrorString, 
              "Attempt to copy event data to non-existant event, %u, %u",
               uiEvent, uiGetNumEvents());
      return(ERROR);
   }

                                     // set event name
   if ((*pacFskEvent)[uiEvent].iSetName((char*)pcEventData->pchName))
   {
                                     // the base class set the app error
      return(ERROR);
   }

                                     // set event name
   if ((*pacFskEvent)[uiEvent].iSetDescription(
                               (char*)pcEventData->pchDescription))
   {
                                     // the base class set the app error
      return(ERROR);
   }

   return(0);
}



/*----------------------------------------------------------------------------*\
|                                                                              |
| Function:  Copy the contents/members of an FSK state from the one provided   |
|                                                                              |
| Returns:   INT                                                               |
|            On failure, a non-zero value is returned.                         |
|                                                                              |
\*----------------------------------------------------------------------------*/
INT
Fsk::
iSetStateData(UINT uiState,          // IN  FSK state to set data members in
              FskState* pcStateData  // IN  state containing data to copy to
                                     //     FSK data members to
             )
{

   vSetModifiedFlag(TRUE);

   if (uiState >= uiGetNumStates()) 
   {
      sprintf(achAppErrorString, 
              "Attempt to copy state data to non-existant state, %u, %u",
               uiState, uiGetNumStates());
      return(ERROR);
   }

                                     // set state name
   if ((*pacFskState)[uiState].iSetName((char*)pcStateData->pchName))
   {
                                     // the base class set the app error
      return(ERROR);
   }

                                     // set state name
   if ((*pacFskState)[uiState].iSetDescription(
                               (char*)pcStateData->pchDescription))
   {
                                     // the base class set the app error
      return(ERROR);
   }

   return(0);
}



/*----------------------------------------------------------------------------*\
|                                                                              |
| Function:  Copy the contents/members of an FSK action block to the one       |
|            in the matrix specified by the event/state                        |
|                                                                              |
| Returns:   INT                                                               |
|            On failure, a non-zero value is returned.                         |
|                                                                              |
\*----------------------------------------------------------------------------*/
INT
Fsk::
iSetActionBlockData(UINT uiEvent,    // IN  FSK event index
                    UINT uiState,    // IN  FSK state index
                    FskActionBlock* pcFskActionBlock  
                                     // IN  action block containing data to 
                                     //     set specified FSK action block 
                                     //     members to
             )
{

   vSetModifiedFlag(TRUE);

   if (uiEvent >= uiGetNumEvents()) 
   {
      sprintf(achAppErrorString, 
              "Attempt to copy action block data to non-existant event, %u, %u",
               uiEvent, uiGetNumEvents());
      LOG_ERROR(achAppErrorString);
      return(ERROR);
   }

   if (uiState >= uiGetNumStates()) 
   {
      sprintf(achAppErrorString, 
              "Attempt to copy action block data to non-existant state, %u, %u",
               uiState, uiGetNumStates());
      LOG_ERROR(achAppErrorString);
      return(ERROR);
   }

   pcFskActionBlocks->rxCellData((ULONG)uiEvent, (ULONG)uiState) =
      *pcFskActionBlock;
   
   return(0);
}



/*----------------------------------------------------------------------------*\
|                                                                              |
| Function:  Copy the contents/members of an FSK event to the one provided     |
|                                                                              |
| Returns:   INT                                                               |
|            On failure, a non-zero value is returned.                         |
|                                                                              |
\*----------------------------------------------------------------------------*/
INT
Fsk::
iGetEventData(UINT uiEvent,          // IN  FSK event to get data members from
              FskEvent* pcEventData  // OU  event to put the contents of the
                                     //     FSK members to 
             )
{

   if (uiEvent >= uiGetNumEvents()) 
   {
      sprintf(achAppErrorString, 
              "Attempt to get event data from non-existant event, %u, %u",
               uiEvent, uiGetNumEvents());
      return(ERROR);
   }

   *pcEventData = (*pacFskEvent)[uiEvent];

   return(0);
}



/*----------------------------------------------------------------------------*\
|                                                                              |
| Function:  Copy the contents/members of an FSK state to the one provided     |
|                                                                              |
| Returns:   INT                                                               |
|            On failure, a non-zero value is returned.                         |
|                                                                              |
\*----------------------------------------------------------------------------*/
INT
Fsk::
iGetStateData(UINT uiState,          // IN  FSK state to get data members from
              FskState* pcStateData  // OU  location to put the FSK data state
                                     //     members in
             )
{

   if (uiState >= uiGetNumStates()) 
   {
      sprintf(achAppErrorString, 
              "Attempt to get state data from non-existant state, %u, %u",
               uiState, uiGetNumStates());
      return(ERROR);
   }

   *pcStateData = (*pacFskState)[uiState];

   return(0);
}



/*----------------------------------------------------------------------------*\
|                                                                              |
| Function:  Copy the contents/members of an FSK action block from the         |
|            one specified in the FSK to the one provided.                     |
|                                                                              |
| Returns:   INT                                                               |
|            On failure, a non-zero value is returned.                         |
|                                                                              |
\*----------------------------------------------------------------------------*/
INT
Fsk::
iGetActionBlockData(UINT uiEvent,    // IN  FSK event index
                    UINT uiState,    // IN  FSK state index
                    FskActionBlock* pcFskActionBlock  
                                     // OU  action block containing data to 
                                     //     set specified FSK action block 
                                     //     members to
             )
{

   if (uiEvent >= uiGetNumEvents()) 
   {
      sprintf(achAppErrorString, 
            "Attempt to get action block data from non-existant event, %u, %u",
             uiEvent, uiGetNumEvents());
      LOG_ERROR(achAppErrorString);
      return(ERROR);
   }

   if (uiState >= uiGetNumStates()) 
   {
      sprintf(achAppErrorString, 
            "Attempt to get action block data from non-existant state, %u, %u",
             uiState, uiGetNumStates());
      LOG_ERROR(achAppErrorString);
      return(ERROR);
   }

   *pcFskActionBlock =
      pcFskActionBlocks->rxCellData((ULONG)uiEvent, (ULONG)uiState);
   
   return(0);
}



/*----------------------------------------------------------------------------*\
|                                                                              |
| Function:  Copy an FSK event to another event in the same FSK.               |
|                                                                              |
| Returns:   INT                                                               |
|            On failure, a non-zero value is returned.                         |
|                                                                              |
\*----------------------------------------------------------------------------*/
INT
Fsk::
iCopyEvent(UINT uiFromEvent,         // IN  FSK event to copy "from"
           UINT uiToEvent            // IN  FSK event to copy "to"
          )
{

   if (uiFromEvent >= uiGetNumEvents()) 
   {
      sprintf(achAppErrorString, 
            "Attempt to copy event data from non-existant event, %u, %u",
             uiFromEvent, uiGetNumEvents());
      return(ERROR);
   }

   if (uiToEvent >= uiGetNumEvents()) 
   {
      sprintf(achAppErrorString, 
            "Attempt to copy event data to non-existant event, %u, %u",
             uiToEvent, uiGetNumEvents());
      return(ERROR);
   }


                                     // copy the action blocks     
   if (pcFskActionBlocks->iCopyRow((ULONG)uiFromEvent, (ULONG)uiToEvent))
   {
      sprintf(achAppErrorString, "Could not copy event %d to %d",
              uiFromEvent, uiToEvent);
      return(ERROR);
   }

                                     // copy the event block     
   if (iSetEventData(uiToEvent, &((*pacFskEvent)[uiFromEvent])))
   {
                                     // TODO, allow "undo" of previous
                                     //       copy, so we can back-out clean...
      return(ERROR);
   }

   
   return(0);
}



/*----------------------------------------------------------------------------*\
|                                                                              |
| Function:  Copy an FSK state to another state in the same FSK.               |
|                                                                              |
| Returns:   INT                                                               |
|            On failure, a non-zero value is returned.                         |
|                                                                              |
\*----------------------------------------------------------------------------*/
INT
Fsk::
iCopyState(UINT uiFromState,         // IN  FSK state to copy "from"
           UINT uiToState            // IN  FSK state to copy "to"
          )
{

   if (uiFromState >= uiGetNumStates()) 
   {
      sprintf(achAppErrorString, 
            "Attempt to copy state data from non-existant state, %u, %u",
             uiFromState, uiGetNumStates());
      return(ERROR);
   }

   if (uiToState >= uiGetNumStates()) 
   {
      sprintf(achAppErrorString, 
            "Attempt to copy state data to non-existant state, %u, %u",
             uiToState, uiGetNumStates());
      return(ERROR);
   }


                                     // copy the action blocks     
   if (pcFskActionBlocks->iCopyColumn((ULONG)uiFromState, (ULONG)uiToState))
   {
      sprintf(achAppErrorString, "Could not copy state %d to %d",
              uiFromState, uiToState);
      return(ERROR);
   }

                                     // copy the state block     
   if (iSetStateData(uiToState, &((*pacFskState)[uiFromState])))
   {
      LOG_ERROR("Internal error");
      return(ERROR);
   }

   
   return(0);
}



/*----------------------------------------------------------------------------*\
|                                                                              |
| Function:  Copy an FSK event from an FSK to another event in a different FSK |
|                                                                              |
| Returns:   INT                                                               |
|            On failure, a non-zero value is returned.                         |
|                                                                              |
\*----------------------------------------------------------------------------*/
INT
Fsk::
iCopyEvent(UINT uiFromEvent,         // IN  FSK index event to copy "from"
           Fsk* pcToFsk,             // OU  FSK to copy event "to"
           UINT uiToEvent            // IN  FSK index event to copy "to"
          )
{
   Fsk* pcFromFsk;           

   pcFromFsk = this;

   if (uiFromEvent >= pcFromFsk->uiGetNumEvents()) 
   {
      sprintf(achAppErrorString, 
            "Attempt to copy event data from non-existant event, %u, %u",
             uiFromEvent, pcFromFsk->uiGetNumEvents());
      return(ERROR);
   }

   if (uiToEvent >= pcToFsk->uiGetNumEvents()) 
   {
      sprintf(achAppErrorString, 
            "Attempt to copy event data to non-existant event, %u, %u",
             uiToEvent, pcToFsk->uiGetNumEvents());
      return(ERROR);
   }

                                     // number of states must be same to
                                     // copy a row of events...
   if (pcFromFsk->uiGetNumStates() != pcToFsk->uiGetNumStates()) 
   {
      sprintf(achAppErrorString, 
            "FSKs have different number of rows, can't copy, %u, %u",
             uiFromEvent, uiToEvent);
      return(ERROR);
   }


                                     // copy the action blocks     
   if (pcToFsk->pcFskActionBlocks->iCopyRow(pcFromFsk->pcFskActionBlocks,
                                            (ULONG)uiFromEvent, 
                                            pcToFsk->pcFskActionBlocks,
                                            (ULONG)uiToEvent))
   {
      sprintf(achAppErrorString, "Could not copy event %d to %d",
              uiFromEvent, uiToEvent);
      return(ERROR);
   }

                                     // copy the event block     
   if (pcToFsk->iSetEventData(uiToEvent, 
                              &(((*(pcFromFsk->pacFskEvent))[uiFromEvent]))))
   {
                                     // TODO, allow "undo" of previous
                                     //       copy, so we can back-out clean...
      return(ERROR);
   }

   
   pcToFsk->vSetModifiedFlag(TRUE);
   return(0);
}



/*----------------------------------------------------------------------------*\
|                                                                              |
| Function:  Copy an FSK state from an FSK to another state in a different FSK |
|                                                                              |
| Returns:   INT                                                               |
|            On failure, a non-zero value is returned.                         |
|                                                                              |
\*----------------------------------------------------------------------------*/
INT
Fsk::
iCopyState(UINT uiFromState,         // IN  FSK state index to copy "from"
           Fsk* pcToFsk,             // OU  FSK to copy state "to"
           UINT uiToState            // IN  FSK state index to copy "to"
          )
{
   Fsk* pcFromFsk;

   pcFromFsk = this;

   if (uiFromState >= pcFromFsk->uiGetNumStates()) 
   {
      sprintf(achAppErrorString, 
            "Attempt to copy state data from non-existant state, %u, %u",
             uiFromState, pcFromFsk->uiGetNumStates());
      return(ERROR);
   }

   if (uiToState >= pcToFsk->uiGetNumStates()) 
   {
      sprintf(achAppErrorString, 
            "Attempt to copy state data to non-existant state, %u, %u",
             uiToState, pcToFsk->uiGetNumStates());
      return(ERROR);
   }

                                     // number of events must be same to
                                     // copy a column of states...
   if (pcFromFsk->uiGetNumEvents() != pcToFsk->uiGetNumEvents()) 
   {
      sprintf(achAppErrorString, 
            "FSKs have different number of states, can't copy, %u, %u",
             uiFromState, uiToState);
      return(ERROR);
   }


                                     // copy the action blocks     
   if (pcToFsk->pcFskActionBlocks->iCopyColumn(pcFromFsk->pcFskActionBlocks,
                                               (ULONG)uiFromState, 
                                               pcToFsk->pcFskActionBlocks,
                                               (ULONG)uiToState))
   {
      sprintf(achAppErrorString, "Could not copy state %d to %d",
              uiFromState, uiToState);
      return(ERROR);
   }

                                     // copy the state block     
   if (pcToFsk->iSetStateData(uiToState, 
                              &(((*(pcFromFsk->pacFskState))[uiFromState]))))
   {
      LOG_ERROR("Internal error");
      return(ERROR);
   }

   
   pcToFsk->vSetModifiedFlag(TRUE);
   return(0);
}


/*----------------------------------------------------------------------------*\
|                                                                              |
| Function:  Return the next state to transition to after the action block at  |
|            the given event/state is executed.                                |
|                                                                              |
| Returns:   INT                                                               |
|            On success, a value of zero is returned.                          |
|            On failure, a non-zero value is returned.                         |
|                                                                              |
\*----------------------------------------------------------------------------*/
INT
Fsk::
iGetNextState(UINT  uiEvent,         // IN  FSK event index
              UINT  uiState,         // IN  FSK state index
              UINT* puiNextState     // OU  the next state to transition to 
                                     //     after this action block function is     
                                     //     executed  
             )
{

   if (uiEvent >= uiGetNumEvents()) 
   {
      sprintf(achAppErrorString, 
              "Attempt to get next state value from non-existant event, %u, %u",
               uiEvent, uiGetNumEvents());
      return(ERROR);
   }

   if (uiState >= uiGetNumStates()) 
   {
      sprintf(achAppErrorString, 
              "Attempt to get next state value from non-existant state, %u, %u",
               uiState, uiGetNumStates());
      return(ERROR);
   }

   *puiNextState =
      *((pcFskActionBlocks->
         rxCellData((ULONG)uiEvent, (ULONG)uiState)).puiNextState);
   
   return(0);
}



/*----------------------------------------------------------------------------*\
|                                                                              |
| Function:  Get the first action block in the FSK.                            |
|                                                                              |
| Returns:   INT                                                               |
|            0 == success                                                      |
|           !0 == no more action blocks                                        |
|                                                                              |
\*----------------------------------------------------------------------------*/
INT
Fsk::
iGetFirstActionBlock(FskActionBlock* pcFskActionBlock,
                                     // OU  action block
                     UINT* puiCurrentEvent,
                                     // OU  the current event
                     UINT* puiCurrentState 
                                     // OU  the current state
                    )
{
   return(pcFskActionBlocks->iGetMatrixCell(
                                 FIRST_CELL_IN_MATRIX, *pcFskActionBlock,
                                 puiCurrentEvent, puiCurrentState));
}



/*----------------------------------------------------------------------------*\
|                                                                              |
| Function:  Get the next action block in the FSK.                             |
|                                                                              |
| Returns:   INT                                                               |
|            0 == success                                                      |
|           !0 == no more action blocks                                        |
|                                                                              |
\*----------------------------------------------------------------------------*/
INT
Fsk::
iGetNextActionBlock(FskActionBlock* pcFskActionBlock,
                                     // OU  action block
                    UINT* puiCurrentEvent,
                                     // OU  the current event
                    UINT* puiCurrentState 
                                     // OU  the current state
                    )
{
   return(pcFskActionBlocks->iGetMatrixCell(
                                 NEXT_CELL_IN_MATRIX, *pcFskActionBlock,
                                 puiCurrentEvent, puiCurrentState));
}



/*----------------------------------------------------------------------------*\
|                                                                              |
| Function:  Delete the given FSK state, adjusting the uiNextState value       |
|            in each action-block as needed.  If the uiNextState value is      |
|            equal to the state being deleted, it is set to                    |
|            LAST_SET_TO_UNASSIGNED_STATE so the calling program can set it    |
|            back in the case of an undelete.  The next time this function is  |
|            called the uiNextState values that equal                          |
|            LAST_SET_TO_UNASSIGNED_STATE  are changed to UNASSIGNED_STATE.    |
|                                                                              |
| Returns:   INT                                                               |
|            On failure, a non-zero value is returned.                         |
|                                                                              |
\*----------------------------------------------------------------------------*/
INT
Fsk::
iDeleteState(UINT uiStateToDelete    // IN  state to delete from the FSK
            )
{

   UINT uiLoop;
   UINT uiNumStates;
   UINT uiCurrentState;
   UINT uiCurrentEvent;
   FskActionBlock cFskActionBlock;

   vSetModifiedFlag(TRUE);

   uiNumStates = uiGetNumStates();

   if (uiNumStates == 1) 
   {
      sprintf(achAppErrorString, 
              "FSK must have at least one state, can't delete last state");
      return(ERROR);
   }

                                     // delete the fsk state action blocks
   if (pcFskActionBlocks->iDeleteColumn(uiStateToDelete))
   {
      sprintf(achAppErrorString, "Could delete FSK state row");
      return(ERROR);
   }

                                     // delete the state block
   if (pacFskState->iDeleteCell(uiStateToDelete))
   {
                                     // if we fail, we can't back out the 
                                     // deletion of the fsk state column,
                                     // so drop core
                                     // TODO, add the capability to back
                                     //       out a deletion of a column.
      return(ERROR);
   }

                                     // set the next state values that 
                                     // equal the state we just deleted to
                                     // unassigned; also, any next state value
                                     // that is greater than the state number
                                     // we deleted, decrement the value.
   if (pcFskActionBlocks->iGetMatrixCell(FIRST_CELL_IN_MATRIX, cFskActionBlock,
                                         &uiCurrentEvent, &uiCurrentState))
   {
                                     // this is an internal error...
      sprintf(achAppErrorString, "Could not get action-block data, %u, %u",
              uiCurrentEvent, uiCurrentState);
      LOG_ERROR(achAppErrorString);
      return(ERROR);
   }


   do
   {

      if (*cFskActionBlock.puiNextState == uiStateToDelete)
      {
                         // set nextState so the upper-level functions 
                         // can change it back if needed (i.e. undelete is 
                         // called).
         cFskActionBlock.vSetNextState(LAST_SET_TO_UNASSIGNED_STATE);
      }
      else if ((*cFskActionBlock.puiNextState > uiStateToDelete) &&
               (*cFskActionBlock.puiNextState < MAX_FSK_STATES))
      {
                                     // decrement the nextState value
         cFskActionBlock.vSetNextState((*cFskActionBlock.puiNextState)-1);
      }
      else if (*cFskActionBlock.puiNextState ==  LAST_SET_TO_UNASSIGNED_STATE)
      {
                                     // set the nextState value
         cFskActionBlock.vSetNextState(UNASSIGNED_STATE);
      }

      if (iSetActionBlockData(uiCurrentEvent, uiCurrentState, &cFskActionBlock))
      {
         sprintf(achAppErrorString, "Could not set action-block data, %u, %u",
                 uiCurrentEvent, uiCurrentState);
         LOG_ERROR(achAppErrorString);
         return(ERROR);
      }

   } while (pcFskActionBlocks->iGetMatrixCell(
                           NEXT_CELL_IN_MATRIX, cFskActionBlock,
                           &uiCurrentEvent, &uiCurrentState) == 0);

   return(0);
}



/*----------------------------------------------------------------------------*\
|                                                                              |
| Function:  Set the default function for action-blocks.  All new action-blocks|
|            will contain this function as the function to call whe the        |
|            action-block is entered.                                          |
|                                                                              |
| Returns:   INT                                                               |
|            On failure, non-zero value is returned.                           |
|                                                                              |
\*----------------------------------------------------------------------------*/
INT
Fsk::
iSetDefaultFunction(CHAR* pchDefaultFunction
                                     // function, with parameters, to use
                                     // as the default function when new   
                                     // action-blocks are created
            )
{ 
   vSetModifiedFlag(TRUE);
   return(cDefaultFskActionBlock.iSetFunction(pchDefaultFunction));
}



/*----------------------------------------------------------------------------*\
|                                                                              |
| Function:  Set the default function next state value.  All new action-blocks |
|            will contain this value as the next state to transition to.       |
|                                                                              |
| Returns:   VOID                                                              |
|                                                                              |
\*----------------------------------------------------------------------------*/
VOID
Fsk::
vSetDefaultNextState(UINT uiDefaultNextState
                                     // nextState value to use
                                     // as the default function when new   
                                     // action-blocks are created
               )
{ 
   vSetModifiedFlag(TRUE);
   cDefaultFskActionBlock.vSetNextState(uiDefaultNextState);
   return;
}



/*----------------------------------------------------------------------------*\
|                                                                              |
| Function:  Write the FSK to the specified file.  If the file exists, don't   |
|            over-wite it unless the over-write flag is non-zero.              |
|                                                                              |
| Returns:   INT                                                               |
|            0             If FSK was successfully written to the specified    |
|                          file                                                |
|            FILE_EXISTS   If the file exists and the over-write flag is zero  |
|            ERROR         If any other type of error is encountered           |
|                                                                              |
\*----------------------------------------------------------------------------*/
INT
Fsk::
iWriteFile(CHAR* pchOutputFile,     // IN  name of file to write FSK to  
           INT   iOverWrite          // IN  if non-zero, overwite the file if
                                     //     it exists
          )
{

   UINT   uiLoop;
   UINT   uiLoop2;
   UINT   uiNumStates;
   UINT   uiNumEvents;
   FskEvent cFskEvent;
   FskState cFskState;
   FskActionBlock cFskActionBlock;
   FILE* phOutputFile;
   struct stat strStatBuf;

   if (!iOverWrite)
   {
                                     // see if file exists
      phOutputFile = fopen(pchOutputFile, "r");
      
      if (phOutputFile != NULL)
      {
                                     // determine if it is a regular file
         stat(pchOutputFile, &strStatBuf);
         fclose(phOutputFile);

         if (strStatBuf.st_mode & S_IFREG)
         {
            return(FILE_EXISTS);
         }
         else
         {
            sprintf(achAppErrorString, 
                    "Can't write \"%s\" - not a regular file",
                    pchOutputFile);
            return(ERROR);
         }
      }
                                     // else, we got an error, let the open
                                     // to write determine the error (we may
                                     // not have read perms, but may have   
                                     // write...)

   }


   phOutputFile = fopen(pchOutputFile, "w");

   if (phOutputFile == NULL)
   {
                                     // we can't write to the file for some
                                     // reason...
      if (errno == EACCES)
      {
         sprintf(achAppErrorString, "Could not write to %s, permission denied",
                 pchOutputFile);
      }
      else
      {
         sprintf(achAppErrorString, "Could not write to %s, errno = %d",
                 pchOutputFile, errno);
      }

      return(errno);
   }

                                     // write the number of events followed
                                     // by the number of states, then the   
                                     // event blocks followed by the state     
                                     // blocks, then each row of states of the 
                                     // action blocks...

   vSetModifiedDate();
   uiNumEvents = uiGetNumEvents();
   uiNumStates = uiGetNumStates();

                                     // write the version..
   fwrite(achFskFileVersionString, FSK_VERSION_STRING_LEN, 1, phOutputFile);

                                     // write the number of events and states
   fwrite(&uiNumEvents, sizeof(UINT), 1, phOutputFile);
   fwrite(&uiNumStates, sizeof(UINT), 1, phOutputFile);

                                     // write the FSK class data
   fwrite(&iModified, sizeof(INT), 1, phOutputFile);
   fwrite(&iUserDefinedErrorHandler, sizeof(INT), 1, phOutputFile);
   fwrite(achLastModifiedDate, MODIFIED_DATE_LEN, 1, phOutputFile);
   fwrite(achDescription, DESCRIPTION_LEN, 1, phOutputFile);
   fwrite(achReturnType, RETURN_TYPE_LEN, 1, phOutputFile);
   fwrite(achHeaderFiles, HEADER_FILES_STRING_LEN, 1, phOutputFile);
   fwrite(&cDefaultFskActionBlock, sizeof(FskActionBlock), 1, phOutputFile);


   for (uiLoop=0; uiLoop<uiNumEvents; uiLoop++)
   {
      
      if (iGetEventData(uiLoop, &cFskEvent))
      {
         sprintf(achAppErrorString, "Could not get event data, %u",
                 uiLoop);
         LOG_ERROR(achAppErrorString);
         fclose(phOutputFile);
         return(ERROR);
      }

      fwrite(&cFskEvent, sizeof(FskEvent), 1, phOutputFile);
   }


   for (uiLoop=0; uiLoop<uiNumStates; uiLoop++)
   {
      
      if (iGetStateData(uiLoop, &cFskState))
      {
         sprintf(achAppErrorString, "Could not get state data, %u",
                 uiLoop);
         LOG_ERROR(achAppErrorString);
         fclose(phOutputFile);
         return(ERROR);
      }

      fwrite(&cFskState, sizeof(FskState), 1, phOutputFile);
   }


   for (uiLoop=0; uiLoop<uiNumEvents; uiLoop++)
   {
      for (uiLoop2=0; uiLoop2<uiNumStates; uiLoop2++)
      {
         if (iGetActionBlockData(uiLoop, uiLoop2, &cFskActionBlock))
         {
            sprintf(achAppErrorString, 
                    "Could not get action block data, %u, %u",
                    uiLoop, uiLoop2);
            LOG_ERROR(achAppErrorString);
            fclose(phOutputFile);
            return(ERROR);
         }

         fwrite(&cFskActionBlock, sizeof(FskActionBlock), 1, phOutputFile);
      }
   }

   fclose(phOutputFile);
   vSetModifiedFlag(FALSE);
   return(0);
}



/*----------------------------------------------------------------------------*\
|                                                                              |
| Function:  Read an FSK from the specified file.                              |
|                                                                              |
| Returns:   INT                                                               |
|            0             If the FSK was sucessfully read from the specified  |
|                          file                                                |
|                                                                              |
|            errno         if an error occurs for which errono is set          |
|            non-zero      on other errors                                     |
|                                                                              |
\*----------------------------------------------------------------------------*/
INT
Fsk::
iReadFile(CHAR* pchInputFile        // IN  name of file to write FSK to  
         )
{

   UINT   uiLoop;
   UINT   uiLoop2;
   UINT   uiNumStates;
   UINT   uiNumEvents;
   FskEvent cFskEvent;
   FskState cFskState;
   FskActionBlock cFskActionBlock;
   FskEvent cTempFskEvent;
   FskState cTempFskState;
   FskActionBlock cTempFskActionBlock;
   FILE* phInputFile;

   phInputFile = fopen(pchInputFile, "r");

   if (phInputFile == NULL)
   {
                                     // we can't open the specified file
      if (errno == EACCES)
      {
         sprintf(achAppErrorString, "Could not open %s, permission denied",
                 pchInputFile);
      }
      else
      {
         sprintf(achAppErrorString, "Could not open file %s, error = %d",
                 pchInputFile, errno);
      }

      return(errno);
   }

                                     // read the version..
   fread(achFskFileVersionString, FSK_VERSION_STRING_LEN, 1, phInputFile);

   if (strcmp(achFskFileVersionString, achFskVersionString) != 0)
   {
                                     // determine if the version of the file is
                                     // compatible with the version of this 
                                     // FSKC release
      for(uiLoop=0; 
          pachCompatibleFskVersions[uiLoop] != (CONST CHAR*)0; 
          uiLoop++)
      {
         if (strcmp(pachCompatibleFskVersions[uiLoop],
                    achFskFileVersionString) == 0)
         {
                                     // the file being read in is compatible   
                                     // with this version of the FSKC software 
            break;
         }
      }

      if (pachCompatibleFskVersions[uiLoop] == (CONST CHAR*)0)
      {
                                     // the file is not compatible with this   
                                     // version of the FSKC software           
         sprintf(achAppErrorString, 
                "CASE Tool version %s is incompatible with FSK file version %s",
                 achFskVersionString, achFskFileVersionString);
         fclose(phInputFile);
         return(ERROR);
      }
                                     // update the version string to the current
                                     // one since the versions are compatible..
      strcpy(achFskFileVersionString, achFskVersionString);
   }

                                     // read the number of events and states    
                                     // from the file
   fread(&uiNumEvents, sizeof(UINT), 1, phInputFile);
   fread(&uiNumStates, sizeof(UINT), 1, phInputFile);

                                     // init the FSK
   if (iInitFsk(uiNumEvents, uiNumStates))
   {
      sprintf(achAppErrorString, "could not create FSK");
      LOG_ERROR(achAppErrorString);
      return(ERROR);
   }

                                     // write the FSK class data
   fread(&iModified, sizeof(INT), 1, phInputFile);
   fread(&iUserDefinedErrorHandler, sizeof(INT), 1, phInputFile);
   fread(achLastModifiedDate, MODIFIED_DATE_LEN, 1, phInputFile);
   fread(achDescription, DESCRIPTION_LEN, 1, phInputFile);
   fread(achReturnType, RETURN_TYPE_LEN, 1, phInputFile);
   fread(achHeaderFiles, HEADER_FILES_STRING_LEN, 1, phInputFile);

   fread(&cTempFskActionBlock, sizeof(FskActionBlock), 1, phInputFile);
   cDefaultFskActionBlock = cTempFskActionBlock;

                                     // load the events, states and action
                                     // blocks from the FSK file...
   for (uiLoop=0; uiLoop<uiNumEvents; uiLoop++)
   {
      
      fread(&cTempFskEvent, sizeof(FskEvent), 1, phInputFile);
      // when we read an event, the pointers are garbage; so, we need to
      // assign it to an object with valid pointers...
      cFskEvent = cTempFskEvent;
      if (iSetEventData(uiLoop, &cFskEvent))
      {
         sprintf(achAppErrorString, "Could not set event data, %u",
                 uiLoop);
         LOG_ERROR(achAppErrorString);
         fclose(phInputFile);
         return(ERROR);
      }
   }


   for (uiLoop=0; uiLoop<uiNumStates; uiLoop++)
   {
      
      fread(&cTempFskState, sizeof(FskState), 1, phInputFile);
      // when we read a state, the pointers are garbage; so, we need to
      // assign it to an object with valid pointers...
      cFskState = cTempFskState;
      if (iSetStateData(uiLoop, &cFskState))
      {
         sprintf(achAppErrorString, "Could not set state data, %u",
                 uiLoop);
         LOG_ERROR(achAppErrorString);
         fclose(phInputFile);
         return(ERROR);
      }
   }


   for (uiLoop=0; uiLoop<uiNumEvents; uiLoop++)
   {
      for (uiLoop2=0; uiLoop2<uiNumStates; uiLoop2++)
      {
         fread(&cTempFskActionBlock, sizeof(FskActionBlock), 1, phInputFile);

         // when we read an action block, the pointers are garbage; so, we
         // need to assign it to an object with valid pointers...
         cFskActionBlock = cTempFskActionBlock;

         // if next state is LAST_SET_TO_UNASSIGNED_STATE, set to 
         // UNASSIGNED_STATE....
         if (*cFskActionBlock.puiNextState ==  LAST_SET_TO_UNASSIGNED_STATE)
         {
                      // set the nextState value
            cFskActionBlock.vSetNextState(UNASSIGNED_STATE);
         }

         if (iSetActionBlockData(uiLoop, uiLoop2, &cFskActionBlock))
         {
            sprintf(achAppErrorString, 
                    "Could not set action block data, %u, %u",
                    uiLoop, uiLoop2);
            LOG_ERROR( achAppErrorString);
            fclose(phInputFile);
            return(ERROR);
         }
      }
   }

   fclose(phInputFile);
   vSetModifiedFlag(FALSE);
   return(0);
}



/*----------------------------------------------------------------------------*\
|                                                                              |
| Function:  Validate the FSK                                                  |
|                                                                              |
| Returns:   INT                                                               |
|            non-zero value on error                                           |
|                                                                              |
\*----------------------------------------------------------------------------*/
INT
Fsk::
iValidate(
         )
{ 
   INT   iResult;
   UINT  uiNumStates;
   UINT  uiNumEvents;
   UINT  uiCurrentState;
   UINT  uiCurrentEvent;
   UINT  uiLoop;
   UINT  uiLoop2;
   CHAR* pchTemp;
   FskEvent cFskEvent;
   FskState cFskState;
   FskEvent cFskEvent2;
   FskState cFskState2;
   FskActionBlock cFskActionBlock;
   
                         // make sure FSK return is not null or just spaces

   for (uiLoop=0; pchReturnType[uiLoop] == ' '; uiLoop++)
   {
      /* NULL LOOP */
   }

   if ((*pchReturnType == '\0') ||
       (pchReturnType[uiLoop] == '\0'))
   {
      sprintf(achAppErrorString,
              "FSK return type not set,\nSet using Edit->Source Settings");
      return(ERROR);
   }


                         // make sure event and state names contain a 
                         // string and the string does not contain
                         // spaces....we could check for embedded new-lines,
                         // *, -, +, =, etc. but we will let the C 
                         // compiler do that job...:-)
   uiNumStates = uiGetNumStates();
   for (uiLoop=0; uiLoop<uiNumStates; uiLoop++)
   {
      if (iGetStateData(uiLoop, &cFskState))
      {
         LOG_ERROR("Internal error");
         return(ERROR);
      }

      if (cFskState.pchName[0] == '\0')
      {
         sprintf(achAppErrorString,
                 "State number %u does not contain a name", uiLoop);
         return(ERROR);
      }
  
      if (strchr(cFskState.pchName, ' ') != NULL)
      {
         sprintf(achAppErrorString,
                 "State name \"%s\" contains a space, state number %u", 
                 (CHAR*)cFskState.pchName, uiLoop);
         return(ERROR);
      }
   }

   uiNumEvents = uiGetNumEvents();
   for (uiLoop=0; uiLoop<uiNumEvents; uiLoop++)
   {
      if (iGetEventData(uiLoop, &cFskEvent))
      {
         LOG_ERROR("Internal error");
         return(ERROR);
      }

      if (cFskEvent.pchName[0] == '\0')
      {
         sprintf(achAppErrorString,
                 "Event number %u does not contain a name", uiLoop);
         return(ERROR);
      }

      if (strchr(cFskEvent.pchName, ' ') != NULL)
      {
         sprintf(achAppErrorString,
                 "Event name \"%s\" contains a space, event number %u", 
                 (CHAR*)cFskEvent.pchName, uiLoop);
         return(ERROR);
      }
   }

                         // check for duplicate state names
   for (uiLoop=0; uiLoop<(uiNumStates-1); uiLoop++)
   {
      if (iGetStateData(uiLoop, &cFskState))
      {
         LOG_ERROR("Internal error");
         return(ERROR);
      }

      for (uiLoop2=(uiLoop+1); uiLoop2<uiNumStates; uiLoop2++)
      {
         if (iGetStateData(uiLoop2, &cFskState2))
         {
            LOG_ERROR("Internal error");
            return(ERROR);
         }

         if (strcmp(cFskState2.pchName, cFskState.pchName) == 0)
         {
            sprintf(achAppErrorString,
                    "Duplicate state names in state %u and %u", 
                    uiLoop, uiLoop2);
            return(ERROR);
         }
      }
   }

                         // check for duplicate event names
   for (uiLoop=0; uiLoop<(uiNumEvents-1); uiLoop++)
   {
      if (iGetEventData(uiLoop, &cFskEvent))
      {
         LOG_ERROR("Internal error");
         return(ERROR);
      }

      for (uiLoop2=(uiLoop+1); uiLoop2<uiNumEvents; uiLoop2++)
      {
         if (iGetEventData(uiLoop2, &cFskEvent2))
         {
            LOG_ERROR("Internal error");
            return(ERROR);
         }

         if (strcmp(cFskEvent2.pchName, cFskEvent.pchName) == 0)
         {
            sprintf(achAppErrorString,
                    "Duplicate event names in event %u and %u", 
                    uiLoop, uiLoop2);
            return(ERROR);
         }
      }
   }


                         // make sure all next state values are less than
                         // the current number of states;  make sure each
                         // function has a beginning and ending parenthesis
   if (iGetFirstActionBlock(&cFskActionBlock,
                            &uiCurrentEvent, &uiCurrentState))
   {
                                  // this is an internal error...
      sprintf(achAppErrorString, "Could not get action-block data, %u, %u",
              uiCurrentEvent, uiCurrentState);
      LOG_ERROR(achAppErrorString);
      return(ERROR);
   }

   do
   {
      if (*cFskActionBlock.puiNextState >= uiNumStates) 
      {
         sprintf(achAppErrorString,
           "Invalid transition state in action-block at location %u,%u\n", 
           uiCurrentEvent, uiCurrentState);
         if (*cFskActionBlock.puiNextState < MAX_FSK_STATES)
         {
            sprintf(&achAppErrorString[strlen(achAppErrorString)],
              "Current next state value = %u, last state number = %u", 
              *cFskActionBlock.puiNextState, uiNumStates-1);
         }
         return(ERROR);
      }

      if ((strchr(cFskActionBlock.pchFunction, '(') == NULL) ||
          (strrchr(cFskActionBlock.pchFunction, ')') == NULL))
      {
         sprintf(achAppErrorString,
           "Paramter list in action-block at location %u,%u does not\n", 
           uiCurrentEvent, uiCurrentState);
         sprintf(&achAppErrorString[strlen(achAppErrorString)],
           "contain beginning and/or ending parenthesis");
         return(ERROR);
      }

   } while (iGetNextActionBlock(&cFskActionBlock,
                                &uiCurrentEvent, &uiCurrentState) == 0);

   return(0);
}



/*----------------------------------------------------------------------------*\
|                                                                              |
| Function:  Generate the FSK source code                                      |
|                                                                              |
| Returns:   INT                                                               |
|            non-zero value on error                                           |
|                                                                              |
\*----------------------------------------------------------------------------*/
INT
Fsk::
iGenerateCSource(CHAR* pchOutputFile,
                                     // IN  name of file to dump source code to
                 CHAR* pchFskName,    
                                     // IN  name of the FSK
                 UINT  uiTransitionHistory,   
                                     // IN  transition history
                 INT   iOverWrite    // IN  overwrite the file if it exists
                )
{ 


   INT   iSecondStringFound;
   UINT  uiLoop;
   UINT  uiLoop2;
   UINT  uiLoop3;
   UINT  uiIndex;
   UINT  uiNumEvents;
   UINT  uiNumStates;
   UINT  uiCurrentEvent; 
   UINT  uiCurrentState;
   UINT  uiTemp;
   CHAR  achHeaderFile[MAX_FILE_AND_PATH_LEN];
   CHAR  achText[100];
   CHAR  achHeaderIfdef[100];
   CHAR  achFunctionName[FUNCTION_NAME_LEN+4];
   CHAR  achCompressedFunctionName[FUNCTION_NAME_LEN+4];
   CHAR  achParameter[FUNCTION_NAME_LEN+4];
   CHAR  achHeaders[HEADER_FILES_STRING_LEN+4];
   CHAR* pchTemp;
   CHAR* pchTemp2;
   CHAR* pchTemp3;
   FILE* phHeaderFile;
   FILE* phSourceFile;
   FskEvent cFskEvent;
   FskState cFskState;
   FskActionBlock cFskActionBlock;
   DynamicArray<CHAR*>  acpDuplicateArray(1); 
   DynamicArray<CHAR*>  acpFunctionArray(1); 
   DynamicMatrix<UINT>* pcFunctionCaseValue;


                         // check for overwrite
   if (!iOverWrite)
   {
      phSourceFile = fopen(pchOutputFile, "r");
      if (phSourceFile != NULL)
      {
         fclose(phSourceFile);
         sprintf(achAppErrorString, "File %s exists", pchOutputFile);
         return(ERROR);
      }
   }

   strcpy(achHeaderFile, pchOutputFile);
   achHeaderFile[strlen(achHeaderFile)-1] = 'h';

   
   phSourceFile = fopen(pchOutputFile, "w");
   if (phSourceFile == NULL)
   {
      if (errno == EACCES)
      {
         sprintf(achAppErrorString, "Could not write to %s, permission denied",
                 pchOutputFile);
      }
      else
      {
         sprintf(achAppErrorString, "Could not write to %s, errno = %d", 
                 pchOutputFile, errno);
      }
      return(errno);
   }

   phHeaderFile = fopen(achHeaderFile, "w");
   if (phHeaderFile == NULL)
   {
      fclose(phSourceFile);
      if (errno == EACCES)
      {
         sprintf(achAppErrorString, "Could not write to %s, permission denied",
                 achHeaderFile);
      }
      else
      {
         sprintf(achAppErrorString, "Could not write to %s", achHeaderFile);
      }
      return(errno);
   }

   uiNumStates = uiGetNumStates();
   uiNumEvents = uiGetNumEvents();

                         // create the header file
   fprintf(phHeaderFile, 
           "/*************************************************************\n");
   fprintf(phHeaderFile, "***\n***\n");

   sprintf(achText, "*** EVENT HEADER FILE FOR : %s\n", pchFskName);
   fprintf(phHeaderFile, "%s", achText);

   fprintf(phHeaderFile, "***\n");

   sprintf(achText, "*** LAST MODIFIED ON      : %s\n", achLastModifiedDate);
   fprintf(phHeaderFile, "%s", achText);

   fprintf(phHeaderFile, 
           "***\n*** Do NOT edit this file, it was automatically generated\n");

   sprintf(achText, 
           "*** by the FSM CASE Tool version %s\n", achFskFileVersionString);
   fprintf(phHeaderFile, "%s", achText);

   fprintf(phHeaderFile, "***\n");
   fprintf(phHeaderFile, 
           "***   FSKC tool Copyright (C) 1997, 2000, 2001  James A. Cureington\n");
   fprintf(phHeaderFile, 
           "***                              tonycu@users.sourceforge.net\n");
   fprintf(phHeaderFile, "***\n");
   fprintf(phHeaderFile, 
           "**************************************************************/\n");
   fprintf(phHeaderFile, "\n");

   strcpy(achHeaderIfdef, pchFskName);
   achHeaderIfdef[strlen(achHeaderIfdef)-4] = '_'; 

   sprintf(achText, "#ifndef %s_H\n", achHeaderIfdef);
   fprintf(phHeaderFile, "%s", achText);
   sprintf(achText, "#define %s_H\n", achHeaderIfdef);
   fprintf(phHeaderFile, "%s", achText);
   fprintf(phHeaderFile, "\n");

   for (uiLoop=0; uiLoop<uiNumEvents; uiLoop++)
   {
      if (iGetEventData(uiLoop, &cFskEvent))
      {
         fclose(phHeaderFile);
         fclose(phSourceFile);
         LOG_ERROR("Internal error");
         return(ERROR);
      }

      sprintf(achText, "#define %s    %%%du\n", 
              cFskEvent.pchName, 
              STATE_NAME_LEN - strlen(cFskEvent.pchName));

      fprintf(phHeaderFile, achText,  uiLoop);
   }

   fprintf(phHeaderFile, "\n");
   sprintf(achText, "#endif /* %s_H */\n", achHeaderIfdef);
   fprintf(phHeaderFile, "%s", achText);

   fclose(phHeaderFile);



                         // create the source file
   fprintf(phSourceFile, 
           "/*************************************************************\n");
   fprintf(phSourceFile, "***\n***\n");

   sprintf(achText, "*** FINITE STATE KERNEL FOR : %s\n", pchFskName);
   fprintf(phSourceFile, "%s", achText);

   fprintf(phSourceFile, "***\n");

   sprintf(achText, "*** LAST MODIFIED ON        : %s\n", achLastModifiedDate);
   fprintf(phSourceFile, "%s", achText);

   fprintf(phSourceFile, 
           "***\n*** Do NOT edit this file, it was automatically generated\n");
   sprintf(achText, 
           "*** by the FSM CASE Tool version %s\n", achFskFileVersionString);
   fprintf(phSourceFile, "%s", achText);

   fprintf(phSourceFile, "***\n");
   fprintf(phSourceFile, 
           "***   FSKC tool Copyright (C) 1997, 2000, 2001  James A. Cureington\n");
   fprintf(phSourceFile, 
           "***                              tonycu@users.sourceforge.net\n");
   fprintf(phSourceFile, "***\n");
   fprintf(phSourceFile, 
           "**************************************************************/\n");
   fprintf(phSourceFile, "\n");


   fprintf(phSourceFile, "#include <stdio.h>\n");

   fprintf(phSourceFile, "\n");

   sprintf(achText, "#define NUM_EVENTS          %u", uiNumEvents);
   fprintf(phSourceFile, "%s\n", achText);

   sprintf(achText, "#define NUM_STATES          %u", uiNumStates);
   fprintf(phSourceFile, "%s\n", achText);

   if (uiTransitionHistory > 0)
   {
      sprintf(achText, "#define TRANSITION_HISTORY  %u", uiTransitionHistory);
      fprintf(phSourceFile, "%s\n", achText);

      fprintf(phSourceFile, "\n");
      fprintf(phSourceFile, 
        "static unsigned int aauiTransitionHistory[TRANSITION_HISTORY][3];\n");
      fprintf(phSourceFile, 
        "static unsigned int uiTransitionIndex = 0;\n");
   }
   fprintf(phSourceFile, "\n\n");


                         // print user header files 
   fprintf(phSourceFile, "/**********************************\n");
   fprintf(phSourceFile, " ***  user header files         ***\n");
   fprintf(phSourceFile, " **********************************/\n");
   fprintf(phSourceFile, "\n");
   
   strcpy(achHeaders, achHeaderFiles);
  
   if (achHeaders != '\0')
   {
                         // remove all new-lines
      while ((pchTemp = strchr(achHeaders, '\n')) != NULL)
      {
         *pchTemp =  ' ';
      }

      if ((pchTemp = strtok(achHeaders, " ")) != NULL)
      {
         do
         {
            sprintf(achText, "#include %s", pchTemp);
            fprintf(phSourceFile, "%s\n", achText);

         } while ((pchTemp = strtok(NULL, " ")) != NULL);
      }
      else
      {
                         // check for single header file
         for (uiLoop=0; uiLoop<strlen(achHeaders); uiLoop++)
         {
            if (achHeaders[uiLoop] != ' ')
            {
               sprintf(achText, "#include %s", achHeaders[uiLoop]);
               fprintf(phSourceFile, "%s\n", achText);
               break;
            }
         }
      }
   }

   fprintf(phSourceFile, "\n\n");


                         // print states 
   fprintf(phSourceFile, "/**********************************\n");
   fprintf(phSourceFile, " ***  states, for comments only ***\n");
   fprintf(phSourceFile, " **********************************/\n");
   fprintf(phSourceFile, "\n");
   fprintf(phSourceFile, "#ifdef DONT_DO\n");

   for (uiLoop=0; uiLoop<uiNumStates; uiLoop++)
   {
      if (iGetStateData(uiLoop, &cFskState))
      {
         fclose(phSourceFile);
         LOG_ERROR("Internal error");
         return(ERROR);
      }

      sprintf(achText, "#define %s_S    %%%du\n", 
              cFskState.pchName, 
              STATE_NAME_LEN - strlen(cFskState.pchName));

      fprintf(phSourceFile, achText,  uiLoop);
   }

   fprintf(phSourceFile, "#endif /* DONT_DO */\n");
   fprintf(phSourceFile, "\n");


   fprintf(phSourceFile, "/**********************************\n");
   fprintf(phSourceFile, " ***  local variables           ***\n");
   fprintf(phSourceFile, " **********************************/\n");


   acpDuplicateArray[0] = NULL; 

                         // 
                         // process all paremters in each 
                         // action-block
                         // 
   if (iGetFirstActionBlock(&cFskActionBlock,
                            &uiCurrentEvent, &uiCurrentState))
   {
                         // this is an internal error...
      fclose(phSourceFile);
      sprintf(achAppErrorString, "Could not get action-block data, %u, %u",
              uiCurrentEvent, uiCurrentState);
      LOG_ERROR(achAppErrorString);
      return(ERROR);
   }

   do
   {
      strcpy(achFunctionName, cFskActionBlock.pchFunction);

                         // remove all new-lines from function
      while ((pchTemp = strchr(achFunctionName, '\n')) != NULL)
      {
         *pchTemp =  ' ';
      }

                         // process the parameter list
      if ((pchTemp = strrchr(achFunctionName, ')')) != NULL)
      {
         *pchTemp =  '\0';
      }
      else
      {
         fclose(phSourceFile);
         sprintf(achAppErrorString, 
                 "Action-block parameter list, %u, %u, does not contain a )",
                 uiCurrentEvent, uiCurrentState);
         return(ERROR);
      }

      if ((pchTemp = strchr(achFunctionName, '(')) != NULL)
      {
         *pchTemp =  '\0';
      }
      else
      {
         fclose(phSourceFile);
         sprintf(achAppErrorString, 
                 "Action-block parameter list, %u, %u, does not contain a (",
                 uiCurrentEvent, uiCurrentState);
         return(ERROR);
      }
      pchTemp++;

      if (*pchTemp == '\0')
      {
         continue;
      }
                         // 
                         // process each action block function
                         // parameter 
                         // add each parameter to the list if it's 
                         // not already there, also remove any 
                         // double+ spaces
                         // 
      pchTemp2 = strtok(pchTemp, ",");
      do
      {
                                  // if only one parameter, proces it
         if (pchTemp2 == NULL)
         {
            pchTemp2 = pchTemp;
         }

                         // a string with a "
         if (strchr(pchTemp2, '"') != NULL)
         {
            continue;
         }

                         // make sure we have at least two strings in the
                         // parameter.  if not, its a single parameter like
                         // a constant or macro, i.e. 1, MAX_XXX, etc.
                         // we are looking for two strings "type name", 
                         // such as "int i"
         for (uiLoop=0, iSecondStringFound=0;
              ((uiLoop<strlen(pchTemp2)) && (!iSecondStringFound)); 
              uiLoop++)
         {
            if (pchTemp2[uiLoop] != ' ')
            { 
                         // found the first string, see if there is a second
               if ((pchTemp3 = strchr(&pchTemp2[uiLoop], ' ')) == NULL)
               {
                  break;
               }
            
               while (*pchTemp3 != '\0')
               {
                  if (*pchTemp3 != ' ')
                  {
                     iSecondStringFound = 1;
                     break;
                  }
                  pchTemp3++;
               }
            } 
         }

         if (!iSecondStringFound)
         {
                         // a single string in parameter
            continue;
         }
 

                         // if parameter has already been printed
                         // don't print it again
                         // parameters in duplicate array have ALL spaces
                         // removed...
         strcpy(achParameter, pchTemp2);
         for (uiLoop=0, uiIndex=0; uiLoop<strlen(achParameter); uiLoop++)
         {
            if (achParameter[uiLoop] != ' ')
            { 
               achParameter[uiIndex] = achParameter[uiLoop];
               uiIndex++;
            } 
         }
 
                         // all spaces, no parameter
         if (uiIndex ==  0)
         {
            continue;
         }

         achParameter[uiIndex] = '\0';

         uiTemp = acpDuplicateArray.uiGetSize();

                         // don't compare acpVar.... containing NULL
                         // ptr, causes core dump.  hence, need to
                         // add if...NULL...
         if (acpDuplicateArray[0] != NULL)
         {
            for (uiLoop=0; uiLoop<uiTemp; uiLoop++)
            {
                if (strcmp(acpDuplicateArray[uiLoop], achParameter) == 0)
                {
                         // parameter is already in list
                   break;
                }
            }

                         // parameter is already in list
            if (uiLoop<uiTemp)
            {
               continue;
            }

                         // add parameter to list
            if (acpDuplicateArray.iInsertCell(uiTemp-1, AFTER))
            {
               fclose(phSourceFile);
               return(ERROR);
            }
         }

         uiTemp = acpDuplicateArray.uiGetSize();
         acpDuplicateArray[uiTemp-1] = (CHAR*) malloc(sizeof(CHAR) * 
                                                     FUNCTION_NAME_LEN+4);
         CHECK_MALLOC(acpDuplicateArray[uiTemp-1]);
         strcpy(acpDuplicateArray[uiTemp-1], achParameter);


                         // remove leading spaces
         while (*pchTemp2 == ' ')
         {
            pchTemp2++;
         }
                         // print the parameter to the source file
         fprintf(phSourceFile, "%s;\n", pchTemp2);


      } while ((pchTemp2 = strtok(NULL, ",")) != NULL);

   } while (iGetNextActionBlock(&cFskActionBlock,
                                &uiCurrentEvent, &uiCurrentState) == 0);

                         // free the parameter list, keeping one
                         // element in array
   uiTemp = acpDuplicateArray.uiGetSize();
   for (uiLoop=(uiTemp-1); uiLoop>0; uiLoop--)
   {
      free(acpDuplicateArray[uiLoop]);
      if (acpDuplicateArray.iDeleteCell(uiLoop))
      {
         fclose(phSourceFile);
         sprintf(achAppErrorString, 
                 "Internal Error: could delete array element");
         LOG_ERROR(achAppErrorString);
         return(ERROR);
      }
   }

   free(acpDuplicateArray[0]);


                         // ok, parameter list has been printed
   fprintf(phSourceFile, "\n\n");


                         // 
                         // print the Next State Table
                         // 
   fprintf(phSourceFile, "\n\n");
   fprintf(phSourceFile, "/**********************************\n");
   fprintf(phSourceFile, " ***  Next State Table          ***\n");
   fprintf(phSourceFile, " **********************************/\n");
   fprintf(phSourceFile, "static unsigned int\n");
   fprintf(phSourceFile, "aauiNextStateTable[NUM_EVENTS][NUM_STATES] = {\n");


   fprintf(phSourceFile, "     {");

   for (uiLoop=0; uiLoop<uiNumEvents; uiLoop++)
   {
      for (uiLoop2=0; uiLoop2<uiNumStates; uiLoop2++)
      {
         if (iGetActionBlockData(uiLoop, uiLoop2, &cFskActionBlock))
         {
            sprintf(achAppErrorString, 
                    "Could not get action block data, %u, %u",
                    uiLoop, uiLoop2);
            LOG_ERROR(achAppErrorString);
            fclose(phSourceFile);
            return(ERROR);
         }

                         // don't print "," on first state
         
         if (uiLoop2 == 0)
         {
            sprintf(achText, "%3u", *cFskActionBlock.puiNextState);
         }
         else
         {
            sprintf(achText, ", %3u", *cFskActionBlock.puiNextState);
         }

         fprintf(phSourceFile, "%s", achText);
      }

                         // print brace for last event in each row
      if (uiLoop != (uiNumEvents-1))
      {
         fprintf(phSourceFile, "},\n");
         fprintf(phSourceFile, "     {");
      }
      else
      {
         fprintf(phSourceFile, "}\n");
         fprintf(phSourceFile, "};\n");
      }
   } 

                         // 
                         // print the Action Block Table
                         // 
   fprintf(phSourceFile, "\n\n");
   fprintf(phSourceFile, "/**********************************\n");
   fprintf(phSourceFile, " ***  Action-block Table        ***\n");
   fprintf(phSourceFile, " **********************************/\n");
   acpDuplicateArray[0] = NULL; 
   acpFunctionArray[0]  = NULL; 

   pcFunctionCaseValue = new DynamicMatrix<UINT> (uiNumEvents, uiNumStates);
   CHECK_NEW(pcFunctionCaseValue);

                         // 
                         // process all functions in each action-block
                         // the duplicate array maintains all the 
                         // functions processed so far with no
                         // embedded spaces, while the function 
                         // array maintains the associated function
                         // as it is in the action-block including spaces, etc.
                         // 
   if (iGetFirstActionBlock(&cFskActionBlock,
                            &uiCurrentEvent, &uiCurrentState))
   {
                         // this is an internal error...
      fclose(phSourceFile);
      sprintf(achAppErrorString, "Could not get action-block data, %u, %u",
              uiCurrentEvent, uiCurrentState);
      LOG_ERROR(achAppErrorString);
      return(ERROR);
   }

   do
   {
      strcpy(achCompressedFunctionName, cFskActionBlock.pchFunction);

                         // remove all new-lines from function
      while ((pchTemp = strchr(achCompressedFunctionName, '\n')) != NULL)
      {
         *pchTemp =  ' ';
      }
                         // save off function name at this point
      strcpy(achFunctionName, achCompressedFunctionName);

                         // remove all spaces
      for (uiLoop=0, uiIndex=0; 
           uiLoop<strlen(achCompressedFunctionName); 
           uiLoop++)
      {
         if (achCompressedFunctionName[uiLoop] != ' ')
         { 
            achCompressedFunctionName[uiIndex] = 
                                      achCompressedFunctionName[uiLoop];
            uiIndex++;
         } 
      }
      achCompressedFunctionName[uiIndex] = '\0';

                         // add a ";" to the end if needed
      if (achCompressedFunctionName[strlen(achCompressedFunctionName)-1] 
                                                                      != ';')
      {
         strcat(achCompressedFunctionName, ";");
      }
                         // if function is iterative add a 1 to the end, else
                         // add a 0.  we need to keep track of this with the
                         // same function since an action-block with a function
                         // can be iterative and non-recusrsive
      if (*cFskActionBlock.piIterative)
      {
         strcat(achCompressedFunctionName, "1");
      }
      else
      {
         strcat(achCompressedFunctionName, "0");
      }

 
                         // if function has already been processed
                         // don't processed it again
      uiTemp = acpDuplicateArray.uiGetSize();

                         // don't compare acpVar.... containing NULL
                         // ptr, causes core dump.  hence, need to
                         // add if...NULL...
      if (acpDuplicateArray[0] != NULL)
      {
         for (uiLoop=0; uiLoop<uiTemp; uiLoop++)
         {
             if (strcmp(acpDuplicateArray[uiLoop], 
                        achCompressedFunctionName) == 0)
             {
                         // function is already in list
                         // set the case value for this function
                pcFunctionCaseValue->rxCellData(uiCurrentEvent, 
                                                uiCurrentState) = uiLoop;
                break;
             }
         }

                         // function is already in list
         if (uiLoop<uiTemp)
         {
            continue;
         }

                         // add function to lists
                         // duplicate and function array are the same size
         if (acpDuplicateArray.iInsertCell(uiTemp-1, AFTER))
         {
            fclose(phSourceFile);
            return(ERROR);
         }

         if (acpFunctionArray.iInsertCell(uiTemp-1, AFTER))
         {
            fclose(phSourceFile);
            return(ERROR);
         }
      }

      uiTemp = acpDuplicateArray.uiGetSize();

      acpDuplicateArray[uiTemp-1] = (CHAR*) malloc(sizeof(CHAR) * 
                                                  FUNCTION_NAME_LEN+4);
      CHECK_MALLOC(acpDuplicateArray[uiTemp-1]);

      acpFunctionArray[uiTemp-1] = (CHAR*) malloc(sizeof(CHAR) * 
                                                  FUNCTION_NAME_LEN+4);
      CHECK_MALLOC(acpFunctionArray[uiTemp-1]);


      strcpy(acpDuplicateArray[uiTemp-1], achCompressedFunctionName);


                         // remove leading spaces from function name
      pchTemp = achFunctionName;

      if (*pchTemp == ' ')
      {
         while (*pchTemp == ' ')
         {
            pchTemp++;
         }
         memmove(achFunctionName, pchTemp, strlen(pchTemp)+1);
      }

      pchTemp = strchr(achFunctionName, '(');
      pchTemp++;

                         // remove definition from each variable
      while ((pchTemp2 = strchr(pchTemp, ',')) != NULL)
      {
                         // clobber the "," so we don't pick it up next time
         *pchTemp2 = ' ';
         pchTemp2--;

                         // backup to last char of name
         while (*pchTemp2 == ' ')
         {
            if (pchTemp2 == pchTemp)
            {
                fclose(phSourceFile);
                sprintf(achAppErrorString, 
                        "Bad parameter list in action-block., %u, %u",
                        uiCurrentEvent, uiCurrentState);
                return(ERROR);
            }
            pchTemp2--;
         }

                         // check for "
         if (*pchTemp2 == '"')
         {
            pchTemp2--;
            while (*pchTemp2 != '"')
            {
               if (pchTemp2 == pchTemp)
               {
                   fclose(phSourceFile);
                   sprintf(achAppErrorString, 
                           "Bad parameter list in action-block.., %u, %u",
                           uiCurrentEvent, uiCurrentState);
                   return(ERROR);
               }
               pchTemp2--;
            }

                         // copy the string
            *pchTemp = '"';
            pchTemp++;
            pchTemp2++;
            while (*pchTemp2 != '"')
            {
               *pchTemp = *pchTemp2;
               pchTemp++;
               pchTemp2++;
            }

            *pchTemp = '"';
            pchTemp++;
            *pchTemp = ',';
            pchTemp++;
            continue;
         }

                         // backup to char just before the name
         while (*pchTemp2 != ' ')
         {
            if (pchTemp2 == pchTemp)
            {
                         // must be a macro/constant value
                         // backup one for the increment outside this loop
                pchTemp2--;
                break;
            }
            pchTemp2--;
         }

         pchTemp2++;
                         // now copy the name
         while (*pchTemp2 != ' ')
         {
            *pchTemp = *pchTemp2;
            pchTemp++;
            pchTemp2++;
         }

         *pchTemp = ',';
         pchTemp++;
      }

                         // process the last parameter
      if ((pchTemp2 = strrchr(achFunctionName, ')')) == NULL)
      {
          fclose(phSourceFile);
          sprintf(achAppErrorString, 
                  "Bad parameter list in action-block...., %u, %u",
                  uiCurrentEvent, uiCurrentState);
          return(ERROR);
      }

      *pchTemp2 = ' ';

                         // if parm list is (), the ptrs will be equal now...
                         // hence, no params to process...
      if (pchTemp != pchTemp2)
      {
         pchTemp2--;

                         // backup to last char of name
         while (*pchTemp2 == ' ')
         {
            if (pchTemp2 == pchTemp)
            {
                         // parm list is "( <spaces> )" or "(x<spaces>)"
               break;
            }
            pchTemp2--;
         }

                         // check for "
         if (*pchTemp2 == '"')
         {
            pchTemp2--;
            while (*pchTemp2 != '"')
            {
               if (pchTemp2 == pchTemp)
               {
                   fclose(phSourceFile);
                   sprintf(achAppErrorString, 
                           "Bad parameter list in action-block......, %u, %u",
                           uiCurrentEvent, uiCurrentState);
                   return(ERROR);
               }
               pchTemp2--;
            }

                         // copy the string
            *pchTemp = '"';
            pchTemp++;
            pchTemp2++;
            while (*pchTemp2 != '"')
            {
               *pchTemp = *pchTemp2;
               pchTemp++;
               pchTemp2++;
            }
   
            *pchTemp = '"';
            pchTemp++;
         }
         else if (pchTemp2 != pchTemp)
         {
                         // parm list is not empty

                         // backup to char just before the name
            while (*pchTemp2 != ' ')
            {
               if (pchTemp2 == pchTemp)
               {
                         // must be a macro/constant value
                   break;
               }
               pchTemp2--;
            }
   
            pchTemp2++;
                         // now copy the name
            while (*pchTemp2 != ' ')
            {
               *pchTemp = *pchTemp2;
               pchTemp++;
               pchTemp2++;
            }
         }
         else /* (pchTemp2 == pchTemp) */
         {
            if (*pchTemp != ' ')
            {
                         // parm list is "(x<spaces>)", i..e (1   ), (1 ), etc.
               pchTemp++;
            }
         }
      }

      strcpy(pchTemp, ");");


                         // if function is iterative add a 1 to the end, else
                         // add a 0.  we need to keep track of this with the
                         // same function since an action-block with a function
                         // can be iterative and non-recusrsive
      if (*cFskActionBlock.piIterative)
      {
         strcat(achFunctionName, "1");
      }
      else
      {
         strcat(achFunctionName, "0");
      }

                         // store the function name in the array
      strcpy(acpFunctionArray[uiTemp-1], achFunctionName);

                         // set the case value for this function
      pcFunctionCaseValue->rxCellData(uiCurrentEvent, 
                                      uiCurrentState) = uiTemp-1;


   } while (iGetNextActionBlock(&cFskActionBlock,
                                &uiCurrentEvent, &uiCurrentState) == 0);


                         // 
                         // now, print the Action-block Table
                         // 
   fprintf(phSourceFile, "static unsigned int\n");
   fprintf(phSourceFile, "aauiActionBlockTable[NUM_EVENTS][NUM_STATES] = {\n");


   fprintf(phSourceFile, "     {");

   for (uiLoop=0; uiLoop<uiNumEvents; uiLoop++)
   {
      for (uiLoop2=0; uiLoop2<uiNumStates; uiLoop2++)
      {
                         // don't print "," on first state
         if (uiLoop2 == 0)
         {
            sprintf(achText, "%3u", pcFunctionCaseValue->rxCellData(uiLoop, 
                                                                   uiLoop2));
         }
         else
         {
            sprintf(achText, ", %3u", pcFunctionCaseValue->rxCellData(uiLoop, 
                                                                     uiLoop2));
         }

         fprintf(phSourceFile, "%s", achText);
      }

                         // print brace for last event in each row
      if (uiLoop != (uiNumEvents-1))
      {
         fprintf(phSourceFile, "},\n");
         fprintf(phSourceFile, "     {");
      }
      else
      {
         fprintf(phSourceFile, "}\n");
         fprintf(phSourceFile, "};\n");
      }
   } 


                         // free the duplicate array
   uiTemp = acpDuplicateArray.uiGetSize();
   for (uiLoop=(uiTemp-1); uiLoop>0; uiLoop--)
   {
      free(acpDuplicateArray[uiLoop]);
      if (acpDuplicateArray.iDeleteCell(uiLoop))
      {
         fclose(phSourceFile);
         sprintf(achAppErrorString, 
                 "Internal Error: could delete array element");
         LOG_ERROR(achAppErrorString);
         return(ERROR);
      }
   }
   free(acpDuplicateArray[0]);

   delete(pcFunctionCaseValue);


                         // 
                         // create static fsk logging function
                         // 
   fprintf(phSourceFile, "\n");
   fprintf(phSourceFile, "/********************************************\n");
   fprintf(phSourceFile, " ***  static fsk history logging function ***\n");
   fprintf(phSourceFile, " ********************************************/\n");
   fprintf(phSourceFile, "static unsigned int\n");
   fprintf(phSourceFile, "uiPrintFskHistory(unsigned int uiEventToReturn)\n");
   fprintf(phSourceFile, "{\n");
   fprintf(phSourceFile, "\n");
   if (uiTransitionHistory > 0)
   {
      fprintf(phSourceFile, "   static unsigned int uiLoop;\n");
      fprintf(phSourceFile, "   static unsigned int uiIndex;\n");
      fprintf(phSourceFile, "\n");
      fprintf(phSourceFile, "   /* dump transitions */\n");
      fprintf(phSourceFile, "   for (uiLoop=0,\n");
      fprintf(phSourceFile, "        uiIndex=uiTransitionIndex;\n");
      fprintf(phSourceFile, "        uiLoop<TRANSITION_HISTORY;\n");
      fprintf(phSourceFile, "        uiLoop++)\n");
      fprintf(phSourceFile, "   {\n");
      fprintf(phSourceFile, "      ");
      fprintf(phSourceFile, 
        "printf(\"current state = %%8u, event = %%8u, case = %%8u\\n\",\n");
      fprintf(phSourceFile, "              ");
      fprintf(phSourceFile, "aauiTransitionHistory[uiIndex][1],\n");
      fprintf(phSourceFile, "              ");
      fprintf(phSourceFile, "aauiTransitionHistory[uiIndex][0],\n");
      fprintf(phSourceFile, "              ");
      fprintf(phSourceFile, "aauiTransitionHistory[uiIndex][2]);\n");
      fprintf(phSourceFile, "\n");
      fprintf(phSourceFile, "      uiIndex++;\n");
      fprintf(phSourceFile, "      ");
      fprintf(phSourceFile, "uiIndex %%= TRANSITION_HISTORY;\n");
      fprintf(phSourceFile, "\n");
      fprintf(phSourceFile, "   }\n");
   }
   fprintf(phSourceFile, "   return(uiEventToReturn);\n");
   fprintf(phSourceFile, "}\n");

   fprintf(phSourceFile, "\n\n");


                         // 
                         // create static fsk abort function
                         // 
   fprintf(phSourceFile, "\n");
   fprintf(phSourceFile, "/***************************\n");
   fprintf(phSourceFile, " ***  abort fsk function ***\n");
   fprintf(phSourceFile, " ***************************/\n");
   fprintf(phSourceFile, "static void\n");
   fprintf(phSourceFile, "vAbortFsk()\n");
   fprintf(phSourceFile, "{\n");
   fprintf(phSourceFile, "   static unsigned int uiReturnValue;\n");
   fprintf(phSourceFile, "   uiReturnValue = uiPrintFskHistory(0);\n");
   fprintf(phSourceFile, "   abort();\n");
   fprintf(phSourceFile, "}\n");

   fprintf(phSourceFile, "\n\n");




                         // 
                         // create static fsk no-op function
                         // 
   fprintf(phSourceFile, "\n");
   fprintf(phSourceFile, "/***********************************\n");
   fprintf(phSourceFile, " ***  static fsk uiNoOp function ***\n");
   fprintf(phSourceFile, " ***********************************/\n");
   fprintf(phSourceFile, "static unsigned int\n");
   fprintf(phSourceFile, "uiNoOp(unsigned int uiNoOpReturnVal)\n");
   fprintf(phSourceFile, "{\n");
   fprintf(phSourceFile, "   return(uiNoOpReturnVal);\n");
   fprintf(phSourceFile, "}\n");

   fprintf(phSourceFile, "\n\n");


                         // 
                         // print the FSK function
                         // 

   fprintf(phSourceFile, "\n");
   fprintf(phSourceFile, "/**********************************\n");
   fprintf(phSourceFile, " ***  FSK FUNCTION              ***\n");
   fprintf(phSourceFile, " **********************************/\n");

   fprintf(phSourceFile, "%s\n", pchReturnType);

   sprintf(achText, "%s", pchFskName);
   achText[strlen(achText)-4] = '\0';
   fprintf(phSourceFile, "%s(unsigned int uiEvent)\n", achText);
   fprintf(phSourceFile, "{\n");
   fprintf(phSourceFile, "\n");

   if (uiTransitionHistory > 0)
   {
      fprintf(phSourceFile, 
              "   static unsigned int uiInitTransitionHistory = 1;\n");
   }

   fprintf(phSourceFile, "   static unsigned int uiState = 0;\n");
   fprintf(phSourceFile, "   static unsigned int uiActionBlockFunction;\n");

   fprintf(phSourceFile, "\n");
   if (uiTransitionHistory > 0)
   {
      fprintf(phSourceFile, "   /* init transition history */\n");
      fprintf(phSourceFile, "   if (uiInitTransitionHistory == 1)\n");
      fprintf(phSourceFile, "   {\n");
      fprintf(phSourceFile, "      static unsigned int uiLoop;\n");
      fprintf(phSourceFile, "      uiInitTransitionHistory = 0;\n");
      fprintf(phSourceFile, "      for (uiLoop=0;\n");
      fprintf(phSourceFile, "           uiLoop<TRANSITION_HISTORY;\n");
      fprintf(phSourceFile, "           uiLoop++)\n");
      fprintf(phSourceFile, "      {\n");
      fprintf(phSourceFile, "         ");
      fprintf(phSourceFile, 
              "aauiTransitionHistory[uiLoop][0] = NUM_EVENTS;\n");
      fprintf(phSourceFile, "         ");
      fprintf(phSourceFile, 
              "aauiTransitionHistory[uiLoop][1] = NUM_STATES;\n");
      fprintf(phSourceFile, "         ");
      fprintf(phSourceFile, 
              "aauiTransitionHistory[uiLoop][2] = NUM_EVENTS;\n");
      fprintf(phSourceFile, "      }\n");
      fprintf(phSourceFile, "   }\n");
      fprintf(phSourceFile, "\n");
   }

   fprintf(phSourceFile, "\n");
   fprintf(phSourceFile, "   /* while loop emulates iterative calls */\n");
   fprintf(phSourceFile, "   /* without overhead of stack pushes... */\n");
   fprintf(phSourceFile, "   while(1)\n");
   fprintf(phSourceFile, "   {\n");

   fprintf(phSourceFile, "    \n");
   fprintf(phSourceFile, "      if (uiEvent >= NUM_EVENTS)\n");
   fprintf(phSourceFile, "      {\n");



   if (*piUserDefinedErrorHandler)
   {
      fprintf(phSourceFile, 
              "         /* user defined error handler in 0,0 */\n");
      fprintf(phSourceFile, "         uiEvent = 0;\n");
      fprintf(phSourceFile, "         uiState = 0;\n");
   }
   else
   {
      fprintf(phSourceFile, "         ");
      fprintf(phSourceFile, 
           "printf(\"Internal Error in: %s\\n\");\n", pchFskName);

      fprintf(phSourceFile, "         ");
      fprintf(phSourceFile, 
        "printf(\"uiEvent>=NUM_EVENTS (%%u>=%%u)\\n\", uiEvent, NUM_EVENTS);\n");
      fprintf(phSourceFile, "         vAbortFsk();\n");
   }


   fprintf(phSourceFile, "      }\n");
   fprintf(phSourceFile, "\n");
   fprintf(phSourceFile, "      ");
   fprintf(phSourceFile, 
        "uiActionBlockFunction = aauiActionBlockTable[uiEvent][uiState];\n");


   fprintf(phSourceFile, "\n");

   if (uiTransitionHistory > 0)
   {
      fprintf(phSourceFile, "      /* save transitions */\n");
      fprintf(phSourceFile, "      ");
      fprintf(phSourceFile, 
        "aauiTransitionHistory[uiTransitionIndex][0] = uiEvent;\n");
      fprintf(phSourceFile, "      ");
      fprintf(phSourceFile, 
        "aauiTransitionHistory[uiTransitionIndex][1] = uiState;\n");
      fprintf(phSourceFile, "      ");
      fprintf(phSourceFile, 
        "aauiTransitionHistory[uiTransitionIndex][2] = ");
      fprintf(phSourceFile, "uiActionBlockFunction;\n");
      fprintf(phSourceFile, "      ");
      fprintf(phSourceFile, 
        "uiTransitionIndex++;\n");
      fprintf(phSourceFile, "      ");
      fprintf(phSourceFile, 
        "uiTransitionIndex %%= TRANSITION_HISTORY;\n");
      fprintf(phSourceFile, "\n");
   }

   fprintf(phSourceFile, "      ");
   fprintf(phSourceFile, "uiState = aauiNextStateTable[uiEvent][uiState];\n");


   fprintf(phSourceFile, "      switch(uiActionBlockFunction)\n");
   fprintf(phSourceFile, "      {\n");


                         // print the functions
   fprintf(phSourceFile, "\n");
   uiTemp = acpFunctionArray.uiGetSize();

   for (uiLoop=0; uiLoop<uiTemp; uiLoop++)
   {
      uiIndex = strlen(acpFunctionArray[uiLoop]);
      if (acpFunctionArray[uiLoop][uiIndex-1] == '0')
      {
                         // non-iterative function, remove the 0
         acpFunctionArray[uiLoop][uiIndex-1] = '\0';

                         // if vAbortFsk function, don't return because it
                         // is a void....

         if (strncmp(acpFunctionArray[uiLoop], "vAbortFsk(", 10) == 0)
         {
            fprintf(phSourceFile, "         ");
            fprintf(phSourceFile, "case %4u: %s\n", 
                                   uiLoop, acpFunctionArray[uiLoop]);
            fprintf(phSourceFile, "               ");
            fprintf(phSourceFile, "     break;   /* NOT REACHED */\n");
         }
         else
         {
                         // remove the ";"
            acpFunctionArray[uiLoop][uiIndex-2] = '\0';
            fprintf(phSourceFile, "         ");
            fprintf(phSourceFile, "case %4u: return(%s);\n", 
                                   uiLoop, acpFunctionArray[uiLoop]);
         }
      }
      else
      {
                         // iterative function, remove the 1
         acpFunctionArray[uiLoop][uiIndex-1] = '\0';
         fprintf(phSourceFile, "         ");
         fprintf(phSourceFile, "case %4u:\n", uiLoop);

         fprintf(phSourceFile, "            {");
         fprintf(phSourceFile, "               /* iterative action-block */\n");
         fprintf(phSourceFile, "               ");
         fprintf(phSourceFile, "uiEvent = %s\n", acpFunctionArray[uiLoop]);

         fprintf(phSourceFile, "               ");
         fprintf(phSourceFile, "break;\n");

         fprintf(phSourceFile, "            }\n");
      }
   }


   fprintf(phSourceFile, "         \n");
   fprintf(phSourceFile, "         default:\n");
   fprintf(phSourceFile, "            {\n");
   fprintf(phSourceFile, "               ");
   fprintf(phSourceFile, 
        "printf(\"Internal Error in: %s\\n\");\n", pchFskName);
   fprintf(phSourceFile, "               ");
   fprintf(phSourceFile, 
   "printf(\"No function to map event to, %%u\\n\",\n");
   fprintf(phSourceFile, "                      ");
   fprintf(phSourceFile, "uiActionBlockFunction);\n");
   fprintf(phSourceFile, "               vAbortFsk();\n");
   fprintf(phSourceFile, "            }\n");
   fprintf(phSourceFile, "      }\n");
   fprintf(phSourceFile, "   }\n");
   fprintf(phSourceFile, "}\n");

   fprintf(phSourceFile, "\n");


                         // free the parameter and function array
   uiTemp = acpFunctionArray.uiGetSize();
   for (uiLoop=(uiTemp-1); uiLoop>0; uiLoop--)
   {
      free(acpFunctionArray[uiLoop]);
      if (acpFunctionArray.iDeleteCell(uiLoop))
      {
         fclose(phSourceFile);
         sprintf(achAppErrorString, 
                 "Internal Error: could delete array element");
         LOG_ERROR(achAppErrorString);
         return(ERROR);
      }
   }

   free(acpFunctionArray[0]);



   fclose(phSourceFile);

   return(0);
}



/*----------------------------------------------------------------------------*\
|                                                                              |
| Function:  Print the action-block at the given location.                     |
|                                                                              |
| Returns:   INT                                                               |
|            non-zero value on error                                           |
|                                                                              |
\*----------------------------------------------------------------------------*/
INT
Fsk::
iPrintActionBlock(VirtualPage* pcVirtualPage,
                                     // OU  virtual page to place action in
                  UINT uiRowNumber,
                                     // IN  column location to place contents of
                                     //     action-block in
                  UINT uiColumnNumber,
                                     // IN  number to place in action-block 
                                     //     at the specified location      
                  FskActionBlock* pcFskActionBlock
                                     // IN  action-block to place at location
                )
{ 
   INT iLoop;

   CHAR  achFunctionName[FUNCTION_NAME_LEN];
   CHAR  aachTextBlock[PRINT_BLOCK_HEIGHT][PRINT_BLOCK_WIDTH];
   CHAR* pchTemp;


   strcpy(achFunctionName, pcFskActionBlock->pchFunction);

                                  // get function name from the function
   pchTemp = strchr(achFunctionName, '(');

   if (pchTemp != NULL)
   {
      *pchTemp = '\0';
   }

                                  // load the array
   for (iLoop=0; iLoop<(PRINT_BLOCK_HEIGHT-1); iLoop++) 
   {
      if (strlen(achFunctionName) >= ((iLoop)*(PRINT_BLOCK_WIDTH-2)))
      {
         strncpy(&(aachTextBlock[iLoop])[1], 
                 &(achFunctionName)[(iLoop)*(PRINT_BLOCK_WIDTH-2)],
                 PRINT_BLOCK_WIDTH-2);
         aachTextBlock[iLoop][PRINT_BLOCK_WIDTH-1] = '\0';
         aachTextBlock[iLoop][0] = ' ';
      }
      else
      {
         aachTextBlock[iLoop][0] = '\0';
      }
   }

   if (*pcFskActionBlock->puiNextState < MAX_FSK_STATES)
   {
      sprintf(aachTextBlock[PRINT_BLOCK_HEIGHT-1], " [%u]%s", 
              *pcFskActionBlock->puiNextState,
              ((*pcFskActionBlock->piIterative)? ",I" : ""));
   }
   else
   {
      sprintf(aachTextBlock[PRINT_BLOCK_HEIGHT-1], " [*]%s", 
              ((*pcFskActionBlock->piIterative)? ",I" : ""));
   }

                                  // if function name is more than allocated 
                                  // space in block, replace the last three 
                                  // visable chars with ...
   if (strlen(achFunctionName) >
       ((PRINT_BLOCK_HEIGHT-1)*(PRINT_BLOCK_WIDTH-2)))
   {
      strcpy(&aachTextBlock[PRINT_BLOCK_HEIGHT-2][PRINT_BLOCK_WIDTH-4],
             "...");
   }


                                  // copy the array to the action-block
                                  // location in the virtual page
   for (iLoop=0; iLoop<PRINT_BLOCK_HEIGHT; iLoop++) 
   {
      if (pcVirtualPage->iWriteLine((uiRowNumber*PRINT_BLOCK_HEIGHT)+
                                     HEADER_OFFSET+uiRowNumber+iLoop+1,
                                    (uiColumnNumber*PRINT_BLOCK_WIDTH)+
                                     (uiColumnNumber+1), 
                                    aachTextBlock[iLoop]))
      {
         LOG_ERROR("iWriteLine failed");
         return(ERROR);
      }
   }

   return(0);
} 



/*----------------------------------------------------------------------------*\
|                                                                              |
| Function:  Print the state-block at the given location.                      |
|                                                                              |
| Returns:   INT                                                               |
|            non-zero value on error                                           |
|                                                                              |
\*----------------------------------------------------------------------------*/
INT
Fsk::
iPrintStateBlock(VirtualPage* pcVirtualPage,
                                     // OU  virtual page to place state in
                 UINT uiStateBlockLocation,
                                     // IN  state location to place contents of
                                     //     state-block in
                 UINT uiStateNumber,
                                     // IN  number to place in state-block 
                                     //     at the specified location      
                 FskState* pcFskState
                                     // IN  state-block to place at location
                )
{ 
   INT iLoop;

   CHAR aachTextBlock[PRINT_BLOCK_HEIGHT][PRINT_BLOCK_WIDTH];

                                  // load the array
   sprintf(aachTextBlock[0], " %u", uiStateNumber);

   for (iLoop=1; iLoop<PRINT_BLOCK_HEIGHT; iLoop++) 
   {
      if (strlen(pcFskState->pchName) >= ((iLoop-1)*(PRINT_BLOCK_WIDTH-2)))
      {
         strncpy(&(aachTextBlock[iLoop])[1], 
                 &(pcFskState->pchName)[(iLoop-1)*(PRINT_BLOCK_WIDTH-2)],
                 PRINT_BLOCK_WIDTH-2);
         aachTextBlock[iLoop][PRINT_BLOCK_WIDTH-1] = '\0';
         aachTextBlock[iLoop][0] = ' ';
      }
      else
      {
         aachTextBlock[iLoop][0] = '\0';
      }
   }

                                  // if name is more than allocated space
                                  // in block, replace the last three 
                                  // visable chars with ...
   if (strlen(pcFskState->pchName) >
       ((PRINT_BLOCK_HEIGHT-1)*(PRINT_BLOCK_WIDTH-2)))
   {
      strcpy(&aachTextBlock[PRINT_BLOCK_HEIGHT-1][PRINT_BLOCK_WIDTH-4],
             "...");
   }


                                  // copy the array to the virtual page
   if (pcVirtualPage->iWriteLine(HEADER_OFFSET+1, 
                                 (uiStateBlockLocation*PRINT_BLOCK_WIDTH)+
                                 (uiStateBlockLocation+1), 
                                 aachTextBlock[0]))
   {
      LOG_ERROR("iWriteLine failed");
      return(ERROR);
   }

   for (iLoop=1; iLoop<PRINT_BLOCK_HEIGHT; iLoop++) 
   {
      if (pcVirtualPage->iWriteLine(HEADER_OFFSET+iLoop+1, 
                                    (uiStateBlockLocation*PRINT_BLOCK_WIDTH)+
                                    (uiStateBlockLocation+1), 
                                    aachTextBlock[iLoop]))
      {
         LOG_ERROR("iWriteLine failed");
         return(ERROR);
      }
   }


   return(0);
} 



/*----------------------------------------------------------------------------*\
|                                                                              |
| Function:  Print the event-block at the given location.                      |
|                                                                              |
| Returns:   INT                                                               |
|            non-zero value on error                                           |
|                                                                              |
\*----------------------------------------------------------------------------*/
INT
Fsk::
iPrintEventBlock(VirtualPage* pcVirtualPage,
                                     // OU  virtual page to place event in
                 UINT uiEventBlockLocation,
                                     // IN  event location to place contents of
                                     //     event-block in
                 UINT uiEventNumber,
                                     // IN  number to place in event-block 
                                     //     at the specified location      
                 FskEvent* pcFskEvent
                                     // IN  event-block to place at location
                )
{ 
   INT iLoop;

   CHAR aachTextBlock[PRINT_BLOCK_HEIGHT][PRINT_BLOCK_WIDTH];

                                  // load the array
   sprintf(aachTextBlock[0], " %u", uiEventNumber);

   for (iLoop=1; iLoop<PRINT_BLOCK_HEIGHT; iLoop++) 
   {
      if (strlen(pcFskEvent->pchName) >= ((iLoop-1)*(PRINT_BLOCK_WIDTH-2)))
      {
         strncpy(&(aachTextBlock[iLoop])[1], 
                 &(pcFskEvent->pchName)[(iLoop-1)*(PRINT_BLOCK_WIDTH-2)],
                 PRINT_BLOCK_WIDTH-2);
         aachTextBlock[iLoop][PRINT_BLOCK_WIDTH-1] = '\0';
         aachTextBlock[iLoop][0] = ' ';
      }
      else
      {
         aachTextBlock[iLoop][0] = '\0';
      }
   }

                                  // if name is more than allocated space
                                  // in block, replace the last three 
                                  // visable chars with ...
   if (strlen(pcFskEvent->pchName) >
       ((PRINT_BLOCK_HEIGHT-1)*(PRINT_BLOCK_WIDTH-2)))
   {
      strcpy(&aachTextBlock[PRINT_BLOCK_HEIGHT-1][PRINT_BLOCK_WIDTH-4],
             "...");
   }

                                  // copy the array to the virtual page
   if (pcVirtualPage->iWriteLine((uiEventBlockLocation*PRINT_BLOCK_HEIGHT)+
                                 (uiEventBlockLocation+HEADER_OFFSET+1), 
                                 1,
                                 aachTextBlock[0]))
   {
      LOG_ERROR("iWriteLine failed");
      return(ERROR);
   }

   for (iLoop=1; iLoop<PRINT_BLOCK_HEIGHT; iLoop++) 
   {
      if (pcVirtualPage->iWriteLine((uiEventBlockLocation*PRINT_BLOCK_HEIGHT)+
                                    (uiEventBlockLocation+HEADER_OFFSET+
                                     iLoop+1),
                                    1,
                                    aachTextBlock[iLoop]))
      {
         LOG_ERROR("iWriteLine failed");
         return(ERROR);
      }
   }


   return(0);
} 



/*----------------------------------------------------------------------------*\
|                                                                              |
| Function:  Create a blank fsk state table.                                   |
|                                                                              |
| Returns:   INT                                                               |
|            non-zero value on error                                           |
|                                                                              |
\*----------------------------------------------------------------------------*/
INT
Fsk::
iCreateBlankPrintStateTable(VirtualPage* pcVirtualPage
                                     // OU  virtual page to create blank 
                                     //     state table on
                           )
{ 
   UINT uiLoop;
   UINT uiLoop2;
   CHAR achLine[CHARS_PER_LINE+1];
   CHAR achSolidLine[CHARS_PER_LINE+1];           // ---------------
   CHAR achBarsAndBlanks[CHARS_PER_LINE+1];       // |      |      |
   CHAR achBarLine[CHARS_PER_LINE+1];             // |------|------|

   achLine[CHARS_PER_LINE] = '\0';
   achSolidLine[CHARS_PER_LINE] = '\0';
   achBarsAndBlanks[CHARS_PER_LINE] = '\0';
   achBarLine[CHARS_PER_LINE] = '\0';


   memset(achSolidLine, '-', CHARS_PER_LINE);
   memset(achBarsAndBlanks, ' ', CHARS_PER_LINE);
   memset(achBarLine, '-', CHARS_PER_LINE);

   achSolidLine[CHARS_PER_LINE-2] = '\0';
   achSolidLine[0] = ' ';

   achBarLine[CHARS_PER_LINE-1] = '\0';

   for (uiLoop=0; uiLoop<CHARS_PER_LINE; uiLoop += (PRINT_BLOCK_WIDTH+1))
   {
      achBarsAndBlanks[uiLoop] = '|';
      achBarLine[uiLoop]       = '|';
   }
  
                         // create the "standard" page 
   pcVirtualPage->vClearPage();

   for (uiLoop=0; 
        uiLoop<(LINES_PER_PAGE-(1+FOOTER_OFFSET+HEADER_OFFSET)); 
        uiLoop++)
   {
      if ((uiLoop%(PRINT_BLOCK_HEIGHT+1)) != 0)
      {
         if (pcVirtualPage->iWriteLine(uiLoop+HEADER_OFFSET, achBarsAndBlanks))
         {
            LOG_ERROR("iWriteLine failed");
            return(ERROR);
         }
      }
      else
      {
         if (pcVirtualPage->iWriteLine(uiLoop+HEADER_OFFSET, achBarLine))
         {
            LOG_ERROR("iWriteLine failed");
            return(ERROR);
         }
      }
   }

   if (pcVirtualPage->iWriteLine(HEADER_OFFSET, achSolidLine))
   {
      LOG_ERROR("iWriteLine failed");
      return(ERROR);
   }

   if (pcVirtualPage->iWriteLine((LINES_PER_PAGE-1-FOOTER_OFFSET), 
                                 achSolidLine))
   {
      LOG_ERROR("iWriteLine failed");
      return(ERROR);
   }

   return(0);   

} 



/*----------------------------------------------------------------------------*\
|                                                                              |
| Function:  Print the FSK state table                                         |
|                                                                              |
| Returns:   INT                                                               |
|            non-zero value on error                                           |
|                                                                              |
\*----------------------------------------------------------------------------*/
INT
Fsk::
iPrintStateTable(CHAR* pchOutputFile,
                                     // IN  name of file to print state table
                                     //     to
                 CHAR* pchFskName,    
                                     // IN  name of the FSK
                 INT   iOverWrite     
                                     // IN  overwrite the file if it exists
                )
{ 

   INT             iReplaceOrAppend;       // first time through, set to 
                                           // replace
   INT             iNumRowBlocks;          // number of row blocks on page
   INT             iNumColumnBlocks;       // number of state blocks on page
   INT             iPageRow;               // the page "row" we are on
   INT             iPageColumn;            // the page "column" we are on
   INT             iBlockRowOffset;        // if we print the state blocks,
                                           // we must offset the rows by this 
                                           // amount
   INT             iBlockColumnOffset;     // if we print the event blocks,
                                           // we must offset the columns by
                                           // this amount
   UINT            uiLoop;
   UINT            uiLoop2;
   UINT            uiNumStates;
   UINT            uiNumEvents;
   UINT            uiStartState;           // the state number to start
                                           // printing under
   UINT            uiStartEvent;           // the event number to start
                                           // printing to the right of
   CHAR            achPageLocation[30];    // the "row-column" page location
   FskState        cFskState;
   FskEvent        cFskEvent;
   FskActionBlock  cFskActionBlock;
   VirtualPage*    pcVirtualPage;

   pcVirtualPage = new VirtualPage(LINES_PER_PAGE, CHARS_PER_LINE);
   CHECK_NEW(pcVirtualPage);

   if (iCreateBlankPrintStateTable(pcVirtualPage))
   {
      LOG_ERROR("iCreateBlankPrintStateTable failed");
      return(ERROR);
   }

                         // create the upper-left state/event block
   if (pcVirtualPage->iWriteLine(HEADER_OFFSET+1, 1, "\\")      ||
       pcVirtualPage->iWriteLine(HEADER_OFFSET+1, 2, "__")      ||
       pcVirtualPage->iWriteLine(HEADER_OFFSET+1, 5, "STATES")  ||
       pcVirtualPage->iWriteLine(HEADER_OFFSET+2, 4, "\\")      ||
       pcVirtualPage->iWriteLine(HEADER_OFFSET+3, 5, "\\")      ||
       pcVirtualPage->iWriteLine(HEADER_OFFSET+3, 6, "_____")   ||
       pcVirtualPage->iWriteLine(HEADER_OFFSET+4, 2, "EVENTS")  ||
       pcVirtualPage->iWriteLine(HEADER_OFFSET+4, 11, "\\"))
   {
      LOG_ERROR("iWriteLine failed");
      return(ERROR);
   }

   if (pcVirtualPage->iWriteCenteredLine(0, pchFskName))
   {
      LOG_ERROR("iWriteCenteredLine failed");
      return(ERROR);
   }

   if (pcVirtualPage->iWriteCenteredLine(1, achLastModifiedDate))
   {
      LOG_ERROR("iWriteCenteredLine failed");
      return(ERROR);
   }

   if (pcVirtualPage->iWriteCenteredLine(LINES_PER_PAGE-1, "0-0"))
   {
      LOG_ERROR("iWriteCenteredLine failed");
      return(ERROR);
   }
   
                         // print the fsk
   uiNumStates    = uiGetNumStates();
   uiNumEvents    = uiGetNumEvents();
   uiStartState   = 0;
   uiStartEvent   = 0;
   iPageRow       = 0;
   iPageColumn    = 0;
   iReplaceOrAppend = REPLACE;

   iNumRowBlocks = (LINES_PER_PAGE - HEADER_OFFSET - FOOTER_OFFSET) / 
                   (PRINT_BLOCK_HEIGHT+1);
   iNumColumnBlocks = (CHARS_PER_LINE-1) / (PRINT_BLOCK_WIDTH+1);


   while (uiStartEvent < uiNumEvents)
   {

      if (iPageRow == 0)
      {
         iBlockRowOffset = 1;
      }
      else
      {
         iBlockRowOffset = 0;
      }

      if (iPageColumn == 0)
      {
         iBlockColumnOffset = 1;
      }
      else
      {
         iBlockColumnOffset = 0;
      }

                         // if we are on the first row of pages/events, print
                         // the associated state blocks
      if (uiStartEvent == 0)
      {
         for (uiLoop=0; 
              (((uiLoop+uiStartState)<uiNumStates) && 
               ((uiLoop+iBlockColumnOffset)<iNumColumnBlocks)); uiLoop++)
         {
            if (iGetStateData(uiLoop+uiStartState, &cFskState))
            {
               sprintf(achAppErrorString, "Could not get state data, %u",
                       uiLoop);
               LOG_ERROR(achAppErrorString);
               return(ERROR);
            }

                         // if we are going to print the event blocks, offset
                         // the states by one
            if (iPrintStateBlock(pcVirtualPage, 
                                 uiLoop+iBlockColumnOffset,
                                 uiLoop+uiStartState, 
                                 &cFskState))
            {
               LOG_ERROR("Internal error");
               return(ERROR);
            }
         }
      }

                         // if we are on the first column of pages/statest,
                         // the the associated event blocks
      if (uiStartState == 0)
      {
         for (uiLoop=0; 
              (((uiLoop+uiStartEvent)<uiNumEvents) && 
               ((uiLoop+iBlockRowOffset)<iNumRowBlocks)); uiLoop++)
         {
            if (iGetEventData(uiLoop+uiStartEvent, &cFskEvent))
            {
               sprintf(achAppErrorString, "Could not get event data, %u",
                       uiLoop);
               LOG_ERROR(achAppErrorString);
               return(ERROR);
            }

                         // if we printed the state blocks, offset
                         // the events by one
            if (iPrintEventBlock(pcVirtualPage, 
                                 uiLoop+iBlockRowOffset,
                                 uiLoop+uiStartEvent, 
                                 &cFskEvent))
            {
               LOG_ERROR("Internal error");
               return(ERROR);
            }
         }
      }


                         // print the action blocks
      for (uiLoop=uiStartEvent; 
           ((uiLoop<uiNumEvents) && 
            ((uiLoop+iBlockRowOffset)<(iNumRowBlocks+uiStartEvent))); 
            uiLoop++)
      {

         for (uiLoop2=uiStartState; 
              ((uiLoop2<uiNumStates) && 
               ((uiLoop2+iBlockColumnOffset)<(iNumColumnBlocks+uiStartState))); 
               uiLoop2++)
         {
            if (iGetActionBlockData(uiLoop,
                                    uiLoop2, 
                                    &cFskActionBlock))
            {
               sprintf(achAppErrorString, 
                       "Could not get action block data, %u, %u",
                       uiLoop, uiLoop2);
               LOG_ERROR(achAppErrorString);
               return(ERROR);
            }

                         // add one to the action block location
                         // if we printed the state/event block
            if (iPrintActionBlock(pcVirtualPage, 
                                  uiLoop-uiStartEvent+iBlockRowOffset,
                                  uiLoop2-uiStartState+iBlockColumnOffset,
                                  &cFskActionBlock))
            {
               LOG_ERROR("Internal error");
               return(ERROR);
            }
         }
      }


      if (pcVirtualPage->iPrintPage(pchOutputFile, iReplaceOrAppend, 
                                    iOverWrite))
      {
         LOG_DEBUG("iWriteLine failed");
         return(ERROR);
      }

                         // if at the end of a row of pages/blocks, move down
                         // and process the next row of pages/blocks
      uiStartState += (iNumColumnBlocks-iBlockColumnOffset); 
      iPageColumn++;

      if (uiStartState >= uiNumStates)
      {
                         // we are at the end of a row of pages/blocks
                         // set counters to process next row of pages/blocks
         uiStartState = 0; 
         uiStartEvent += (iNumRowBlocks-iBlockRowOffset); 
         iPageRow++;
         iPageColumn = 0;
      }

      iReplaceOrAppend = APPEND;

                         // create a new page
      if (iCreateBlankPrintStateTable(pcVirtualPage))
      {
         LOG_ERROR("Internal Error");
         return(ERROR);
      }


                         // setup the new page
      if (pcVirtualPage->iWriteLeftJustifiedLine(LINES_PER_PAGE-1, 
                                                 pchFskName))
      {
         LOG_ERROR("iWriteLeftJustifiedLine failed");
         return(ERROR);
      }

      if (pcVirtualPage->iWriteRightJustifiedLine(LINES_PER_PAGE-1, 
                                                  achLastModifiedDate))
      {
         LOG_ERROR("iWriteRightJustifiedLine failed");
         return(ERROR);
      }

      sprintf(achPageLocation, " %d-%d ", iPageRow, iPageColumn);

      if (pcVirtualPage->iWriteCenteredLine(LINES_PER_PAGE-1, 
                                            achPageLocation))
      {
         LOG_ERROR("iWriteCenteredLine failed");
         return(ERROR);
      }

      if (pcVirtualPage->iWriteCenteredLine(LINES_PER_PAGE-1, 
                                            achPageLocation))
      {
         LOG_ERROR("iWriteCenteredLine failed");
         return(ERROR);
      }

   } /* while (uiStartEvent < uiNumEvents) */
   
   delete(pcVirtualPage);

   return(0);
}



/*----------------------------------------------------------------------------*\
|                                                                              |
| Function:  Print the FSK state/event descriptions                            |
|                                                                              |
| Returns:   INT                                                               |
|            non-zero value on error                                           |
|                                                                              |
\*----------------------------------------------------------------------------*/
INT
Fsk::
iPrintDescriptions(CHAR* pchOutputFile,
                                     // IN  name of file to print state 
                                     //     description to
                        CHAR* pchFskName,    
                                     // IN  name of the FSK
                        INT   iDescriptionType,
                                     // IN  type of description table to print
                                     //     STATE_DESCRIPTIONS, or 
                                     //     EVENT_DESCRIPTIONS
                        INT   iOverWrite     
                                     // IN  overwrite the file if it exists
                       )
{ 

   INT             iReplaceOrAppend;       // first time through, set to 
                                           // replace

   INT             iCharsPerDescriptionLine = 55;
                                           // number of characters on each
                                           // description line
   INT             iPageNumber;
   INT             iLinesInNameBlock;
   INT             iLinesInDescriptionBlock;
   INT             iTotalLinesPrintedOnPage;
   INT             iEndOfPage;
   UINT            uiLoop;
   UINT            uiLoop2;
   UINT            uiStartBlock;
   UINT            uiNumBlocks;
   CHAR            achPageNumber[20];      // the page number
   CHAR            achTemp[200];
   CHAR            achBlockName[STATE_NAME_LEN + EVENT_NAME_LEN + 
                                NAME_BLOCK_WIDTH];
   CHAR            achBlockDescription[DESCRIPTION_LEN + 
                                       DESCRIPTION_BLOCK_WIDTH];
   CHAR            chTemp;
   CHAR*           pchTemp;
   FskState        cFskState;
   FskEvent        cFskEvent;
   VirtualPage*    pcVirtualPage;

   CHAR achSolidLine[CHARS_PER_LINE+1];           // ---------------
   CHAR achBarsAndBlanks[CHARS_PER_LINE+1];       // |      |      |
   CHAR achBarLine[CHARS_PER_LINE+1];             // |------|------|

   CHAR achBlankLine[CHARS_PER_LINE+1];

   achSolidLine[CHARS_PER_LINE] = '\0';
   achBarsAndBlanks[CHARS_PER_LINE] = '\0';
   achBarLine[CHARS_PER_LINE] = '\0';
   achBlankLine[CHARS_PER_LINE] = '\0';

   memset(achSolidLine, '-', CHARS_PER_LINE);
   memset(achBarsAndBlanks, ' ', CHARS_PER_LINE);
   memset(achBarLine, '-', CHARS_PER_LINE);
   memset(achBlankLine, ' ', CHARS_PER_LINE);

   achSolidLine[CHARS_PER_LINE-1] = '\0';
   achSolidLine[0] = ' ';

   achBarLine[CHARS_PER_LINE-1] = '\0';

   achBarsAndBlanks[0] = '|';
   achBarLine[0]       = '|';
   achBarsAndBlanks[NUMBER_BLOCK_WIDTH] = '|';
   achBarLine[NUMBER_BLOCK_WIDTH]       = '|';
   achBarsAndBlanks[NUMBER_BLOCK_WIDTH + NAME_BLOCK_WIDTH] = '|';
   achBarLine[NUMBER_BLOCK_WIDTH + NAME_BLOCK_WIDTH]       = '|';
   achBarsAndBlanks[NUMBER_BLOCK_WIDTH + NAME_BLOCK_WIDTH + 
                                  DESCRIPTION_BLOCK_WIDTH] = '|';
   achBarLine[NUMBER_BLOCK_WIDTH + NAME_BLOCK_WIDTH + 
                                  DESCRIPTION_BLOCK_WIDTH] = '|';
  

   if (iDescriptionType == EVENT_DESCRIPTIONS)
   {
      uiNumBlocks = uiGetNumEvents();
   }
   else if (iDescriptionType == STATE_DESCRIPTIONS)
   {
      uiNumBlocks = uiGetNumStates();
   }
   else
   {
      sprintf(achAppErrorString, "Bad description type, %d", iDescriptionType);
      LOG_ERROR(achAppErrorString);
      return(ERROR);
   }

   pcVirtualPage = new VirtualPage(LINES_PER_PAGE, CHARS_PER_LINE);
   CHECK_NEW(pcVirtualPage);

   uiStartBlock = 0;
   uiLoop       = 0;
   iPageNumber  = 0;
   iReplaceOrAppend = REPLACE;
                         // print the descriptions
   while (uiLoop<uiNumBlocks)
   {

      pcVirtualPage->vClearPage();

                         // the number of blocks on a page varies on the
                         // number of lines each description takes up
                         // print the header
      if (iDescriptionType == EVENT_DESCRIPTIONS)
      {
         sprintf(achTemp, "%s Event Descriptions", pchFskName);
      }
      else 
      {
         sprintf(achTemp, "%s State Descriptions", pchFskName);
      }

      if (pcVirtualPage->iWriteCenteredLine(0, achTemp))
      {
         LOG_ERROR("iWriteCenteredLine failed");
         return(ERROR);
      }

      if (pcVirtualPage->iWriteCenteredLine(1, achLastModifiedDate))
      {
         LOG_ERROR("iWriteCenteredLine failed");
         return(ERROR);
      }

      if ((pcVirtualPage->iWriteLine(HEADER_OFFSET, achSolidLine))          ||
          (pcVirtualPage->iWriteLine(HEADER_OFFSET+1, achBarsAndBlanks))    ||
          (pcVirtualPage->iWriteLine(HEADER_OFFSET+2, achBarLine))          ||
          (pcVirtualPage->iWriteLine(HEADER_OFFSET+1, 2, "NUMBER"))         ||
          (pcVirtualPage->iWriteLine(HEADER_OFFSET+1, NUMBER_BLOCK_WIDTH+9, 
                                                              "NAME"))      ||
          (pcVirtualPage->iWriteLine(HEADER_OFFSET+1, 
                                  NUMBER_BLOCK_WIDTH+NAME_BLOCK_WIDTH+20, 
                                                 "DESCRIPTION")))
      {
         LOG_ERROR("iWriteLine failed");
         return(ERROR);
      }

      iTotalLinesPrintedOnPage = HEADER_OFFSET + 2;
      iEndOfPage               = 0;

                         // print a page worth of state/event descriptions
      for (uiLoop=uiStartBlock; uiLoop<uiNumBlocks; uiLoop++)
      {

         if (iDescriptionType == EVENT_DESCRIPTIONS)
         {
            if (iGetEventData(uiLoop, &cFskEvent))
            {
               sprintf(achAppErrorString, "Could not get event data, %u",
                       uiLoop);
               LOG_ERROR(achAppErrorString);
               return(ERROR);
            }
            strcpy(achBlockName, cFskEvent.pchName);
            strcpy(achBlockDescription, cFskEvent.pchDescription);
         }
         else 
         {
            if (iGetStateData(uiLoop, &cFskState))
            {
               sprintf(achAppErrorString, "Could not get state data, %u",
                       uiLoop);
               LOG_ERROR(achAppErrorString);
               return(ERROR);
            }
            strcpy(achBlockName, cFskState.pchName);
            strcpy(achBlockDescription, cFskState.pchDescription);
         }


                         // print the state or event block name
         for (iLinesInNameBlock=1; ;iLinesInNameBlock++)
         {

                         // if we can't print this block in the remaining
                         // lines on the page, back out the block we just 
                         // printed, minus two for page number and the
                         // last bar line
            if ((iTotalLinesPrintedOnPage + iLinesInNameBlock ) >= 
                (LINES_PER_PAGE-2))
            {
               for (uiLoop2=iTotalLinesPrintedOnPage;
                    uiLoop2<LINES_PER_PAGE;
                    uiLoop2++)
               {
                  if (pcVirtualPage->iWriteLine(uiLoop2, achBlankLine))
                  {
                     LOG_ERROR("iWriteLine failed");
                     return(ERROR);
                  }
               }

               iLinesInNameBlock = 0;
               iLinesInDescriptionBlock = 0;
               iEndOfPage = 1;
               break;
            }

            chTemp = achBlockName[iLinesInNameBlock*(NAME_BLOCK_WIDTH-3)];
            achBlockName[iLinesInNameBlock*(NAME_BLOCK_WIDTH-3)] = '\0';

            if (pcVirtualPage->iWriteLine(iTotalLinesPrintedOnPage + 
                                          iLinesInNameBlock, 
                                          achBarsAndBlanks))
            {
               LOG_ERROR("iWriteLine failed");
               return(ERROR);
            }

            if (pcVirtualPage->iWriteLine((iTotalLinesPrintedOnPage  +
                                                 iLinesInNameBlock),
                                          NUMBER_BLOCK_WIDTH+2,
                                          &achBlockName[(iLinesInNameBlock-1)*
                                                        (NAME_BLOCK_WIDTH-3)]))
            {
               LOG_ERROR("iWriteLine failed");
               return(ERROR);
            }



                         // if the char is at the very end of the name
            if (chTemp == '\0')
            {
               break;
            }
                         // replace the null we put in the string with the
                         // char we took out
            achBlockName[iLinesInNameBlock*(NAME_BLOCK_WIDTH-3)] = chTemp;

            if (strlen(achBlockName) <= 
                (iLinesInNameBlock*(NAME_BLOCK_WIDTH-3)))
            {
                         // past the end, we have printed the complete name
               break;
            }
         }

         if (iEndOfPage)
         {
            break;
         }

                         // print the state or event block number
         sprintf(achTemp, "%u", uiLoop);
         if (pcVirtualPage->iWriteLine(iTotalLinesPrintedOnPage + 1,
                                       2, achTemp))
         {
            LOG_ERROR("iWriteLine failed");
            return(ERROR);
         }

                         // change all new-line chars to blanks
         while ((pchTemp = strchr(achBlockDescription, '\n')) != NULL)
         {
            *pchTemp =  ' ';
         }

                         // print the state/event description
         for (iLinesInDescriptionBlock=1; ;iLinesInDescriptionBlock++)
         {

                         // if we can't print the text in the remaining
                         // lines on the page, back out the block we just 
                         // printed. minus two for page number and the
                         // last bar line
            if ((iTotalLinesPrintedOnPage + iLinesInDescriptionBlock ) >= 
                (LINES_PER_PAGE-2))
            {
               for (uiLoop2=iTotalLinesPrintedOnPage;
                    uiLoop2<LINES_PER_PAGE;
                    uiLoop2++)
               {
                  if (pcVirtualPage->iWriteLine(uiLoop2, achBlankLine))
                  {
                     LOG_ERROR("iWriteLine failed");
                     return(ERROR);
                  }
               }

               iLinesInDescriptionBlock = 0;
               iLinesInDescriptionBlock = 0;
               iEndOfPage = 1;
               break;
            }


            chTemp = achBlockDescription[iLinesInDescriptionBlock*
                                          (DESCRIPTION_BLOCK_WIDTH-3)];

            achBlockDescription[iLinesInDescriptionBlock*
                                          (DESCRIPTION_BLOCK_WIDTH-3)] = '\0';

                         // the number of lines in the description has 
                         // exceeded the number in the name block, so we
                         // must print the bars for this line...           
            if (iLinesInDescriptionBlock > iLinesInNameBlock)
            { 
               if (pcVirtualPage->iWriteLine(iTotalLinesPrintedOnPage + 
                                             iLinesInDescriptionBlock, 
                                             achBarsAndBlanks))
               {
                  LOG_ERROR("iWriteLine failed");
                  return(ERROR);
               }
            } 

            if (pcVirtualPage->iWriteLine(
                                (iTotalLinesPrintedOnPage  +
                                    iLinesInDescriptionBlock),
                                 NUMBER_BLOCK_WIDTH + NAME_BLOCK_WIDTH + 2, 
                                 &achBlockDescription[ 
                                    (iLinesInDescriptionBlock-1)*
                                     (DESCRIPTION_BLOCK_WIDTH-3)]))
            {
               LOG_ERROR("iWriteLine failed");
               return(ERROR);
            }



                         // if the char is at the very end of the name
            if (chTemp == '\0')
            {
               break;
            }
                         // replace the null we put in the string with the
                         // char we took out
            achBlockDescription[iLinesInDescriptionBlock*
                                 (DESCRIPTION_BLOCK_WIDTH-3)] = chTemp;

            if (strlen(achBlockDescription) <= 
                (iLinesInDescriptionBlock*(DESCRIPTION_BLOCK_WIDTH-3)))
            {
                         // past the end, we have printed the complete name
               break;
            }
         }

         if (iEndOfPage)
         {
            break;
         }


         iTotalLinesPrintedOnPage += MAX(iLinesInNameBlock, 
                                         iLinesInDescriptionBlock);

         if (pcVirtualPage->iWriteLine(iTotalLinesPrintedOnPage + 1,
                                       achBarLine))
         {
            LOG_ERROR("iWriteLine failed");
            return(ERROR);
         }

         iTotalLinesPrintedOnPage++;


         uiStartBlock = uiLoop+1;

      }

                         // decrement for the barline we last printed 
      if ((uiLoop>=uiNumBlocks) ||
          (iEndOfPage))
      {
         iTotalLinesPrintedOnPage--;
      }

                         // make the last line solid
      if (pcVirtualPage->iWriteLine(iTotalLinesPrintedOnPage + 1,
                                    achSolidLine))
      {
         LOG_ERROR("iWriteLine failed");
         return(ERROR);
      }

                         // print the page number
      iPageNumber++;
      sprintf(achPageNumber, "%d", iPageNumber);
      if (pcVirtualPage->iWriteCenteredLine(LINES_PER_PAGE-1, achPageNumber))
      {
         LOG_ERROR("iWriteCenteredLine failed");
         return(ERROR);
      }

      if (pcVirtualPage->iPrintPage(pchOutputFile, iReplaceOrAppend, 
                                    iOverWrite))
      {
         LOG_DEBUG("iWriteLine failed");
         return(ERROR);
      }

      iReplaceOrAppend = APPEND;

   } /* while (uiLoop<uiNumBlocks) */


   return(0);
}



/*----------------------------------------------------------------------------*\
|                                                                              |
| Function:  Print the FSK state descriptions                                  |
|                                                                              |
| Returns:   INT                                                               |
|            non-zero value on error                                           |
|                                                                              |
\*----------------------------------------------------------------------------*/
INT
Fsk::
iPrintStateDescriptions(CHAR* pchOutputFile,
                                     // IN  name of file to print state 
                                     //     description to
                        CHAR* pchFskName,    
                                     // IN  name of the FSK
                        INT   iOverWrite     
                                     // IN  overwrite the file if it exists
                       )
{ 

   if (iPrintDescriptions(pchOutputFile, pchFskName, STATE_DESCRIPTIONS,
                          iOverWrite))
   {
      LOG_ERROR("Internal Error: Could Not Print STATE Descriptions");
      return(ERROR);
   }

   return(0);
}



/*----------------------------------------------------------------------------*\
|                                                                              |
| Function:  Print the FSK event descriptions                                  |
|                                                                              |
| Returns:   INT                                                               |
|            non-zero value on error                                           |
|                                                                              |
\*----------------------------------------------------------------------------*/
INT
Fsk::
iPrintEventDescriptions(CHAR* pchOutputFile,
                                     // IN  name of file to print event 
                                     //     description to
                        CHAR* pchFskName,    
                                     // IN  name of the FSK
                        INT   iOverWrite     
                                     // IN  overwrite the file if it exists
                       )
{ 
   if (iPrintDescriptions(pchOutputFile, pchFskName, EVENT_DESCRIPTIONS,
                          iOverWrite))
   {
      LOG_ERROR("Internal Error: Could Not Print EVENT Descriptions");
      return(ERROR);
   }
   return(0);
}



/*----------------------------------------------------------------------------*\
|                                                                              |
| Function:  Check for infinite states in FSK (a state that does not           |
|            contain action-blocks that transition to another state)           |
|                                                                              |
| Returns:   INT                                                               |
|            0       no infinite states found.                                 |
|            WARNING indicates infinite states found                           |
|            ERROR   indicates we encountered an error                         |
|                                                                              |
\*----------------------------------------------------------------------------*/
INT
Fsk::
iInfiniteStateCheck(CHAR** ppchMessage
                                     // CHAR**  message allocated and returned
                                     //         if non-zero returned, calling 
                                     //         routine must free the alloced 
                                     //         mem
                   )
{ 
   INT   iResult;
   INT   iReturnValue = 0;
   UINT  uiLoop;
   UINT  uiLoop2;
   UINT  uiNumStates;
   UINT  uiNumEvents;
   UINT  uiCurrentState;
   CHAR* pchTemp;
   FskActionBlock cFskActionBlock;
   
   *ppchMessage = NULL;

   uiNumStates = uiGetNumStates();
   uiNumEvents = uiGetNumEvents();

   for (uiLoop=0; uiLoop<uiNumStates; uiLoop++)
   {

      uiCurrentState = uiLoop;

      for (uiLoop2=0; uiLoop2<uiNumEvents; uiLoop2++)
      {
         if (iGetActionBlockData(uiLoop2, uiLoop, &cFskActionBlock))
         {
            sprintf(achAppErrorString, 
                    "Could not get action block data, %u, %u",
                    uiLoop2, uiLoop);
            LOG_ERROR(achAppErrorString);
            return(ERROR);
         }

         if (uiCurrentState != *cFskActionBlock.puiNextState)
         {
                         // not an infinite state
            break;
         }
      }

      if (uiCurrentState == *cFskActionBlock.puiNextState)
      {
                         // an infinite state
         if (iReturnValue == 0)
         {
            *ppchMessage = (CHAR*) malloc(150);
            CHECK_MALLOC(*ppchMessage);
            strcpy(*ppchMessage, 
                   "WARNING: The following states are infinite :\n");

            iReturnValue = WARNING;
         }
         else
         {
            *ppchMessage = (CHAR*) realloc(*ppchMessage,
                                           (strlen(*ppchMessage) + 80));
            CHECK_MALLOC(*ppchMessage);
         }

         sprintf((*ppchMessage)+strlen(*ppchMessage),
                  "                                            %u\n", 
                  uiCurrentState);
      }
   }


   return(iReturnValue);
}
