/*----------------------------------------------------------------------------*\
|
| Copyright (C) 1997, 2000  James A. Cureington
|                           tonyc@acm.org
| 
| This program is free software; you can redistribute it and/or
| modify it under the terms of the GNU General Public License
| as published by the Free Software Foundation; either version 2
| of the License, or any later version.
| 
| This program is distributed in the hope that it will be useful,
| but WITHOUT ANY WARRANTY; without even the implied warranty of
| MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
| GNU General Public License for more details.
| 
| You should have received a copy of the GNU General Public License
| along with this program; if not, write to the Free Software
| Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
|
|                                                                              
| Tony Cureington                                                              
| September 21, 1996                                                           
|                                                                              
| Module:  DynamicMatrix template implementations                              
|          This file contains templates for the DynamicMatrix templates.       
|          The DynamicMatrix templates allow insertion and deletion of matrix  
|          rows and columns, as well as copying of rows and columns between    
|          different matrixes.                                                 
|                                                                              
| $Id: d_matrix.C,v 1.1 2001/03/04 00:05:14 tonycu Exp $                         
|                                                                              
| $Log: d_matrix.C,v $
| Revision 1.1  2001/03/04 00:05:14  tonycu
| initial check-in at sourceforge...
|
| Revision 1.10  2000/05/24 03:19:01  tony
| Updated all copyright dates.
|
| Revision 1.9  1999/09/13 01:21:13  tony
|
| Updated copyright notice date
|
| Revision 1.8  1999/09/12 22:03:56  tony
|
| Changed #defines to const and inline functions...
|
 * Revision 1.7  1997/03/26  07:07:15  tony
 *      Fixed defects uncovered from test plan
 *
 * Revision 1.6  1997/03/01  23:24:15  tony
 *      Made the following changes:
 *           - Add version to FSK file for validation when reading in
 *           - Check for " in parameter list
 *           - Add CVS Id to source and header file
 *           - Add copyright notice to header files
 *           - Add EDIT menu item and associated entries
 *           - Make mouse button 3 cause the commnad pop-up to be displayed
 *             instead of mouse button 2
 *           - Add box to insert header files into
 *           - Changed pchz to pch
 *           - Allow macros to be used in parameter list
 *           - Add function and associated calls to check for infinite state
 *             condition; this is a warning, not an error condition
 *           - Add over-ride abort function capability to GUI
 *           - Changed "modified save pop-up" window to be displayed before
 *             any window is displayed to read and RSK, or create a new one
 *
 * Revision 1.5  1997/01/26  04:57:18  tony
 * Made changes to:
 *      - insert user defined default function next state when a state or
 *        event is inserted
 *      - stub out future functions
 *      - change tabs to spaces
 *
 * Revision 1.4  1997/01/13  08:22:57  tony
 * added code to:
 *        - delete states
 *        - insert states
 *        - search dynamic matrix diagonally
 *        - iGetMatrixCell, to chain through the matrix cells using the links
 *
 * Revision 1.3  1996/12/13  04:16:21  tony
 * 1. Moved public arrays in classes to private and added const pointers to
 *    the private members
 *
 * 2. Added assignment operators to classes
 *
 * 3. Changed the dynamic array class - replaced malloc/free with new/delete
 *    so it could work with all objects
 *
 * Revision 1.2  1996/11/27  06:16:03  tony
 * added more member functions to classes and cleaned up code a bit
 *
|* Revision 1.1  1996/09/28  16:41:45  tony
|* initial check-in
|*                                                                      
|                                                                              
\*----------------------------------------------------------------------------*/
#include <stdio.h>
#include <errno.h>
#include <string.h>

#include <d_matrix.h>

STATIC CHAR *pchDmatrixId = "@(#) $Id: d_matrix.C,v 1.1 2001/03/04 00:05:14 tonycu Exp $";


/*----------------------------------------------------------------------------*\
|                                                                              |
| Function:  Constructor to allocate a DynamicMatrix of the specified size for |
|            the specified data type.                                          |
|                                                                              |
| Returns:   none                                                              |
|            On failure, abort is called.                                      |
|                                                                              |
\*----------------------------------------------------------------------------*/
template<class DataType>
DynamicMatrix<DataType>::
DynamicMatrix(ULONG ulNumRows,       // IN  number of rows in matrix
              ULONG ulNumColumns     // IN  number of columns in matrix
             ) 
{
   ULONG ulRowIndex;
   ULONG ulColumnIndex;
   MatrixCell<DataType>* pxMatrixCell;
   MatrixCell<DataType>* pxMatrixRow;
   MatrixCell<DataType>* pxMatrixColumn;

                              // if we have one or more rows, we must
                              // have at least one column;  if we have
                              // one or more columns, we must have at
                              // least one row...  hence, if either row
                              // or column is equal to zero, the other
                              // must be equal to zero as well...     
   if ((ulNumRows == 0) || (ulNumColumns == 0))
   {
      if (ulNumRows != ulNumColumns)
      {
         LOG_ERROR("If number of rows or columns is 0, both must be 0 ");
         abort();
      }

      ulNumMatrixRows = 0;
      ulNumMatrixColumns = 0;
      pxMatrixStart = NULL;
      return;
   }

   ulNumMatrixRows = ulNumRows;
   ulNumMatrixColumns = ulNumColumns;

                              // all is good, create the matrix...    

                              // build the first row of columns,
                              // seperate from the rest of the matrix
                              // so we can set the starting location 
                              // outside the main processing loop for
                              // a little increase in performance... 
   pxMatrixStart = new MatrixCell<DataType>;
   CHECK_NEW(pxMatrixStart);

   pxMatrixCell = pxMatrixStart;

   for (ulColumnIndex=1; ulColumnIndex<ulNumColumns; ulColumnIndex++)
   {
      pxMatrixCell->pxNextColumn = new MatrixCell<DataType>;
      CHECK_NEW(pxMatrixCell->pxNextColumn);
      pxMatrixCell->pxNextColumn->pxPreviousColumn = pxMatrixCell;
      pxMatrixCell = pxMatrixCell->pxNextColumn;
   }

                              // build the second through Nth rows of 
                              // columns
   pxMatrixColumn = pxMatrixStart;
   for (ulRowIndex=1; ulRowIndex<ulNumRows; ulRowIndex++)
   {
      pxMatrixCell = new MatrixCell<DataType>;
      CHECK_NEW(pxMatrixCell);
      pxMatrixColumn->pxNextRow = pxMatrixCell;
      pxMatrixCell->pxPreviousRow = pxMatrixColumn;
      pxMatrixRow = pxMatrixColumn;
      for (ulColumnIndex=1; ulColumnIndex<ulNumColumns; ulColumnIndex++)
      {
                              // create and link the next column
         pxMatrixCell->pxNextColumn = new MatrixCell<DataType>;
         CHECK_NEW(pxMatrixCell->pxNextColumn);
         pxMatrixCell->pxNextColumn->pxPreviousColumn = pxMatrixCell;
         pxMatrixCell = pxMatrixCell->pxNextColumn;
                              // link the cell to the previous row
         pxMatrixColumn = pxMatrixColumn->pxNextColumn;
         pxMatrixColumn->pxNextRow = pxMatrixCell;
         pxMatrixCell->pxPreviousRow = pxMatrixColumn;
      }
      pxMatrixColumn = pxMatrixRow->pxNextRow;
   }
}



/*----------------------------------------------------------------------------*\
|                                                                              |
| Function:  Destructor to deallocate all memory associated with a previousley |
|            instantiated DynamicMatrix.                                       |
|                                                                              |
| Returns:   none                                                              |
|            On failure, ASSERT is called                                      |
|                                                                              |
\*----------------------------------------------------------------------------*/
template<class DataType>
DynamicMatrix<DataType>::
~DynamicMatrix() 
{
   ULONG ulRow;
   ULONG ulColumn;
   MatrixCell<DataType>* pxMatrixCell;
   MatrixCell<DataType>* pxNextMatrixCell;

                              // a few sanity checks...
   if ((ulNumMatrixRows == 0) || (ulNumMatrixColumns == 0))
   {
      if (ulNumMatrixRows != ulNumMatrixColumns)
      {
         LOG_ERROR("If number of rows or columns is 0, both must be 0 ");
         abort();
      }
   }

   if (pxMatrixStart == NULL)
   {
      return;
   }

                              // when freeing memory associated with 
                              // linked lists, we usually use recursion.
                              // we are concerned with the amount of   
                              // stack space we would use for a large  
                              // matrix...  so, we don't use recursion.. 
   for (ulRow=0; ulRow<ulNumMatrixRows; ulRow++)
   {
      pxMatrixCell = pxMatrixStart;
      pxMatrixStart = pxMatrixStart->pxNextRow;
      for (ulColumn=0; ulColumn<ulNumMatrixColumns; ulColumn++)
      {
         ASSERT(pxMatrixCell != NULL);
         pxNextMatrixCell = pxMatrixCell->pxNextColumn;
         delete pxMatrixCell;
         pxMatrixCell = pxNextMatrixCell;
      }
   }

   ulNumMatrixRows    = 0;
   ulNumMatrixColumns = 0;
   pxMatrixStart = NULL;
}


/*----------------------------------------------------------------------------*\
|                                                                              |
| Function:  Return the number of rows in the DynamicMatrix.                   |
|                                                                              |
| Returns:   ULONG                                                             |
|                                                                              |
\*----------------------------------------------------------------------------*/
template<class DataType> ULONG  
DynamicMatrix<DataType>::
ulGetNumRows() 
{
   return(ulNumMatrixRows);
}



/*----------------------------------------------------------------------------*\
|                                                                              |
| Function:  Returns the number of columns in the DynamicMatrix.               |
|                                                                              |
| Returns:   ULONG                                                             |
|                                                                              |
\*----------------------------------------------------------------------------*/
template<class DataType> ULONG
DynamicMatrix<DataType>::
ulGetNumColumns() 
{
   return(ulNumMatrixColumns);
}



/*----------------------------------------------------------------------------*\
|                                                                              |
| Function:  Return a reference to the data at the matrix cell at the          |
|            specified row and column.                                         |
|                                                                              |
| Returns:   DataType&                                                         |
|            On failure, ASSERT is called                                      |
|                                                                              |
\*----------------------------------------------------------------------------*/
template<class DataType> DataType& 
DynamicMatrix<DataType>::
rxCellData(ULONG ulMatrixRow,         // IN  matrix row
           ULONG ulMatrixColumn       // IN  matrix column
           )
{

   ULONG ulLoop;
   ULONG ulCurrentRow;
   ULONG ulCurrentColumn;
   MatrixCell<DataType>* pxMatrixCell;

                              // ulMatrixRow is zero based and                                       
                              // ulNumMatrixRows is one based
   if ((ulMatrixRow >= ulNumMatrixRows) ||
       (ulMatrixColumn >= ulNumMatrixColumns))
   {
      sprintf(achErrorString, "bad matrix location [%u,%u], %uX%u",
              ulMatrixRow, ulMatrixColumn, ulNumMatrixRows, ulNumMatrixColumns);
      LOG_ERROR(achErrorString);

      ASSERT(ulMatrixRow < ulNumMatrixRows);
      ASSERT(ulMatrixColumn < ulNumMatrixColumns);
   }

   pxMatrixCell = pxMatrixStart;


   ulCurrentRow     = 0;
   ulCurrentColumn  = 0;
                              // traverse links diagonally
   while ((ulCurrentRow<ulMatrixRow) && (ulCurrentColumn<ulMatrixColumn))
   {
      pxMatrixCell = pxMatrixCell->pxNextRow;
      pxMatrixCell = pxMatrixCell->pxNextColumn;
      ulCurrentRow++;
      ulCurrentColumn++;
   }

                              // traverse links to specified row        
   for (ulLoop=ulCurrentRow; ulLoop<ulMatrixRow; ulLoop++)
   {
      pxMatrixCell = pxMatrixCell->pxNextRow;
   }

                              // traverse links to specified column     
   for (ulLoop=ulCurrentColumn; ulLoop<ulMatrixColumn; ulLoop++)
   {
      pxMatrixCell = pxMatrixCell->pxNextColumn;
   }

                              // we're at the specified row/column      
                              // matrix cell                            
   return(pxMatrixCell->xCellData);
}



/*----------------------------------------------------------------------------*\
|                                                                              |
| Function:  Get the first or next cell in the matrix.  This is usefull for    |
|            searching the matrix.                                             |
|                                                                              |
| Returns:   INT                                                               |
|               0 = success                                                    |
|              !0 = end of list                                                |
|                                                                              |
\*----------------------------------------------------------------------------*/
template<class DataType> INT
DynamicMatrix<DataType>::
iGetMatrixCell(INT iFirstOrNext,     // IN  set to FIRST_CELL_IN_MATRIX or
                                     //     NEXT_CELL_IN_MATRIX
               DataType& rxCellData, // OU  the next matrix cell
               UINT* puiEvent,       // OU  the event number of the returned
                                     //     cell
               UINT* puiState        // OU  the state Number of the returned
                                     //     cell
           )
                                     // get the first or next matrix cell
                                     // if iFirstOrNext == FIRST_CELL_IN_MATRIX
                                     // get the first cell in the matrix  
                                     // if iFirstOrNext == NEXT_CELL_IN_MATRIX
                                     // get the next cell in the matrix
{

   INT iRight = 1;
   INT iLeft  = 2;
   STATIC INT  iDirection;
   STATIC INT  iEndOfMatrix;
   STATIC UINT uiCurrentEvent;
   STATIC UINT uiCurrentState;
   STATIC MatrixCell<DataType>* pxMatrixCell;


   if (iFirstOrNext == FIRST_CELL_IN_MATRIX)
   {
                              // initialize the values
      pxMatrixCell = pxMatrixStart;
      iDirection = iRight;
      iEndOfMatrix = 0;
      uiCurrentEvent = 0;
      uiCurrentState = 0;
      *puiEvent = uiCurrentEvent;
      *puiState = uiCurrentState;
      rxCellData = pxMatrixCell->xCellData;
      return(0);
   }

   if (iEndOfMatrix)
   {
      return(ERROR);
   }

   if (iDirection == iRight)
   {
                              // moving right along the columns
      if (pxMatrixCell->pxNextColumn != NULL)
      {
         uiCurrentState++;
         pxMatrixCell = pxMatrixCell->pxNextColumn;
      }
      else
      {
                              // move down a row and reverse direction
         if (pxMatrixCell->pxNextRow == NULL)
         {
                              // at end of matrix
            iEndOfMatrix = 1;
            return(-1);
         }

         uiCurrentEvent++;
         pxMatrixCell = pxMatrixCell->pxNextRow;
         iDirection = iLeft;
      }
   } 
   else /* iDirection == iLeft */
   {
                              // moving left along the columns
      if (pxMatrixCell->pxPreviousColumn != NULL)
      {
         uiCurrentState--;
         pxMatrixCell = pxMatrixCell->pxPreviousColumn;
      }
      else
      {
                              // move down a row and reverse direction
         if (pxMatrixCell->pxNextRow == NULL)
         {
                              // at end of matrix
            iEndOfMatrix = 1;
            return(-1);
         }

         uiCurrentEvent++;
         pxMatrixCell = pxMatrixCell->pxNextRow;
         iDirection = iRight;
      }
   }

   *puiEvent = uiCurrentEvent;
   *puiState = uiCurrentState;
   rxCellData = pxMatrixCell->xCellData;

   return(0);
}

/*----------------------------------------------------------------------------*\
|                                                                              |
| Function:  Insert a new row in the DynamicMatrix before or after the         |
|            specified row.                                                    |
|                                                                              |
| Returns:   INT                                                               |
|            On failure, a non-zero value is returned.                         |
|                                                                              |
\*----------------------------------------------------------------------------*/
template<class DataType> INT
DynamicMatrix<DataType>::
iInsertRow(ULONG ulRowInsertionPoint,
                                     // IN  matrix row to insert before or 
                                     //     after 
           INT iInsertDirection
                                     // IN  enumeration specifying to insert
                                     //     the new matrix row before or after
                                     //     the specified row             
          )
{

   ULONG ulLoop;
   MatrixCell<DataType>* pxMatrixCell;
   MatrixCell<DataType>* pNewMatrixCell;
   MatrixCell<DataType>* pxPreviousNewMatrixCell;

   if (ulNumMatrixRows == 0)
   {
      LOG_ERROR("attempt to insert row in matrix with 0 rows");
      return(1); 
   }

   if (ulRowInsertionPoint >= ulNumMatrixRows)
   {
      sprintf(achErrorString, 
              "attempt to insert matrix row outside matrix boundry, %u, %uX%u",
              ulRowInsertionPoint+1, ulNumMatrixRows, ulNumMatrixColumns); 
      LOG_ERROR(achErrorString);
      return(1); 
   }


                                     // find the row to insert before or after
   pxMatrixCell = pxMatrixStart;
                                     // traverse links to specified row        
   for (ulLoop=0; ulLoop<ulRowInsertionPoint; ulLoop++)
   {
      pxMatrixCell = pxMatrixCell->pxNextRow;
   }

   pxPreviousNewMatrixCell = NULL;
                                     // insert matrix row at appropriate 
                                     // location   
   if (iInsertDirection == AFTER)
   {
                                     // insert matrix row below (after) the
                                     // current one
      for (ulLoop=0; ulLoop<ulNumMatrixColumns; ulLoop++)
      {
         pNewMatrixCell = new MatrixCell<DataType>;
         CHECK_NEW(pNewMatrixCell);
                                     // link new cell with the corresponding
                                     // one in the row below and above the 
                                     // current one

                                     // if we are inserting after the last
                                     // matrix row, there is no need to link
                                     // to the row after the current one...
         if (pxMatrixCell->pxNextRow != NULL)
         {
            pNewMatrixCell->pxNextRow = pxMatrixCell->pxNextRow;
            pxMatrixCell->pxNextRow->pxPreviousRow = pNewMatrixCell;
         }
         pxMatrixCell->pxNextRow = pNewMatrixCell;
         pNewMatrixCell->pxPreviousRow = pxMatrixCell;
                                     // link the columns
         if (pxPreviousNewMatrixCell == NULL)
         {
            pxPreviousNewMatrixCell = pNewMatrixCell;
         }
         else
         {
            pxPreviousNewMatrixCell->pxNextColumn = pNewMatrixCell;
            pNewMatrixCell->pxPreviousColumn = pxPreviousNewMatrixCell;
            pxPreviousNewMatrixCell = pNewMatrixCell;
         }
         pxMatrixCell = pxMatrixCell->pxNextColumn;
      }
   }
   else if (iInsertDirection == BEFORE)
   {
                                     // insert matrix row above (before) the
                                     // current one
      for (ulLoop=0; ulLoop<ulNumMatrixColumns; ulLoop++)
      {
         pNewMatrixCell = new MatrixCell<DataType>;
         CHECK_NEW(pNewMatrixCell);
                                     // link new cell with the corresponding
                                     // one in the column below and above the 
                                     // current one

                                     // if we are inserting before the first
                                     // matrix row, there is no need to link
                                     // to the row before the current one...
         if (pxMatrixCell->pxPreviousRow != NULL)
         {
            pNewMatrixCell->pxPreviousRow = pxMatrixCell->pxPreviousRow;
            pxMatrixCell->pxPreviousRow->pxNextRow = pNewMatrixCell;
         }
         pxMatrixCell->pxPreviousRow = pNewMatrixCell;
         pNewMatrixCell->pxNextRow = pxMatrixCell;
                                     // link the columns
         if (pxPreviousNewMatrixCell == NULL)
         {
            pxPreviousNewMatrixCell = pNewMatrixCell;
         }
         else
         {
            pxPreviousNewMatrixCell->pxNextColumn = pNewMatrixCell;
            pNewMatrixCell->pxPreviousColumn = pxPreviousNewMatrixCell;
            pxPreviousNewMatrixCell = pNewMatrixCell;
         }
         pxMatrixCell = pxMatrixCell->pxNextColumn;
      }

      if (ulRowInsertionPoint == 0)
      {
                              // if we inserted before (above) the first
                              // row of the matrix, move the matrix 
                              // starting ptr
         pxMatrixStart = pxMatrixStart->pxPreviousRow;
      }
   }
   else 
   {
      LOG_ERROR("Invalid insert direction");
      return(1);
   }

                                     // increment matrix row count       
   ulNumMatrixRows++;

   return(0);
}



/*----------------------------------------------------------------------------*\
|                                                                              |
| Function:  Insert a new column in the DynamicMatrix before or after the      |
|            specified column.                                                 |
|                                                                              |
| Returns:   INT                                                               |
|            On failure, a non-zero value is returned.                         |
|                                                                              |
\*----------------------------------------------------------------------------*/
template<class DataType> INT
DynamicMatrix<DataType>::
iInsertColumn(ULONG ulColumnInsertionPoint,
                                     // IN  matrix column to insert before or 
                                     //     after 
              INT iInsertDirection
                                     // IN  enumeration specifying to insert
                                     //     the new matrix column before or
                                     //     after the specified column             
             )
{

   ULONG ulLoop;
   MatrixCell<DataType>* pxMatrixCell;
   MatrixCell<DataType>* pNewMatrixCell;
   MatrixCell<DataType>* pxPreviousNewMatrixCell;

   if (ulNumMatrixColumns == 0)
   {
      LOG_ERROR("attempt to insert column in matrix with 0 columns");
      return(1); 
   }

   if (ulColumnInsertionPoint >= ulNumMatrixColumns)
   {
      sprintf(achErrorString, 
              "attempt to insert column outside matrix boundry, %u, %uX%u",
              ulColumnInsertionPoint+1, ulNumMatrixRows, ulNumMatrixColumns); 
      LOG_ERROR(achErrorString);
      return(1); 
   }

                                     // find the column to insert before or 
                                     // after
   pxMatrixCell = pxMatrixStart;
                                     // traverse links to specified row        
   for (ulLoop=0; ulLoop<ulColumnInsertionPoint; ulLoop++)
   {
      pxMatrixCell = pxMatrixCell->pxNextColumn;
   }

   pxPreviousNewMatrixCell = NULL;
                                     // insert matrix column at appropriate 
                                     // location   
   if (iInsertDirection == AFTER)
   {
                                     // insert matrix column after the
                                     // current one
      for (ulLoop=0; ulLoop<ulNumMatrixRows; ulLoop++)
      {
         pNewMatrixCell = new MatrixCell<DataType>;
         CHECK_NEW(pNewMatrixCell);
                                     // link new cell with the corresponding
                                     // one in the column to the right and
                                     // the current one

                                     // if we are inserting after the last
                                     // matrix column, there is no need to link
                                     // to the column after the current one...
         if (pxMatrixCell->pxNextColumn != NULL)
         {
            pNewMatrixCell->pxNextColumn = pxMatrixCell->pxNextColumn;
            pxMatrixCell->pxNextColumn->pxPreviousColumn = pNewMatrixCell;
         }
         pxMatrixCell->pxNextColumn = pNewMatrixCell;
         pNewMatrixCell->pxPreviousColumn = pxMatrixCell;
                                     // link the columns
         if (pxPreviousNewMatrixCell == NULL)
         {
            pxPreviousNewMatrixCell = pNewMatrixCell;
         }
         else
         {
            pxPreviousNewMatrixCell->pxNextRow = pNewMatrixCell;
            pNewMatrixCell->pxPreviousRow = pxPreviousNewMatrixCell;
            pxPreviousNewMatrixCell = pNewMatrixCell;
         }
         pxMatrixCell = pxMatrixCell->pxNextRow;
      }
   }
   else if (iInsertDirection == BEFORE)
   {
                                     // insert matrix column before the
                                     // current one
      for (ulLoop=0; ulLoop<ulNumMatrixRows; ulLoop++)
      {
         pNewMatrixCell = new MatrixCell<DataType>;
         CHECK_NEW(pNewMatrixCell);
                                     // link new cell with the corresponding
                                     // one in the column to the left and
                                     // the current one

                                     // if we are inserting before the first
                                     // matrix column, there is no need to link
                                     // to the column before the current one...
         if (pxMatrixCell->pxPreviousColumn != NULL)
         {
            pNewMatrixCell->pxPreviousColumn = pxMatrixCell->pxPreviousColumn;
            pxMatrixCell->pxPreviousColumn->pxNextColumn = pNewMatrixCell;
         }
         pxMatrixCell->pxPreviousColumn = pNewMatrixCell;
         pNewMatrixCell->pxNextColumn = pxMatrixCell;
                                     // link the columns
         if (pxPreviousNewMatrixCell == NULL)
         {
            pxPreviousNewMatrixCell = pNewMatrixCell;
         }
         else
         {
            pxPreviousNewMatrixCell->pxNextRow = pNewMatrixCell;
            pNewMatrixCell->pxPreviousRow = pxPreviousNewMatrixCell;
            pxPreviousNewMatrixCell = pNewMatrixCell;
         }
         pxMatrixCell = pxMatrixCell->pxNextRow;
      }

      if (ulColumnInsertionPoint == 0)
      {
                              // if we inserted before the first column
                              // of the matrix, move the matrix starting ptr
         pxMatrixStart = pxMatrixStart->pxPreviousColumn;
      }
   }
   else 
   {
      LOG_ERROR("Invalid insert direction");
      return(1);
   }

                                     // increment matrix row count       
   ulNumMatrixColumns++;

   return(0);
}



/*----------------------------------------------------------------------------*\
|                                                                              |
| Function:  Delete the specified DynamicMatrix row.                           |
|                                                                              |
| Returns:   INT                                                               |
|            On failure, a non-zero value is returned.                         |
|                                                                              |
\*----------------------------------------------------------------------------*/
template<class DataType> INT
DynamicMatrix<DataType>::
iDeleteRow(ULONG ulRowToDelete // IN  matrix row to delete
          )
{

   ULONG ulLoop;
   MatrixCell<DataType>* pxMatrixCell;
   MatrixCell<DataType>* pxPreviousMatrixCell;
   MatrixCell<DataType>* pxNextMatrixCell;

   if (ulNumMatrixRows == 0)
   {
      LOG_ERROR("attempt to delete row from matrix with 0 rows");
      return(1); 
   }

   if (ulRowToDelete >= ulNumMatrixRows)
   {
      sprintf(achErrorString, 
              "attempt to delete matrix row outside matrix boundry, %u, %uX%u",
              ulRowToDelete+1, ulNumMatrixRows, ulNumMatrixColumns); 
      LOG_ERROR(achErrorString);
      return(1); 
   }

                                     // only one row to delete
   if (ulNumMatrixRows == 1)
   {
      delete pxMatrixStart;
      pxMatrixStart=NULL;
      ulNumMatrixRows = 0;
      ulNumMatrixColumns = 0;
      return(0);
   }

                                     // if deleteing first row
   if (ulRowToDelete == 0)
   {
      pxMatrixStart = pxMatrixStart->pxNextRow;
      pxMatrixCell = pxMatrixStart;

      for (ulLoop=0; ulLoop<ulNumMatrixColumns; ulLoop++)
      {
         delete pxMatrixCell->pxPreviousRow;
         pxMatrixCell->pxPreviousRow = NULL;
         pxMatrixCell = pxMatrixCell->pxNextColumn;
      }

      ulNumMatrixRows--;
      return(0);
   }
                                     // find the row to delete
                                     // traverse links to specified row        
   pxMatrixCell = pxMatrixStart;
   for (ulLoop=0; ulLoop<ulRowToDelete; ulLoop++)
   {
      pxMatrixCell = pxMatrixCell->pxNextRow;
   }

                                     // if deleteing last row
   if (ulRowToDelete == (ulNumMatrixRows-1))
   {
      pxMatrixCell = pxMatrixCell->pxPreviousRow;

      for (ulLoop=0; ulLoop<ulNumMatrixColumns; ulLoop++)
      {
         delete pxMatrixCell->pxNextRow;
         pxMatrixCell->pxNextRow = NULL;
         pxMatrixCell = pxMatrixCell->pxNextColumn;
      }

      ulNumMatrixRows--;
      return(0);
   }

                                     // at this point, we know there is
                                     // a row above and below the one we
                                     // are going to delete
   pxPreviousMatrixCell = pxMatrixCell->pxPreviousRow;
   pxNextMatrixCell = pxMatrixCell->pxNextRow;

                                     // link the previous cells to the next
                                     // cells...                           
   for (ulLoop=0; ulLoop<ulNumMatrixColumns; ulLoop++)
   {
      pxPreviousMatrixCell->pxNextRow = pxNextMatrixCell;
      pxNextMatrixCell->pxPreviousRow = pxPreviousMatrixCell;

      pxPreviousMatrixCell = pxPreviousMatrixCell->pxNextColumn;
      pxNextMatrixCell= pxNextMatrixCell->pxNextColumn;
   }

   ulNumMatrixRows--;

                                     // now delete the row...
   for (ulLoop=0; ulLoop<(ulNumMatrixColumns-1); ulLoop++)
   {
      pxMatrixCell = pxMatrixCell->pxNextColumn;
      delete pxMatrixCell->pxPreviousColumn;
   }
   delete pxMatrixCell;


   return(0);
}



/*----------------------------------------------------------------------------*\
|                                                                              |
| Function:  Delete the specified DynamicMatrix column.                        |
|                                                                              |
| Returns:   INT                                                               |
|            On failure, a non-zero value is returned.                         |
|                                                                              |
\*----------------------------------------------------------------------------*/
template<class DataType> INT
DynamicMatrix<DataType>::
iDeleteColumn(ULONG ulColumnToDelete
                                     // IN  matrix column to delete
              )
{

   ULONG ulLoop;
   MatrixCell<DataType>* pxMatrixCell;
   MatrixCell<DataType>* pxPreviousMatrixCell;
   MatrixCell<DataType>* pxNextMatrixCell;

   if (ulNumMatrixColumns == 0)
   {
      LOG_ERROR("attempt to delete column from matrix with 0 columns");
      return(1); 
   }

   if (ulColumnToDelete >= ulNumMatrixColumns)
   {
      sprintf(achErrorString, 
              "attempt to delete columns outside matrix boundry, %u, %uX%u",
              ulColumnToDelete+1, ulNumMatrixRows, ulNumMatrixColumns); 
      LOG_ERROR(achErrorString);
      return(1); 
   }

                                     // only one column to delete
   if (ulNumMatrixColumns == 1)
   {
      delete pxMatrixStart;
      pxMatrixStart=NULL;
      ulNumMatrixRows = 0;
      ulNumMatrixColumns = 0;
      return(0);
   }

                                     // if deleteing first columns
   if (ulColumnToDelete == 0)
   {
      pxMatrixStart = pxMatrixStart->pxNextColumn;
      pxMatrixCell = pxMatrixStart;

      for (ulLoop=0; ulLoop<ulNumMatrixRows; ulLoop++)
      {
         delete pxMatrixCell->pxPreviousColumn;
         pxMatrixCell->pxPreviousColumn = NULL;
         pxMatrixCell = pxMatrixCell->pxNextRow;
      }

      ulNumMatrixColumns--;
      return(0);
   }
                                     // find the column to delete
                                     // traverse links to specified column
   pxMatrixCell = pxMatrixStart;
   for (ulLoop=0; ulLoop<ulColumnToDelete; ulLoop++)
   {
      pxMatrixCell = pxMatrixCell->pxNextColumn;
   }

                                     // if deleteing last column
   if (ulColumnToDelete == (ulNumMatrixColumns-1))
   {
      pxMatrixCell = pxMatrixCell->pxPreviousColumn;

      for (ulLoop=0; ulLoop<ulNumMatrixRows; ulLoop++)
      {
         delete pxMatrixCell->pxNextColumn;
         pxMatrixCell->pxNextColumn = NULL;
         pxMatrixCell = pxMatrixCell->pxNextRow;
      }

      ulNumMatrixColumns--;
      return(0);
   }

                                     // at this point, we know there is
                                     // a column to the left and right of 
                                     // the one we are going to delete
   pxPreviousMatrixCell = pxMatrixCell->pxPreviousColumn;
   pxNextMatrixCell = pxMatrixCell->pxNextColumn;

                                     // link the previous cells to the next
                                     // cells...                           
   for (ulLoop=0; ulLoop<ulNumMatrixRows; ulLoop++)
   {
      pxPreviousMatrixCell->pxNextColumn = pxNextMatrixCell;
      pxNextMatrixCell->pxPreviousColumn = pxPreviousMatrixCell;

      pxPreviousMatrixCell = pxPreviousMatrixCell->pxNextRow;
      pxNextMatrixCell= pxNextMatrixCell->pxNextRow;
   }

   ulNumMatrixColumns--;

                                     // now delete the column...
   for (ulLoop=0; ulLoop<(ulNumMatrixRows-1); ulLoop++)
   {
      pxMatrixCell = pxMatrixCell->pxNextRow;
      delete pxMatrixCell->pxPreviousRow;
   }
   delete pxMatrixCell;


   return(0);
}



/*----------------------------------------------------------------------------*\
|                                                                              |
| Function:  Copy a matrix row to another matrix row within the same matrix.   |
|                                                                              |
| Returns:   INT                                                               |
|            On failure, a non-zero value is returned.                         |
|                                                                              |
\*----------------------------------------------------------------------------*/
template<class DataType> INT
DynamicMatrix<DataType>::
iCopyRow(ULONG ulFromRow,            // IN  matrix row to copy cells from
         ULONG ulToRow               // IN  matrix row to copy "from" cells to 
        )
{

   ULONG ulLoop;
   MatrixCell<DataType>* pFromMatrixCell;
   MatrixCell<DataType>* pToMatrixCell;


   if (ulNumMatrixRows == 0)
   {
      LOG_ERROR("attempt to copy row in matrix with 0 rows");
      return(1); 
   }

   if (ulFromRow >= ulNumMatrixRows)
   {
      sprintf(achErrorString, 
              "attempt to copy matrix row from invalid row, %u, %uX%u",
              ulFromRow+1, ulNumMatrixRows, ulNumMatrixColumns); 
      LOG_ERROR(achErrorString);
      return(1); 
   }

   if (ulToRow >= ulNumMatrixRows)
   {
      sprintf(achErrorString, 
              "attempt to copy matrix row to invalid row, %u, %uX%u",
              ulToRow+1, ulNumMatrixRows, ulNumMatrixColumns); 
      LOG_ERROR(achErrorString);
      return(1); 
   }

   if (ulFromRow == ulToRow)
   {
                                     // nothing to copy...    
      return(0); 
   }


                                     // find the row to copy from and to
   pFromMatrixCell = pxMatrixStart;
   for (ulLoop=0; ulLoop<ulFromRow; ulLoop++)
   {
      pFromMatrixCell = pFromMatrixCell->pxNextRow;
   }

   pToMatrixCell = pxMatrixStart;
   for (ulLoop=0; ulLoop<ulToRow; ulLoop++)
   {
      pToMatrixCell = pToMatrixCell->pxNextRow;
   }

                                     // now, copy the data
   for (ulLoop=0; ulLoop<ulNumMatrixColumns; ulLoop++)
   {
      pToMatrixCell->xCellData = pFromMatrixCell->xCellData;

      pFromMatrixCell = pFromMatrixCell->pxNextColumn;
      pToMatrixCell= pToMatrixCell->pxNextColumn;
   }

   return(0);
}



/*----------------------------------------------------------------------------*\
|                                                                              |
| Function:  Copy a matrix column to another matrix column within the same     |
|            matrix.                                                           |
|                                                                              |
| Returns:   INT                                                               |
|            On failure, a non-zero value is returned.                         |
|                                                                              |
\*----------------------------------------------------------------------------*/
template<class DataType> INT
DynamicMatrix<DataType>::
iCopyColumn(ULONG ulFromColumn,      // IN  matrix column to copy cells from
            ULONG ulToColumn         // IN  matrix column to copy "from" cells
                                     //     to
           )
{

   INT ulLoop;
   MatrixCell<DataType>* pFromMatrixCell;
   MatrixCell<DataType>* pToMatrixCell;


   if (ulNumMatrixColumns == 0)
   {
      LOG_ERROR("attempt to insert column in matrix with 0 columns");
      return(1); 
   }

   if (ulFromColumn >= ulNumMatrixColumns)
   {
      sprintf(achErrorString, 
              "attempt to copy matrix column from invalid column, %u, %uX%u",
              ulFromColumn+1, ulNumMatrixRows, ulNumMatrixColumns); 
      LOG_ERROR(achErrorString);
      return(1); 
   }

   if (ulToColumn >= ulNumMatrixColumns)
   {
      sprintf(achErrorString, 
              "attempt to copy matrix column to invalid column, %u, %uX%u",
              ulToColumn+1, ulNumMatrixRows, ulNumMatrixColumns); 
      LOG_ERROR(achErrorString);
      return(1); 
   }

   if (ulFromColumn == ulToColumn)
   {
                                     // nothing to copy...    
      return(0); 
   }


                                     // find the column to copy from and to
   pFromMatrixCell = pxMatrixStart;
   for (ulLoop=0; ulLoop<ulFromColumn; ulLoop++)
   {
      pFromMatrixCell = pFromMatrixCell->pxNextColumn;
   }

   pToMatrixCell = pxMatrixStart;
   for (ulLoop=0; ulLoop<ulToColumn; ulLoop++)
   {
      pToMatrixCell = pToMatrixCell->pxNextColumn;
   }

                                     // now, copy the data
   for (ulLoop=0; ulLoop<ulNumMatrixRows; ulLoop++)
   {
      pToMatrixCell->xCellData = pFromMatrixCell->xCellData;

      pFromMatrixCell = pFromMatrixCell->pxNextRow;
      pToMatrixCell= pToMatrixCell->pxNextRow;
   }

   return(0);
}



/*----------------------------------------------------------------------------*\
|                                                                              |
| Function:  Copy a matrix row to a row in a different matrix.                 |
|                                                                              |
| Returns:   INT                                                               |
|            On failure, a non-zero value is returned.                         |
|                                                                              |
\*----------------------------------------------------------------------------*/
template<class DataType> INT
DynamicMatrix<DataType>::
iCopyRow(DynamicMatrix<DataType>* pFromMatrix,
                                     // IN  matrix to copy row from
         ULONG ulFromRow,            // IN  matrix row to copy cells from 
         DynamicMatrix<DataType>* pToMatrix,
                                     // IO  matrix to copy "from" row to
         ULONG ulToRow               // IN  matrix row to copy "from" cells to
        )
{

   INT ulLoop;
   MatrixCell<DataType>* pFromMatrixCell;
   MatrixCell<DataType>* pToMatrixCell;


                                     // make sure the number of columns in each
                                     // matrix row are the same
   if (pFromMatrix->ulNumMatrixColumns != pToMatrix->ulNumMatrixColumns)
   {
      sprintf(achErrorString,
              "attempt to cp row in matrixes with diff size columns, %u, %u",
               pFromMatrix->ulNumMatrixColumns, pToMatrix->ulNumMatrixColumns);
      LOG_ERROR(achErrorString);
      return(1); 
   }

                                     // make sure we have non-zero size matrix
   if (pFromMatrix->ulNumMatrixRows == 0)
   {
      LOG_ERROR("attempt to copy row from matrix with 0 rows");
      return(1); 
   }

   if (pToMatrix->ulNumMatrixRows == 0)
   {
      LOG_ERROR("attempt to copy row to matrix with 0 rows");
      return(1); 
   }

                                     // make sure "to" and "from" are within
                                     // their matrix boundaries
   if ((ulFromRow >= pFromMatrix->ulNumMatrixRows) || (ulFromRow < 0))
   {
      sprintf(achErrorString, 
              "attempt to copy matrix row from invalid row, %u, %uX%u",
              ulFromRow+1, ulNumMatrixRows, ulNumMatrixColumns); 
      LOG_ERROR(achErrorString);
      return(1); 
   }

   if ((ulToRow >= pToMatrix->ulNumMatrixRows) || (ulToRow < 0))
   {
      sprintf(achErrorString, 
              "attempt to copy matrix row to invalid row, %u, %uX%u",
              ulFromRow+1, ulNumMatrixRows, ulNumMatrixColumns); 
      LOG_ERROR(achErrorString);
      return(1); 
   }


                                     // find the row to copy "from" and "to"
   pFromMatrixCell = pFromMatrix->pxMatrixStart;
   for (ulLoop=0; ulLoop<ulFromRow; ulLoop++)
   {
      pFromMatrixCell = pFromMatrixCell->pxNextRow;
   }

   pToMatrixCell   = pToMatrix->pxMatrixStart;
   for (ulLoop=0; ulLoop<ulToRow; ulLoop++)
   {
      pToMatrixCell = pToMatrixCell->pxNextRow;
   }

                                     // now, copy the data
   for (ulLoop=0; ulLoop<pToMatrix->ulNumMatrixColumns; ulLoop++)
   {
      pToMatrixCell->xCellData = pFromMatrixCell->xCellData;

      pFromMatrixCell = pFromMatrixCell->pxNextColumn;
      pToMatrixCell= pToMatrixCell->pxNextColumn;
   }

   return(0);
}



/*----------------------------------------------------------------------------*\
|                                                                              |
| Function:  Copy a matrix column to a column in a different matrix.           |
|                                                                              |
| Returns:   INT                                                               |
|            On failure, a non-zero value is returned.                         |
|                                                                              |
\*----------------------------------------------------------------------------*/
template<class DataType> INT
DynamicMatrix<DataType>::
iCopyColumn(DynamicMatrix<DataType>* pFromMatrix,
                                     // IN  matrix to copy column from
            ULONG ulFromRow,         // IN  matrix column to copy cells from
            DynamicMatrix<DataType>* pToMatrix,
                                     // IO  matrix to copy column to
            ULONG ulToRow            // IN  matrix column to copy "from" cells
                                     //     to
           )
{

   INT ulLoop;
   MatrixCell<DataType>* pFromMatrixCell;
   MatrixCell<DataType>* pToMatrixCell;


                                     // make sure the number of rows in each
                                     // matrix column are the same
   if (pFromMatrix->ulNumMatrixRows != pToMatrix->ulNumMatrixRows)
   {
      sprintf(achErrorString,
              "attempt to cp column in matrixes with diff size rows, %u, %u",
               pFromMatrix->ulNumMatrixRows, pToMatrix->ulNumMatrixRows);
      LOG_ERROR(achErrorString);
      return(1); 
   }

                                     // make sure we have non-zero size matrix
   if (pFromMatrix->ulNumMatrixColumns == 0)
   {
      LOG_ERROR("attempt to copy column from matrix with 0 columns");
      return(1); 
   }

   if (pToMatrix->ulNumMatrixColumns == 0)
   {
      LOG_ERROR("attempt to copy column to matrix with 0 columns");
      return(1); 
   }


                                     // make sure "to" and "from" are within
                                     // their matrix boundaries
   if (ulFromRow >= pFromMatrix->ulNumMatrixColumns)
   {
      sprintf(achErrorString, 
              "attempt to copy matrix column from invalid column, %u, %uX%u",
              ulFromRow+1, ulNumMatrixRows, ulNumMatrixColumns); 
      LOG_ERROR(achErrorString);
      return(1); 
   }

   if (ulToRow >= pToMatrix->ulNumMatrixColumns)
   {
      sprintf(achErrorString, 
              "attempt to copy matrix column to invalid column, %u, %uX%u",
              ulFromRow+1, ulNumMatrixRows, ulNumMatrixColumns); 
      LOG_ERROR(achErrorString);
      return(1); 
   }


                                     // find the column to copy "from" and "to"
   pFromMatrixCell = pFromMatrix->pxMatrixStart;
   for (ulLoop=0; ulLoop<ulFromRow; ulLoop++)
   {
      pFromMatrixCell = pFromMatrixCell->pxNextColumn;
   }

   pToMatrixCell   = pToMatrix->pxMatrixStart;
   for (ulLoop=0; ulLoop<ulToRow; ulLoop++)
   {
      pToMatrixCell = pToMatrixCell->pxNextColumn;
   }

                                     // now, copy the data
   for (ulLoop=0; ulLoop<pToMatrix->ulNumMatrixRows; ulLoop++)
   {
      pToMatrixCell->xCellData = pFromMatrixCell->xCellData;

      pFromMatrixCell = pFromMatrixCell->pxNextRow;
      pToMatrixCell= pToMatrixCell->pxNextRow;
   }

   return(0);
}



/*----------------------------------------------------------------------------*\
|                                                                              |
| Function:  Return a pointer to the specified matrix row.                     |
|                                                                              |
| Returns:   DataType*                                                         |
|            On failure, a NULL pointer is returned.                           |
|                                                                              |
\*----------------------------------------------------------------------------*/
template<class DataType> MatrixCell<DataType>*
DynamicMatrix<DataType>::
pxGetRow(ULONG ulRowToGet             // IN  matrix row to delete
        )
{

   INT ulLoop;
   MatrixCell<DataType>* pxMatrixCell;

   if (ulNumMatrixRows == 0)
   {
      LOG_ERROR("attempt to get row from matrix with 0 rows");
      return((MatrixCell<DataType>*) NULL); 
   }

   if ((ulRowToGet >= ulNumMatrixRows) || (ulRowToGet < 0))
   {
      sprintf(achErrorString, 
              "attempt to get matrix row outside matrix boundry, %u, %uX%u",
              ulRowToGet+1, ulNumMatrixRows, ulNumMatrixColumns); 
      LOG_ERROR(achErrorString);
      return((MatrixCell<DataType>*) NULL); 
   }

                                     // find the row to return a ptr to
   pxMatrixCell = pxMatrixStart;
   for (ulLoop=0; ulLoop<ulRowToGet; ulLoop++)
   {
      pxMatrixCell = pxMatrixCell->pxNextRow;
   }

   return(pxMatrixCell);
}



/*----------------------------------------------------------------------------*\
|                                                                              |
| Function:  Return a pointer to the specified matrix column.                  |
|                                                                              |
| Returns:   DataType*                                                         |
|            On failure, a NULL pointer is returned.                           |
|                                                                              |
\*----------------------------------------------------------------------------*/
template<class DataType> MatrixCell<DataType>*
DynamicMatrix<DataType>::
pxGetColumn(ULONG ulColumnToGet       // IN  matrix row to delete
           )
{

   INT ulLoop;
   MatrixCell<DataType>* pxMatrixCell;

   if (ulNumMatrixColumns == 0)
   {
      LOG_ERROR("attempt to get column from matrix with 0 columns");
      return((MatrixCell<DataType>*) NULL); 
   }

   if ((ulColumnToGet >= ulNumMatrixColumns) || (ulColumnToGet < 0))
   {
      sprintf(achErrorString, 
              "attempt to get matrix column outside matrix boundry, %u, %uX%u",
              ulColumnToGet+1, ulNumMatrixRows, ulNumMatrixColumns); 
      LOG_ERROR(achErrorString);
      return((MatrixCell<DataType>*) NULL); 
   }

                                     // find the column to return a ptr to
   pxMatrixCell = pxMatrixStart;
   for (ulLoop=0; ulLoop<ulColumnToGet; ulLoop++)
   {
      pxMatrixCell = pxMatrixCell->pxNextColumn;
   }

   return(pxMatrixCell);
}


