/*----------------------------------------------------------------------------*\
|
| Copyright (C) 1997, 2000  James A. Cureington
|                           tonyc@acm.org
| 
| This program is free software; you can redistribute it and/or
| modify it under the terms of the GNU General Public License
| as published by the Free Software Foundation; either version 2
| of the License, or any later version.
| 
| This program is distributed in the hope that it will be useful,
| but WITHOUT ANY WARRANTY; without even the implied warranty of
| MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
| GNU General Public License for more details.
| 
| You should have received a copy of the GNU General Public License
| along with this program; if not, write to the Free Software
| Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
|
|
| Tony Cureington                                                              
| September 14, 1996                                                           
|                                                                              
| Module:  DynamicArray template implementations                               
|          This file contains templates for the DynamicArray templates.        
|          The DynamicArray templates allow insertion and deletion of array    
|          cells from the array.  As cells are added, memory is allocated for 
|          them dynamically.  When memory is allocated, more than that required 
|          by a single cell is allocated.  This prevents the next insertion of 
|          an array cell from needing to allocate more memory, therefore 
|          increasing the performance.  In addition, a hysteresis is 
|          incorporated to prevent a "yo-yo" effect when a cell is allocated 
|          and then emediately deallocated. 
|                                                                              
| $Id: d_array.C,v 1.1 2001/03/04 00:05:14 tonycu Exp $                          
|                                                                              
| $Log: d_array.C,v $
| Revision 1.1  2001/03/04 00:05:14  tonycu
| initial check-in at sourceforge...
|
| Revision 1.9  2000/05/24 03:19:01  tony
| Updated all copyright dates.
|
| Revision 1.8  1999/09/13 01:21:13  tony
|
| Updated copyright notice date
|
| Revision 1.7  1999/09/12 22:03:56  tony
|
| Changed #defines to const and inline functions...
|
 * Revision 1.6  1997/03/01  23:24:14  tony
 *      Made the following changes:
 *           - Add version to FSK file for validation when reading in
 *           - Check for " in parameter list
 *           - Add CVS Id to source and header file
 *           - Add copyright notice to header files
 *           - Add EDIT menu item and associated entries
 *           - Make mouse button 3 cause the commnad pop-up to be displayed
 *             instead of mouse button 2
 *           - Add box to insert header files into
 *           - Changed pchz to pch
 *           - Allow macros to be used in parameter list
 *           - Add function and associated calls to check for infinite state
 *             condition; this is a warning, not an error condition
 *           - Add over-ride abort function capability to GUI
 *           - Changed "modified save pop-up" window to be displayed before
 *             any window is displayed to read and RSK, or create a new one
 *
 * Revision 1.5  1997/01/26  04:57:18  tony
 * Made changes to:
 *      - insert user defined default function next state when a state or
 *        event is inserted
 *      - stub out future functions
 *      - change tabs to spaces
 *
 * Revision 1.4  1997/01/13  08:22:56  tony
 * added code to:
 *        - delete states
 *        - insert states
 *        - search dynamic matrix diagonally
 *        - iGetMatrixCell, to chain through the matrix cells using the links
 *
 * Revision 1.3  1996/12/13  04:16:20  tony
 * 1. Moved public arrays in classes to private and added const pointers to
 *    the private members
 *
 * 2. Added assignment operators to classes
 *
 * 3. Changed the dynamic array class - replaced malloc/free with new/delete
 *    so it could work with all objects
 *
 * Revision 1.2  1996/11/27  06:16:03  tony
 * added more member functions to classes and cleaned up code a bit
 *
|* Revision 1.1  1996/09/28  16:41:44  tony
|* initial check-in
|*                                                                      
|                                                                              
\*----------------------------------------------------------------------------*/
#include <stdio.h>
#include <errno.h>
#include <string.h>

#include <d_array.h>

STATIC CHAR *pchDarrayId = "@(#) $Id: d_array.C,v 1.1 2001/03/04 00:05:14 tonycu Exp $";

/*----------------------------------------------------------------------------*\
|                                                                              |
| Function:  Constructor to allocate a DynamicArray of the specified size for  |
|            the specified data type.                                          |
|                                                                              |
| Returns:   none                                                              |
|            On failure, abort is called.                                      |
|                                                                              |
\*----------------------------------------------------------------------------*/
template<class DataType>
DynamicArray<DataType>::
DynamicArray(size_t uiNewArraySize   // IN  size of the new array
            ) 
{
   if (uiNewArraySize > 0)
   {                                 // allocate memory for array 
      uiUserArraySize    = uiNewArraySize;
      uiActualArraySize  = uiNewArraySize;

      pxArray = new DataType[uiUserArraySize];
      CHECK_NEW(pxArray);

      pxInitializedArrayCell = new DataType[1];
      CHECK_NEW(pxInitializedArrayCell);
   }
   else
   {
      uiUserArraySize         = 0;
      uiActualArraySize       = 0;
      pxArray                 = NULL;
      pxInitializedArrayCell  = NULL;
   }

   uiArrayIncreaseAmount = 30;
   uiArrayHysteresis     = 15;
}



/*----------------------------------------------------------------------------*\
|                                                                              |
| Function:  Destructor to deallocate all memory associated with a previousley |
|            instantiated DynamicArray.                                        |
|                                                                              |
| Returns:   none                                                              |
|                                                                              |
\*----------------------------------------------------------------------------*/
template<class DataType>
DynamicArray<DataType>::
~DynamicArray() 
{
   delete[] pxArray;
   delete[] pxInitializedArrayCell;
}



/*----------------------------------------------------------------------------*\
|                                                                              |
| Function:  Index into a DynamicArray and return a reference to the data      |
|            element at the specified location.  Also make sure the index is   |
|            within the array boundaries.                                      |
|                                                                              |
| Returns:   DataType&                                                         | 
|            Reference to the data element at the specified location.          |
|            On failure, abort is called since a return value is not           |
|            usually check for an operator of this type.                       |
|                                                                              |
\*----------------------------------------------------------------------------*/
template<class DataType> DataType& 
DynamicArray<DataType>::
operator[](size_t uiIndex            // IN  index of the array cell to return
          )
{
   
                                     // check for errors                                                   
   if ((uiUserArraySize == 0) ||
       (uiIndex >= uiUserArraySize))
   {
      sprintf(achErrorString, "bad value %u, %u, %u",
              uiUserArraySize, uiIndex, uiUserArraySize); 
      LOG_ERROR(achErrorString);

                                     // we have errors, since this is an 
                                     // over-loaded operator, and we don't
                                     // usually check for errors on this type
                                     // of operator, stop everything here and 
                                     // dump, core so we can see whats going on
      abort();
   }

   return(pxArray[uiIndex]);
}



/*----------------------------------------------------------------------------*\
|                                                                              |
| Function:  Deallocate all memory allocated for the DynamicArray, and         |
|            allocates a new array of the specified size.                      |
|                                                                              |
| Returns:   INT                                                               |
|            On failure, a non-zero value is returned.                         |
|                                                                              |
\*----------------------------------------------------------------------------*/
template<class DataType> INT
DynamicArray<DataType>::
iSetSize(size_t uiNewArraySize  // IN  number of cells to allocate in the 
                                //     new dynamic array
        )
{

   if (pxArray != NULL)
   {                                 // delete existing array memory
      delete[] pxArray;
      delete[] pxInitializedArrayCell;
   }

   if (uiNewArraySize > 0)
   {                                 // allocate memory for new array 
      uiUserArraySize    = uiNewArraySize;
      uiActualArraySize  = uiNewArraySize;
      pxArray            = new DataType[uiNewArraySize];
      CHECK_NEW(pxArray);
      pxInitializedArrayCell = new DataType[1];
      CHECK_NEW(pxInitializedArrayCell);
   }
   else
   {
      uiUserArraySize         = 0;
      uiActualArraySize       = 0;
      pxArray                 = NULL;
      pxInitializedArrayCell  = NULL;
   }
   return(0);
}



/*----------------------------------------------------------------------------*\
|                                                                              |
| Function:  Return the size of the DynamicArray.                              |
|                                                                              |
| Returns:   UINT                                                              |
|            The size of the DynamicArray.                                     |
|                                                                              |
\*----------------------------------------------------------------------------*/
template<class DataType> UINT
DynamicArray<DataType>::
uiGetSize() 
{
   return(uiUserArraySize);
}



/*----------------------------------------------------------------------------*\
|                                                                              |
| Function:  Insert an array cell before or after the specified array cell.    |
|                                                                              |
| Returns:   INT                                                               |
|            On failure, a non-zero value is returned.                         |
|                                                                              |
\*----------------------------------------------------------------------------*/
template<class DataType> INT
DynamicArray<DataType>::
iInsertCell(size_t uiIndex,          // IN  array cell index to insert a cell
                                     //     before or after
            INT iInsertDirection
                                     // IN  enumeration specifying whether to
                                     //     insert the cell before or after  
                                     //     the given index                  
           ) 
{

   UINT uiLoop;
   DataType* pxNewArray;

   if (uiUserArraySize == 0)
   {
      LOG_ERROR("attempt to insert cell in array of size 0");
      return(1); 
   }

   if (uiIndex >= uiUserArraySize)
   {
      sprintf(achErrorString, 
              "attempt to insert cell outside array boundry, %u, %u",
              uiIndex+1, uiUserArraySize); 
      LOG_ERROR(achErrorString);
      return(1); 
   }

                                     // usr array size should never be greater
                                     // than the actual array size            
   ASSERT(uiUserArraySize <= uiActualArraySize);

   if (uiUserArraySize == uiActualArraySize)
   {
                                     // we need to add more cells to the array
      uiActualArraySize += uiArrayIncreaseAmount;

      pxNewArray = new DataType [uiActualArraySize];
      CHECK_NEW(pxNewArray);

                                     // copy pxArray to pxNewArray 
      for (uiLoop=0; uiLoop<uiUserArraySize; uiLoop++)
      {
         pxNewArray[uiLoop] = pxArray[uiLoop];
      }
                                     // delete the array
      delete[] pxArray;
                                     // set the array to the new array
      pxArray = pxNewArray;
   }

                                     // insert cell at appropriate location   
   if (iInsertDirection == AFTER)
   {
      if ((uiIndex+1) == uiUserArraySize)
      {
                                     // insert cell after last array cell       
                                     // just increasing the array size inserts 
                                     // a new cell after last cell of the array
         uiUserArraySize++;
         pxArray[uiUserArraySize] = pxInitializedArrayCell[0];
      }
      else
      {
                                     // insert cell after a cell between the 
                                     // first and last cells of the array                    
         for (uiLoop=uiUserArraySize; uiLoop>=uiIndex+1; uiLoop--)
         {
            pxArray[uiLoop] = pxArray[uiLoop-1];
         }

         pxArray[uiIndex+1] = pxInitializedArrayCell[0];

         uiUserArraySize++;
      }
   }
   else if (iInsertDirection == BEFORE)
   {
      if (uiIndex == 0)
      {
                                     // insert cell before first cell of array
                                     // hence, shift all cells to the right by
                                     // one                                   
         for (uiLoop=uiUserArraySize; uiLoop>0; uiLoop--)
         {
            pxArray[uiLoop] = pxArray[uiLoop-1];
         }

         pxArray[0] = pxInitializedArrayCell[0];
         uiUserArraySize++;
      }
      else
      {
                                     // insert cell before a cell between the 
                                     // first and last cells of the array                    
         uiUserArraySize++;
         for (uiLoop=uiUserArraySize; uiLoop>=uiIndex; uiLoop--)
         {
            pxArray[uiLoop] = pxArray[uiLoop-1];
         }

         pxArray[uiIndex] = pxInitializedArrayCell[0];
      }
   }
   else
   {
      LOG_ERROR("invalid array insert direction");
      return(1);
   }
   return(0);
}



/*----------------------------------------------------------------------------*\
|                                                                              |
| Function:  Delete the array cell at the specified location from the array.   |
|                                                                              |
| Returns:   INT                                                               |
|            On failure, a non-zero value is returned.                         |
|                                                                              |
\*----------------------------------------------------------------------------*/
template<class DataType> INT
DynamicArray<DataType>::
iDeleteCell(size_t uiIndex           // IN  index of the array cell to delete
                                     //     from the array                   
           )
{

   UINT uiLoop;
   DataType* pxNewArray;

   ASSERT(uiUserArraySize <= uiActualArraySize);

   if (uiUserArraySize == 0)
   {
      LOG_ERROR("attempt to delete cell from array of size 0");
      return(1); 
   }

   if (uiIndex > (uiUserArraySize-1))
   {
      sprintf(achErrorString, 
              "attempt to delete cell out of array boundary %u, %u",
               uiIndex, (uiUserArraySize-1));
      LOG_ERROR(achErrorString);
      return(1); 
   }

   if ((uiUserArraySize == 1) ||
       (uiIndex == (uiUserArraySize-1)))
   {
                              // only one cell in array or we are deleting
                              // the last cell in the array               
      uiUserArraySize--;
   }
   else
   {
                              // we need to shift all cells after the given  
                              // cell to the left by one                     
         for (uiLoop=uiIndex+1; uiLoop<uiUserArraySize; uiLoop++)
         {
            pxArray[uiLoop-1] = pxArray[uiLoop];
         }
         uiUserArraySize--;
   }

                              // we need the below test so the value does not                       
                              // wrap...                                                            
   if (uiActualArraySize > (uiUserArraySize + uiArrayHysteresis))
   {
                              // we add the hysteresis so we don't bounce back                      
                              // and forth on an increment boundary                                 
      if (uiArrayIncreaseAmount <= (uiActualArraySize - 
                                    (uiUserArraySize + uiArrayHysteresis)))
      {
                              // delete unused cell from the array                                  
         uiActualArraySize -= uiArrayIncreaseAmount;

         pxNewArray = new DataType [uiActualArraySize];
         CHECK_NEW(pxNewArray);

                              // copy pxArray to pxNewArray 
         for (uiLoop=0; uiLoop<uiUserArraySize; uiLoop++)
         {
            pxNewArray[uiLoop] = pxArray[uiLoop];
         }
                              // delete the array
         delete[] pxArray;
                              // set the array to the new array
         pxArray = pxNewArray;

      }
   }
   return(0);
}


