/*
 * Some common ELF note handling routines
 *
 * This software is being released under the GNU Public Licence.
 *
 * Copyright 1999, Jeremy Fitzhardinge <jeremy@goop.org>
 */
#include <elf.h>
#include <stdlib.h>
#include <stdio.h>

#include "notes.h"
#define roundup(x, y)  ((((x)+((y)-1))/(y))*(y))

/* Calculate the size of a set of notes in an ELF file */
int notesize(struct note *note)
{
	int sz = 0;

	for(; note != NULL; note = note->next)
	{
		sz += sizeof(Elf32_Nhdr);
		sz += roundup(strlen(note->name), 4);
		sz += roundup(note->datasz, 4);
	}

	return sz;
}

/* Generate a formatted note segment for an ELF file */
int notefmt(struct note *note, char *buf, int bufsz)
{
	int sz = notesize(note);
	char *cp;

	if (sz == 0 || bufsz < sz)
		return 0;

	cp = buf;

	for(; note != NULL; cp += notesize(note), note = note->next)
	{
		Elf32_Nhdr *hdr = (Elf32_Nhdr *)cp;
		hdr->n_namesz = strlen(note->name);
		hdr->n_descsz = note->datasz;
		hdr->n_type = note->type;
		memcpy(cp + sizeof(*hdr), note->name, strlen(note->name));
		memcpy(cp + sizeof(*hdr) + roundup(hdr->n_namesz, 4),
		       note->data, note->datasz);
	}

	return cp - buf;
}

/* Parse a note segment */
struct note *noteparse(const char *buf, int buflen, struct note *notes)
{
	while(buflen > sizeof(Elf32_Nhdr)) {
		Elf32_Nhdr *nhdr = (Elf32_Nhdr *)buf;
		struct note *note;
		char *cp;
		int sz;
		
		sz = sizeof(Elf32_Nhdr) + 
			roundup(nhdr->n_namesz, 4) + 
			roundup(nhdr->n_descsz, 4);
		if (buflen < sz)
			break;

		note = malloc(sizeof(*note));
		note->next = notes;
		notes = note;

		note->type = nhdr->n_type;

		note->name = cp = malloc(nhdr->n_namesz + 1);
		memcpy(cp, buf + sizeof(*nhdr), nhdr->n_namesz);
		cp[nhdr->n_namesz] = 0;

		note->datasz = nhdr->n_descsz;
		note->data = malloc(nhdr->n_descsz);
		memcpy(note->data, 
		       buf + sizeof(*nhdr) + roundup(nhdr->n_namesz, 4), 
		       nhdr->n_descsz);

		buf += sz;
		buflen -= sz;
	}

	if (buflen != 0)
		printf("noteparse: %d bytes left over\n", buflen);
	
	return notes;
}
