/*
 * Add a PT_NOTE segment to an ELF executable
 *
 * This software is being released under the GNU Public Licence.
 *
 * Copyright 1999, Jeremy Fitzhardinge <jeremy@goop.org>
 * Modified for general purpose use by Jordan Mendelson <jordy@wserv.com>
 */

#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <fcntl.h>
#include <assert.h>
#include <sys/mman.h>
#include <sys/param.h>
#include <sys/stat.h>

#include <elf.h>

#include "notes.h"

#ifdef DEBUG
#define DBG(x)	x
#else
#define DBG(x)
#endif

#define ELF_PAGESIZE	4096

/* Segments in the ELF file */
struct segment {
	struct segment *next;
	int idx, origidx;

	Elf32_Phdr ph;
};

static struct segment *seg_insert(struct segment *list, const Elf32_Phdr *ph, int idx)
{
	struct segment *seg;
	struct segment **prev;
	struct segment *newseg;
	int lastidx = -1;

	for(prev = &list, seg = list;
	    seg != NULL; 
	    prev = &seg->next, seg = seg->next) {
		if (ph->p_offset <= seg->ph.p_offset)
			break;
	}

	newseg = malloc(sizeof(*newseg));

	newseg->next = seg;
	newseg->ph = *ph;
	*prev = newseg;

	if (idx == -1) {
		lastidx = 0;
		for(seg = list; seg != NULL; seg = seg->next)
			seg->idx = lastidx++;
	}

	return list;
}

static struct segment *seg_delete(struct segment *list, int idx)
{
	struct segment *seg, **prev;

	for(prev = &list, seg = list;
	    seg != NULL;
	    prev = &seg->next, seg = seg->next)
		if (seg->idx == idx) {
			*prev = seg->next;
			free(seg);
			break;
		}

	return list;
}

static off_t seg_freespace(struct segment *list, off_t size)
{
	off_t space = 0;
	struct segment *seg;
	
	for(seg = list; seg != NULL; seg = seg->next) {
		if ((space + size) <= seg->ph.p_offset)
			return space;
		space = seg->ph.p_offset + seg->ph.p_filesz;
	}

	return -1;
}

static const char *progname;

static const Elf32_Ehdr *parse_elf(const char *map)
{
	const char *badness = NULL;
	const Elf32_Ehdr *ehdr = (const Elf32_Ehdr *)map;

	if (memcmp(ehdr->e_ident, ELFMAG, SELFMAG) != 0) 
		badness = "bad ELF magic";

	/* XXX prototype code - clean all this up */
	if (!badness && ehdr->e_ident[EI_CLASS] != ELFCLASS32)
		badness = "can only deal with 32-bit exes";
	if (!badness && ehdr->e_ident[EI_DATA] != ELFDATA2LSB)
		badness = "can only deal with LE exes";

	if (!badness && ehdr->e_type != ET_EXEC)
		badness = "can only deal with executable ELF files";

	if (!badness && (
		ehdr->e_phentsize != sizeof(Elf32_Phdr) ||
		ehdr->e_shentsize != sizeof(Elf32_Shdr)))
		badness = "mismatched size for phdr or shdr";

	if (badness) {
		fprintf(stderr, "%s: can't use elf file: %s\n",
			progname, badness);
		return NULL;
	}

	return ehdr;
}

int main(int argc, char **argv)
{
	int i, err = 0;
	const char *exename;
	off_t len, outlen, templen, pos;
	struct stat st;
	int infd, outfd, tempfd;
	char *inmap, *outmap;
	const Elf32_Ehdr *inhdr;
	const Elf32_Phdr *inphdr;
	Elf32_Ehdr *outhdr;
	Elf32_Phdr *outphdr;
	struct note *notes = NULL;
	struct note *note;
	struct segment *seglist = NULL;
	struct segment *seg;
	int noteidx;
	int phdrs, type, typespecified;
	off_t noteoff;
	off_t notesz;
	off_t outbase;
	char *namebuf, *filename, *notename, tempbuf[512];
	progname = argv[0];
	filename = notename = namebuf = NULL;
	typespecified = 0;
	type = 18;

	while((i = getopt(argc, argv, "hf:n:o:t:")) != -1) {
		switch(i) {
			case 'f':
				filename = strdup(optarg);
				break;
			case 'o':
				namebuf = strdup(optarg);
				break;
			case 'n':
				notename = strdup(optarg);
				break;
			case 't':
				typespecified++;
				type = atoi(optarg);
				break;
			case 'h':
			default:
				err++;
		}
	}

	if (err || argc != optind+1) {
		fprintf(stderr, "Usage: %s [-f filename] [-n notename] [-o outname] [-t type] exename\n",
			progname);
		fprintf(stderr, "       -n\tThe name of the note you wish to view/change\n");
		fprintf(stderr, "       -f\tChange notename to the contents of filename\n");
		fprintf(stderr, "       -o\tName the new executable this\n");
		fprintf(stderr, "       -t\tSet the type to this\n");

		return 1;
	}

	exename = argv[optind];

	if ((infd = open(exename, O_RDONLY)) == -1) {
		fprintf(stderr, "%s: can't open %s: %s\n",
			progname, exename, strerror(errno));
		return 1;
	}

	if (fstat(infd, &st) == -1) {
		fprintf(stderr, "%s: can't stat input: %s\n",
			progname, strerror(errno));
		return 1;
	}

	len = st.st_size;

	if ((inmap = mmap(0, len, PROT_READ, MAP_PRIVATE, infd, 0)) == (char *)-1) {
		fprintf(stderr, "%s: can't mmap %s: %s\n",
			progname, exename, strerror(errno));
		return 1;
	}

	inhdr = parse_elf(inmap);

	if (inhdr == NULL)
		return 1;

	inphdr = (Elf32_Phdr *)(inmap+inhdr->e_phoff);

	/* 
	 * Look for existing notes
	 */
	for(i = 0; i < inhdr->e_phnum; i++) {
		const Elf32_Phdr *ph = &inphdr[i];

		seglist = seg_insert(seglist, ph, i);

		if (ph->p_type == PT_NOTE)
			notes = noteparse(inmap + ph->p_offset, 
					  ph->p_filesz, notes);
	}

#ifdef DEBUG
	printf("notes found:\n");
	for(note = notes; note != NULL; note = note->next)
		printf("note %s: type %d, %d bytes\n",
		       note->name, note->type, note->datasz);
#endif

	for (note = notes; note != NULL; note = note->next) {
		if (notename == NULL) {
			if (typespecified && note->type != type)
				continue;

			printf("note %s: type %d, %d bytes\n", note->name,
				note->type, note->datasz);
			printf("%s\n", note->data);
		} else if (!strcmp(note->name, notename)) {
			if (filename == NULL) {
				printf("note %s: type %d, %d bytes\n", note->name,
					note->type, note->datasz);
				printf("%s\n", note->data);
				exit(0);
			} else
				break;
		}
	}
	if (notename == NULL) {
		exit(0);
	}
	if (filename == NULL) {
		fprintf(stderr, "Could not find note %s\n", notename);
		exit(-1);
	}

	if (note == NULL) {
		note = malloc(sizeof(*note));
		note->name = strdup(notename);
		note->type = type;
		note->data = NULL;
		note->datasz = 0;

		note->next = notes;
		notes = note;
	}

	if ((tempfd = open(filename, O_RDONLY)) == -1) {
		fprintf(stderr, "Can't open %s: %s", filename, strerror(errno));
	}

	if (note->data)
		free(note->data);

	note->data = NULL;
	pos = 0;
	while ((templen = read(tempfd, (void *)tempbuf, 512)) > 0) {
		if (note->data == NULL) {
			note->data = malloc(templen);
			if (note->data == NULL) {
				perror("malloc");
				exit(-1);
			}
		} else {
			char *ptr;

			ptr = realloc(note->data, templen +pos);
			if (ptr == NULL) {
				perror("realloc");
				exit(-1);
			} else
				note->data = ptr;
			
		}
		bcopy(tempbuf, note->data+pos, templen);
		pos += templen;
        }
#ifdef DEBUG
	printf("read data:\n");
	printf("%s\n", note->data);
#endif

	note->datasz = pos;
	notesz = notesize(notes);

#ifdef DEBUG	
	printf("notes inserted:\n");
	for(note = notes; note != NULL; note = note->next)
		printf("note %s: type %d, %d bytes\n",
		       note->name, note->type, note->datasz);
#endif

	outlen = 0;
	for(seg = seglist; seg != NULL; seg = seg->next) {
		if (outlen < seg->ph.p_offset+seg->ph.p_filesz)
			outlen = seg->ph.p_offset+seg->ph.p_filesz;
		DBG(printf("segment %d: start %d len %d\n",
			   seg->idx, seg->ph.p_offset, seg->ph.p_filesz));
	}

	noteidx = -1;
	for(i = 0; i < inhdr->e_phnum; i++)
		if (inphdr[i].p_type == PT_NOTE) {
			noteidx = i;
			seglist = seg_delete(seglist, i);
		}

	noteoff = seg_freespace(seglist, notesz);

	if (namebuf == NULL) {
		namebuf = (char *)malloc(strlen(exename)+5);
		sprintf(namebuf, "%s.new", exename);
	}
	if ((outfd = open(namebuf, O_RDWR|O_TRUNC|O_CREAT/*|O_EXCL*/, st.st_mode)) == -1) {
		fprintf(stderr, "%s: can't create temp file %s: %s\n",
			progname, namebuf, strerror(errno));
		return 1;
	}

	phdrs = inhdr->e_phnum;

	if (noteoff != -1 && noteidx != -1) {
		DBG(printf("space for %d bytes of notes at %ld, idx %d\n",
			   notesz, noteoff, noteidx));
		outbase = 0;
	} else {
		if (noteidx == -1)
			phdrs++;

		noteoff = sizeof(Elf32_Ehdr) + sizeof(Elf32_Phdr) * phdrs;
		
		outbase = roundup(noteoff + notesz, ELF_PAGESIZE);
		
		outlen += outbase;
	}

	DBG(printf("outbase=%d\n", outbase));
	
	if (ftruncate(outfd, outlen) == -1) {
		fprintf(stderr, "%s: can't set output file size to %ld: %s\n",
			progname, outlen, strerror(errno));
		return 1;
	}

	outmap = mmap(0, outlen, PROT_READ|PROT_WRITE, MAP_SHARED, outfd, 0);
	if (outmap == (char *)-1) {
		fprintf(stderr, "%s: can't mmap %s: %s\n",
			progname, namebuf, strerror(errno));
		return 1;
	}

	/* outhdr & outphdr == original ones */
	outhdr = (Elf32_Ehdr *)(outmap+outbase);
	outphdr = (Elf32_Phdr *)(outmap+outbase+inhdr->e_phoff);

	for(seg = seglist; seg != NULL; seg = seg->next) {
		memcpy(outmap + seg->ph.p_offset + outbase,
		       inmap + seg->ph.p_offset, seg->ph.p_filesz);
		if (seg->ph.p_offset != 0)
			seg->ph.p_offset += outbase;
	}

	outhdr->e_phoff += outbase;
	outhdr->e_shoff = 0;
	outhdr->e_shnum = 0;
	outhdr->e_shstrndx = 0;

	if (outbase != 0) {
		Elf32_Phdr ph;

		for(seg = seglist; seg != NULL; seg = seg->next)
			if (seg->ph.p_type == PT_LOAD && seg->ph.p_offset == 0)
				break;
		if (seg != NULL) {
			seg->ph.p_filesz += outbase;
			seg->ph.p_memsz += outbase;
			seg->ph.p_vaddr -= outbase;
			seg->ph.p_paddr -= outbase;
		} else
			fprintf(stderr, "%s: no segment mapping start!\n",
				progname);
		
		memset(&ph, 0, sizeof(ph));
		ph.p_type = PT_NOTE;
		ph.p_offset = noteoff;
		ph.p_vaddr = ph.p_paddr = 0;
		ph.p_filesz = ph.p_memsz = notesz;
		ph.p_flags = PF_R;
		ph.p_align = 4;

		seglist = seg_insert(seglist, &ph, -1);

		/* Set up real header + pheaders */
		outhdr = (Elf32_Ehdr *)outmap;
		DBG(printf("setting up %d new headers for %d, phdr off %d\n",
			   phdrs, outbase, inhdr->e_phoff));
		memcpy(outmap, inmap, sizeof(Elf32_Ehdr));

		outhdr->e_phnum = phdrs;
		outhdr->e_shoff = 0;
		outhdr->e_shnum = 0;
		outhdr->e_shstrndx = 0;

		outphdr = (Elf32_Phdr *)(outmap + outhdr->e_phoff);
		memset(outphdr, 0, phdrs * sizeof(Elf32_Phdr));

		for(seg = seglist; seg != NULL; seg = seg->next) {
			Elf32_Phdr *ph = &outphdr[seg->idx];

			DBG(printf("doing phdr %d->%d, type %d\n",
				   seg->origidx, seg->idx, seg->ph.p_type));
			
			*ph = seg->ph;
			
			if (seg->ph.p_type == PT_NOTE)
				noteidx = seg->idx;
			if (seg->ph.p_type == PT_PHDR) {
				ph->p_offset = sizeof(*outhdr);
				ph->p_vaddr -= outbase;
				ph->p_paddr -= outbase;
				ph->p_memsz = ph->p_filesz = phdrs * sizeof(Elf32_Phdr);
			}
		}
	}
	
	outphdr[noteidx].p_filesz = notesz;
	outphdr[noteidx].p_memsz = notesz;
	notefmt(notes, outmap + outphdr[noteidx].p_offset, notesz);

	return 0;
}
