static char rcsid[] = "$Id: dialog.c,v 1.2 1992/01/16 21:10:54 jtsillas Exp $";

/*****************************************************************************
 *
 *  Copyright 1989 The University of Texas at Austin
 *  Copyright 1990 Microelectronics and Computer Technology Corporation
 *  Copyright 1990 Thomson Consumer Electronics, Inc.
 *  Copyright 1991 Bull HN Worldwide Info Systems, Inc.
 *
 *****************************************************************************/

/*  dialog.c
 *
 *    Create the dialogue window where the user enter gdb commands, and
 *    provide action procs to make a text widget behave like a terminal.
 *
 *    InsertSpace():	Prevent user from deleting past the prompt (action proc
 *			for DELETE or BACKSPACE) (static).
 *    Dispatch():	Send an input command line to gdb (static).
 *    SigInt():		Send SIGINT to gdb (action proc for Ctrl-C) (static).
 *    SigEof():		Send an EOF signal to gdb (action proc for 
 *                      Ctrl-D) (static).
 *    SigQuit():	Send SIGQUIT to gdb (action proc for Ctrl-\) (static).
 *    CreateDialogWindow(): Create dialog window and install action table.
 *    AppendDialogText(): Append string to dialog window.
 *    CopyPrimaryEnd(): Copy the current selection to the end (static).
 */

#include <signal.h>
#include "global.h"
#include <Xm/Xm.h>
#include <Xm/Text.h>

static void InsertSpace(Widget, XEvent*, String*, Cardinal*);
static void Dispatch(Widget, XEvent*, String*, Cardinal*);
static void SigInt(Widget, XEvent*, String*, Cardinal*);
static void SigEof(Widget, XEvent*, String*, Cardinal*);
static void SigQuit(Widget, XEvent*, String*, Cardinal*);
static void CopyPrimaryEnd(Widget, XEvent*, String*, Cardinal*);

Widget	dialogWindow;			/* text window as a gdb terminal */
Boolean FalseSignal = FALSE;		/* set to TRUE before self-generated
					   interrupt/quit signals */
static XmTextPosition  StartPos;      	/* starting position of input text */
static XmTextPosition  LastPos;         /* End of the input text */

/*  This procedure prevents the user from deleting past the prompt, or
 *  any text appended by AppendDialogText() to the dialog window.
 *  It checks the last position of text, if it matches StartPos, set
 *  by AppendDialogText(), it inserts a space so that delete-previous-
 *  character() can only delete the space character.
 */
static void InsertSpace(w, event, params, num_params)
     Widget w;
     XEvent *event;
     String *params;
     Cardinal *num_params;
{
  if (XmTextGetInsertionPosition(w) <= StartPos) {
    XmTextInsert(w, LastPos, " ");
    XmTextSetInsertionPosition(w, LastPos + 1);
  }
}

/*  Dispatch() is invoked on every <CR>.
 *  It collects text from the dialog window and sends it to gdb.
 *  If the string is a command to gdb (Prompt would be TRUE),
 *  it is stored in the global variable, Command.
 */
static void Dispatch(w, event, params, num_params)
    Widget w;
    XEvent *event;
    String *params;
    Cardinal *num_params; 
{
  char *DialogText;
  /* 
    For GDB, '\n' means exec previous command again.
    default command is space+CR, so that we never send
    CR to gdb (the repeat is managed here)
    */
  static char gdb_command[LINESIZ] = " \n";
  char s[LINESIZ];
  
  DialogText = XmTextGetString(dialogWindow);
  strcpy(s, DialogText + StartPos);
  StartPos = StartPos + strlen(s);
  
  if (Prompt) 
    {
      if (gdb_source_command(s,FALSE))	/* filter source command 
					   (& do not display source command) */
	{
	  strcpy(gdb_command," \n");	/* do not execute anything 
					   if next command is '\n' */
	  return;
	}
      /* When we send \n to gdb, it executes the last command,
	 so better tell mxgdb what gdb is doing */
      if (strcmp(s, "\n"))
      strcpy(gdb_command,s);
      else
	strcpy(s,gdb_command);
      send_command(s);
    }
  else 
      write_gdb(s);

  XtFree(DialogText);
}


/*  Sends an interrupt signal, SIGINT, to gdb.
 *  Simulates the action of the INTR character (ctrl-C).
 */
static void SigInt(w, event, params, num_params)
     Widget w;
     XEvent *event;
     String *params;
     Cardinal *num_params;
{
  FalseSignal = TRUE;
#ifdef DEBUG
  if(debug)
    fprintf(stderr, "SigInt: a ctrl-C was received.\n");
#endif
#ifdef SYSV
  kill(gdbpid, SIGINT);		/* Whatever your system inteprets as */
#else				/* the break character. This character */
  write_gdb("\03");		/* should cause gdb to send SIGINT to */
#endif				/* the inferior process. */
}


/*  Sends an EOF signal to gdb. (ctrl-D) */
static void SigEof(w, event, params, num_params)
     Widget w;
     XEvent *event;
     String *params;
     Cardinal *num_params;
{
  write_gdb("\04");
}


/*  Sends a QUIT signal, SIGQUIT, to gdb. 
 *  Simulates the action of the QUIT character (ctrl-\) 
 */
static void SigQuit(w, event, params, num_params)
     Widget w;
     XEvent *event;
     String *params;
     Cardinal *num_params;
{
  FalseSignal = TRUE;
  write_gdb("\034");
}

/* 
 * This Action checks for a selection in the widget w. If there is
 * such a selection then insert it at the current cursor pos otherwise
 * look for a primary selection from the outside. If ANY selections exist
 * we always copy to the end of the Text (so that it acts like an xterm).
 */
static void CopyPrimaryEnd(w, event, params, num_params)
     Widget w;
     XEvent *event;
     String *params;
     Cardinal *num_params;
{
  char *selection;
  XmTextPosition last_pos;
  
  /*
   * Not sure if this is a bug but it appears that changing the
   * position of the cursor while there is a selection destroys the
   * the selection. So in order to allow selection from the same Text
   * I first Get a possible selection from it and then go to the end
   * of the file for insertion.
   */
  selection = XmTextGetSelection(w);
  last_pos = XmTextGetLastPosition(w);

  if (selection)
    {
      XmTextInsert(w, last_pos, selection);
      XtFree(selection);
    }
  else
    {
      XmTextSetInsertionPosition(w, last_pos);
      XtCallActionProc(w, "copy-primary", event, NULL, 0);
    }
}

/* 
 *  Dialog window has its own set of translations for editing.
 *  Special action procedures for keys Delete/Backspace, Carriage Return,
 *  Ctrl-U, Ctrl-C, Ctrl-D, Ctrl-\, and word selection.
 */
void CreateDialogWindow(parent)
     Widget parent;
{
  static XtActionsRec dialog_actions[] = {
    {"SigInt", (XtActionProc) SigInt},
    {"SigEof", (XtActionProc) SigEof},
    {"SigQuit", (XtActionProc) SigQuit},
    {"InsertSpace", (XtActionProc) InsertSpace},
    {"Dispatch", (XtActionProc) Dispatch},
    {"CopyPrimaryEnd", (XtActionProc) CopyPrimaryEnd},
    {NULL, NULL}
  };
  
  dialogWindow = XmCreateScrolledText(parent, "dialogWindow",
				      NULL, 0);
  XtManageChild(dialogWindow);
  
  XtAppAddActions(app_context, dialog_actions, XtNumber(dialog_actions));
}
 
void AppendDialogText(s)
     char   *s;
{
  LastPos = XmTextGetLastPosition(dialogWindow);
  XmTextInsert(dialogWindow, LastPos, s);
  StartPos = LastPos + strlen(s);
  XmTextSetInsertionPosition(dialogWindow, StartPos);
  LastPos = StartPos;
}
