// Copyright 1994, 1995 Michael Chastain
// Licensed under the Gnu Public License, Version 2
//
// File: WhString.cc
//   String class.
//   Strings may contain nulls and are *not* null-terminated.
//   This is a concrete class.
//
// File Created:	18 Jan 1994		Michael Chastain
// Last Reviewed:	12 Sep 1994		Michael Chastain
// Last Edited:		08 Nov 1995		Michael Chastain

#include <string.h>

#include <ErAbort.hh>
#include <ErFatal.hh>
#include <ErMem.hh>
#include <ErPtr.hh>
#include <WhString.hh>



// Used to format digits.
static const char acDigitFmt [] = "0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZ$";



// Constructor.
WhString::WhString( )
    : lcData_	( )
{
    ;
}



// Constructor.
WhString::WhString( const char * pcOld )
    : lcData_	( )
{
    if ( pcOld == 0 )
	ErPtr( );
    lcData_.appendArray( pcOld, ::strlen( pcOld ) );
}



// Constructor.
WhString::WhString( const char * pcOld, int ncOldMax )
    : lcData_	( )
{
    if ( pcOld == 0 )
	ErPtr( );

    lcData_.clear( ncOldMax );
    for ( int icOld = 0; icOld < ncOldMax; ++icOld )
    {
	if ( pcOld[icOld] == '\0' )
	    break;
	lcData_.appendVal( pcOld[icOld] );
    }
}



// Return an address for a classic string.
//   Strings may contain nulls and are *not* null-terminated.
const char * WhString::address( ) const
{
    return lcData_.address( );
}



// Check whether a string is representable as 'const char *'.
//   This means: nul terminated and no interior nulls.
void WhString::checkCcs( ) const
{
    if ( lcData_.count( ) == 0 )
	ErFatal( "WhString::checkCcs: not ccs." );

    for ( int icData = 0; icData < lcData_.count( ); ++icData )
    {
	if ( ( lcData_[icData] == '\0' ) != ( icData == lcData_.count( ) - 1 ) )
	    ErFatal( "WhString::checkCcs: not ccs." );
    }
}



// Return whether two strings match, with case.
bool WhString::matchWithCase( const WhString & strMatch ) const
{
    if ( lcData_.count( ) != strMatch.lcData_.count( ) )
	return false;
    return ::memcmp( lcData_.address( ), strMatch.lcData_.address( ),
	lcData_.count( ) ) == 0;
}



// Append a string, unformatted.
void WhString::appStrRaw( const char * pcAppend )
{
    if ( pcAppend == 0 )
	ErPtr( );
    lcData_.appendArray( pcAppend, ::strlen( pcAppend ) );
}



// Append a string, unformatted.
void WhString::appStrRaw( const WhString & strAppend )
{
    lcData_.appendArray( strAppend.lcData_.address( ), strAppend.lcData_.count( ) );
}



// Append one character, formatted.
void WhString::appChrFmt( char cValue )
{
    if      ( cValue >= 0x20 && cValue <= 0x7E )
    {
	lcData_.appendVal( cValue );
    }
    else if ( cValue == '\n' )
    {
	lcData_.appendVal( '\\' );
	lcData_.appendVal( 'n'  );
    }
    else if ( cValue == '\r' )
    {
	lcData_.appendVal( '\\' );
	lcData_.appendVal( 'r'  );
    }
    else if ( cValue == '\t' )
    {
	lcData_.appendVal( '\\' );
	lcData_.appendVal( 't'  );
    }
    else if ( cValue == '\0' )
    {
	lcData_.appendVal( '\\' );
	lcData_.appendVal( '0'  );
    }
    else
    {
	lcData_.appendVal( '\\' );
	lcData_.appendVal( 'x'  );
	lcData_.appendVal( acDigitFmt[((unsigned char) cValue) / 16] );
	lcData_.appendVal( acDigitFmt[((unsigned char) cValue) % 16] );
    }
}



// Append an integer, formatted.
//   This code is tense.
void WhString::appIntFmt( unsigned int iValue, int iBase, int nDigit )
{
    // Check 'iBase'.
    if ( iBase < 2 || iBase > 36 )
	ErAbort( "WhString::appIntFmt: bad base." );

    // Check 'nDigit'.
    if ( nDigit < 0 )
	ErAbort( "WhString::appIntFmt: negative count." );

    // Generated digits.
    //   Use a fixed array for speed.
    //   If fixed array overflows, cutover to list.
    //   Note that the array is forward but the list is backward!
    const int		nDigitFwdMax	= 64;	// Arbitrary
    char		acDigitFwd	[nDigitFwdMax];
    int			iDigitFwd	= nDigitFwdMax;
    WhList <char> *	plcDigitRev	= 0;

    // Sign flags.
    bool		fSign		= iBase == 10;
    bool		fOverflow	= false;

    // Handle signed output.
    if ( fSign && int( iValue ) < 0 )
    {
	// Output sign.
	lcData_.appendVal( '-' );

	if ( 0 - int( iValue ) < 0 )
	{
	    // This is the 2's complement most negative integer.
	    //   E.g. 0x80000000 on 32-bit processor.
	    fOverflow = true;

	    // Have to bias the positive value to avoid overflow.
	    iValue = -1 - int( iValue );

	    // Always use list for this hard case.
	    plcDigitRev = new WhList <char>;
	    if ( plcDigitRev == 0 )
		ErMem( );
	}
	else
	{
	    // Just convert to positive.
	    iValue = 0 - int( iValue );
	}
    }

    // Convert to list of digits.
    while ( iValue > 0 )
    {
	// Cutover when fixed array overflows.
	if ( plcDigitRev == 0 && iDigitFwd == 0 )
	{
	    plcDigitRev = new WhList <char>;
	    if ( plcDigitRev == 0 )
		ErMem( );
	    for ( int iDigit = nDigitFwdMax - 1; iDigit >= 0; --iDigit )
		plcDigitRev->appendVal( acDigitFwd[iDigit] );
	}

	char cDigit = acDigitFmt[iValue % iBase];
	iValue /= iBase;

	if ( plcDigitRev == 0 )
	    acDigitFwd[--iDigitFwd] = cDigit;
	else
	    plcDigitRev->appendVal( cDigit );
    }

    // Handle most-negative-integer overflow.
    //   It was offset by 1, and looks like "7463847412" right now.
    //   Add 1 to its magnitude in analog fashion.
    if ( fOverflow )
    {
	// Create a new list to hold addition result.
	WhList <char>	lcDigitRevNew;
	int		iCarry		= 1;

	for ( int iDigit = 0; iDigit < plcDigitRev->count( ); ++iDigit )
	{
	    char cDigit = char( int( (*plcDigitRev)[iDigit] ) + iCarry );
	    if ( cDigit == acDigitFmt[iBase] )
		cDigit = acDigitFmt[0];
	    else
		iCarry = 0;
	    lcDigitRevNew.appendVal( cDigit );
	}

	if ( iCarry > 0 )
	    lcDigitRevNew.appendVal( acDigitFmt[1] );

	*plcDigitRev = lcDigitRevNew;
    }

    if ( plcDigitRev == 0 )
    {
	// Generate leading pad.
	for ( int iPad = nDigitFwdMax - iDigitFwd; iPad < nDigit; ++iPad )
	    lcData_.appendVal( acDigitFmt[0] );

	// Generate main body.
	lcData_.appendArray( &acDigitFwd[iDigitFwd], nDigitFwdMax - iDigitFwd );
    }
    else
    {
	// Generate leading pad.
	for ( int iPad = plcDigitRev->count( ); iPad < nDigit; ++iPad )
	    lcData_.appendVal( acDigitFmt[0] );

	// Reverse main body.
	for ( int iDigit = plcDigitRev->count( ) - 1; iDigit >= 0; --iDigit )
	    lcData_.appendVal( (*plcDigitRev)[iDigit] );

	// Delete the dynamic list.
	delete plcDigitRev;
	plcDigitRev = 0;
    }
}



// Append a pointer, formatted.
void WhString::appPtrFmt( const void * pValue )
{
    if ( pValue == 0 )
    {
	lcData_.appendVal( '0' );
    }
    else
    {
	const unsigned int uValue = (unsigned int) pValue;

	lcData_.appendVal( '0' );
	lcData_.appendVal( 'x' );
	lcData_.appendVal( acDigitFmt[( uValue & (0xF << (7 * 4)) ) >> (7 * 4)] );
	lcData_.appendVal( acDigitFmt[( uValue & (0xF << (6 * 4)) ) >> (6 * 4)] );
	lcData_.appendVal( acDigitFmt[( uValue & (0xF << (5 * 4)) ) >> (5 * 4)] );
	lcData_.appendVal( acDigitFmt[( uValue & (0xF << (4 * 4)) ) >> (4 * 4)] );
	lcData_.appendVal( acDigitFmt[( uValue & (0xF << (3 * 4)) ) >> (3 * 4)] );
	lcData_.appendVal( acDigitFmt[( uValue & (0xF << (2 * 4)) ) >> (2 * 4)] );
	lcData_.appendVal( acDigitFmt[( uValue & (0xF << (1 * 4)) ) >> (1 * 4)] );
	lcData_.appendVal( acDigitFmt[( uValue & (0xF << (0 * 4)) ) >> (0 * 4)] );
    }
}



// Append a string, formatted.
//   Beware aliasing.
void WhString::appStrFmt( const char * pcValue, int ncValueMax )
{
    if ( pcValue == 0 )
	ErPtr( );

    WhString strFmt;
    for ( int icValue = 0; icValue < ncValueMax; ++icValue )
    {
	if ( pcValue[icValue] == '\0' )
	    break;
	strFmt.appChrFmt( pcValue[icValue] );
    }
    appStrRaw( strFmt );
}
