// Copyright 1994, 1995 Michael Chastain
// Licensed under the Gnu Public License, Version 2
//
// File: WhList.cc
//   List template.
//   This is a template.
//
// File Created:	23 Jan 1994		Michael Chastain
// Last Reviewed:	24 Apr 1995		Michael Chastain
// Last Edited:		01 Nov 1995		Michael Chastain

#include <ErAbort.hh>
#include <ErMem.hh>
#include <ErPtr.hh>
#include <WhList.hh>



// Copy constructor.
template <class T> WhList<T>::WhList( const WhList <T> & lOld )
    : nData_	(0)
    , nDataMax_	(0)
    , pData_	(0)
{
    *this = lOld;
}



// Assignment operator.
template <class T> void WhList<T>::operator = ( const WhList <T> & lOld )
{
    if ( this != &lOld )
    {
	clear( lOld.nData_ );
	for ( int iDataOld = 0; iDataOld < lOld.nData_; ++iDataOld )
	    pData_[iDataOld] = lOld.pData_[iDataOld];
	nData_ = lOld.nData_;
    }
}



// Destructor.
template <class T> WhList<T>::~WhList( )
{
    if ( pData_ != 0 )
	delete [] pData_;
    pData_ = 0;
}



// Clear with new maximum size.
//   The list can always expand so 0 is always cool.
//   A non-0 value is a hint for performance.
template <class T> void WhList<T>::clear( int nDataMaxNew )
{
    // Check count.
    if ( nDataMaxNew < 0 )
	ErAbort( "WhList<T>::clear: negative count." );

    // Reset index.
    nData_ = 0;

    // Expand if needed.
    if ( nDataMaxNew > nDataMax_ )
    {
	// Allocate new array.
	T * pDataNew = new T [nDataMaxNew];
	if ( pDataNew == 0 )
	    ErMem( );

	// Substitute in new array.
	if ( pData_ != 0 )
	    delete [] pData_;
	pData_    = pDataNew;
	nDataMax_ = nDataMaxNew;
    }
}



// Set the item count.
//   This is a low-level high-speed interface.
template <class T> void WhList<T>::setCount( int nDataNew )
{
    if ( nDataNew < 0 || nDataNew > nDataMax_ )
	ErAbort( "WhList<T>::setCount: count out of range." );
    nData_ = nDataNew;
}



// Append an array of data.
template <class T> void WhList<T>::appendArray( const T * pDataNew, int nDataNew )
{
    // Check pointer.
    if ( pDataNew == 0 )
    {
	if ( nDataNew == 0 )
	    return;
	ErPtr( );
    }

    // Check count.
    if ( nDataNew < 0 )
	ErAbort( "WhList<T>::append: negative count." );

    // Expand if needed.
    if ( nDataNew > nDataMax_ - nData_ )
    {
	// Anti-aliasing if source is moving.
	if ( pDataNew >= pData_ && pDataNew < pData_ + nDataMax_ )
	{
	    WhList <T> lDataNewCopy;
	    lDataNewCopy.appendArray( pDataNew, nDataNew );
	    appendArray( lDataNewCopy.address( ), lDataNewCopy.count( ) );
	    return;
	}

	// Expand.
	expand( nDataNew );
    }

    // Copy in the data.
    for ( int iDataNew = 0; iDataNew < nDataNew; ++iDataNew )
	pData_[nData_ + iDataNew] = pDataNew[iDataNew];

    // Record new count.
    nData_ += nDataNew;
}



// Remove a datum.
template <class T> void WhList<T>::remove( int iDataRemove )
{
    if ( iDataRemove < 0 || iDataRemove >= nData_ )
	ErRange( );
    --nData_;
    for ( int iData = iDataRemove; iData < nData_; ++iData )
	pData_[iData] = pData_[iData + 1];
}



// Expand.
//   Can either give exact size requested or round up.
template <class T> void WhList<T>::expand( int nDataNew )
{
    // Check count.
    if ( nDataNew < 0 )
	ErAbort( "WhList<T>::expand: negative count." );

    // Compute new maximum and check it.
    int nDataMaxNew = nData_ + nDataNew;
    if ( nDataMaxNew < nData_ )
	ErAbort( "WhList<T>::expand: count overflow." );

    // Expand if needed.
    if ( nDataMaxNew > nDataMax_ )
    {
	// Heuristic sizes for first allocation.
	const int nDataMaxFirst = sizeof(T) <= sizeof(char) ? 240 :
				  sizeof(T) <= sizeof(int)  ?  16 :
							        4 ;

	// Round up if not exact.
	if ( nDataMaxNew < nDataMaxFirst ) nDataMaxNew = nDataMaxFirst;
	if ( nDataMaxNew < 2 * nDataMax_ ) nDataMaxNew = 2 * nDataMax_;

	// Allocate new array.
	T * pDataNew = new T [nDataMaxNew];
	if ( pDataNew == 0 )
	    ErMem( );

	// Copy from old to new.
	for ( int iData = 0; iData < nData_; ++iData )
	    pDataNew[iData] = pData_[iData];

	// Substitute in new array.
	if ( pData_ != 0 )
	    delete [] pData_;
	pData_    = pDataNew;
	nDataMax_ = nDataMaxNew;
    }
}
