// Copyright 1995 Michael Chastain
// Licensed under the Gnu Public License, Version 2
//
// File: WhLap.cc
//   List of auto pointers.
//   This is a template.
//
// File Created:	28 Oct 1995		Michael Chastain
// Last Edited:		02 Nov 1995		Michael Chastain

#include <ErAbort.hh>
#include <ErMem.hh>
#include <ErPtr.hh>
#include <WhLap.hh>



// Destructor.
template <class T> WhLap<T>::~WhLap( )
{
    reset( );
}



// Remove a datum.
template <class T> void WhLap<T>::remove( int iDataRemove )
{
    if ( iDataRemove < 0 || iDataRemove >= nData_ )
	ErRange( );
    --nData_;
    delete ppData_[iDataRemove];
    for ( int iData = iDataRemove; iData < nData_; ++iData )
	ppData_[iData] = ppData_[iData + 1];
    ppData_[nData_] = 0;
}



// Reset the list.
//   Destroys existing data.
template <class T> void WhLap<T>::reset( )
{
    for ( int iData = nData_ - 1; iData >= 0; --iData )
    {
	delete ppData_[iData];
	ppData_[iData] = 0;
    }

    if ( ppData_ != 0 )
	delete [] ppData_;

    ppData_   = 0;
    nData_    = 0;
    nDataMax_ = 0;
}



// Reset with new maximum size.
//   The list can always expand so 0 is always cool.
//   A non-0 value is a hint for performance.
template <class T> void WhLap<T>::reset( int nDataMaxNew )
{
    // Check count.
    if ( nDataMaxNew < 0 )
	ErAbort( "WhLap<T>::clear: negative count." );

    // Reset to empty.
    reset( );

    // Pre-allocate maximum size.
    if ( nDataMaxNew > 0 )
    {
	nDataMax_ = nDataMaxNew;
	ppData_   = new T * [nDataMaxNew];
	if ( ppData_ == 0 )
	    ErMem( );
    }
}



// Append by linear pointer.
template <class T> void WhLap<T>::appendAp( T * & pData )
{
    if ( pData == 0 )
	ErPtr( );
    if ( nData_ >= nDataMax_ )
	expand( 1 );
    ppData_[nData_++] = pData;
    pData = 0;
}



// Expand.
//   Can either give exact size requested or round up.
template <class T> void WhLap<T>::expand( int nDataNew )
{
    // Check count.
    if ( nDataNew < 0 )
	ErAbort( "WhLap<T>::expand: negative count." );

    // Compute new maximum and check it.
    int nDataMaxNew = nData_ + nDataNew;
    if ( nDataMaxNew < nData_ )
	ErAbort( "WhLap<T>::expand: count overflow." );

    // Expand if needed.
    if ( nDataMaxNew > nDataMax_ )
    {
	// Heuristic sizes for allocation.
	const int nDataMaxFirst = 16;
	if ( nDataMaxNew < nDataMaxFirst ) nDataMaxNew = nDataMaxFirst;
	if ( nDataMaxNew < 2 * nDataMax_ ) nDataMaxNew = 2 * nDataMax_;

	// Allocate new array.
	T ** ppDataNew = new T * [nDataMaxNew];
	if ( ppDataNew == 0 )
	    ErMem( );

	// Copy from old to new.
	for ( int iData = 0; iData < nData_; ++iData )
	    ppDataNew[iData] = ppData_[iData];

	// Substitute in new array.
	if ( ppData_ != 0 )
	    delete [] ppData_;
	ppData_   = ppDataNew;
	nDataMax_ = nDataMaxNew;
    }
}
