// Copyright 1995 Michael Chastain
// Licensed under the Gnu Public License, Version 2
//
// File: WhImage.cc
//   An mmap'ed image of a file.
//   This is an external resource handler class.
//
// File Created:	25 Oct 1995		Michael Chastain
// Last Edited:		28 Oct 1995		Michael Chastain

#include <sys/types.h>
#include <sys/mman.h>
#include <sys/stat.h>
#include <errno.h>
#include <fcntl.h>
#include <string.h>
#include <unistd.h>

#include <ErAbort.hh>
#include <ErFatal.hh>
#include <WhImage.hh>



// Constructor.
WhImage::WhImage( const WhString & strName )
    : fImage_	( false )
    , strError_	(	)
    , fdImage_	( -1	)
    , pcImage_	( 0	)
    , ncImage_	( 0	)
{
    // Make name.
    WhString strNameNul( strName );
    strNameNul.appChrRaw( '\0' );
    strNameNul.checkCcs( );

    // Open file.
    ::errno = 0;
    fdImage_ = ::open( strNameNul.address( ), O_RDONLY, 0644 );
    if ( fdImage_ == -1 )
    {
	strError_.appStrRaw( ::strerror( ::errno ) );
	return;
    }

    // Get size.
    struct stat statBuf;
    if ( ::fstat( fdImage_, &statBuf ) != 0 )
    {
	strError_.appStrRaw( ::strerror( ::errno ) );
	if ( ::close( fdImage_ ) != 0 )
	    ErFatal( "WhImage::WhImage: failed close." );
	return;
    }
    ncImage_ = statBuf.st_size;

    // Map.
    ::errno = 0;
    pcImage_ = ::mmap( 0, ncImage_, PROT_READ, MAP_FILE | MAP_PRIVATE,
	fdImage_, 0 );
    if ( ::errno != 0 )
    {
	strError_.appStrRaw( ::strerror( ::errno ) );
	if ( ::close( fdImage_ ) != 0 )
	    ErFatal( "WhImage::WhImage: failed close." );
	return;
    }

    // Success.
    fImage_ = true;
}



// Destructor.
WhImage::~WhImage( )
{
    finalize( );
}



// Finalizer.
void WhImage::finalize( )
{
    if ( fImage_ )
    {
	// g++ 2.6.3: no 'const_cast' yet.
	if ( ::munmap( * (char **) &pcImage_, ncImage_ ) != 0 )
	    ErFatal( "WhImage::finalize: failed munmap." );
	if ( ::close( fdImage_ ) != 0 )
	    ErFatal( "WhImage::finalize: failed close." );
    }
    fImage_ = false;
}



// Return data address.
const char * WhImage::address( ) const
{
    if ( isError( ) )
	ErAbort( "WhImage::address: not bound." );
    return pcImage_;
}



// Return data count.
int WhImage::count( ) const
{
    if ( isError( ) )
	ErAbort( "WhImage::count: not bound." );
    return ncImage_;
}
