// Copyright 1995 Michael Chastain
// Licensed under the Gnu Public License, Version 2
//
// File: WhFileOut.cc
//   File class.
//   This is an external resource handler class.
//
// File Created:	25 Oct 1995		Michael Chastain
// Last Edited:		06 Nov 1995		Michael Chastain

#include <sys/types.h>
#include <errno.h>
#include <fcntl.h>
#include <unistd.h>

#include <ErAbort.hh>
#include <ErFatal.hh>
#include <WhFileOut.hh>
#include <WhString.hh>



// Constructor.
WhFileOut::WhFileOut( const WhString & strName, TyOpen tyOpen )
    : strError_		(	)
    , fFile_		( false	)
    , fdFile_		( -1	)
{
    // Check name.
    WhString strNameNul( strName );
    strNameNul.appChrRaw( '\0' );
    strNameNul.checkCcs( );

    // Translate the mode.
    int    wFlags = 0;
    mode_t wMode  = 0;
    switch ( tyOpen )
    {
    default:
	ErAbort( "WhFileOut::openName: bad enum." );
    case tyOpenWriteData:
	wFlags = O_RDWR|O_TRUNC|O_CREAT;
	wMode  = 0644;
	break;
    case tyOpenWriteExe:
	wFlags = O_RDWR|O_TRUNC|O_CREAT|O_EXCL;
	wMode  = 0755;
	break;
    }

    // Open the file.
    ::errno = 0;
    fdFile_ = ::open( strNameNul.address( ), wFlags, wMode );
    if ( fdFile_ == -1 )
    {
	strError_.appStrRaw( ::strerror( ::errno ) );
	return;
    }

    // Success.
    fFile_ = true;
}



// Destructor.
WhFileOut::~WhFileOut( )
{
    finalize( );
}



// Finalizer.
void WhFileOut::finalize( )
{
    if ( fFile_ )
    {
	if ( ::close( fdFile_ ) != 0 )
	    ErFatal( "WhFileOut::finalize: failed close." );
    }
    fFile_ = false;
}    



// Write an array of chars.
void WhFileOut::writeArrayChar( const char * pcWrite, int ncWrite )
{
    // Check fd.
    if ( !fFile_ )
	ErAbort( "WhFileOut::writeArrayChar: not bound." );

    // Write all chars.
    for ( int icWriteTotal = 0; icWriteTotal < ncWrite; )
    {
	// Write some chars.
	const int ncWriteSome = ::write( fdFile_,
	    pcWrite + icWriteTotal, ncWrite - icWriteTotal );

	// Handle errors.
	if ( ncWriteSome  > ncWrite - icWriteTotal )
	    ErFatal( "WhFileOut::writeArrayChar: wild write." );
	if ( ncWriteSome  < 0 )
	    ErFatal( "WhFileOut::writeArrayChar: failed write." );
	if ( ncWriteSome == 0 )
	    ErFatal( "WhFileOut::writeArrayChar: early eof." );

	// Advance.
	icWriteTotal += ncWriteSome;
    }
}
