// Copyright 1994, 1995 Michael Chastain
// Licensed under the Gnu Public License, Version 2
//
// File: TraceMain.cc
//   Main program for mec-trace.
//
// File Created:	21 Jun 1994		Michael Chastain
// Last Edited:		18 Nov 1995		Michael Chastain

// Program version.
static const char acVersion [] = "0.3";



// Unix library.
#include <sys/types.h>
#include <sys/wait.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

// My library classes.
#include <CxFetch.hh>
#include <ErFatal.hh>
#include <ErMem.hh>
#include <ErPtr.hh>
#include <ErUser.hh>
#include <EvBase.hh>
#include <EvFirst.hh>
#include <EvType.hh>
#include <PrProc.hh>
#include <WhFileOut.hh>
#include <WhFlatOut.hh>
#include <WhString.hh>

// My application classes.
#include <TraceArg.hh>



// Override builtin 'new' and 'delete'.
//   g++ 2.6.0: profiler won't profile the builtins.
void *	operator new		( size_t s )        { return ::malloc( s ); }
void *	operator new    []	( size_t s )        { return ::malloc( s ); }
void	operator delete		( void *p, size_t ) { ::free( p ); }
void	operator delete []	( void *p, size_t ) { ::free( p ); }



// Local functions.
int	do_trace		( const TraceArg & );



// Entry point.
int main( int, const char * argv [] )
{
    // Collect arguments.
    TraceArg argTrace( argv );

    // Handle argument errors.
    if ( argTrace.isError( ) )
    {
	argTrace.showUsage( acVersion );
	return 2;
    }

    // Handle '-h' (help).
    if ( argTrace.hasHelp( ) )
    {
	argTrace.showUsage( acVersion );
	return 0;
    }

    // Trace.
    return do_trace( argTrace );
}



// Trace a process.
int do_trace( const TraceArg & argTrace )
{
    // Spawn a child.
    PrProc * pprocTrace = new PrProc( argTrace.getTargetArgv( ) );
    if ( pprocTrace == 0 )
	ErMem( );

    // Create a context.
    CxFetch cxTrace( pprocTrace );
    PrProc & procTrace = cxTrace.getProc( );

    // This scope is for the output file.
    {
	// Open output file.
	WhFileOut fileTrace( argTrace.getNameTrace( ), WhFileOut::tyOpenWriteData );
	if ( fileTrace.isError( ) )
	{
	    ErUser( argTrace.getNameSelf( ), "trace file",
		argTrace.getNameTrace( ), fileTrace.getError( ) );
	    return 1;
	}

	// Open output flat.
	WhFlatOut flatOut( 2 * 65536 );
	flatOut.putInt( 0x0300FACE );

	// Remember most recent EvSci.
	EvBase * pevSciLast = new EvFirst;

	// Monitor child's behaviour.
	for ( ; ; )
	{
	    // Wait for child.
	    procTrace.waitOne( );
	    const TyEvent tyEvFetch = procTrace.getTyEvent( );

	    // Create an event.
	    EvBase * pevFetch = TyEventCreateBase( tyEvFetch );
	    if ( pevFetch == 0 )
		ErPtr( );

	    // Stamp time.
	    pevFetch->setStamp( procTrace.getTimeSecond( ), procTrace.getTimeMicro( ) );

	    // Set up pointers.
	    cxTrace.setPtr( tyEvFetch == tyEvSci
		? pevFetch->ptrEvSci( ) : pevSciLast->ptrEvSci( ),
		pevFetch->ptrEvSco( ) );

	    // Fetch-o-rama.
	    pevFetch->fetch( cxTrace );

	    // Warn about unreplayable events.
	    if ( pevFetch->isModelFail( ) )
	    {
		WhString strWarn;
		strWarn.appStrRaw( "Warning: unreplayable event: "	);
		strWarn.appStrRaw( pevFetch->getModelFail( )	);
		strWarn.appStrRaw( ".\n"				);
		ErUser( strWarn );
	    }

	    // Output event.
	    flatOut.putInt( int( tyEvFetch ) );
	    pevFetch->toFlat( flatOut );

	    // Flush output.
	    if ( flatOut.count( ) > 65536 )
	    {
		fileTrace.writeArrayChar( flatOut.address( ), flatOut.count( ) );
		flatOut.clear( );
	    }

	    // Track last evSci seen.
	    if ( tyEvFetch == tyEvSci )
	    {
		delete pevSciLast;
		pevSciLast = pevFetch;
	    }
	    else
	    {
		delete pevFetch;
	    }

	    // Break on exit.
	    if ( tyEvFetch == tyEvExit )
		break;

	    // Continue.
	    procTrace.ptraceCont( );
	}

	// Flush output.
	fileTrace.writeArrayChar( flatOut.address( ), flatOut.count( ) );
	flatOut.clear( );
    }

    // Failed exec?
    if ( !procTrace.hasExeced( ) )
    {
	// Failed.
	int istatusExit = procTrace.getStatusWait( );
	ErUser( argTrace.getNameSelf( ), "run file",
	    argTrace.getTargetArgv( )[0],
	    ::strerror( WEXITSTATUS( istatusExit ) ) );

	// Unlink output file.
	WhString strNameTraceNul( argTrace.getNameTrace( ) );
	strNameTraceNul.appChrRaw( '\0' );
	strNameTraceNul.checkCcs( );
	if ( ::unlink( strNameTraceNul.address( ) ) != 0 )
	    ErFatal( "do_trace: failed unlink." );
	return 2;
    }
    else
    {
	// Succeeded.
	return 0;
    }
}
