// Copyright 1994, 1995 Michael Chastain
// Licensed under the Gnu Public License, Version 2
//
// File: PrProc.hh
//   Process handler.
//   This is an external resource handler class.
//
// File Created:	04 Oct 1994		Michael Chastain
// Last Reviewed:	23 Feb 1995		Michael Chastain
// Last Edited:		18 Nov 1995		Michael Chastain

#if !defined(PR_PROC_HH)
#define PR_PROC_HH

#include <sys/types.h>
#include <sys/resource.h>

#include <ErPtr.hh>
#include <EvType.hh>
#include <MmSpace.hh>
#include <MmType.hh>
#include <WhLap.hh>
#include <WhList.hh>

class	CxStore;
class	PrBpt;



class	PrProc
{
    // Execution states.
    public:
	enum	StExec
	{
	    stExecRunning,	// Running.
	    stExecStopped,	// Stopped.
	    stExecKilled,	// Killed.
	    stExecExited	// Exited.
	};

    // Life cycle methods.
    public:
				PrProc		( const char * [ ] );
				PrProc		( MmWord, pid_t	   );
				~PrProc		(		   );
	void			finalize	(		   );


    // Forbidden copiers.
    private:
				PrProc		( const PrProc &   );
	void			operator =	( const PrProc &   );

    // Pid interface.
    public:
	pid_t			getPgid		(		   ) const;
	pid_t			getPid		(		   ) const;
	pid_t			getPpid		(		   ) const;

    // Accessors.
    public:
	TyEvent			getTyEvent	(		   ) const;
	int			getStatusWait	(		   ) const;
	int			getTimeSecond	(		   ) const;
	int			getTimeMicro	(		   ) const;
	struct rusage &		getRusage	(		   ) const;
	MmWord			getSysEntry	(		   ) const;
	MmWord			getSubEntry	(		   ) const;
	bool			hasExeced	(		   ) const;

    // Execution interface.
    public:
	void			killProc	(		   );
	void			ptraceCont	(		   );
	void			ptraceKill	(		   );
	void			ptraceStep	(		   );
	void			waitOne		(		   );
	static CxStore &	waitMulti	( WhLap<CxStore> & );

    // Wait helpers.
    private:
	void			waitUpdate	( const PrBpt *	   );
	bool			waitIsSignal	(		   );
	bool			waitIsStep	(		   );
	bool			waitIsBpt	( const PrBpt *	   );
	void			waitDoSc	(		   );

    // Register interface.
    public:
	MmWord			fetchReg	( int		   ) const;
	MmWord			fetchRegFl	(		   ) const;
	MmWord			fetchRegPc	(		   ) const;
	MmWord			fetchRegSp	(		   ) const;
	MmWord			fetchRegSc	(		   ) const;
	void			fetchAllReg	( WhList<MmWord> & ) const;
	void			fetchListReg	( int, int,
						  WhList<MmWord> & ) const;
	void			storeReg	( int, MmWord      );
	void			storeRegFl	( MmWord	   );

    // Address-space interface.
    public:
	bool			fetchWord	( TyMmSpace, MmAddr,
						  MmWord &	   ) const;
	bool			fetchListWord	( TyMmSpace, MmAddr, int,
						  WhList<MmWord> & ) const;
	bool			storeWord	( TyMmSpace, MmAddr,
						  MmWord	   );
	bool			storeListWord	( TyMmSpace, MmAddr, int,
						  const WhList <MmWord> &
						  		   );

    // Instance data.
    private:
	StExec			stExec_;	// Execution state.
	TyEvent			stEv_;		// Event state.
	TyEvent			tyEv_;		// Current event type.
	pid_t			pid_;		// My process.
	pid_t			ppid_;		// My parent.
	int			istatusWait_;	// Last 'wait' status.
	int			itimeSecond_;	// Last 'wait' time.
	int			itimeMicro_;	// Last 'wait' time.
	struct rusage *		prusageWait_;	// Last 'wait' rusage.
	MmWord			wSysEntry_;	// Sys entry.
	MmWord			wSubEntry_;	// Sub entry.
	bool			fExeced_;	// Has exec'ed.
	bool			fStep_;		// Is stepping.
};



// Inline methods.

inline pid_t PrProc::getPid( ) const
{
    return pid_;
}

inline pid_t PrProc::getPpid( ) const
{
    return ppid_;
}

inline TyEvent PrProc::getTyEvent( ) const
{
    return tyEv_;
}

inline int PrProc::getStatusWait( ) const
{
    return istatusWait_;
}

inline int PrProc::getTimeSecond( ) const
{
    return itimeSecond_;
}

inline int PrProc::getTimeMicro( ) const
{
    return itimeMicro_;
}

inline struct rusage & PrProc::getRusage( ) const
{
    if ( prusageWait_ == 0 )
	ErPtr( );
    return *prusageWait_;
}

inline MmWord PrProc::getSysEntry( ) const
{
    return wSysEntry_;
}

inline MmWord PrProc::getSubEntry( ) const
{
    return wSubEntry_;
}

inline bool PrProc::hasExeced( ) const
{
    return fExeced_;
}



#endif
