// Copyright 1994, 1995 Michael Chastain
// Licensed under the Gnu Public License, Version 2
//
// File: PrProc.cc
//   Process handler.
//   This is an external resource handler class.
//
// File Created:	04 Oct 1994		Michael Chastain
// Last Reviewed:	23 Feb 1995		Michael Chastain
// Last Edited:		19 Nov 1995		Michael Chastain

#if defined(MEC_TARGET_LIX)
#include <linux/unistd.h>
#endif

#if defined(MEC_TARGET_SUN)
#include <FixHeadSun.hh>
#include <sys/user.h>
#endif

#include <sys/types.h>
#include <sys/ptrace.h>
#include <sys/signal.h>
#include <sys/time.h>
#include <sys/wait.h>
#include <errno.h>
#include <string.h>
#include <time.h>
#include <unistd.h>

#include <CxStore.hh>
#include <ErAbort.hh>
#include <ErFatal.hh>
#include <ErMem.hh>
#include <ErPtr.hh>
#include <MmInsn.hh>
#include <MmScr.hh>
#include <MmSpace.hh>
#include <MmType.hh>
#include <PrBpt.hh>
#include <PrProc.hh>



// Small machine-dependent functions.

#if defined(MEC_TARGET_LIX)

static inline int ptrace( int tyReq, int pid, unsigned char * addr, int data )
{
    return ptrace( tyReq, pid, (int) addr, data );
}

static inline int FetchSpace( TyMmSpace tyMmSpace )
{
    if ( tyMmSpace == tyMmData ) return PTRACE_PEEKDATA; // Fast normal case.
    if ( tyMmSpace == tyMmUser ) return PTRACE_PEEKUSR;
    if ( tyMmSpace == tyMmText ) return PTRACE_PEEKTEXT;
    return PTRACE_PEEKDATA;
}

static inline int StoreSpace( TyMmSpace tyMmSpace )
{
    if ( tyMmSpace == tyMmData ) return PTRACE_POKEDATA; // Fast normal case.
    if ( tyMmSpace == tyMmUser ) return PTRACE_POKEUSR;
    if ( tyMmSpace == tyMmText ) return PTRACE_POKETEXT;
    return PTRACE_POKEDATA;
}

MmWord PrProc::fetchRegFl( ) const { return fetchReg(      EFL - EBX ); }
MmWord PrProc::fetchRegPc( ) const { return fetchReg(      EIP - EBX ); }
MmWord PrProc::fetchRegSp( ) const { return fetchReg(     UESP - EBX ); }
MmWord PrProc::fetchRegSc( ) const { return fetchReg( ORIG_EAX - EBX ); }

void PrProc::storeRegFl( MmWord w ) { storeReg(  EFL - EBX, w ); }

void PrProc::fetchAllReg( WhList <MmWord> & lwReg ) const
{
    fetchListReg( 0, 17, lwReg );
}

#endif



// Small machine-dependent functions.

#if defined(MEC_TARGET_SUN)

static inline int ptrace( enum ptracereq tyReq, int pid, unsigned char * addr,
    int data )
{
    return ptrace( tyReq, pid, addr, data, 0 );
}

static inline enum ptracereq FetchSpace( TyMmSpace tyMmSpace )
{
    if ( tyMmSpace == tyMmData ) return PTRACE_PEEKDATA; // Fast normal case.
    if ( tyMmSpace == tyMmUser ) return PTRACE_PEEKUSER;
    if ( tyMmSpace == tyMmText ) return PTRACE_PEEKTEXT;
    return PTRACE_PEEKDATA;
}

static inline enum ptracereq StoreSpace( TyMmSpace tyMmSpace )
{
    if ( tyMmSpace == tyMmData ) return PTRACE_POKEDATA; // Fast normal case.
    if ( tyMmSpace == tyMmUser ) return PTRACE_POKEUSER;
    if ( tyMmSpace == tyMmText ) return PTRACE_POKETEXT;
    return PTRACE_POKEDATA;
}

MmWord PrProc::fetchRegFl( ) const { return fetchReg( PS ); }
MmWord PrProc::fetchRegPc( ) const { return fetchReg( PC ); }
MmWord PrProc::fetchRegSp( ) const { return fetchReg( SP ); }
MmWord PrProc::fetchRegSc( ) const { return fetchReg(  7 ); }

void PrProc::storeRegFl( MmWord w ) { storeReg( PS, w ); }

void PrProc::fetchAllReg( WhList <MmWord> & lwRegRet ) const
{
    fetchListReg( 0, 8, lwRegRet );
}

#endif



// Constructor: spawn.
PrProc::PrProc( const char * argv [] )
    : stExec_		( stExecRunning	)
    , stEv_		( tyEvBlank	)
    , tyEv_		( tyEvBlank	)
    , pid_		( 0		)
    , ppid_		( ::getpid( )	)
    , istatusWait_	( 0		)
    , itimeSecond_	( 0		)
    , itimeMicro_	( 0		)
    , prusageWait_	( 0		)
    , wSysEntry_	( 0		)
    , wSubEntry_	( 0		)
    , fExeced_		( false		)
    , fStep_		( false		)
{
    // Check parameters.
    if ( argv == 0 || argv[0] == 0 )
	ErPtr( );

    // Fork the process.
    //   Parent returns.
    //   Child -must never- return.
    pid_ = ::fork( );
    switch ( pid_ )
    {
    case -1: ErFatal( "PrProc::PrProc: failed fork." );
    case 0:  break;
    default: return;
    }

    // Enable tracing.
    if ( ::ptrace( PTRACE_TRACEME, ::getpid( ), 0, 0 ) != 0 )
	ErFatal( "PrProc::PrProc: failed ptrace." );

    // Trap for the parent.
    //   No system calls allowed after this until the exec.
    //   If you try, replay will give mismatches.
    if ( ::kill( ::getpid( ), SIGTRAP ) != 0 )
	ErFatal( "PrProc::PrProc: failed kill." );

    // Exec the process.
    //   g++ 2.6.3: I believe this prototype is lacking some 'const'.
    ::execvp( argv[0], (char * const []) argv );
    ::_exit( ::errno );
}



// Constructor: attach.
PrProc::PrProc( MmWord wpidAttach, pid_t ppidAttach )
    : stExec_		( stExecStopped	)
    , stEv_		( tyEvSco	)
    , tyEv_		( tyEvSco	)
    , pid_		( wpidAttach	)
    , ppid_		( ppidAttach	)
    , istatusWait_	( 0		)
    , itimeSecond_	( 0		)
    , itimeMicro_	( 0		)
    , prusageWait_	( new rusage	)
    , wSysEntry_	( 0		)
    , wSubEntry_	( 0		)
    , fExeced_		( false		)
    , fStep_		( false		)
{
    if ( prusageWait_ == 0 )
	ErMem( );

    if ( ::ptrace( PTRACE_ATTACH, pid_, 0, 0 ) != 0 )
	ErFatal( "PrProc::PrProc: failed ptrace." );
    if ( ::wait4( pid_, 0, 0, 0 ) != pid_ )
	ErFatal( "PrProc::PrProc: failed wait4." );
}



// Destructor.
PrProc::~PrProc( )
{
    finalize( );
}



// Finalize.
void PrProc::finalize( )
{
    if ( stExec_ == stExecRunning || stExec_ == stExecStopped )
    {
	if ( ::kill( pid_, SIGKILL ) != 0 )
	    ErFatal( "PrProc::finalize: failed kill." );
	stExec_ = stExecKilled;
    }

    if ( stExec_ == stExecKilled )
    {
	if ( ::wait4( pid_, 0, 0, 0 ) != pid_ )
	    ErFatal( "PrProc::finalize: failed wait." );
	stExec_ = stExecExited;
    }

    if ( prusageWait_ != 0 )
	delete prusageWait_;
    prusageWait_ = 0;
}



// Get process group id.
pid_t PrProc::getPgid( ) const
{
    ::errno = 0;

#if defined(MEC_TARGET_LIX)
    const pid_t pgid = ::getpgid( pid_ );
#endif

#if defined(MEC_TARGET_SUN)
#if 0
    // Man page is confusing!
    const pid_t pgid = ::getpgrp( pid_ );
#else
    const pid_t pgid = 0;
    ErAbort( "PrProc::getPgid: not implemented." );
#endif
#endif

    if ( ::errno != 0 )
	ErFatal( "PrProc::getPgid: failed getpgid." );
    return pgid;
}



// Kill the process.
void PrProc::killProc( )
{
    // Bullets into corpse.
    if ( stExec_ == stExecExited )
	return;

    // Kill.
    if ( ::kill( pid_, SIGKILL ) != 0 )
	ErFatal( "PrProc::killProc: failed kill." );

    // Move to new state.
    stExec_ = stExecKilled;
}



// Continue the process.
void PrProc::ptraceCont( )
{
    // Check state.
    if ( stExec_ != stExecStopped )
	ErAbort( "PrProc::ptraceCont: bad state." );

    // Continue.
    const int iSignal = (tyEv_ == tyEvSignal) ? WSTOPSIG(istatusWait_) : 0;
    if ( ::ptrace( PTRACE_SYSCALL, pid_, 0, iSignal ) != 0 )
	ErFatal( "PrProc::ptraceCont: failed ptrace." );

    // Move to new state.
    stExec_ = stExecRunning;
}



// Kill the process.
void PrProc::ptraceKill( )
{
    // Bullets into corpse.
    if ( stExec_ == stExecExited )
	return;

    // Kill.
    if ( ::ptrace( PTRACE_KILL, pid_, 0, 0 ) != 0 )
	ErFatal( "PrProc::ptraceKill: failed ptrace." );

    // Move to new state.
    stExec_ = stExecKilled;
}



// Step the process.
void PrProc::ptraceStep( )
{
    // Check state.
    if ( stExec_ != stExecStopped )
	ErAbort( "PrProc::stepProc: bad state." );

    // Step.
    if ( ::ptrace( PTRACE_SINGLESTEP, pid_, 0, 0 ) != 0 )
	ErFatal( "PrProc::stepProc: failed ptrace." );

    // Move to new state.
    fStep_  = true;
    stExec_ = stExecRunning;
}



// Wait for status change in process.
void PrProc::waitOne( )
{
    // Wait for this process.
    if ( ::wait4( pid_, &istatusWait_, 0, prusageWait_ ) != pid_ )
	ErFatal( "PrProc::waitOne: failed wait." );

    // Mark time.
    struct timeval timeval;
    if ( ::gettimeofday( &timeval, 0 ) != 0 )
	ErFatal( "PrProc::waitOne: failed gettimeofday." );
    itimeSecond_ = timeval.tv_sec;
    itimeMicro_  = timeval.tv_usec;

    // Set status.
    waitUpdate( 0 );
}



// Wait for status change in one of multi processes.
CxStore & PrProc::waitMulti( WhLap <CxStore> & lcxWait )
{
    // Wait for any process.
    ::errno = 0;
    int istatusWait;
    struct rusage rusageWait;
    const pid_t pidWait = ::wait4( -1, &istatusWait, 0, &rusageWait );
    if ( ::errno != 0 )
	ErFatal( "PrProc::waitMulti: failed wait." );

    // Mark time.
    struct timeval timeval;
    if ( ::gettimeofday( &timeval, 0 ) != 0 )
	ErFatal( "PrProc::waitMulti: failed gettimeofday." );

    // Find process which changed.
    for ( int icxWait = 0; icxWait < lcxWait.count( ); ++icxWait )
    {
	CxStore & cxWait   = lcxWait.refNonConst( icxWait );
	PrProc  & procWait = cxWait.getProc( );
	if ( procWait.pid_ == pidWait )
	{
	    procWait.istatusWait_ = istatusWait;
	    procWait.itimeSecond_ = timeval.tv_sec;
	    procWait.itimeMicro_  = timeval.tv_usec;
	    if ( procWait.prusageWait_ != 0 )
		*procWait.prusageWait_ = rusageWait;
	    procWait.waitUpdate( &cxWait.getBpt( ) );
	    return cxWait;
	}
    }

    // Not found!
    ErFatal( "PrProc::waitMulti: unlisted process." );
    return lcxWait.refNonConst( -1 );
}



// Update state of a process after wait.
void PrProc::waitUpdate( const PrBpt * pbpt )
{
    // Check state.
    if ( stExec_ != stExecRunning && stExec_ != stExecKilled )
	ErAbort( "PrProc::waitUpdate: bad state." );

    // Advance exec state.
    if      ( WIFSTOPPED  (istatusWait_) )
    {
	stExec_ = stExecStopped;
	stEv_   = stEv_;
	if      ( waitIsSignal (      ) ) tyEv_ = tyEvSignal;
	else if ( waitIsStep   (      ) ) tyEv_ = tyEvStep;
	else if ( waitIsBpt    ( pbpt ) ) tyEv_ = tyEvBpt;
	else      waitDoSc     (      );
    }
    else if ( WIFSIGNALED (istatusWait_) || WIFEXITED (istatusWait_) )
    {
	stExec_ = stExecExited;
	stEv_   = tyEvExit;
	tyEv_   = tyEvExit;
    }
    else
    {
	// Can't happen according to 'man 2 wait'.
	ErAbort( "PrProc::waitUpdate: bad status." );
    }

    // Track whether ever exec'ed.
    if ( tyEv_ == tyEvExec )
	fExeced_ = true;
}



// Recognize a signal.
bool PrProc::waitIsSignal( )
{
    return WSTOPSIG(istatusWait_) != SIGTRAP;
}



// Recognize a single-step.
bool PrProc::waitIsStep( )
{
    const bool fStepOld = fStep_;
    fStep_ = false;
    return fStepOld;
}



// Recognize a breakpoint.
bool PrProc::waitIsBpt( const PrBpt * pbpt )
{
    if ( pbpt != 0 )
    {
	const MmAddr addrInsn = MmAddr( fetchRegPc( ) );
	if ( addrInsn != 0 )
	    return pbpt->isBpt( addrInsn - insnBpt[0] * sizeof(MmInsn) );
    }
    return false;
}



// Recognize a system call.
void PrProc::waitDoSc( )
{
    // Advance event state.
    switch ( stEv_ )
    {
    default:        ErAbort( "PrProc::waitDoSc: bad state." );
    case tyEvBlank: stEv_ = tyEvFirst; break;
    case tyEvFirst: stEv_ = tyEvSci;   break;
    case tyEvExec:  stEv_ = tyEvSco;   break;
    case tyEvSco:   stEv_ = tyEvSci;   break;

    case tyEvSci:
	switch ( wSysEntry_ )
	{
	default:
	    stEv_ = tyEvSco;
	    break;

#if defined(MEC_TARGET_LIX)
	case __NR_execve:
	    {
		MmScr scrExec( MmScr::tyScrInt );
		scrExec.fetchProc( *this );
		stEv_ = scrExec.isError( ) ? tyEvSco : tyEvExec;
	    }
	    break;

	case __NR_vm86:
	    stEv_ = tyEvSci;
	    break;
#endif
	}
	break;
    }

    // Fetch system call number.
    if ( stEv_ == tyEvSci )
    {
	wSysEntry_ = fetchRegSc( );
	wSubEntry_ = 0;

#if defined(MEC_TARGET_LIX)
	if ( wSysEntry_ == __NR_ipc || wSysEntry_ == __NR_socketcall )
	    wSubEntry_ = fetchReg( 0 );
#endif

    }

    // Type follows state.
    tyEv_ = stEv_;
}



// Fetch a register.
MmWord PrProc::fetchReg( int iReg ) const
{
    // Check state.
    if ( stExec_ != stExecStopped )
	ErAbort( "PrProc::fetchReg: bad state." );

    // Check register.
    if ( iReg < 0 )
	ErAbort( "PrProc::fetchReg: negative index." );

#if defined(MEC_TARGET_LIX)
    // Fetch from process.
    const int wAddrReg = (EBX + iReg) * sizeof(MmWord);
    ::errno = 0;
    const MmWord wReg = ::ptrace( PTRACE_PEEKUSR, pid_, wAddrReg, 0 );
    if ( ::errno != 0 )
	ErFatal( "PrProc::fetchReg: failed fetch." );
    return wReg;
#endif

#if defined(MEC_TARGET_SUN)
    // Fetch from process.
    struct user user;
    const MmAddr addrReg =
	MmAddr( MmAddr( &user.u_arg[iReg] ) - MmAddr( &user ) );
    ::errno = 0;
    const MmWord wReg = ::ptrace( PTRACE_PEEKUSER, pid_, addrReg, 0 );
    if ( ::errno != 0 )
	ErFatal( "PrProc::fetchReg: failed fetch." );
    return wReg;
#endif
}



// Fetch a list of registers.
void PrProc::fetchListReg( int iRegFirst, int nReg, WhList <MmWord> & lwReg )
    const
{
    // Check count.
    if ( nReg < 0 )
	ErAbort( "PrProc::fetchListReg: negative count." );

    // Fetch registers.
    lwReg.clear( nReg );
    for ( int iReg = 0; iReg < nReg; ++iReg )
	lwReg.appendVal( fetchReg( iRegFirst + iReg ) );
}



// Store a register.
void PrProc::storeReg( int iReg, MmWord wReg )
{
    // Check state.
    if ( stExec_ != stExecStopped )
	ErAbort( "PrProc::storeReg: bad state." );

    // Check register.
    if ( iReg < 0 )
	ErAbort( "PrProc::storeReg: negative index." );

#if defined(MEC_TARGET_LIX)
    // Store to process.
    const int wAddrReg = (EBX + iReg) * sizeof(MmWord);
    ::errno = 0;
    ::ptrace( PTRACE_POKEUSR, pid_, wAddrReg, wReg );
    if ( ::errno != 0 )
	ErFatal( "PrProc::storeReg: failed store." );
#endif

#if defined(MEC_TARGET_SUN)
    // Store to process.
    struct user user;
    const MmAddr addrReg =
	MmAddr( MmAddr( &user.u_arg[iReg] ) - MmAddr( &user ) );
    ::errno = 0;
    ::ptrace( PTRACE_POKEUSER, pid_, addrReg, wReg );
    if ( ::errno != 0 )
	ErFatal( "PrProc::storeReg: failed store." );
#endif
}



// Fetch a word from user struct, text space, or data space.
bool PrProc::fetchWord( TyMmSpace tyMmSpace, MmAddr addrFetch,
    MmWord & wFetchRet ) const
{
    // Check state.
    if ( stExec_ != stExecStopped )
	ErAbort( "PrProc::fetchWord: bad state." );

    // Fetch the word.
    ::errno = 0;
    wFetchRet = ::ptrace( FetchSpace( tyMmSpace ), pid_, addrFetch, 0 );
    return ::errno == 0;
}



// Fetch a list of words from user struct, text space, or data space.
bool PrProc::fetchListWord( TyMmSpace tyMmSpace, MmAddr addrFirst,
    int nbFetch, WhList <MmWord> & lwFetchRet ) const
{
    // Check state.
    if ( stExec_ != stExecStopped )
	ErAbort( "PrProc::fetchListWord: bad state." );

    // Check count.
    if ( nbFetch < 0 )
	ErAbort( "PrProc::fetchListWord: negative count." );

    // Size return value.
    const int nwFetch = (nbFetch + sizeof(MmWord) - 1) / sizeof(MmWord);
    lwFetchRet.clear( nwFetch );

    // Fetch with low-speed but standard 'ptrace'.
    ::errno = 0;
    for ( MmAddr addrFetch  = addrFirst;
		 addrFetch  < addrFirst + nbFetch;
		 addrFetch += sizeof(MmWord) )
    {
	// Fetch one word.
	const MmWord wFetch = ::ptrace( FetchSpace( tyMmSpace ),
	    pid_, addrFetch, 0 );
	if ( ::errno != 0 )
	    return false;
	lwFetchRet.appendVal( wFetch );
    }

    // Success.
    return true;
}



// Store a word to user struct, text space, or data space.
bool PrProc::storeWord( TyMmSpace tyMmSpace, MmAddr addrStore, MmWord wStore )
{
    // Check state.
    if ( stExec_ != stExecStopped )
	ErAbort( "PrProc::storeWord: bad state." );

    // Store the word.
    ::errno = 0;
    ::ptrace( StoreSpace( tyMmSpace ), pid_, addrStore, wStore );
    return ::errno == 0;
}



// Store a list of words to user struct, text space, or data space.
bool PrProc::storeListWord( TyMmSpace tyMmSpace, MmAddr addrFirst,
    int nbStore, const WhList <MmWord> & lwStore )
{
    // Check state.
    if ( stExec_ != stExecStopped )
	ErAbort( "PrProc::storeListWord: bad state." );

    // Check count.
    if ( nbStore < 0 )
	ErAbort( "PrProc::storeListWord: negative count." );

    // Check size.
    if ( lwStore.count( ) * sizeof(MmWord) != nbStore * sizeof(MmByte) )
	ErAbort( "PrProc::storeListWord: count mismatch." );

    // Generic low speed implementation.
    ::errno = 0;
    int iwStore = 0;
    for ( MmAddr addrStore  = addrFirst;
		 addrStore  < addrFirst + nbStore;
		 addrStore += sizeof(MmWord) )
    {
	::ptrace( StoreSpace( tyMmSpace ), pid_, addrStore, lwStore[iwStore] );
	if ( ::errno != 0 )
	    return false;
	++iwStore;
    }

    // Success.
    return true;
}
