// Copyright 1995 Michael Chastain
// Licensed under the Gnu Public License, Version 2
//
// File: PrMap.cc
//   Target address map.
//   This is a concrete class.
//
// File Created:	24 Apr 1995		Michael Chastain
// Last Edited:		08 Nov 1995		Michael Chastain

#include <ErAbort.hh>
#include <ErMem.hh>
#include <ErUser.hh>
#include <MmArea.hh>
#include <MmSpace.hh>
#include <MmType.hh>
#include <PrMap.hh>
#include <WhLap.hh>
#include <WhString.hh>



// Constructor.
PrMap::PrMap( )
    : larea_	( )
{
    MmArea * pareaBlank = new MmArea( );
    if ( pareaBlank == 0 )
	ErMem( );
    pareaBlank->setCover( tyMmBlank );
    larea_.appendAp( pareaBlank );
}



// Get current 'brk' address.
MmAddr PrMap::getAddrBrk( ) const
{
    MmAddr addrBrk = 0;
    for ( int iArea = 0; iArea < countArea( ); ++iArea )
    {
	const MmArea & area = getArea( iArea );
	switch ( area.getTyMmSpace( ) )
	{
	default:
	    break;
	case tyMmText:
	case tyMmData:
	case tyMmBss:
	case tyMmBrk:
	    addrBrk = area.getAddrMax( );
	    break;
	}
    }
    return addrBrk;
}



// Get maximum stack pointer.
MmAddr PrMap::getAddrStackMax( ) const
{
    return mmAddrStackMax;
}



// Merge an area.
void PrMap::mergeArea( const MmArea & areaNew )
{
    // Get scalars for short names.
    const TyMmSpace tyMmSpaceNew = areaNew.getTyMmSpace ( );
    const MmAddr    addrNewMin   = areaNew.getAddrMin   ( );
    const MmAddr    addrNewMax   = areaNew.getAddrMax   ( );
    const MmAddr    addrMapMin   = larea_[0].getAddrMin( );
    const MmAddr    addrMapMax   = larea_[larea_.count( ) - 1].getAddrMax( );

    // Check bounds.
    if ( addrNewMin < addrMapMin || addrNewMax > addrMapMax )
	ErAbort( "PrMap::mergeArea: bad address." );

    // Make new areas.
    WhLap <MmArea> lareaNew;
    for ( int iareaOld = 0; iareaOld < larea_.count( ); ++iareaOld )
    {
	// Get old area.
	const MmArea &  areaOld      = larea_[iareaOld];
	const TyMmSpace tyMmSpaceOld = areaOld.getTyMmSpace ( );
	const MmAddr    addrOldMin   = areaOld.getAddrMin   ( );
	const MmAddr    addrOldMax   = areaOld.getAddrMax   ( );

	// 4!/(2*2) == 6 cases for endpoints to fall.
	if ( addrOldMin <= addrNewMin )
	{
	    if      ( addrOldMax <= addrNewMin )
	    {
		addArea( lareaNew, tyMmSpaceOld, addrOldMin, addrOldMax );
	    }
	    else if ( addrOldMax <= addrNewMax )
	    {
		addArea( lareaNew, tyMmSpaceOld, addrOldMin, addrNewMin );
		addArea( lareaNew, tyMmSpaceNew, addrNewMin, addrOldMax );
	    }
	    else
	    {
		addArea( lareaNew, tyMmSpaceOld, addrOldMin, addrNewMin );
		addArea( lareaNew, tyMmSpaceNew, addrNewMin, addrNewMax );
		addArea( lareaNew, tyMmSpaceOld, addrNewMax, addrOldMax );
	    }
	}
	else
	{
	    if      ( addrNewMax <= addrOldMin )
	    {
		addArea( lareaNew, tyMmSpaceOld, addrOldMin, addrOldMax );
	    }
	    else if ( addrNewMax <= addrOldMax )
	    {
		addArea( lareaNew, tyMmSpaceNew, addrOldMin, addrNewMax );
		addArea( lareaNew, tyMmSpaceOld, addrNewMax, addrOldMax );
	    }
	    else
	    {
		addArea( lareaNew, tyMmSpaceNew, addrOldMin, addrOldMax );
	    }
	}
    }

    // Check new area list.
    if ( lareaNew[0].getAddrMin( )                     != addrMapMin
    ||   lareaNew[lareaNew.count( ) - 1].getAddrMax( ) != addrMapMax )
	ErAbort( "PrMap::mergeArea: mangled address." );

    // Copy in new area list.
    larea_.reset( lareaNew.count( ) );
    for ( int iareaNew = 0; iareaNew < lareaNew.count( ); ++iareaNew )
    {
	MmArea * pareaCopy = new MmArea( lareaNew[iareaNew] );
	if ( pareaCopy == 0 )
	    ErMem( );
	larea_.appendAp( pareaCopy );
    }

#if 0
    // List for inspection.
    WhString strList;
    strList.appStrRaw( "PrMap::mergeArea: "	);
    strList.appIntFmt( larea_.count( )		);
    strList.appChrRaw( '\n'			);
    for ( int iarea = 0; iarea < larea_.count( ); ++iarea )
	larea_[iarea].fmtStr( strList );
    ErUser( strList );
#endif
}



// Add area to end.
void PrMap::addArea( WhLap <MmArea> & lareaNew,
    TyMmSpace tyMmSpace, MmAddr addrMin, MmAddr addrMax )
{
    if ( addrMin > addrMax )
	ErAbort( "PrMap::addArea: bad address." );

    if ( addrMin < addrMax )
    {
	if ( lareaNew.count( ) > 0 )
	{
	    const MmArea & areaLast = lareaNew[lareaNew.count( ) - 1];
	    if ( addrMin != areaLast.getAddrMax( ) )
		ErAbort( "PrMap::addArea: not adjacent." );
	    if ( tyMmSpace == areaLast.getTyMmSpace( ) )
	    {
		addrMin = areaLast.getAddrMin( );
		lareaNew.remove( lareaNew.count( ) - 1 );
	    }
	}

	MmArea * pareaAdd = new MmArea( tyMmSpace, addrMin, addrMax );
	if ( pareaAdd == 0 )
	    ErMem( );
	lareaNew.appendAp( pareaAdd );
    }
}
