// Copyright 1995 Michael Chastain
// Licensed under the Gnu Public License, Version 2
//
// File: EvSci.cc
//   An event, type 'system call ingoing'.
//   This is a leaf class.
//
// File Created:	11 Apr 1995		Michael Chastain
// Last Edited:		16 Nov 1995		Michael Chastain

#include <CxFetch.hh>
#include <CxStore.hh>
#include <ErAbort.hh>
#include <ErFatal.hh>
#include <EvLog.hh>
#include <EvSci.hh>
#include <EvSco.hh>
#include <EvType.hh>
#include <MmSeg.hh>
#include <MmType.hh>
#include <PrProc.hh>
#include <PrSl.hh>
#include <ScLine.hh>
#include <WhFlatIn.hh>
#include <WhFlatOut.hh>
#include <WhList.hh>
#include <WhString.hh>



// Constructor.
EvSci::EvSci( )
    : fArgTried_	( false	)
    , fArgFetched_	( false )
    , fBound_		( false	)
    , wSysEntry_	( 0	)
    , wSubEntry_	( 0	)
    , lwArg_		(	)
{
    ;
}



// Destructor.
EvSci::~EvSci( )
{
    ;
}



// RTTI.
const EvSci * EvSci::ptrEvSci ( ) const
{
    return this;
}



// Flat input (combiner).
void EvSci::fromFlatEv( WhFlatIn & flatIn )
{
    flatIn.getBool	( fArgTried_	);
    flatIn.getBool	( fArgFetched_	);
    flatIn.getBool	( fBound_	);
    flatIn.getWord	( wSysEntry_	);
    flatIn.getWord	( wSubEntry_	);
    flatIn.getListWord	( lwArg_	);
}



// Flat output (combiner).
void EvSci::toFlatEv( WhFlatOut & flatOut ) const
{
    flatOut.putBool	( fArgTried_	);
    flatOut.putBool	( fArgFetched_	);
    flatOut.putBool	( fBound_	);
    flatOut.putWord	( wSysEntry_	);
    flatOut.putWord	( wSubEntry_	);
    flatOut.putListWord	( lwArg_	);
}



// Format to string (combiner).
void EvSci::fmtStrEv( WhString & strRet ) const
{
    if ( !fBound_ )
	ErAbort( "EvSci::fmtStrEv: not bound." );
    ScLine scLine;
    scLine.initFromSce( wSysEntry_, wSubEntry_ );
    scLine.fmtStrEvSci( strRet, *this );
}



// Fetch from process.
void EvSci::fetch( CxFetch & cxFetch )
{
    // Check bound.
    if ( fBound_ )
	ErAbort( "EvSci::fetch: is bound." );

    // Fetch sys entry # and sub entry #.
    wSysEntry_ = cxFetch.getProc( ).getSysEntry( );
    wSubEntry_ = cxFetch.getProc( ).getSubEntry( );

    // Fetch fetch fetch.
    ScLine scLine;
    scLine.initFromSce ( wSysEntry_, wSubEntry_ );
    scLine.fetchSci    ( cxFetch.getProc( ), fArgTried_, fArgFetched_, lwArg_ );
    scLine.fetchSeg    ( cxFetch, *this );

    // That's all, folks.
    fBound_ = true;
}



// Store to process.
void EvSci::storeAw( CxStore & ) const
{
    ;
}



// Store to process.
void EvSci::storeBc( CxStore & cxStore ) const
{
    // Get process.
    PrProc & procStore = cxStore.getProc( );

    // Check synchronization.
    if ( procStore.getTyEvent( ) != tyEvSci )
	ErFatal( "EvSci::storeBc: mismatched event." );

    // Check bound.
    if ( !fBound_ )
	ErAbort( "EvSci::storeBc: not bound." );

    // Compare sys entry # and sub entry #.
    if ( wSysEntry_ != procStore.getSysEntry( )
    ||   wSubEntry_ != procStore.getSubEntry( ) )
	ErFatal( "EvSci::storeBc: entry mismatch." );

    // Get our ScLine.
    ScLine scLine;
    scLine.initFromSce( wSysEntry_, wSubEntry_ );

    // Compare arguments.
    {
	WhList <MmWord> lwArgRefetch;
	bool fArgTry;
	bool fArgRefetch;
	scLine.fetchSci( procStore, fArgTry, fArgRefetch, lwArgRefetch );
	if ( !matchArg( fArgRefetch, lwArgRefetch ) )
	    ErFatal( "EvSci::storeBc: arg mismatch." );
    }

    // Stash datasets.
    const EvSco * pevScoNext = cxStore.getLog( ).getEvScoNext( );
    if ( pevScoNext != 0 )
    {
	for ( int ids = 0; ids < pevScoNext->countDs( ); ++ids )
	{
	    cxStore.getStashLink( ).stashLink( pevScoNext->getDs( ids ),
		cxStore.getStashData( ) );
	}
    }

    // Annul system call.
    scLine.smashProc( procStore, pevScoNext != 0 && !pevScoNext->isScrError( ) );
}



// Get an argument.
MmWord EvSci::getArg( int iArg ) const
{
    if ( !fArgTried_ || !fArgFetched_ )
	ErAbort( "EvSci::getArg: not bound." );
    return lwArg_[iArg];
}



// Get sys entry.
MmWord EvSci::getSysEntry( ) const
{
    if ( !fBound_ )
	ErAbort( "EvSci::getSysEntry: not bound." );
    return wSysEntry_;
}



// Get sub entry.
MmWord EvSci::getSubEntry( ) const
{
    if ( !fBound_ )
	ErAbort( "EvSci::getSysEntry: not bound." );
    return wSubEntry_;
}



// Return whether I have arguments.
bool EvSci::hasArg( ) const
{
    if ( !fArgTried_ )
	ErAbort( "EvSci::hasArg: not bound." );
    return fArgFetched_;
}



// Match arguments.
bool EvSci::matchArg( bool fArgFetched, const WhList <MmWord> & lwArg ) const
{
    if ( !fArgTried_ )
	ErAbort( "EvSci::matchArg: not bound." );

    if ( fArgFetched != fArgFetched_ )
	return false;

    if ( fArgFetched )
    {
	if ( lwArg.count( ) != lwArg_.count( ) )
	    return false;
	for ( int iArg = 0; iArg < lwArg.count( ); ++iArg )
	{
	    if ( lwArg[iArg] != lwArg_[iArg] )
		return false;
	}
    }

    return true;
}
