// Copyright 1995 Michael Chastain
// Licensed under the Gnu Public License, Version 2
//
// File: ControlFilterLix.cc
//   Event filter.
//
// File Created:	31 Oct 1995		Michael Chastain
// Last Edited:		19 Nov 1995		Michael Chastain

#include <linux/unistd.h>
#include <sys/ptrace.h>
#include <sys/user.h>
#include <errno.h>
#include <signal.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>

#include <CxStore.hh>
#include <ErAbort.hh>
#include <ErFatal.hh>
#include <ErMem.hh>
#include <EvType.hh>
#include <MmInsn.hh>
#include <MmScr.hh>
#include <MmSeg.hh>
#include <MmSpace.hh>
#include <MmType.hh>
#include <PrProc.hh>
#include <PrSl.hh>
#include <TySegLix.hh>
#include <WhLap.hh>
#include <WhList.hh>
#include <WhString.hh>

#include <ControlState.hh>
#include <ControlViewer.hh>

#define __NR_raw_ptrace __NR_ptrace
_syscall4( int, raw_ptrace, MmWord, w0, MmWord, w1, MmWord, w2, MmWord, w3 );



// Local functions.
void	viewer_execve_sci_bc	( CxStore &, WhLap <CxStore> & );
void	viewer_execve_sco_aw	( CxStore &, WhLap <CxStore> & );
void	viewer_fork_sci_bc	( CxStore &, WhLap <CxStore> & );
void	viewer_fork_sco_aw	( CxStore &, WhLap <CxStore> & );
void	viewer_kill_sci_bc	( CxStore &, WhLap <CxStore> & );
void	viewer_kill_sco_aw	( CxStore &, WhLap <CxStore> & );
void	viewer_ptrace_sci_bc	( CxStore &, WhLap <CxStore> & );
void	viewer_ptrace_sco_aw	( CxStore &, WhLap <CxStore> & );
void	viewer_wait4_sci_bc	( CxStore &, WhLap <CxStore> & );
void	viewer_wait4_sco_aw	( CxStore &, WhLap <CxStore> & );
void	viewer_waitpid_sci_bc	( CxStore &, WhLap <CxStore> & );
void	viewer_waitpid_sco_aw	( CxStore &, WhLap <CxStore> & );



// Viewer, before continue.
void viewer_bc( CxStore & cxViewer, WhLap <CxStore> & lcxControl )
{
    PrProc & procViewer = cxViewer.getProc( );

    if ( procViewer.getTyEvent( ) == tyEvSci )
    {
	switch ( procViewer.getSysEntry( ) )
	{
	default:           break;
	case __NR_clone:   ErFatal( "viewer_bc: cannot handle clone." );
	case __NR_execve:  viewer_execve_sci_bc  ( cxViewer, lcxControl ); break;
	case __NR_fork:    viewer_fork_sci_bc    ( cxViewer, lcxControl ); break;
	case __NR_kill:    viewer_kill_sci_bc    ( cxViewer, lcxControl ); break;
	case __NR_ptrace:  viewer_ptrace_sci_bc  ( cxViewer, lcxControl ); break;
	case __NR_wait4:   viewer_wait4_sci_bc   ( cxViewer, lcxControl ); break;
	case __NR_waitpid: viewer_waitpid_sci_bc ( cxViewer, lcxControl ); break;
	}
    }
}



// Viewer, after wait.
void viewer_aw( CxStore & cxViewer, WhLap <CxStore> & lcxControl )
{
    PrProc & procViewer = cxViewer.getProc( );

    if ( procViewer.getTyEvent( ) == tyEvSco )
    {
	switch ( procViewer.getSysEntry( ) )
	{
	default:           break;
	case __NR_clone:   ErFatal( "viewer_aw: cannot handle clone." );
	case __NR_execve:  viewer_execve_sco_aw  ( cxViewer, lcxControl ); break;
	case __NR_fork:    viewer_fork_sco_aw    ( cxViewer, lcxControl ); break;
	case __NR_kill:    viewer_kill_sco_aw    ( cxViewer, lcxControl ); break;
	case __NR_ptrace:  viewer_ptrace_sco_aw  ( cxViewer, lcxControl ); break;
	case __NR_wait4:   viewer_wait4_sco_aw   ( cxViewer, lcxControl ); break;
	case __NR_waitpid: viewer_waitpid_sco_aw ( cxViewer, lcxControl ); break;
	}
    }
}



// Viewer, 'execve', sci, before continue.
void viewer_execve_sci_bc( CxStore & cxViewer, WhLap <CxStore> & )
{
    // Get process.
    PrProc & procViewer = cxViewer.getProc( );

    // Smash name.
    MmSeg segName;
    segName.fetchProc( procViewer, tyMmData, tySegStrNul,
	MmAddr( procViewer.fetchReg( 0 ) ), 1 );
    if ( segName.hasArea( ) && segName.hasData( ) )
    {
	segName.checkCcs( );
	const PrSd & stashDataViewer = cxViewer.getStashData( );
	if ( stashDataViewer.getNameExecData( 0 ).matchWithCase(
	    (const char *) segName.address( ) ) )
	{
	    WhString strNameExecLinkNul( stashDataViewer.getNameExecLink( 0 ) );
	    strNameExecLinkNul.appChrRaw( '\0' );
	    strNameExecLinkNul.checkCcs( );
	    segName.smashData( (const MmByte *) strNameExecLinkNul.address( ),
		strNameExecLinkNul.count( ) );
	    segName.storeProc( procViewer );
	    cxViewer.setTrackState( stTrackExecer );
	}
    }
}



// Viewer, 'execve', sco, after wait.
void viewer_execve_sco_aw( CxStore & cxViewer, WhLap <CxStore> & )
{
    MmScr scrViewer( MmScr::tyScrInt );
    scrViewer.fetchProc( cxViewer.getProc( ) );
    if ( !scrViewer.isError( ) )
    {
	cxViewer.getBpt( ).clear( );
    }
}



// Viewer, 'fork', sci, before continue.
void viewer_fork_sci_bc( CxStore & cxViewer, WhLap <CxStore> & )
{
    // Get process.
    PrProc & procViewer = cxViewer.getProc( );

    // Fetch restoration segment.
    MmSeg * psegViewerCode = new MmSeg;
    if ( psegViewerCode == 0 )
	ErMem( );
    psegViewerCode->fetchProc( procViewer, tyMmText, tySegListInsn,
	MmAddr( procViewer.fetchRegPc( ) ), int( insnJumpDot[0] ) );
    if ( !psegViewerCode->hasArea( ) || !psegViewerCode->hasData( ) )
	ErFatal( "viewer_fork_sci_bc: failed fetch." );

    // Smash code at return address to capture free process.
    MmSeg segSmashCode( *psegViewerCode );
    segSmashCode.smashData( &insnJumpDot[1],
	int( insnJumpDot[0] ) * sizeof(MmInsn) );
    segSmashCode.storeProc( procViewer );

    // Save.
    WhList <MmWord> lwArgSci;
    WhLap  <MmSeg>  lsegSci;
    procViewer.fetchListReg( 0, 0, lwArgSci );
    lsegSci.appendAp( psegViewerCode );
    cxViewer.setSci( lwArgSci, lsegSci );
}



// Viewer, 'fork', sco, after wait.
void viewer_fork_sco_aw( CxStore & cxViewer, WhLap <CxStore> & lcxControl )
{
    // Get process.
    PrProc & procViewer = cxViewer.getProc( );

    // Restore insn.
    cxViewer.getSegSci( 0 ).storeProc( procViewer );

    // Capture child.
    MmScr scrFork( MmScr::tyScrInt );
    scrFork.fetchProc( procViewer );
    if ( !scrFork.isError( ) )
    {
	// Create process.
	PrProc * pprocFork = new PrProc( scrFork.getWord( 0 ), procViewer.getPid( ) );
	if ( pprocFork == 0 )
	    ErMem( );

	// Restore insn.
	cxViewer.getSegSci( 0 ).storeProc( *pprocFork );

	// Create stash.
	WhString strDirStash;
	strDirStash.appStrRaw( "/tmp" );
	strDirStash.appStrRaw( "/stash-" );
	strDirStash.appIntFmt( scrFork.getWord( 0 ) );
	PrSl * pstashLinkFork = new PrSl( strDirStash );
	if ( pstashLinkFork == 0 )
	    ErMem( );

	// Create log.
	EvLog * plogFork = new EvLog( cxViewer.getLog( ) );
	if ( plogFork == 0 )
	    ErMem( );
	plogFork->rewind( );

	// Create context.
	CxStore * pcxFork = new CxStore( stTrackViewer, stRunRunnable,
	    pprocFork, cxViewer.getStashData( ), pstashLinkFork, plogFork );
	if ( pcxFork == 0 )
	    ErMem( );

	// Append child to control list.
	lcxControl.appendAp( pcxFork );
    }
}



// Viewer, 'kill,' sci, before continue.
void viewer_kill_sci_bc( CxStore & cxViewer, WhLap <CxStore> & )
{
    // Get process.
    PrProc & procViewer = cxViewer.getProc( );

    // Save.
    WhList <MmWord> lwArgSci;
    WhLap  <MmSeg>  lsegSci;
    procViewer.fetchListReg( 0, 2, lwArgSci );
    cxViewer.setSci( lwArgSci, lsegSci );

    // Annul.
    procViewer.storeReg( 0, MmWord( ~0 ) );
    procViewer.storeReg( 1, MmWord( ~0 ) );
}



// Viewer, 'kill', sco, after wait.
void viewer_kill_sco_aw( CxStore & cxViewer, WhLap <CxStore> & lcxControl )
{
    // Get process.
    PrProc & procViewer = cxViewer.getProc( );

    // Get and restore arguments.
    const MmWord wpidTarget = cxViewer.getArgSci( 0 );
    const MmWord wisig      = cxViewer.getArgSci( 1 );
    procViewer.storeReg( 0, wpidTarget );
    procViewer.storeReg( 1, wisig      );

    // Get target process.
    CxStore * pcxTarget = 0;
    if ( int( wpidTarget ) > 0 )
    {
	for ( int icx = 0; icx < lcxControl.count( ); ++icx )
	{
	    CxStore & cxTry = lcxControl.refNonConst( icx );
	    if ( cxTry.getProc( ).getPid( ) == wpidTarget )
	    {
		pcxTarget = &cxTry;
		break;
	    }
	}
    }

    // Declare synthetic return value.
    MmScr scrViewer( MmScr::tyScrInt );

    // Limited emulation.
    if ( pcxTarget != 0 && wisig == SIGKILL )
    {
	// Proxy kill.
	pcxTarget->getProc( ).killProc( );
	pcxTarget->setTrackState( stTrackViewer );
	if ( pcxTarget->getRunState( ) != stRunExited )
	    pcxTarget->setRunState( stRunRunning );
	scrViewer.setValue( 0, 0 );
    }
    else
    {
	// Deny kill.
	::fprintf( stderr, "Deny kill( %d, %d ).\n",
	    int( wpidTarget ), int( wisig ) );
	scrViewer.setError( EACCES );
    }

    // Store into viewer.
    scrViewer.storeProc( procViewer );
}




// Viewer, 'ptrace', sci, before continue.
void viewer_ptrace_sci_bc( CxStore & cxViewer, WhLap <CxStore> & )
{
    // Get process.
    PrProc & procViewer = cxViewer.getProc( );

    // Save.
    WhList <MmWord> lwArgSci;
    WhLap  <MmSeg>  lsegSci;
    procViewer.fetchListReg( 0, 4, lwArgSci );
    cxViewer.setSci( lwArgSci, lsegSci );

    // Annul.
    procViewer.storeReg( 0, MmWord( ~0 ) );
}



// Viewer, 'ptrace', sco, after wait.
void viewer_ptrace_sco_aw( CxStore & cxViewer, WhLap <CxStore> & lcxControl )
{
    // Get process.
    PrProc & procViewer = cxViewer.getProc( );

    // Get and restore arguments.
    const MmWord wRequest    = cxViewer.getArgSci( 0 );
    const MmWord wpidTarget  = cxViewer.getArgSci( 1 );
    const MmWord waddrTarget = cxViewer.getArgSci( 2 );
    const MmAddr  addrTarget = MmAddr( waddrTarget );
    const MmWord wdataTarget = cxViewer.getArgSci( 3 );
    procViewer.storeReg( 0, wRequest );

    // Declare synthetic return value.
    MmScr scrViewer( MmScr::tyScrInt );

    // Viewer getting traced.
    if ( wRequest == PTRACE_TRACEME )
    {
	if ( procViewer.getPpid( ) == ::getpid( ) )
	    ErFatal( "viewer_ptrace_sco_aw: PTRACE_TRACEME." );
	if ( cxViewer.isTraceMe( ) )
	{
	    scrViewer.setError( EPERM );
	}
	else
	{
	    cxViewer.setTraceMe( true );
	    scrViewer.setValue( 0, 0 );
	}
	scrViewer.storeProc( procViewer );
	return;
    }

    // Get target process.
    CxStore * pcxTarget = 0;
    for ( int icx = 0; icx < lcxControl.count( ); ++icx )
    {
	CxStore & cxTry = lcxControl.refNonConst( icx );
	if ( cxTry.getProc( ).getPid( ) == wpidTarget )
	{
	    pcxTarget = &cxTry;
	    break;
	}
    }

    // Unimplemented attach.
    if ( wRequest == PTRACE_ATTACH )
	ErFatal( "viewer_ptrace_sco_aw: PTRACE_ATTACH not implemented." );

    // Unknown or untraced target.
    if ( pcxTarget == 0 || !pcxTarget->isTraceMe( ) )
    {
	scrViewer.setError( ESRCH );
	scrViewer.storeProc( procViewer );
	return;
    }

    // Get and verify process.
    PrProc & procTarget = pcxTarget->getProc( );
    if ( procTarget.getPpid( ) != procViewer.getPid( ) )
    {
	scrViewer.setError( ESRCH );
	scrViewer.storeProc( procViewer );
	return;
    }

    // Request on non-ready process.
    if ( wRequest != PTRACE_KILL && pcxTarget->getRunState( ) != stRunReady )
    {
	scrViewer.setError( ESRCH );
	scrViewer.storeProc( procViewer );
	return;
    }

    // Get target pid.
    const pid_t pidTarget = procTarget.getPid( );

    // Dispatch.
    bool fDeny = false;
    switch ( wRequest )
    {
    default:
	ErFatal( "viewer_ptrace_sco_aw: request not implemented." );

    case PTRACE_TRACEME:
	ErFatal( "viewer_ptrace_sco_aw: PTRACE_TRACEME." );

    case PTRACE_PEEKTEXT:
    case PTRACE_PEEKDATA:
    case PTRACE_PEEKUSR:
	{
	    MmWord wData;
	    if ( procViewer.fetchWord( tyMmData, MmAddr( wdataTarget ), wData ) )
	    {
		::errno = 0;
		::raw_ptrace( wRequest, pidTarget, waddrTarget, MmWord( &wData ) );
		const MmWord wErrno = ::errno;
		if ( wErrno == 0 )
		    scrViewer.setValue( 0, 0 );
		else
		    scrViewer.setError( wErrno );
		if ( !procViewer.storeWord( tyMmData, MmAddr( wdataTarget ), wData ) )
		    scrViewer.setError( EFAULT );
	    }
	    else
	    {
		scrViewer.setError( EFAULT );
	    }
	}
	break;

    case PTRACE_POKETEXT:
    case PTRACE_POKEDATA:
	// Linux 1.3.41: no distinction at OS level.
	// Gdb 4.13: always writes to D space, even for breakpoints!
	{
	    // Fetch target word.
	    MmWord wdataFetch;
	    if ( !procTarget.fetchWord( tyMmText, addrTarget, wdataFetch )
	    ||   !procTarget.storeWord( tyMmText, addrTarget, wdataFetch ) )
	    {
		scrViewer.setError( EFAULT );
	    }
	    else if ( pcxTarget->getTrackState( ) != stTrackReplay
	    ||        pcxTarget->getBpt( ).changeWord( addrTarget, wdataFetch, wdataTarget ) )
	    {
		::errno = 0;
		::raw_ptrace( wRequest, pidTarget, waddrTarget, wdataTarget );
		if ( ::errno != 0 )
		    ErFatal( "viewer_ptrace_sco_aw: failed ptrace." );
		scrViewer.setValue( 0, 0 );
	    }
	    else
	    {
		fDeny = true;
	    }
	}
	break;

    case PTRACE_POKEUSR:
	{
	    struct user user;
	    if ( pcxTarget->getTrackState( ) != stTrackReplay 
	    ||   waddrTarget == MmAddr( &user.u_debugreg[6] ) - MmAddr( &user )
	    ||   waddrTarget == EIP * 4 && MmAddr( wdataTarget ) == pcxTarget->getAddrCont( ) )
	    {
		::errno = 0;
		::raw_ptrace( wRequest, pidTarget, waddrTarget, wdataTarget );
		const MmWord wErrno = ::errno;
		if ( wErrno == 0 )
		    scrViewer.setValue( 0, 0 );
		else
		    scrViewer.setError( wErrno );
	    }
	    else
	    {
		fDeny = true;
	    }
	}
	break;

    case PTRACE_CONT:
    case PTRACE_SINGLESTEP:
    case PTRACE_SYSCALL:
	if ( wdataTarget == 0 )
	{
	    pcxTarget->setStep     ( wRequest == PTRACE_SINGLESTEP	);
	    pcxTarget->setTraceSc  ( wRequest == PTRACE_SYSCALL		);
	    pcxTarget->setRunState ( stRunRunnable			);
	    scrViewer.setValue( 0, 0 );
	}
	else
	{
	    fDeny = true;
	}
	break;

    case PTRACE_KILL:
	procTarget.ptraceKill( );
	pcxTarget->setTrackState( stTrackViewer );
	if ( pcxTarget->getRunState( ) != stRunExited )
	    pcxTarget->setRunState( stRunRunning );
	scrViewer.setValue( 0, 0 );
	break;

    case PTRACE_ATTACH:
	ErFatal( "viewer_ptrace_sco_aw: PTRACE_ATTACH not implemented." );
	break;

    case PTRACE_DETACH:
	ErFatal( "viewer_ptrace_sco_aw: PTRACE_DETACH not implemented." );
	break;
    }

    // Denial.
    if ( fDeny )
    {
	::fprintf( stderr, "Deny ptrace( %u, %u, %p, 0x%X ).\n",
	    wRequest, wpidTarget, addrTarget, wdataTarget );
	scrViewer.setError( EACCES );
    }

    // Store synthetic return value.
    scrViewer.storeProc( procViewer );
}



// Viewer, 'wait4', sci, before continue.
void viewer_wait4_sci_bc( CxStore & cxViewer, WhLap <CxStore> & )
{
    // Get process.
    PrProc & procViewer = cxViewer.getProc( );

    // Save.
    WhList <MmWord> lwArgSci;
    WhLap  <MmSeg>  lsegSci;
    procViewer.fetchListReg( 0, 4, lwArgSci );
    cxViewer.setSci( lwArgSci, lsegSci );

    // Annul.
    procViewer.storeReg( 0, MmWord( ~0 ) );
    procViewer.storeReg( 1, MmWord( ~0 ) );
}



// Viewer, 'wait4', sco, after wait.
void viewer_wait4_sco_aw( CxStore & cxViewer, WhLap <CxStore> & )
{
    cxViewer.setRunState( stRunWaiting );
}



// Viewer, 'waitpid', sci, before continue.
void viewer_waitpid_sci_bc( CxStore & cxViewer, WhLap <CxStore> & )
{
    // Get process.
    PrProc & procViewer = cxViewer.getProc( );

    // Save.
    WhList <MmWord> lwArgSci;
    WhLap  <MmSeg>  lsegSci;
    procViewer.fetchListReg( 0, 3, lwArgSci );
    cxViewer.setSci( lwArgSci, lsegSci );

    // Annul.
    procViewer.storeReg( 0, MmWord( ~0 ) );
    procViewer.storeReg( 1, MmWord( ~0 ) );
}



// Viewer, 'waitpid', sco, after wait.
void viewer_waitpid_sco_aw( CxStore & cxViewer, WhLap <CxStore> & )
{
    cxViewer.setRunState( stRunWaiting );
}
