// Copyright 1994, 1995 Michael Chastain
// Licensed under the Gnu Public License, Version 2
//
// File: WhString.cc
//   String class.
//   Strings may contain nulls and are *not* null-terminated.
//   This is a concrete class.
//
// File Created:	18 Jan 1994		Michael Chastain
// Last Reviewed:	12 Sep 1994		Michael Chastain
// Last Edited:		02 Jul 1995		Michael Chastain

#include <WhAbort.h>
#include <WhString.h>



// Used to format digits.
static const char lcDigitFmt [] = "0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZ$";



// Constructor.
WhString::WhString( )
    : lcData_	( )
{
    ;
}



// Constructor.
WhString::WhString( const char * pstrOld )
    : lcData_	( )
{
    if ( pstrOld == 0 )
	WhAbort( "WhString::WhString: zero pointer." );

    for ( int istrOld = 0; pstrOld[istrOld] != '\0'; ++istrOld )
	lcData_.append( pstrOld[istrOld] );
}



// Constructor.
WhString::WhString( const char * pstrOld, int sstrOldMax )
    : lcData_	( )
{
    if ( pstrOld == 0 )
	WhAbort( "WhString::WhString: zero pointer." );

    for ( int istrOld  = 0;
	      istrOld  < sstrOldMax && pstrOld[istrOld] != '\0';
	      istrOld += sizeof(char) )
    {
	lcData_.append( pstrOld[istrOld] );
    }
}



// Return an address for a classic string.
//   Strings may contain nulls and are *not* null-terminated.
const char * WhString::address( ) const
{
    return lcData_.address( );
}



// Check whether a string is representable as 'const char *'.
// This means: null terminated and no interior nulls.
bool WhString::checkCcs( ) const
{
    if ( this->count( ) == 0 || (*this)[this->count( ) - 1] != '\0' )
	return false;

    for ( int iChar = 0; iChar < this->count( ) - 1; ++iChar )
    {
	if ( (*this)[iChar] == '\0' )
	    return false;
    }

    return true;
}



// Return whether two strings match, with case.
bool WhString::matchWithCase( const WhString & strMatch ) const
{
    if ( this->count( ) != strMatch.count( ) )
	return false;

    for ( int iChar = 0; iChar < this->count( ); ++iChar )
    {
	if ( (*this)[iChar] != strMatch[iChar] )
	    return false;
    }

    return true;
}



// Replace one character with another.
void WhString::replaceChr( char cLeft, char cRight )
{
    WhList <char> lcDataNew;
    lcDataNew.clear( this->count( ) );

    for ( int iChar = 0; iChar < this->count( ); ++iChar )
    {
	const char cOld = (*this)[iChar];
	lcDataNew.append( cOld == cLeft ? cRight : cOld );
    }

    lcData_ = lcDataNew;
}



// Append a string, unformatted.
//   Beware aliasing.
void WhString::appStrRaw( const char * pstrAppend )
{
    if ( pstrAppend == 0 )
	WhAbort( "WhString::appStrRaw: zero pointer." );

    if ( lcData_.isInRange( pstrAppend ) )
    {
	WhString strAppendCopy( pstrAppend );
	appStrRaw( strAppendCopy );
    }
    else
    {
	for ( int istrAppend          = 0;
	      pstrAppend[istrAppend] != '\0';
	      istrAppend             += 1 )
	{
	    lcData_.append( pstrAppend[istrAppend] );
	}
    }
}



// Append a string, unformatted.
//   Beware aliasing.
void WhString::appStrRaw( const WhString & strAppend )
{
    lcData_.append( strAppend.lcData_ );
}




// Append one character, formatted.
//   Inlined helper function.
inline void WhString::appChrFmtPriv( char cValue )
{
    if      ( cValue >= 0x20 && cValue <= 0x7E )
    {
	lcData_.append( cValue );
    }
    else if ( cValue == '\n' )
    {
	lcData_.append( '\\' );
	lcData_.append( 'n'  );
    }
    else if ( cValue == '\r' )
    {
	lcData_.append( '\\' );
	lcData_.append( 'r'  );
    }
    else if ( cValue == '\t' )
    {
	lcData_.append( '\\' );
	lcData_.append( 't'  );
    }
    else if ( cValue == '\0' )
    {
	lcData_.append( '\\' );
	lcData_.append( '0'  );
    }
    else
    {
	lcData_.append( '\\' );
	lcData_.append( 'x'  );
	lcData_.append( lcDigitFmt[((unsigned char) cValue) / 16] );
	lcData_.append( lcDigitFmt[((unsigned char) cValue) % 16] );
    }
}



// Append one character, formatted.
void WhString::appChrFmt( char cValue )
{
    appChrFmtPriv( cValue );
}



// Append an integer, formatted.
//   This code is tense.
void WhString::appIntFmt(
    unsigned int iValue, unsigned int iBase, unsigned int nDigit )
{
    // Validate 'iBase'.
    if ( iBase < 2 || iBase > 36 )
	WhAbort( "WhString::appIntFmt: bad base." );

    // Validate 'nDigit'.
    if ( int( nDigit ) < 0 )
	WhAbort( "WhString::appIntFmt: bad # digits." );

    // Generated digits.
    //   Use a fixed array for speed.
    //   If fixed array overflows, cutover to list.
    //   Note that the array is forward but the list is backward!
    const int		nDigitFwdMax	= 64;	// Arbitrary
    char		acDigitFwd	[nDigitFwdMax];
    int			iDigitFwd	= nDigitFwdMax;
    WhList <char> *	plcDigitRev	= 0;

    // Sign flags.
    bool		fSign		= iBase == 10;
    bool		fOverflow	= false;

    // Handle signed output.
    if ( fSign && int( iValue ) < 0 )
    {
	// Output sign.
	lcData_.append( '-' );

	if ( 0 - int( iValue ) < 0 )
	{
	    // This is the 2's complement most negative integer.
	    //   E.g. 0x80000000 on 32-bit processor.
	    fOverflow = true;

	    // Have to bias the positive value to avoid overflow.
	    iValue = -1 - int( iValue );

	    // Always use list for this hard case.
	    plcDigitRev = new WhList <char>;
	    if ( plcDigitRev == 0 )
		WhAbort( "WhString::appIntFmt: out of memory." );
	}
	else
	{
	    // Just convert to positive.
	    iValue = 0 - int( iValue );
	}
    }

    // Convert to list of digits.
    while ( iValue > 0 )
    {
	// Cutover when fixed array overflows.
	if ( plcDigitRev == 0 && iDigitFwd == 0 )
	{
	    plcDigitRev = new WhList <char>;
	    if ( plcDigitRev == 0 )
		WhAbort( "WhString::appIntFmt: out of memory." );
	    for ( int iDigit = nDigitFwdMax - 1; iDigit >= 0; --iDigit )
		plcDigitRev->append( acDigitFwd[iDigit] );
	}

	char cDigit = lcDigitFmt[iValue % iBase];
	iValue /= iBase;

	if ( plcDigitRev == 0 )
	    acDigitFwd[--iDigitFwd] = cDigit;
	else
	    plcDigitRev->append( cDigit );
    }

    // Handle most-negative-integer overflow.
    //   It was offset by 1, and looks like "7463847412" right now.
    //   Add 1 to its magnitude in analog fashion.
    if ( fOverflow )
    {
	// Create a new list to hold addition result.
	WhList <char>	lcDigitRevNew;
	int		iCarry		= 1;

	for ( int iDigit = 0; iDigit < plcDigitRev->count( ); ++iDigit )
	{
	    char cDigit = char( int( (*plcDigitRev)[iDigit] ) + iCarry );
	    if ( cDigit == lcDigitFmt[iBase] )
		cDigit = lcDigitFmt[0];
	    else
		iCarry = 0;
	    lcDigitRevNew.append( cDigit );
	}

	if ( iCarry > 0 )
	    lcDigitRevNew.append( lcDigitFmt[1] );

	*plcDigitRev = lcDigitRevNew;
    }

    if ( plcDigitRev == 0 )
    {
	// Generate leading pad.
	for ( int iPad = nDigitFwdMax - iDigitFwd; iPad < nDigit; ++iPad )
	    lcData_.append( lcDigitFmt[0] );

	// Generate main body.
	lcData_.append( &acDigitFwd[iDigitFwd], nDigitFwdMax - iDigitFwd );
    }
    else
    {
	// Generate leading pad.
	for ( int iPad = plcDigitRev->count( ); iPad < nDigit; ++iPad )
	    lcData_.append( lcDigitFmt[0] );

	// Reverse main body.
	for ( int iDigit = plcDigitRev->count( ) - 1; iDigit >= 0; --iDigit )
	    lcData_.append( (*plcDigitRev)[iDigit] );

	// Delete the dynamic list.
	delete plcDigitRev;
	plcDigitRev = 0;
    }
}



// Append a pointer, formatted.
void WhString::appPtrFmt( const void * pValue )
{
    if ( pValue == 0 )
    {
	lcData_.append( '0' );
    }
    else
    {
	const unsigned int uValue = (unsigned int) pValue;

	lcData_.append( '0' );
	lcData_.append( 'x' );
	lcData_.append( lcDigitFmt[( uValue & (0xF << (7 * 4)) ) >> (7 * 4)] );
	lcData_.append( lcDigitFmt[( uValue & (0xF << (6 * 4)) ) >> (6 * 4)] );
	lcData_.append( lcDigitFmt[( uValue & (0xF << (5 * 4)) ) >> (5 * 4)] );
	lcData_.append( lcDigitFmt[( uValue & (0xF << (4 * 4)) ) >> (4 * 4)] );
	lcData_.append( lcDigitFmt[( uValue & (0xF << (3 * 4)) ) >> (3 * 4)] );
	lcData_.append( lcDigitFmt[( uValue & (0xF << (2 * 4)) ) >> (2 * 4)] );
	lcData_.append( lcDigitFmt[( uValue & (0xF << (1 * 4)) ) >> (1 * 4)] );
	lcData_.append( lcDigitFmt[( uValue & (0xF << (0 * 4)) ) >> (0 * 4)] );
    }
}



// Append a string, formatted.
//   Beware aliasing.
void WhString::appStrFmt( const char * pstrValue )
{
    if ( pstrValue == 0 )
	WhAbort( "WhString::appStrFmt: zero pointer." );

    if ( lcData_.isInRange( pstrValue ) )
    {
	WhString strValueCopy( pstrValue );
	appStrFmt( strValueCopy );
    }
    else
    {
	for ( int icValue = 0; pstrValue[icValue] != '\0'; ++icValue )
	    appChrFmtPriv( pstrValue[icValue] );
    }
}



// Append a string, formatted.
//   Beware aliasing.
void WhString::appStrFmt( const char * pstrValue, int sstrValueMax )
{
    if ( pstrValue == 0 )
	WhAbort( "WhString::appStrFormat: zero pointer." );

    if ( lcData_.isInRange( pstrValue ) )
    {
	WhString strValueCopy( pstrValue, sstrValueMax );
	appStrFmt( strValueCopy );
    }
    else
    {
	for ( int icValue  = 0;
		  icValue  < sstrValueMax && pstrValue[icValue] != '\0';
		  icValue += sizeof(char) )
	{
	    appChrFmtPriv( pstrValue[icValue] );
	}
    }
}



// Append a string, formatted.
//  Beware aliasing.
void WhString::appStrFmt( const WhString & strValue )
{
    if ( this == &strValue )
    {
	WhString strValueCopy( strValue );
	appStrFmt( strValueCopy );
    }
    else
    {
	for ( int icValue  = 0;
		  icValue  < strValue.count( );
		  icValue += sizeof(char) )
	    appChrFmtPriv( strValue[icValue] );
    }
}
