// Copyright 1994, 1995 Michael Chastain
// Licensed under the Gnu Public License, Version 2
//
// File: WhList.cc
//   List template.
//   This is a template.
//
// File Created:	23 Jan 1994		Michael Chastain
// Last Reviewed:	24 Apr 1995		Michael Chastain
// Last Edited:		05 May 1995		Michael Chastain

#include <WhAbort.h>
#include <WhList.h>



// Constructor.
template <class T> WhList<T>::WhList( )
    : nData_	(0)
    , nDataMax_	(0)
    , pData_	(0)
{
    ;
}



// Copy constructor.
template <class T> WhList<T>::WhList( const WhList <T> & lOld )
    : nData_	(0)
    , nDataMax_	(0)
    , pData_	(0)
{
    *this = lOld;
}



// Assignment operator.
template <class T> const WhList <T> & WhList<T>::operator = (
    const WhList <T> & lOld )
{
    if ( this != &lOld )
    {
	clear( lOld.nData_ );
	for ( int iDataOld = 0; iDataOld < lOld.nData_; ++iDataOld )
	    pData_[iDataOld] = lOld.pData_[iDataOld];
	nData_ = lOld.nData_;
    }
    return *this;
}



// Destructor.
template <class T> WhList<T>::~WhList( )
{
    if ( pData_ != 0 )
    {
	delete [] pData_;
	pData_ = 0;
    }
}



// Append.
template <class T> void WhList<T>::append( const T & dataNew )
{
    // Expand if necessary.
    if ( nData_ >= nDataMax_ )
    {
	// Anti-aliasing if source is moving.
	if ( isInRange( &dataNew ) )
	{
	    T dataNewCopy( dataNew );
	    append( dataNewCopy );
	    return;
	}
	expand( 1 );
    }

    // Copy in the data.
    pData_[nData_] = dataNew;
    ++nData_;
}



// Append.
template <class T> void WhList<T>::append( const T * pDataNew, int nDataNew )
{
    // Validate append count.
    if ( nDataNew < 0 )
	WhAbort( "WhList<T>::append: negative count." );

    // Validate pointer.
    if ( nDataNew > 0 && pDataNew == 0 )
	WhAbort( "WhList<T>::append: zero pointer." );

    // Validate bounded length.
    if ( nData_ + nDataNew < nData_ )
	WhAbort( "WhList<T>::append: count overflow." );

    // Expand if needed.
    if ( nData_ + nDataNew > nDataMax_ )
    {
	// Anti-aliasing if source is moving.
	if ( isInRange( pDataNew ) )
	{
	    WhList <T> lDataNewCopy;
	    lDataNewCopy.append( pDataNew, nDataNew );
	    append( lDataNewCopy );
	    return;
	}

	// Expand.
	expand( nDataNew );
    }

    // Copy in the data.
    for ( int iDataNew = 0; iDataNew < nDataNew; ++iDataNew )
	pData_[nData_ + iDataNew] = pDataNew[iDataNew];

    // Record new count.
    nData_ += nDataNew;
}



// Append.
template <class T> void WhList<T>::append( const WhList<T> & lDataNew )
{
    append( lDataNew.address( ), lDataNew.count( ) );
}



// Clear, with optional new maximum size.
//   The list can always expand so 0 is always cool.
//   A non-0 value is a hint for performance.
template <class T> void WhList<T>::clear( int nDataMaxNew )
{
    // Verify count.
    if ( nDataMaxNew < 0 )
	WhAbort( "WhList<T>::clear: negative count." );

    // Reset index.
    nData_ = 0;

    // Expand if needed.
    if ( nDataMaxNew > nDataMax_ )
    {
	// Allocate new array.
	T * pDataNew = new T [nDataMaxNew];
	if ( pDataNew == 0 )
	    WhAbort( "WhList<T>::clear: out of memory." );

	// Substitute in new array.
	if ( pData_ != 0 )
	    delete [] pData_;
	pData_    = pDataNew;
	nDataMax_ = nDataMaxNew;
    }
}



// Remove.
template <class T> void WhList<T>::removeOne( int iData )
{
    // Check index.
    if ( iData < 0 || iData >= nData_ )
	rangeError( iData );

    // Copy tail down.
    for ( int iDataNew = iData; iDataNew < nData_ - 1; ++iDataNew )
	pData_[iDataNew] = pData_[iDataNew+1];

    // Remove tail.
    --nData_;
}



// Expand.
//   Can either give exact size requested or round up.
template <class T> void WhList<T>::expand( int nDataNew )
{
    // Verify count.
    if ( nDataNew < 0 )
	WhAbort( "WhList<T>::expand: negative count." );

    // Compute new maximum and verify it.
    int nDataMaxNew = nData_ + nDataNew;
    if ( nDataMaxNew < nData_ )
	WhAbort( "WhList<T>::expand: count overflow." );

    // Expand if needed.
    if ( nDataMaxNew > nDataMax_ )
    {
	// Heuristic sizes for first allocation.
	const int nDataMaxFirst = sizeof(T) <= sizeof(char) ? 240 :
				  sizeof(T) <= sizeof(int)  ?  16 :
							        4 ;

	// Round up if not exact.
	if ( nDataMaxNew < nDataMaxFirst ) nDataMaxNew = nDataMaxFirst;
	if ( nDataMaxNew < 2 * nDataMax_ ) nDataMaxNew = 2 * nDataMax_;

	// Allocate new array.
	T * pDataNew = new T [nDataMaxNew];
	if ( pDataNew == 0 )
	    WhAbort( "WhList<T>::expand: out of memory." );

	// Copy from old to new.
	for ( int iData = 0; iData < nData_; ++iData )
	    pDataNew[iData] = pData_[iData];

	// Substitute in new array.
	if ( pData_ != 0 )
	    delete [] pData_;
	pData_    = pDataNew;
	nDataMax_ = nDataMaxNew;
    }
}



// Report a range error.
template <class T> void WhList<T>::rangeError( int ) const
{
    WhAbort( "WhList<T>::rangeError: index out of range." );
}
