// Copyright 1995 Michael Chastain
// Licensed under the Gnu Public License, Version 2
//
// File: WhFile.cc
//   File class.
//   This is an external resource handler class.
//
// File Created:	01 Mar 1995		Michael Chastain
// Last Edited:		02 Jul 1995		Michael Chastain

#include <sys/types.h>
#include <sys/stat.h>
#include <errno.h>
#include <fcntl.h>
#include <unistd.h>

#include <WhAbort.h>
#include <WhFile.h>
#include <WhList.h>
#include <WhString.h>



// Constructor.
WhFile::WhFile( )
    : strName_		( 	)
    , fOpen_		( false	)
    , fOutput_		( false	)
    , errnoLast_	( 0	)
    , fdFile_		( -1	)
    , icFile_		( 0	)
    , ncFile_		( 0	)
{
    ;
}



// Destructor.
WhFile::~WhFile( )
{
    if ( fOpen_ )
    {
	if ( ::close( fdFile_ ) != 0 )
	    WhAbort( "WhFile::~WhFile: failed close." );

	if ( fOutput_ )
	{
	    WhString strNameNul( strName_ );
	    strNameNul.appChrRaw( '\0' );
	    if ( ::unlink( strNameNul.address( ) ) != 0 )
		WhAbort( "WhFile::~WhFile: failed unlink." );
	}
    }
}    



// Open a file by name.
MmRet WhFile::openName( const WhString & strName, TyOpen tyOpen )
{
    // Validate state.
    if ( fOpen_ )
	return MmRetRaise( mmRetFileIsOpen );

    // Validate name.
    WhString strNameNul( strName );
    strNameNul.appChrRaw( '\0' );
    if ( !strNameNul.checkCcs( ) )
	return MmRetRaise( mmRetFileNameNotCcs );

    // Translate the mode.
    int    wFlags = 0;
    mode_t wMode  = 0;
    switch ( tyOpen )
    {
    default:
	WhAbort( "WhFile::openName: bad enum." );
    case tyOpenRead:
	wFlags = O_RDONLY;
	wMode  = 0644;
	break;
    case tyOpenWriteData:
	wFlags = O_RDWR|O_TRUNC|O_CREAT;
	wMode  = 0644;
	break;
    case tyOpenWriteExe:
	wFlags = O_RDWR|O_TRUNC|O_CREAT|O_EXCL;
	wMode  = 0755;
	break;
    }

    // Open the file.
    ::errno = 0;
    const int fdNew = ::open( strNameNul.address( ), wFlags, wMode );
    errnoLast_ = ::errno;
    if ( fdNew == -1 )
	return MmRetRaise( mmRetFileWontOpen );

    // Get its size.
    struct stat statBuf;
    ::errno = 0;
    const int iStatus = ::fstat( fdNew, &statBuf );
    errnoLast_ = ::errno;
    if ( iStatus == -1 )
    {
	::close( fdNew );
	return MmRetRaise( mmRetFileWontFstat );
    }

    // Success, save state.
    strName_ = strName;
    fOpen_   = true;
    fOutput_ = (tyOpen != tyOpenRead);
    fdFile_  = fdNew;
    icFile_  = 0;
    ncFile_  = statBuf.st_size;
    return mmRetOk;
}



// Unlink a file by name.
MmRet WhFile::unlinkName( const WhString & strName )
{
    // Validate state.
    if ( fOpen_ )
	return MmRetRaise( mmRetFileIsOpen );

    // Validate name.
    WhString strNameNul( strName );
    strNameNul.appChrRaw( '\0' );
    if ( !strNameNul.checkCcs( ) )
	return MmRetRaise( mmRetFileNameNotCcs );

    // Unlink the file.
    ::errno = 0;
    const int iStatus = ::unlink( strNameNul.address( ) );
    errnoLast_ = ::errno;
    if ( iStatus == -1 )
	return MmRetRaise( mmRetFileWontUnlink );

    // That's all, folks.
    strName_ = strName;
    return mmRetOk;
}



// Close a file.
MmRet WhFile::closeFile( )
{
    // Validate state.
    if ( !fOpen_ )
	return MmRetRaise( mmRetFileNotOpen );

    // Close the file.
    ::errno = 0;
    const int iStatus = ::close( fdFile_ );
    errnoLast_ = ::errno;
    if ( iStatus == -1 )
	return MmRetRaise( mmRetFileWontClose );

    // Success, save state.
    fOpen_ = false;
    return mmRetOk;
}



// Read a list of characters.
MmRet WhFile::readListChar( WhList <char> & lcRead )
{
    // Validate state.
    if ( !fOpen_ )
	return MmRetRaise( mmRetFileNotOpen );
    if ( fOutput_ )
	return MmRetRaise( mmRetFileIsOutput );

    // Count chars to read.
    const int ncReadTotal = ncFile_ - icFile_;
    if ( ncReadTotal < 0 )
	return MmRetRaise( mmRetFileCountNegative );

    // Size the return list.
    lcRead.clear( ncReadTotal );

    // Read all chars.
    for ( int icReadTotal = 0; icReadTotal < ncReadTotal; )
    {
	// Read some chars.
	char rgcReadBuf [4096];

	::errno = 0;
	const int ncReadSome = ::read( fdFile_, rgcReadBuf,
	    ncReadTotal - icReadTotal < sizeof(rgcReadBuf)
		? ncReadTotal - icReadTotal : sizeof(rgcReadBuf) );
	errnoLast_ = ::errno;

	// Handle error.
	if ( ncReadSome  < 0 )
	    return MmRetRaise( mmRetFileWontRead );

	// Handle EOF.
	if ( ncReadSome == 0 )
	    return MmRetRaise( mmRetFileEofPremature );

	// Copy into return list.
	lcRead.append( rgcReadBuf, ncReadSome );

	// Advance.
	icFile_       += ncReadSome;
	icReadTotal   += ncReadSome;
    }

    // That's all, folks.
    return mmRetOk;
}



// Write a list of chars.
MmRet WhFile::writeListChar( const WhList <char> & lcWrite )
{
    // Validate state.
    if ( !fOpen_ )
	return MmRetRaise( mmRetFileNotOpen );
    if ( !fOutput_ )
	return MmRetRaise( mmRetFileNotOutput );

    // Write all chars.
    for ( int icWriteTotal = 0; icWriteTotal < lcWrite.count( ); )
    {
	// Write some chars.
	::errno = 0;
	int ncWriteSome = ::write( fdFile_,
	    lcWrite.address( ) + icWriteTotal,
	    lcWrite.count( ) - icWriteTotal );
	errnoLast_ = ::errno;
	if ( ncWriteSome  < 0 )
	    return MmRetRaise( mmRetFileWontWrite );
	if ( ncWriteSome == 0 )
	    return MmRetRaise( mmRetFileEofPremature );

	// Advance.
	icFile_      += ncWriteSome;
	icWriteTotal += ncWriteSome;
    }

    // That's all, folks.
    return mmRetOk;
}
