// Copyright 1995 Michael Chastain
// Licensed under the Gnu Public License, Version 2
//
// File: ScLine.cc
//   System call line.
//   This is a concrete class.
//
// These are the methods.
//
// File Created:	10 Apr 1995		Michael Chastain
// Last Edited:		22 Sep 1995		Michael Chastain

#include <EvBase.h>
#include <EvSci.h>
#include <EvSco.h>
#include <MmSeg.h>
#include <MmType.h>
#include <PrProc.h>
#include <ScLine.h>
#include <WhAbort.h>
#include <WhString.h>

#if defined(MEC_TARGET_LIX)
#include <sys/mman.h>
#include <TySegEnumLix.h>
#endif



// Format to string.
void ScLine::fmtStrEvSci( WhString & strRet, const EvSci & evSciFmt ) const
{
    // Format name.
    strRet.appStrRaw( pstrName_ );
    if ( !fKnown_ )
    {
	strRet.appStrRaw( "::"				);
	strRet.appIntFmt( evSciFmt.getSysEntry( )	);
    }
    strRet.appStrRaw( " (" );

    // Format arguments.
    if ( pstrTyArg_ == 0 )
	WhAbort( "ScLine::fmtStrEvSci: zero pointer." );
    int nArgFmt = 0;
    for ( int iArg = 0; pstrTyArg_[iArg] != '\0'; ++iArg )
    {
	// Get formatting char.
	const char ctyArg = pstrTyArg_[iArg];
	if ( ctyArg == '-' )
	    continue;

	// Comma for args past first arg.
	if ( nArgFmt > 0 )
	    strRet.appStrRaw( ", " );
	++nArgFmt;

	if ( !evSciFmt.hasArg( ) )
	{
	    // Arg not available.
	    strRet.appStrRaw( "...." );
	}
	else
	{
	    // Discriminate on type.
	    switch ( ctyArg )
	    {
	    default:
		WhAbort( "ScLine::fmtStrEvSci: bad char." );

	    case 'D':
		strRet.appIntFmt( evSciFmt.getArg( iArg ) );
		break;

	    case 'O':
		strRet.appChrFmt( '0' );
		strRet.appIntFmt( evSciFmt.getArg( iArg ), 8, 0 );
		break;

	    case 'P':
		strRet.appPtrFmt( MmAddr( evSciFmt.getArg( iArg ) ) );
		break;

	    case 'X':
		strRet.appStrFmt( "0x" );
		strRet.appIntFmt( evSciFmt.getArg( iArg ), 16, 8 );
		break;
	    }
	}
    }

    // Format end of arguments.
    if ( nArgFmt == 0 )
	strRet.appChrRaw( ' ' );
    strRet.appChrRaw( ')' );

    // Format segments and flats.
    if ( evSciFmt.getSeg( ).count( ) > 0 || evSciFmt.getFlat( ).count( ) > 0 )
    {
	strRet.appChrRaw( '\n' );
	evSciFmt.fmtAllSeg  ( strRet );
	evSciFmt.fmtAllFlat ( strRet );
	strRet.appStrRaw( "       " );
    }

    // Format end.
    if ( tyScr_ == PrScr::tyScrNot )
	strRet.appStrRaw( ".\n" );
    else
	strRet.appStrRaw( " = " );
}



// Fetch arguments.
MmRet ScLine::fetchArg( const PrProc & procFetch, bool & fArgRet,
    WhList <MmWord> & lwArgRet ) const
{
    // Count arguments.
    if ( pstrTyArg_ == 0 )
	WhAbort( "ScLine::fetchArg: zero pointer." );
    int nArg = 0;
    while ( pstrTyArg_[nArg] != '\0' )
	++nArg;

    // Clear return values.
    fArgRet = false;
    lwArgRet.clear( nArg );

    // Fetch arguments.
    switch ( tyArg_ )
    {
    default:
	WhAbort( "ScLine::fetchArg: bad enum." );

    case tyArgMem:
	{
	    MmWord wAddrArg;
	    if ( !procFetch.fetchReg( iArgArg_, wAddrArg ) )
		WhAbort( "ScLine::fetchArg: failed fetch." );
	    for ( int iArg = 0; iArg < nArg; ++iArg )
	    {
		MmWord wArg;
		if ( procFetch.fetchWord( MmArea::tyAreaData,
		    MmAddr( wAddrArg ) + iArg * sizeof(MmWord), wArg, 0 ) )
		{
		    lwArgRet.append( wArg );
		}
	    }
	    if ( lwArgRet.count( ) == nArg )
		fArgRet = true;
	}
	break;

    case tyArgReg:
	if ( !procFetch.fetchListReg( iArgArg_, nArg, lwArgRet ) )
	    WhAbort( "ScLine::fetchArg: failed fetch." );
	fArgRet = true;
	break;
    }

    return mmRetOk;
}



// Fetch registers which will be smashed on replay.
MmRet ScLine::fetchRegSmash( const PrProc & procFetch,
    WhList <MmWord> & lwSmashRet ) const
{
    int nwSmash = 0;
    switch ( tySmash_ )
    {
    default:		WhAbort( "ScLine::fetchRegSmash: bad enum." );
    case tySmashCant:	nwSmash = 0; break;
    case tySmashDef:	nwSmash = 1; break;
    case tySmashFile0:	nwSmash = 0; break;
    case tySmashMmap:	nwSmash = 0; break;
    case tySmashMpi0:	nwSmash = 1; break;
    case tySmashNop:	nwSmash = 0; break;
    case tySmashPair:	nwSmash = 2; break;
    case tySmashReplay:	nwSmash = 0; break;
    case tySmashZero:	nwSmash = 1; break;
    }

    if ( !procFetch.fetchListReg( 0, nwSmash, lwSmashRet ) )
	WhAbort( "ScLine::fetchRegSmash: failed fetch." );

    return mmRetOk;
}



// Fetch sys call return value.
MmRet ScLine::fetchScr( const PrProc & procFetch, PrScr & scrRet ) const
{
    scrRet.fetchProc( procFetch, tyScr_ );
    return mmRetOk;
}



// Fetch segments.
MmRet ScLine::fetchSeg( bool fIngoing, const PrProc & procFetch,
    const MmMap & mapFetch, const EvSci & evSciArg, const EvSco & evScoScr,
    EvBase & evTarget ) const
{
    // Mark unsmashable as 'unknown'.
    if ( tySmash_ == tySmashCant )
	evTarget.setUnknown( true );

    // Need arguments to fetch segments.
    if ( evSciArg.hasArg( ) )
    {
	// Fetch ordinary segments.
	for ( int imsLine = 0; imsLine < nmsLine_; ++imsLine )
	{
	    if ( plmsLine_ == 0 )
		WhAbort( "ScLine::fetchSeg: zero pointer." );
	    MmRetCheck( plmsLine_[imsLine].fetchSeg( fIngoing, procFetch,
		mapFetch, evSciArg, evScoScr, evTarget ) );
	}

	// Fetch special segments.
	const FnFetch * pfnFetch = fIngoing ? pfnFetchIn_ : pfnFetchOut_;
	if ( pfnFetch != 0 )
	{
	    MmRetCheck( (*pfnFetch)( procFetch, mapFetch, evSciArg, evScoScr,
		evTarget ) );
	}
    }

    // That's all, folks.
    return mmRetOk;
}



// Smash registers to annual a system call.
MmRet ScLine::smash( PrProc & procSmash, const EvSci & evSciArg )
{
    switch ( tySmash_ )
    {
    default:
	WhAbort( "ScLine::smash: bad enum." );

    case tySmashCant:
	break;

    case tySmashDef:
	if ( !procSmash.storeReg( 0, MmWord( ~0 ) ) )
	    WhAbort( "ScLine::smash: failed store." );
	break;

    case tySmashFile0:
	{
	    // Smash current name.
	    //   It might have different address on replay (initial execve).
	    MmWord w0;
	    if ( !procSmash.fetchReg( 0, w0 ) )
		WhAbort( "ScLine::smash: failed fetch." );
	    MmSeg segSmashName;
	    MmRetCheck( segSmashName.fetchProc( procSmash, MmArea::tyAreaData,
		tySegStrNul, MmAddr( w0 ), 1 ) );
	    segSmashName.replaceByte( MmByte( '/' ), MmByte( '%' ) );
	    MmRetCheck( segSmashName.storeProc( procSmash ) );
	}
	break;

    case tySmashMmap:
#if defined(MEC_TARGET_LIX)
	{
	    // Smash an argument word.
	    MmWord wAddrArg;
	    if ( !procSmash.fetchReg( 0, wAddrArg ) )
		WhAbort( "ScLine::smash: failed fetch." );
	    if ( !procSmash.storeWord( MmArea::tyAreaData,
		MmAddr( wAddrArg ) + 3 * sizeof(MmWord),
		(evSciArg.getArg( 3 ) & ~MAP_TYPE)|MAP_PRIVATE|MAP_ANONYMOUS,
		0 ) )
	    {
		WhAbort( "ScLine::smash: failed store." );
	    }
	}
#endif

#if defined(MEC_TARGET_SUN)
    	WhAbort( "ScLine::smash: unimplemented enum." );
#endif
	break;

    case tySmashMpi0:
	if ( !procSmash.storeReg( 0, MmWord( ~0 ) >> 1 ) )
	    WhAbort( "ScLine::smash: failed store." );
	break;

    case tySmashNop:
    	break;

    case tySmashPair:
    	if ( !procSmash.storeReg( 0, MmWord( ~0 ) )
    	||   !procSmash.storeReg( 1, MmWord( ~0 ) ) )
	    WhAbort( "ScLine::smash: failed store." );
	break;

    case tySmashReplay:
    	break;

    case tySmashZero:
    	if ( !procSmash.storeReg( 0, 0 ) )
	    WhAbort( "ScLine::smash: failed store." );
	break;
    }

    // That's all, folks.
    return mmRetOk;
}
