// Copyright 1994, 1995 Michael Chastain
// Licensed under the Gnu Public License, Version 2
//
// File: ReplayMain.cc
//   Main program for mec-replay.
//
// File Created:	21 Jun 1994		Michael Chastain
// Last Edited:		14 Sep 1995		Michael Chastain

// Program version.
static const char strVersion [] = "0.1";



// Unix library.
#include <linux/unistd.h>
#include <stdio.h>
#include <stdlib.h>

// My library classes.
#include <EvHistory.h>
#include <EvSci.h>
#include <MmFlat.h>
#include <MmMap.h>
#include <MmRet.h>
#include <MmSeg.h>
#include <MmType.h>
#include <PrProc.h>
#include <TySegEnumLix.h>
#include <WhAbort.h>
#include <WhFile.h>
#include <WhList.h>
#include <WhString.h>

// My application classes.
#include <ReplayArg.h>



// Override builtin 'new' and 'delete'.
//   g++ 2.6.0: profiler won't profile the builtins.
void *	operator new		( size_t s )        { return ::malloc( s ); }
void *	operator new    []	( size_t s )        { return ::malloc( s ); }
void	operator delete		( void *p, size_t ) { ::free( p ); }
void	operator delete []	( void *p, size_t ) { ::free( p ); }



// Local functions.
MmRet		do_replay	( const ReplayArg & );




// Entry point.
int main( int argc, const char * const * argv, const char * const * envp )
{
    // Collect arguments.
    ReplayArg argReplay( argc, argv, envp );

    // Handle argument errors.
    if ( argReplay.isError( ) )
    {
	argReplay.showUsage( strVersion );
	return 1;
    }

    // Handle '-h' (help).
    if ( argReplay.hasHelp( ) )
    {
	argReplay.showUsage( strVersion );
	return 0;
    }

    // Replay.
    const MmRet mmRet = do_replay( argReplay );
    if ( mmRet != mmRetOk )
    {
	::fflush  ( stdout );
	::fprintf ( stderr, "main: MmRet code %d.\n", int( mmRet ) );
	return 2;
    }

    // That's all, folks.
    return 0;
}



// Replay a process.
MmRet do_replay( const ReplayArg & argReplay )
{
    // Declare list of events.
    EvHistory evhReplay;

    // Read input file into list of events.
    {
	MmFlat flatIn;
	{
	    WhFile fileIn;
	    MmRetCheck( fileIn.openName( argReplay.getFileIn( ),
		WhFile::tyOpenRead ) );
	    MmRetCheck( flatIn.fromFile( fileIn ) );
	    MmRetCheck( fileIn.closeFile( ) );
	}
	MmRetCheck( evhReplay.fromFlat( flatIn ) );
    }

    // Dump only.
    if ( argReplay.hasNoEffect( ) )
    {
	for ( ; !evhReplay.isAtEnd( ); evhReplay.advEvCur( ) )
	{
	    WhString strDump;
	    evhReplay.getEvCur( ).fmtStr( strDump );
	    if ( ::fwrite( strDump.address( ), sizeof(char),
		strDump.count( ), stdout ) != strDump.count( ) )
	    {
		return MmRetRaise( mmRetReplayWontFwrite );
	    }
	}
	return mmRetOk;
    }

    // Get reference to first exec call and its segments.
    if ( evhReplay.count( ) < 2 )
	return MmRetRaise( mmRetReplayFileShort );
    const EvSci * pevSciExec = evhReplay.getEvIndex( 1 ).ptrEvSci( );
    if ( pevSciExec == 0 || pevSciExec->getSysEntry( ) != __NR_execve )
	return MmRetRaise( mmRetReplayEventNotExec );
    const WhList <MmSeg> & lsegInExec = pevSciExec->getSeg( );

    // Create pointer to name.
    if ( lsegInExec.count( ) < 1 )
	return MmRetRaise( mmRetReplayExecNoSeg );
    const char * pstrExecName = (const char * ) lsegInExec[0].address( );

    // Create pointers to strings.
    WhList <const char *> lpstrArg;
    WhList <const char *> lpstrEnv;
    for ( int isegInExec = 0; isegInExec < lsegInExec.count( ); ++isegInExec )
    {
	const MmSeg & segInExec = lsegInExec[isegInExec];
	if ( segInExec.getItySeg( ) == tySegStrNulArg )
	    lpstrArg.append( (const char *) segInExec.address( ) );
	if ( segInExec.getItySeg( ) == tySegStrNulEnv )
	    lpstrEnv.append( (const char *) segInExec.address( ) );
    }
    lpstrArg.append( (const char *) 0 );
    lpstrEnv.append( (const char *) 0 );

    // Spawn a child.
    PrProc procReplay;
    MmMap mapReplay;
    procReplay.spawnProc( pstrExecName,
	MmAddr( pevSciExec->getArg( 1 ) ) == 0 ? 0 : lpstrArg.address( ),
	MmAddr( pevSciExec->getArg( 2 ) ) == 0 ? 0 : lpstrEnv.address( ) );

    // Declare wait-list of children.
    WhList <PrProc *> lpProcWait;
    lpProcWait.append( &procReplay );

    // Replay the child.
    for ( ; ; )
    {
	// Wait for child.
	PrProc * pProcWait;
	PrProc::waitProc( lpProcWait, pProcWait );
	if ( pProcWait != &procReplay )
	    return MmRetRaise( mmRetReplayWaitMismatch );

	// Process replay input.
	switch ( procReplay.getExecState( ) )
	{
	default:
	    WhAbort( "do_replay: bad state." );

	case PrProc::stExecExitNormal:
	    ::printf( "Exit: status %d.\n", procReplay.getExitNormal( ) );
	    return mmRetOk;

	case PrProc::stExecExitSignal:
	    ::printf( "Exit: signal %d.\n", procReplay.getExitSignal( ) );
	    return mmRetOk;

	case PrProc::stExecStopSignal:
	    ::printf( "Signal: %d.\n", procReplay.getStopSignal( ) );
	    procReplay.contProc( );
	    break;

	case PrProc::stExecStopSyscall:
	    // Update replay.
	    evhReplay.getEvCur( ).mergeMap( mapReplay );

	    // Store into the process.
	    // Broken calls:
	    //   vm86				affects address map
	    //   sigaction signal sigreturn	need signal handling
	    //   pause sigsuspend		need kick-start to continue
	    MmRetCheck( evhReplay.storeAfterWait  ( procReplay ) );
	    MmRetCheck( evhReplay.storeBeforeCont ( procReplay ) );

	    // Advance the history.
	    evhReplay.advEvCur( );

	    // Continue child execution.
	    procReplay.contProc( );
	    break;
	}
    }
}
