// Copyright 1995 Michael Chastain
// Licensed under the Gnu Public License, Version 2
//
// File: PrScrSun.cc
//   System call return value.
//   This is a concrete class.
//
// SunOS 4.1.3: I am guessing semantics from field names in 'struct user'!
//
// File Created:	19 Jun 1995		Michael Chastain
// Last Edited:		26 Jun 1995		Michael Chastain

#include <sys/types.h>
#include <sys/ptrace.h>
#include <sys/user.h>
#include <errno.h>


#include <MmArea.h>
#include <PrScr.h>
#include <WhAbort.h>



// Constructor.
PrScr::PrScr( )
    : tyScr_	( tyScrNot	)
    , fBound_	( false		)
    , wError_	( 0     	)
    , w1_	( 0     	)
    , w2_	( 0     	)
{
    ;
}



// Flat input.
MmRet PrScr::fromFlat( MmFlat & flatFrom )
{
    int ityScr;
    MmRetCheck( flatFrom.getInt  ( ityScr  ) );
    switch ( ityScr )
    {
    default:		return MmRetRaise( mmRetFlatEnumBad );
    case tyScrAddr:	tyScr_ = tyScrAddr; break;
    case tyScrHex:	tyScr_ = tyScrHex;  break;
    case tyScrInt:	tyScr_ = tyScrInt;  break;
    case tyScrNot:	tyScr_ = tyScrNot;  break;
    case tyScrOct:	tyScr_ = tyScrOct;  break;
    }
    MmRetCheck( flatFrom.getBool ( fBound_ ) );
    MmRetCheck( flatFrom.getWord ( wError_ ) );
    MmRetCheck( flatFrom.getWord ( w1_     ) );
    MmRetCheck( flatFrom.getWord ( w2_     ) );
    return mmRetOk;
}



// Flat output.
void PrScr::toFlat( MmFlat & flatTo ) const
{
    flatTo.putInt  ( int( tyScr_ ) );
    flatTo.putBool ( fBound_       );
    flatTo.putWord ( wError_       );
    flatTo.putWord ( w1_           );
    flatTo.putWord ( w2_           );
}



// Fetch from process.
void PrScr::fetchProc( const PrProc & procFetch, TyScr tyScr )
{
    struct user user;

    const MmAddr addrError = MmAddr( MmAddr(&user.u_error) - MmAddr( &user ) );
    if ( !procFetch.fetchWord( MmArea::tyAreaUser, addrError, wError_, 0 ) )
	WhAbort( "PrScr::fetchProc: cannot fetch." );

    const MmAddr addr1 = MmAddr( MmAddr( &user.u_rval1 ) - MmAddr( &user ) );
    if ( !procFetch.fetchWord( MmArea::tyAreaUser, addr1, w1_, 0 ) )
	WhAbort( "PrScr::fetchProc: cannot fetch." );

    const MmAddr addr2 = MmAddr( MmAddr( &user.u_rval2 ) - MmAddr( &user ) );
    if ( !procFetch.fetchWord( MmArea::tyAreaUser, addr2, w2_, 0 ) )
	WhAbort( "PrScr::fetchProc: cannot fetch." );

    tyScr_  = tyScr;
    fBound_ = true;
}



// Store to process.
void PrScr::storeProc( PrProc & procStore ) const
{
    if ( !fBound_ )
	WhAbort( "PrScr::storeProc: not bound." );

    struct user user;

    const MmAddr addrError = MmAddr( MmAddr(&user.u_error) - MmAddr( &user ) );
    if ( !procStore.storeWord( MmArea::tyAreaUser, addrError, wError_, 0 ) )
	WhAbort( "PrScr::storeProc: cannot store." );

    const MmAddr addr1 = MmAddr( MmAddr( &user.u_rval1 ) - MmAddr( &user ) );
    if ( !procStore.storeWord( MmArea::tyAreaUser, addr1, w1_, 0 ) )
	WhAbort( "PrScr::storeProc: cannot store." );

    const MmAddr addr2 = MmAddr( MmAddr( &user.u_rval2 ) - MmAddr( &user ) );
    if ( !procStore.storeWord( MmArea::tyAreaUser, addr2, w2_, 0 ) )
	WhAbort( "PrScr::storeProc: cannot store." );
}



// Format to string.
void PrScr::fmtScr( WhString & strRet ) const
{
    if ( !fBound_ )
	WhAbort( "PrScr::fmtScr: not bound." );

    if ( isError( ) )
    {
	fmtError( strRet );
    }
    else
    {
	switch ( tyScr_ )
	{
	default:
	    WhAbort( "PrScr::fmtStr: bad enum." );

	case tyScrAddr:
	    strRet.appPtrFmt( MmAddr( w1_ ) );
	    break;

	case tyScrHex:	
	    strRet.appStrRaw( "0x" );
	    strRet.appIntFmt( w1_, 16, 8 );
	    break;

	case tyScrInt:
	    strRet.appIntFmt( w1_ );
	    break;

	case tyScrNot:
	    strRet.appStrRaw( "(UNEXPECTED!) " "0x" );
	    strRet.appIntFmt( w1_, 16, 8 );
	    break;

	case tyScrOct:
	    strRet.appChrRaw( '0' );
	    strRet.appIntFmt( w1_, 8, 0 );
	    break;
	}
    }
    strRet.appChrRaw( '\n' );
}



// Get a word.
MmWord PrScr::getWord( int iWord ) const
{
    if ( !fBound_ )
	WhAbort("PrScr::getWord: not bound." );
    if ( isError( ) )
	WhAbort( "PrScr::getWord: is error." );

    switch ( iWord )
    {
    default: WhAbort( "PrScr::getWord: bad index." );
    case 0:  return wError_;
    case 1:  return w1_;
    case 2:  return w2_;
    }
}



// Return whether value represents an error.
bool PrScr::isError( ) const
{
    if ( !fBound_ )
	WhAbort( "PrScr::isError: not bound." );

    return ( ( wError_ & 0xFF000000 ) >> 24 ) != 0;
}



// Set value to represent a given error.
void PrScr::setError( TyScr tyScr, MmWord wError )
{
    tyScr_  = tyScr;
    wError_ = (wError & 0x000000FF) << 24;
    w1_     = 0;
    w2_     = 0;
    fBound_ = true;

    if ( !isError( ) )
	WhAbort( "PrScr::setError: not error." );
}



// Set value to represent a given non-error value.
void PrScr::setValue( TyScr tyScr, MmWord w1, MmWord w2 )
{
    tyScr_  = tyScr;
    wError_ = 0;
    w1_     = w1;
    w2_     = w2;

    if ( isError( ) )
	WhAbort( "PrScr::setValue: not error." );
}



// Format a string from an error value.
void PrScr::fmtError( WhString & strRet ) const
{
    strRet.appIntFmt( ( wError_ & 0xFF000000 ) >> 24 );
    switch ( ( wError_ & 0xFF000000 ) >> 24 )
    {
#define	when  break; case
    default:			strRet.appStrRaw( " (UNKNOWN)"		);
    case EPERM:			strRet.appStrRaw( " (EPERM)"		);
    case ENOENT:		strRet.appStrRaw( " (ENOENT)"		);
    case ESRCH:			strRet.appStrRaw( " (ESRCH)"		);
    case EINTR:			strRet.appStrRaw( " (EINTR)"		);
    case EIO:			strRet.appStrRaw( " (EIO)"		);
    case ENXIO:			strRet.appStrRaw( " (ENXIO)"		);
    case E2BIG:			strRet.appStrRaw( " (E2BIG)"		);
    case ENOEXEC:		strRet.appStrRaw( " (ENOEXEC)"		);
    case EBADF:			strRet.appStrRaw( " (EBADF)"		);
    case ECHILD:		strRet.appStrRaw( " (ECHILD)"		);
    case EAGAIN:		strRet.appStrRaw( " (EAGAIN)"		);
    case ENOMEM:		strRet.appStrRaw( " (ENOMEM)"		);
    case EACCES:		strRet.appStrRaw( " (EACCES)"		);
    case EFAULT:		strRet.appStrRaw( " (EFAULT)"		);
    case ENOTBLK:		strRet.appStrRaw( " (ENOTBLK)"		);
    case EBUSY:			strRet.appStrRaw( " (EBUSY)"		);
    case EEXIST:		strRet.appStrRaw( " (EEXIST)"		);
    case EXDEV:			strRet.appStrRaw( " (EXDEV)"		);
    case ENODEV:		strRet.appStrRaw( " (ENODEV)"		);
    case ENOTDIR:		strRet.appStrRaw( " (ENOTDIR)"		);
    case EISDIR:		strRet.appStrRaw( " (EISDIR)"		);
    case EINVAL:		strRet.appStrRaw( " (EINVAL)"		);
    case ENFILE:		strRet.appStrRaw( " (ENFILE)"		);
    case EMFILE:		strRet.appStrRaw( " (EMFILE)"		);
    case ENOTTY:		strRet.appStrRaw( " (ENOTTY)"		);
    case ETXTBSY:		strRet.appStrRaw( " (ETXTBSY)"		);
    case EFBIG:			strRet.appStrRaw( " (EFBIG)"		);
    case ENOSPC:		strRet.appStrRaw( " (ENOSPC)"		);
    case ESPIPE:		strRet.appStrRaw( " (ESPIPE)"		);
    case EROFS:			strRet.appStrRaw( " (EROFS)"		);
    case EMLINK:		strRet.appStrRaw( " (EMLINK)"		);
    case EPIPE:			strRet.appStrRaw( " (EPIPE)"		);
    case EDOM:			strRet.appStrRaw( " (EDOM)"		);
    case ERANGE:		strRet.appStrRaw( " (ERANGE)"		);
    case EWOULDBLOCK_sys:	strRet.appStrRaw( " (EWOULDBLOCK_sys)"	);
    case EINPROGRESS:		strRet.appStrRaw( " (EINPROGRESS)"	);
    case EALREADY:		strRet.appStrRaw( " (EALREADY)"		);
    case ENOTSOCK:		strRet.appStrRaw( " (ENOTSOCK)"		);
    case EDESTADDRREQ:		strRet.appStrRaw( " (EDESTADDRREQ)"	);
    case EMSGSIZE:		strRet.appStrRaw( " (EMSGSIZE)"		);
    case EPROTOTYPE:		strRet.appStrRaw( " (EPROTOTYPE)"	);
    case ENOPROTOOPT:		strRet.appStrRaw( " (ENOPROTOOPT)"	);
    case EPROTONOSUPPORT:	strRet.appStrRaw( " (EPROTONOSUPPORT)"	);
    case ESOCKTNOSUPPORT:	strRet.appStrRaw( " (ESOCKTNOSUPPORT)"	);
    case EOPNOTSUPP:		strRet.appStrRaw( " (EOPNOTSUPP)"	);
    case EPFNOSUPPORT:		strRet.appStrRaw( " (EPFNOSUPPORT)"	);
    case EAFNOSUPPORT:		strRet.appStrRaw( " (EAFNOSUPPORT)"	);
    case EADDRINUSE:		strRet.appStrRaw( " (EADDRINUSE)"	);
    case EADDRNOTAVAIL:		strRet.appStrRaw( " (EADDRNOTAVAIL)"	);
    case ENETDOWN:		strRet.appStrRaw( " (ENETDOWN)"		);
    case ENETUNREACH:		strRet.appStrRaw( " (ENETUNREACH)"	);
    case ENETRESET:		strRet.appStrRaw( " (ENETRESET)"	);
    case ECONNABORTED:		strRet.appStrRaw( " (ECONNABORTED)"	);
    case ECONNRESET:		strRet.appStrRaw( " (ECONNRESET)"	);
    case ENOBUFS:		strRet.appStrRaw( " (ENOBUFS)"		);
    case EISCONN:		strRet.appStrRaw( " (EISCONN)"		);
    case ENOTCONN:		strRet.appStrRaw( " (ENOTCONN)"		);
    case ESHUTDOWN:		strRet.appStrRaw( " (ESHUTDOWN)"	);
    case ETOOMANYREFS:		strRet.appStrRaw( " (ETOOMANYREFS)"	);
    case ETIMEDOUT:		strRet.appStrRaw( " (ETIMEDOUT)"	);
    case ECONNREFUSED:		strRet.appStrRaw( " (ECONNREFUSED)"	);
    case ELOOP:			strRet.appStrRaw( " (ELOOP)"		);
    case ENAMETOOLONG:		strRet.appStrRaw( " (ENAMETOOLONG)"	);
    case EHOSTDOWN:		strRet.appStrRaw( " (EHOSTDOWN)"	);
    case EHOSTUNREACH:		strRet.appStrRaw( " (EHOSTUNREACH)"	);
    case ENOTEMPTY:		strRet.appStrRaw( " (ENOTEMPTY)"	);
    case EPROCLIM:		strRet.appStrRaw( " (EPROCLIM)"		);
    case EUSERS:		strRet.appStrRaw( " (EUSERS)"		);
    case EDQUOT:		strRet.appStrRaw( " (EDQUOT)"		);
    case ESTALE:		strRet.appStrRaw( " (ESTALE)"		);
    case EREMOTE:		strRet.appStrRaw( " (EREMOTE)"		);
    case ENOSTR:		strRet.appStrRaw( " (ENOSTR)"		);
    case ETIME:			strRet.appStrRaw( " (ETIME)"		);
    case ENOSR:			strRet.appStrRaw( " (ENOSR)"		);
    case ENOMSG:		strRet.appStrRaw( " (ENOMSG)"		);
    case EBADMSG:		strRet.appStrRaw( " (EBADMSG)"		);
    case EIDRM:			strRet.appStrRaw( " (EIDRM)"		);
    case EDEADLK:		strRet.appStrRaw( " (EDEADLK)"		);
    case ENOLCK:		strRet.appStrRaw( " (ENOLCK)"		);
    case ENONET:		strRet.appStrRaw( " (ENONET)"		);
    case ERREMOTE:		strRet.appStrRaw( " (ERREMOTE)"		);
    case ENOLINK:		strRet.appStrRaw( " (ENOLINK)"		);
    case EADV:			strRet.appStrRaw( " (EADV)"		);
    case ESRMNT:		strRet.appStrRaw( " (ESRMNT)"		);
    case ECOMM:			strRet.appStrRaw( " (ECOMM)"		);
    case EPROTO:		strRet.appStrRaw( " (EPROTO)"		);
    case EMULTIHOP:		strRet.appStrRaw( " (EMULTIHOP)"	);
    case EDOTDOT:		strRet.appStrRaw( " (EDOTDOT)"		);
    case EREMCHG:		strRet.appStrRaw( " (EREMCHG)"		);
    case ENOSYS:		strRet.appStrRaw( " (ENOSYS)"		);
#undef	when
    }
}
