// Copyright 1994, 1995 Michael Chastain
// Licensed under the Gnu Public License, Version 2
//
// File: PrProc.h
//   Process handler.
//   This is an external resource handler class.
//
// File Created:	04 Oct 1994		Michael Chastain
// Last Reviewed:	23 Feb 1995		Michael Chastain
// Last Edited:		04 Sep 1995		Michael Chastain

#if !defined(PR_PROC_H)
#define PR_PROC_H

#include <sys/types.h>

#include <MmArea.h>
#include <MmType.h>
#include <PrTime.h>
#include <WhList.h>

class	EvEvent;
class	EvSci;
class	EvSco;
class	PrScr;



class	PrProc
{
    // Constructor and destructor.
    public:
				PrProc		(		   );
				~PrProc		(		   );

    // Forbidden copiers.
    private:
				PrProc		( const PrProc &   );
	const PrProc &		operator =	( const PrProc &   );

    // Enumeration of execution states.
    public:
	enum	StExec
	{
	    stExecBlank,	// Process constructed.
	    stExecExitNormal,	// Process exited normally.
	    stExecExitSignal,	// Process exited with signal.
	    stExecRunning,	// Process is running.
	    stExecStopSignal,	// Process stopped for signal.
	    stExecStopSyscall	// Process stopped for system call.
	};

    // Enumeration of system-call states.
    public:
	enum	StSys
	{
	    stSysBlank,		// Process constructed.
	    stSysFirst,		// Process in initial trap.
	    stSysSci,		// Process entered system call.
	    stSysExec,		// Process trapped during exec.
	    stSysSco 		// Process exited system call.
	};

    // Execution interface.
    public:
	StExec			getExecState	(		   ) const;
	StSys			getSysState	(		   ) const;
	int			getExitNormal	(		   ) const;
	int			getExitSignal	(		   ) const;
	int			getStopSignal	(		   ) const;
	const PrTime &		getTimeWait	(		   ) const;
	void			contProc	(		   );
	void			spawnProc	( const char *,
						  const char * const *,
						  const char * const *
						  		   );
	static void		waitProc	( WhList <PrProc *> &,
						  PrProc * &	   );

    // Wait helpers.
    private:
	void			waitUpdate	( int,
						  const PrTime &   );
	bool			waitIsSyscall	( int		   ) const;
	void			waitDoSyscall	(		   );

    // Register interface.
    public:
	bool			fetchReg	( int, MmWord &    ) const;
	bool			fetchRegPc	( MmWord &	   ) const;
	bool			fetchRegSp	( MmWord &	   ) const;
	bool			fetchListReg	( int, int,
						  WhList<MmWord> & ) const;
	bool			fetchSysEntry	( MmWord&, MmWord& ) const;
	bool			storeReg	( int, MmWord      );
	bool			storeSysEntry	( MmWord	   );

    // Address-space interface.
    public:
	bool			fetchWord	( MmArea::TyArea,
					          MmAddr, MmWord &,
					          PrScr *	   ) const;
	bool			storeWord	( MmArea::TyArea,
						  MmAddr, MmWord,
						  PrScr *	   );

    // Directory follow interface.
    public:
	bool			isFollowRoot	(		   ) const;
	bool			isFollowCwd	(		   ) const;
	void			setFollow	( bool, bool	   );
	void			followDir	( const EvSci *,
						  const EvSco *	   );

    // Instance data.
    private:
	StExec			stExec_;	// Execution state.
	StSys			stSys_;		// System call state.
	pid_t			pid_;		// My process.
	bool			fWait_;		// Has waited.
	PrTime			timeWait_;	// Last 'wait' time.
	int			iStatusWait_;	// Last 'wait' status.
	bool			fSysEntry_;	// Has a sys entry cached.
	MmWord			wSysEntry_;	// Sys entry.
	MmWord			wSubEntry_;	// Sub entry.
	bool			fFollowRoot_;	// Is following root.
	bool			fFollowCwd_;	// Is following cwd.
};



// Inline methods.

inline PrProc::StExec PrProc::getExecState( ) const
{
    return stExec_;
}

inline PrProc::StSys PrProc::getSysState( ) const
{
    return stSys_;
}

inline bool PrProc::isFollowRoot( ) const
{
    return fFollowRoot_;
}

inline bool PrProc::isFollowCwd( ) const
{
    return fFollowCwd_;
}

inline void PrProc::setFollow( bool fFollowRoot, bool fFollowCwd )
{
    fFollowRoot_ = fFollowRoot;
    fFollowCwd_  = fFollowCwd;
}



#endif
