// Copyright 1995 Michael Chastain
// Licensed under the Gnu Public License, Version 2
//
// File: ExeLix.cc
//   Interpret exe structures.
//
// File Created:	06 May 1995		Michael Chastain
// Last Reviewed:	08 Sep 1995		Michael Chastain
// Last Edited:		08 Sep 1995		Michael Chastain

#include <linux/a.out.h>

#include <Exe.h>
#include <MmArea.h>
#include <MmFlat.h>
#include <MmSeg.h>
#include <MmType.h>
#include <TySegEnumLix.h>
#include <WhList.h>



// Page alignment functions.
//   Linux 1.3.24: see 'PAGE_ALIGN' in 'include/asm-i386/page.h'.
inline int    PAGE_ALIGN(int    a) { return (a + (4096-1)) &~ (4096-1); }
inline MmAddr PAGE_ALIGN(MmAddr a) { return MmAddr(PAGE_ALIGN(int(a))); }



// Extract segment list from an executable.
//   Linux 1.3.24: see 'load_aout_binary' in 'fs/exec.c'.
MmRet ExeExeToSeg( const MmFlat & flatExe, WhList <MmSeg> & lsegExeRet )
{
    // Clear return value.
    lsegExeRet.clear( 4 );

    // Get reference to header.
    const WhList <char> & lcExe = flatExe.getDataRaw( );
    if ( lcExe.count( ) < sizeof(struct exec) )
	return MmRetRaise( mmRetExeEofPremature );
    const struct exec & exec = * (const struct exec *) lcExe.address( );

    // Check header.
    if ( N_MAGIC(exec) != OMAGIC
    &&   N_MAGIC(exec) != ZMAGIC
    &&   N_MAGIC(exec) != QMAGIC )
	return MmRetRaise( mmRetExeMagicBad );
    if ( lcExe.count( ) < N_TXTOFF(exec) + exec.a_text + exec.a_data )
	return MmRetRaise( mmRetExeEofPremature );

    // Map over previous areas.
    MmArea areaClear;
    areaClear.setCover( MmArea::tyAreaBlank );
    MmSeg segOutClear;
    segOutClear.setAreaFree( areaClear );
    lsegExeRet.append( segOutClear );

    // Map text and data areas.
    if ( N_MAGIC(exec) == OMAGIC )
    {
	// Map combined text+data area.
	const MmAddr addrTextData = MmAddr( N_TXTADDR(exec) );
	const int sTextData = PAGE_ALIGN( exec.a_text + exec.a_data );
	MmSeg segOutTextData;
	segOutTextData.setAreaFree( MmArea( MmArea::tyAreaData,
	    addrTextData, addrTextData + sTextData ) );
	lsegExeRet.append( segOutTextData );
    }
    else
    {
	// Map text area.
	const MmAddr addrText = MmAddr( N_TXTADDR(exec) );
	const int sText = PAGE_ALIGN( exec.a_text );
	MmSeg segOutText;
	segOutText.setAreaFree( MmArea( MmArea::tyAreaText,
	    addrText, addrText + sText ) );
	lsegExeRet.append( segOutText );

	// Map data area.
	const MmAddr addrData = addrText + sText;
	const int sData = PAGE_ALIGN( exec.a_data );
	MmSeg segOutData;
	segOutData.setAreaFree( MmArea( MmArea::tyAreaData,
	    addrData, addrData + sData ) );
	lsegExeRet.append( segOutData );
    }

    // Map bss area.
    const MmAddr addrDataEnd =
	MmAddr( N_TXTADDR(exec) + exec.a_text + exec.a_data );
    const MmAddr addrBss = PAGE_ALIGN( addrDataEnd );
    const int sBss = PAGE_ALIGN( addrDataEnd + exec.a_bss ) - addrBss;
    MmSeg segOutBss;
    segOutBss.setAreaFree( MmArea( MmArea::tyAreaBss,
	addrBss, addrBss + sBss ) );
    lsegExeRet.append( segOutBss );

    // That's all, folks.
    return mmRetOk;
}



// Interpret a library.
//   Linux 1.3.24: see 'load_aout_library' in 'fs/exec.c'.
MmRet ExeLibToSeg( const MmFlat & flatExe, WhList <MmSeg> & lsegExeRet )
{
    // Clear return value.
    lsegExeRet.clear( 2 );

    // Get reference to header.
    const WhList <char> & lcExe = flatExe.getDataRaw( );
    if ( lcExe.count( ) < sizeof(struct exec) )
	return MmRetRaise( mmRetExeEofPremature );
    const struct exec & exec = * (const struct exec *) lcExe.address( );

    // Check header.
    if ( N_MAGIC(exec) != ZMAGIC && N_MAGIC(exec) != QMAGIC )
	return MmRetRaise( mmRetExeMagicBad );
    if ( lcExe.count( ) < N_TXTOFF(exec) + exec.a_text + exec.a_data )
	return MmRetRaise( mmRetExeEofPremature );

    // Map combined text+data area.
    const MmAddr addrTextData = MmAddr( exec.a_entry &~ (4096-1) );
    const int sTextData = PAGE_ALIGN( exec.a_text + exec.a_data );
    MmSeg segOutTextData;
    segOutTextData.setAreaFree( MmArea( MmArea::tyAreaLib,
	addrTextData, addrTextData + sTextData ) );
    lsegExeRet.append( segOutTextData );

    // Map bss area.
    const MmAddr addrBss = addrTextData + sTextData;
    const int sBss =
	PAGE_ALIGN( exec.a_text + exec.a_data + exec.a_bss ) - sTextData;
    MmSeg segOutBss;
    segOutBss.setAreaFree( MmArea( MmArea::tyAreaLib,
	addrBss, addrBss + sBss ) );
    lsegExeRet.append( segOutBss );

    // That's all, folks.
    return mmRetOk;
}
