// Copyright 1994, 1995 Michael Chastain
// Licensed under the Gnu Public License, Version 2
//
// File: EvBase.h
//   An event.
//   This is an abstract base class.
//
// File Created:	29 Jul 1994		Michael Chastain
// Last Edited:		03 Sep 1995		Michael Chastain

#if !defined(EV_BASE_H)
#define EV_BASE_H

#include <MmFlat.h>
#include <MmMap.h>
#include <MmRet.h>
#include <MmSeg.h>
#include <PrProc.h>
#include <PrTime.h>
#include <WhList.h>
#include <WhString.h>



// Derived classes for down-casting.
class	EvSci;
class	EvSco;

class	EvBase
{
    // Constructors, assignment operator, destructor, and copier.
    public:
				EvBase		(	           );
				EvBase		( const EvBase &   );
	const EvBase &		operator =	( const EvBase &   );
	virtual			~EvBase		(		   );
	virtual EvBase *	copy		(		   ) const = 0;

    // RTTI.
    public:
	virtual const EvSci   * ptrEvSci	(		   ) const;
	virtual const EvSco   * ptrEvSco	(		   ) const;

    // Reference counts.
    public:
	int			countRef	(		   ) const;
	void			addRef		(		   );
	void			removeRef	(		   );

    // Flat interface.
    public:
	MmRet			fromFlat	( MmFlat &	   );
	void			toFlat		( MmFlat &	   ) const;

    // Flat interface (combiner).
    private:
	virtual MmRet		fromFlatEv	( MmFlat &	   )       = 0;
	virtual void		toFlatEv	( MmFlat &         ) const = 0;

    // Unknown attribute.
    public:
	bool			isUnknown	(		   ) const;
	void			setUnknown	( bool		   );

    // Time stamp.
    public:
	bool			hasStamp	(		   ) const;
	const PrTime &		getStamp	(		   ) const;
	void			setStamp	( const PrTime &   );

    // String interface.
    public:
	void			fmtStr		( WhString &       ) const;

    // String interface (combiner).
    private:
	virtual void		fmtStrEv	( WhString &	   ) const = 0;

    // Process interface.
    public:
	virtual MmRet		fetch		( const PrProc &,
						  const MmMap &,
						  const EvSci *    )       = 0;
	virtual MmRet		storeAfterWait	( PrProc &,
						  const EvSci *	   ) const = 0;
	virtual MmRet		storeBeforeCont	( PrProc &,
						  const EvSci *    ) const = 0;

    // Segment interface.
    public:
	void			appSeg		( const MmSeg &	   );
    	void			clearSeg	(		   );
	void			fmtAllSeg	( WhString &	   ) const;
	const WhList <MmSeg> &	getSeg		(		   ) const;
	const MmSeg *		getSegMatch	( MmAddr, int	   ) const;

    // Map interface.
    public:
	void			mergeMap	( MmMap &	   ) const;

    // Flat interface (#2).
    public:
	void			appFlat		( const MmFlat &   );
	void			clearFlat	(		   );
	void			fmtAllFlat	( WhString &	   ) const;
	const WhList <MmFlat> &	getFlat		(		   ) const;

    // Instance data.
    private:
	int			nRef_;		// Reference count.
	bool			fUnknown_;	// Event not known.
	bool			fStamp_;	// Time stamp.
	PrTime			timeStamp_;	// Time stamp.
	WhList <MmSeg>		lseg_;		// Data segments.
	WhList <MmFlat>		lflat_;		// Flats.
};



// Inline methods.

inline int EvBase::countRef( ) const
{
    return nRef_;
}

inline void EvBase::addRef( )
{
    ++nRef_;
}

inline void EvBase::removeRef( )
{
    --nRef_;
}

inline bool EvBase::isUnknown( ) const
{
    return fUnknown_;
}

inline void EvBase::setUnknown( bool fUnknown )
{
    fUnknown_ = fUnknown;
}

inline bool EvBase::hasStamp( ) const
{
    return fStamp_;
}

inline void EvBase::appSeg( const MmSeg & segApp )
{
    lseg_.append( segApp );
}

inline const WhList <MmSeg> & EvBase::getSeg( ) const
{
    return lseg_;
}

inline void EvBase::appFlat( const MmFlat & flatApp )
{
    lflat_.append( flatApp );
}

inline const WhList <MmFlat> & EvBase::getFlat( ) const
{
    return lflat_;
}

inline void EvBase::clearSeg( )
{
    lseg_.clear( );
}

inline void EvBase::clearFlat( )
{
    lflat_.clear( );
}



#endif
