// Copyright 1994, 1995 Michael Chastain
// Licensed under the Gnu Public License, Version 2
//
// File: EvBase.cc
//   An event.
//   This is an abstract base class.
//
// File Created:	29 Jul 1994		Michael Chastain
// Last Edited:		14 Sep 1995		Michael Chastain

#include <EvBase.h>
#include <MmFlat.h>
#include <MmSeg.h>
#include <WhAbort.h>



// Constructor.
EvBase::EvBase( )
    : nRef_		( 1	)
    , fUnknown_		( false )
    , fStamp_		( false	)
    , timeStamp_	(	)
    , lseg_		(	)
    , lflat_		(	)
{
    ;
}



// Copy constructor.
EvBase::EvBase( const EvBase & evBaseOld )
    : nRef_		( 1			)
    , fUnknown_		( evBaseOld.fUnknown_	)
    , fStamp_		( evBaseOld.fStamp_	)
    , timeStamp_	( evBaseOld.timeStamp_	)
    , lseg_		( evBaseOld.lseg_	)
    , lflat_		( evBaseOld.lflat_	)
{
    ;
}



// Assignment operator.
const EvBase & EvBase::operator = ( const EvBase & evBaseOld )
{
    fUnknown_	= evBaseOld.fUnknown_;
    fStamp_	= evBaseOld.fStamp_;
    timeStamp_	= evBaseOld.timeStamp_;
    lseg_	= evBaseOld.lseg_;
    lflat_	= evBaseOld.lflat_;
    return *this;
}



// Destructor.
EvBase::~EvBase( )
{
    ;
}



// RTTI.
const EvSci   * EvBase::ptrEvSci   ( ) const { return 0; }
const EvSco   * EvBase::ptrEvSco   ( ) const { return 0; }



// Flat input.
MmRet EvBase::fromFlat( MmFlat & flatFrom )
{
    MmRetCheck( flatFrom.getBool	( fUnknown_	) );
    MmRetCheck( flatFrom.getBool	( fStamp_	) );
    MmRetCheck( timeStamp_.fromFlat	( flatFrom	) );

    int nseg;
    MmRetCheck( flatFrom.getInt( nseg ) );
    if ( nseg < 0 )
	return MmRetRaise( mmRetFlatCountNegative );

    lseg_.clear( nseg );
    for ( int iseg = 0; iseg < nseg; ++iseg )
    {
	MmSeg seg;
	MmRetCheck( seg.fromFlat( flatFrom ) );
	lseg_.append( seg );
    }

    int nflat;
    MmRetCheck( flatFrom.getInt( nflat ) );
    if ( nflat < 0 )
	return MmRetRaise( mmRetFlatCountNegative );

    lflat_.clear( nflat );
    for ( int iflat = 0; iflat < nflat; ++iflat )
    {
	MmFlat flat;
	MmRetCheck( flat.fromFlat( flatFrom ) );
	lflat_.append( flat );
    }

    MmRetCheck( fromFlatEv( flatFrom ) );
    return mmRetOk;
}



// Flat output.
void EvBase::toFlat( MmFlat & flatTo ) const
{
    flatTo.putBool	( fUnknown_	);
    flatTo.putBool	( fStamp_	);
    timeStamp_.toFlat	( flatTo	);

    flatTo.putInt( lseg_.count( ) );
    for ( int iseg = 0; iseg < lseg_.count( ); ++iseg )
	lseg_[iseg].toFlat( flatTo );

    flatTo.putInt( lflat_.count( ) );
    for ( int iflat = 0; iflat < lflat_.count( ); ++iflat )
	lflat_[iflat].toFlat( flatTo );
    toFlatEv( flatTo );
}



// Get time stamp.
const PrTime & EvBase::getStamp( ) const
{
    if ( !fStamp_ )
	WhAbort( "EvBase::getStamp: not bound." );
    return timeStamp_;
}



// Set time stamp.
void EvBase::setStamp( const PrTime & timeStamp )
{
    fStamp_    = true;
    timeStamp_ = timeStamp;
}



// Format to string.
void EvBase::fmtStr( WhString & strRet ) const
{
    if ( hasStamp( ) && ptrEvSco( ) == 0 )
    {
	strRet.appIntFmt( getStamp( ).getSecond ( )		);
	strRet.appChrRaw( '.'					);
	strRet.appIntFmt( getStamp( ).getMicro  ( ), 10, 6	);
	strRet.appStrRaw( "  "					);
    }

    if ( fUnknown_ )
	strRet.appStrRaw( "<<UNKNOWN>> " );

    fmtStrEv( strRet );
}



// Format all segments.
void EvBase::fmtAllSeg( WhString & strRet ) const
{
    for ( int iseg = 0; iseg < getSeg( ).count( ); ++iseg )
    {
	const MmSeg & seg = getSeg( )[iseg];
	if ( seg.isBound( ) )
	    seg.fmtSeg( strRet );
    }
}



// Return first segment matching an address range (if any).
//   Callers beware: adding more segs -> array expansion -> invalid pointers!
const MmSeg * EvBase::getSegMatch( MmAddr addrMatch, int sMatch ) const
{
    for ( int iseg = 0; iseg < getSeg( ).count( ); ++iseg )
    {
	const MmSeg & seg = getSeg( )[iseg];
	if ( seg.isBound      ( )
	&&   seg.getAddrFirst ( ) <= addrMatch
	&&   seg.getAddrLast  ( ) >= addrMatch + sMatch )
	{
	    return &seg;
	}
    }
    return 0;
}



// Merge address spaces from an event into my address space map.
void EvBase::mergeMap( MmMap & mapMerge ) const
{
    for ( int isegMerge = 0; isegMerge < getSeg( ).count( ); ++isegMerge )
    {
	const MmSeg & segMerge = getSeg( )[isegMerge];
	if ( segMerge.isAreaNew( ) )
	    mapMerge.mergeArea( segMerge.getArea( ) );
    }
}



// Format all flats.
void EvBase::fmtAllFlat( WhString & strRet ) const
{
    for ( int iflat = 0; iflat < getFlat( ).count( ); ++iflat )
    {
	const MmFlat & flat = getFlat( )[iflat];
	strRet.appStrRaw( "    <flat> "			);
	strRet.appIntFmt( flat.getDataRaw( ).count( )	);
	strRet.appChrRaw( '\n'				);
    }
}
