// Copyright 1995 Michael Chastain
// Licensed under the Gnu Public License, Version 2
//
// File: ControlMain.cc
//   Main program for mec-control.
//
// File Created:	05 Jun 1995		Michael Chastain
// Last Edited:		14 Sep 1995		Michael Chastain

// Program version.
static const char strVersion [] = "0.1";



// Unix library.
#include <linux/unistd.h>
#include <sys/ptrace.h>
#include <errno.h>
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>

// My library classes.
#include <EvEvent.h>
#include <EvHistory.h>
#include <EvSci.h>
#include <EvSco.h>
#include <MmFlat.h>
#include <MmMap.h>
#include <MmRet.h>
#include <MmSeg.h>
#include <MmType.h>
#include <PrProc.h>
#include <PrScr.h>
#include <TySegEnumLix.h>
#include <WhAbort.h>
#include <WhFile.h>
#include <WhList.h>
#include <WhString.h>

// My application classes.
#include <ControlArg.h>



// Override builtin 'new' and 'delete'.
//   g++ 2.6.0: profiler won't profile the builtins.
void *	operator new		( size_t s )        { return ::malloc( s ); }
void *	operator new    []	( size_t s )        { return ::malloc( s ); }
void	operator delete		( void *p, size_t ) { ::free( p ); }
void	operator delete []	( void *p, size_t ) { ::free( p ); }



// Local functions.
MmRet		do_control	( const ControlArg & );
void		filter_ptrace	( PrProc &, PrProc &, MmMap &, pid_t,
				  const EvSci *, EvHistory & );




// Entry point.
int main( int argc, const char * const * argv, const char * const * envp )
{
    // Collect arguments.
    ControlArg argControl( argc, argv, envp );

    // Handle argument errors.
    if ( argControl.isError( ) )
    {
	argControl.showUsage( strVersion );
	return 1;
    }

    // Handle '-h' (help).
    if ( argControl.hasHelp( ) )
    {
	argControl.showUsage( strVersion );
	return 0;
    }

    // Control.
    const MmRet mmRet = do_control( argControl );
    if ( mmRet != mmRetOk )
    {
	::fflush  ( stdout );
	::fprintf ( stderr, "main: MmRet code %d.\n", int( mmRet ) );
	return 2;
    }

    // That's all, folks.
    return 0;
}



// Control a process.
MmRet do_control( const ControlArg & argControl )
{
    // Declare list of events.
    EvHistory evhReplay;

    // Read input file into list of events.
    {
	MmFlat flatIn;
	{
	    WhFile fileIn;
	    MmRetCheck( fileIn.openName( argControl.getFileIn( ),
		WhFile::tyOpenRead ) );
	    MmRetCheck( flatIn.fromFile( fileIn ) );
	    MmRetCheck( fileIn.closeFile( ) );
	}
	MmRetCheck( evhReplay.fromFlat( flatIn ) );
    }

    // Get reference to first exec call and its segments.
    if ( evhReplay.count( ) < 2 )
	return MmRetRaise( mmRetControlFileShort );
    const EvSci * pevSciExec = evhReplay.getEvIndex( 1 ).ptrEvSci( );
    if ( pevSciExec == 0 || pevSciExec->getSysEntry( ) != __NR_execve )
	return MmRetRaise( mmRetControlEventNotExec );
    const WhList <MmSeg> & lsegInExec = pevSciExec->getSeg( );

    // Create pointer to name.
    if ( lsegInExec.count( ) < 1 )
	return MmRetRaise( mmRetControlExecNoSeg );
    const char * pstrExecName = (const char * ) lsegInExec[0].address( );

    // Create pointers to strings.
    WhList <const char *> lpstrArgReplay;
    WhList <const char *> lpstrEnvReplay;
    for ( int isegInExec = 0; isegInExec < lsegInExec.count( ); ++isegInExec )
    {
	const MmSeg & segInExec = lsegInExec[isegInExec];
	if ( segInExec.getItySeg( ) == tySegStrNulArg )
	    lpstrArgReplay.append( (const char *) segInExec.address( ) );
	if ( segInExec.getItySeg( ) == tySegStrNulEnv )
	    lpstrEnvReplay.append( (const char *) segInExec.address( ) );
    }
    lpstrArgReplay.append( (const char *) 0 );
    lpstrEnvReplay.append( (const char *) 0 );

    // Declare replay child.
    PrProc procReplay;
    MmMap mapReplay;
    pid_t pidChild;

    // Create arguments for viewer.
    WhList <const char *> lpstrArgViewer;
    lpstrArgViewer.append( (const char *) "/usr/bin/gdb" );
    lpstrArgViewer.append( pstrExecName );
    lpstrArgViewer.append( (const char *) 0 );

    // Spawn a viewer.
    PrProc procViewer;
    MmMap mapViewer;
    procViewer.spawnProc( "/usr/bin/gdb", lpstrArgViewer.address( ),
	argControl.getEnvp( ) );
    procViewer.setFollow( true, true );

    // Declare wait-list of children.
    WhList <PrProc *> lpProcWait;
    lpProcWait.append( &procViewer );
    lpProcWait.append( &procReplay );

    // Control the children.
    EvEvent evViewerSciLast;
    for ( ; ; )
    {
	// Wait for viewer.
	PrProc * pProcWait;
	PrProc::waitProc( lpProcWait, pProcWait );
	if ( pProcWait != &procViewer )
	    return MmRetRaise( mmRetControlWaitMismatch );

	// Process viewer input.
	switch ( procViewer.getExecState( ) )
	{
	default:
	    WhAbort( "do_control: bad state." );

	case PrProc::stExecExitNormal:
	    ::printf( "Exit: status %d.\n", procViewer.getExitNormal( ) );
	    ::fflush( stdout );
	    return mmRetOk;

	case PrProc::stExecExitSignal:
	    ::printf( "Exit: signal %d.\n", procViewer.getExitSignal( ) );
	    ::fflush( stdout );
	    return mmRetOk;

	case PrProc::stExecStopSignal:
	    ::printf( "Signal: %d.\n", procViewer.getStopSignal( ) );
	    ::fflush( stdout );
	    procViewer.contProc( );
	    break;

	case PrProc::stExecStopSyscall:
	    // Fetch an event from viewer.
	    EvEvent evViewer;
	    evViewer.initFromProc( procViewer );
	    MmRetCheck( evViewer.fetch( procViewer, mapViewer,
		evViewerSciLast.ptrEvSci( ) ) );

	    // Filter some events.
	    if ( procViewer.getSysState( ) == PrProc::stSysSco )
	    {
		const EvSci * pevSci = evViewerSciLast.ptrEvSci( );
		const EvSco * pevSco = evViewer.ptrEvSco( );
		if ( pevSci == 0 || pevSco == 0 )
		    WhAbort( "do_control: zero pointer." );

		switch ( pevSci->getSysEntry( ) )
		{
		default:
		    break;

		case __NR_clone:
		    return MmRetRaise( mmRetControlCantClone );

		case __NR_fork:
		    if ( !pevSco->isScrError( ) )
		    {
			// Kill debugger's child.
			pidChild = pid_t( pevSco->getScrWord( 0 ) );
			::errno = 0;
			::kill( pidChild, SIGKILL );
			if ( ::errno != 0 )
			    return MmRetRaise( mmRetControlWontKill );

			// Spawn replay child.
			procReplay.spawnProc( pstrExecName,
			    MmAddr( pevSciExec->getArg( 1 ) ) == 0
				? 0 : lpstrArgReplay.address( ),
			    MmAddr( pevSciExec->getArg( 2 ) ) == 0
				? 0 : lpstrEnvReplay.address( ) );
			return MmRetRaise( mmRetControlWontExec );
		    }
		    break;

		case __NR_ptrace:
		    filter_ptrace( procViewer, procReplay, mapReplay,
			pidChild, pevSci, evhReplay );
		    break;

		case __NR_kill:
		case __NR_wait4:
		case __NR_waitpid:
		    break;
		}
	    }

	    // Update viewer.
	    evViewer.mergeMap( mapViewer );
	    procViewer.followDir(
		evViewerSciLast.ptrEvSci( ), evViewer.ptrEvSco( ) );

	    // Record last EvSci.
	    if ( evViewer.ptrEvSci( ) != 0 )
		evViewerSciLast = evViewer;

	    // Continue child execution.
	    procViewer.contProc( );
	    break;
	}
    }
}



// Viewer wants to ptrace on replay process.
// The call has already executed and probably failed,
//   because the original debugger child is dead.
// I will perform call on the replay child and store results into viewer.
// For calls to children I don't know about, I warn and do nothing.
//
//   Linux 1.2.9: see 'sys_ptrace' in 'arch/i386/kernel/ptrace.c'.
void filter_ptrace( PrProc & procViewer, PrProc & procReplay,
    MmMap & mapReplay, pid_t pidChild, const EvSci * pevSci,
    EvHistory & evhReplay )
{
    // Fetch arguments.
    if ( pevSci == 0 )
	WhAbort( "filter_ptrace: zero pointer." );
    const MmWord wRequest   =          pevSci->getArg( 0 );
    const pid_t  pidRequest = pid_t  ( pevSci->getArg( 1 ) );
    const MmAddr addrTarget = MmAddr ( pevSci->getArg( 2 ) );
    const MmWord wTarget    =          pevSci->getArg( 3 );

    // Warn about viewer trying to become traced.
    if ( wRequest == PTRACE_TRACEME )
    {
	::printf( "Viewer called PTRACE_TRACEME.\n" );
	::fflush( stdout );
	return;
    }

    // Warn about pid's other than known child.
    if ( pidRequest != pidChild )
    {
	::printf( "Viewer called ptrace( %d, %d, %p, 0x%X ).\n",
	    wRequest, pidRequest, addrTarget, wTarget );
	::fflush( stdout );
	return;
    }

    // Declare synthetic return value.
    PrScr scrViewer;

    // Perform request on replay process.
    switch ( wRequest )
    {
    default:
    case PTRACE_TRACEME:
	WhAbort( "filter_ptrace: unknown request." );

    case PTRACE_PEEKTEXT:
	{
	    MmWord wFetch;
	    procReplay.fetchWord( MmArea::tyAreaText, addrTarget, wFetch,
		&scrViewer );
	    if ( !procViewer.storeWord( MmArea::tyAreaData, MmAddr( wTarget ),
		wFetch, 0 ) )
	    {
		scrViewer.setError( PrScr::tyScrInt, EFAULT );
	    }
	}
	break;

    case PTRACE_PEEKDATA:
	{
	    MmWord wFetch;
	    procReplay.fetchWord( MmArea::tyAreaData, addrTarget, wFetch,
		&scrViewer );
	    if ( !procViewer.storeWord( MmArea::tyAreaData, MmAddr( wTarget ),
		wFetch, 0 ) )
	    {
		scrViewer.setError( PrScr::tyScrInt, EFAULT );
	    }
	}
	break;

    case PTRACE_PEEKUSR:
	{
	    MmWord wFetch;
	    procReplay.fetchWord( MmArea::tyAreaUser, addrTarget, wFetch,
		&scrViewer );
	    if ( !procViewer.storeWord( MmArea::tyAreaData, MmAddr( wTarget ),
		wFetch, 0 ) )
	    {
		scrViewer.setError( PrScr::tyScrInt, EFAULT );
	    }
	}
	break;

    case PTRACE_POKETEXT:
	// Need verification that this poke is to set or clear breakpoint.
	procReplay.storeWord( MmArea::tyAreaText, addrTarget, wTarget,
	    &scrViewer );
	break;

    case PTRACE_POKEDATA:
	// Should always disallow this!
	procReplay.storeWord( MmArea::tyAreaData, addrTarget, wTarget,
	    &scrViewer );
	break;

    case PTRACE_POKEUSR:
	// Need to understand why debugger would use this.
	procReplay.storeWord( MmArea::tyAreaUser, addrTarget, wTarget,
	    &scrViewer );
	break;

    case PTRACE_CONT:
    case PTRACE_SINGLESTEP:
    case PTRACE_SYSCALL:
	{
	    // Declare wait-list of children.
	    WhList <PrProc *> lpProcWait;
	    lpProcWait.append( &procViewer );
	    lpProcWait.append( &procReplay );

	    for ( ; ; )
	    {
		// Continue replay process.
		evhReplay.storeBeforeCont( procReplay );
		procReplay.contProc( );

		// Wait for replay.
		PrProc * pProcWait;
		PrProc::waitProc( lpProcWait, pProcWait );
		if ( pProcWait != &procReplay )
		    WhAbort( "filter_ptrace: wait mismatch." );

		// Discriminate on event type.
		if ( procReplay.getExecState( ) == PrProc::stExecStopSyscall )
		{
		    // Even in history (a syscall).
		    evhReplay.advEvCur( );
		    evhReplay.storeAfterWait( procReplay );
		    if ( wRequest == PTRACE_SYSCALL )
			break;
		}
		else
		{
		    // Event not in history (e.g. a breakpoint or exit).
		    break;
		}
	    }

	    // Set return value.
	    scrViewer.setValue( PrScr::tyScrInt, 0, 0 );
	}
	break;

    case PTRACE_KILL:
	WhAbort( "filter_ptrace: request unimplemented." );
	break;

    case PTRACE_ATTACH:
	WhAbort( "filter_ptrace: request unimplemented." );
	break;

    case PTRACE_DETACH:
	WhAbort( "filter_ptrace: request unimplemented." );
	break;
    }

    // Store return value into viewer.
    scrViewer.storeProc( procViewer );
}
