/* bdmcpu16.c - routines to talk to CPU16 target system
 * Copyright (C) 1992 by Scott Howard, all rights reserved
 * Permission is hereby granted to freely copy and use this code or derivations thereof
 * as long as no charge is made to anyone for its use
 */

#include	<stdio.h>
#include	<stdarg.h>
#include	"bdmcalls.h"
#include	"bdm.h"
#include	"trgtstat.h"
#include	"regs-16.h"
#include	"bdmerror.h"

/* BDM commands for CPU16 */

#define         BDM_RREGM       0x1780	/* Read multiple CPU registers */
#define         BDM_WREGM       0x1781	/* Write multiple CPU registers */
#define		BDM_RDMAC	0x178A	/* Read MAC registers */
#define		BDM_WRMAC	0x178B	/* Write MAC registers */
#define         BDM_RPCSP       0x1782	/* Read PC and SP registers */
#define         BDM_WPCSP       0x1783	/* Write PC and SP registers */
#define         BDM_READ        0x1784	/* Read Data Memory (Byte/Word/Long) */
#define         BDM_WRITE       0x1785	/* Write Data Memory (Byte/Word/Long) */
#define		BDM_RPMEM	0x1786	/* Read Program Memory (Word only) */
#define		BDM_WPMEM	0x1787	/* Write Program Memory (Word only) */
#define         BDM_GO          0x1788	/* Execute starting at address in PC */
#define         BDM_NOP         0x1789	/* No Operation */
#define		BDM_RESYNC     	0x0000	/* Used when we lose bit sync with target */
#define		BDM_REGMASK	0x7f

/* these codes are added to BDM_READ/BDM_WRITE to specify size */

#define         BDM_BYTESIZE    0x0000
#define         BDM_WORDSIZE    0x4000
#define         BDM_LONGSIZE    0x8000

/* define status codes returned from CPU16 */

#define         BDM_NOTREADY    0x10000
#define         BDM_BERR        0x10001
#define         BDM_ILLEGAL     0x1FFFF
#define         BDM_CMDCMPLTE   0x0FFFF

static void bdm_clrerror (void);	/* routine to re-sync with target */
static BYTE fc = SupervisorData;	/* function code used for mem accesses */
static char last_rw;			/* last mem access was read (1) or write (0) */
char RegsValid = 0;			/* record which RAM images of target registers are OK */
static LONG last_addr;			/* address of last mem access */
WORD go_cmd = BDM_GO,			/* value of 'GO' command (for Restart ()/Go () ) */
	CommandBitCount = 17;		/* number of bits in a BDM command */
static WORD RegValues [REG_MAX];	/* our RAM image of target registers */
static Initted = 0;			/* 1 = Init () was previously called */

		/* masks for RegsValid */

#define	PCSPRegsValid	1
#define	DSPRegsValid	2
#define	CPURegsValid	4

/* set_fc is an internal function which saves current function code
 * original function code is restored by restore_fc () below
 */

int set_fc (void)
{
	return 0;
}

int restore_fc (void)
{
	return 0;
}

/* ValidPorts returns unsigned integer where each bit position
 * indicates an installed LPT port (Bit 0 = LPT1, bit 1 = LPT2, etc)
 * This will have to go and mess around in /proc/ioports to find out...
 * For now we just assume a single one, /dev/lp1
 */

unsigned ValidPorts (void)
{
	return 1;
}

/* SetFC is called by user program to select desired memory space for
 * subsequent accesses to target memory
 * use the function code as defined by the CPU16/CPU32/68k processors
 * eg. 1 = user data, 2 = user code, 5 = supervisor data, etc
 * on CPU16 only 5 (supervisor data) and 6 (supervisor code) are valid
 */

void SetFC (unsigned NewFC)
{
	if (NewFC == SupervisorData || NewFC == SupervisorCode)
		fc = NewFC;
}

/* DeInit must be called before program quits or shells to DOS
 * un-do any bad things which have been done to the PC's LPT hardware
 * in order to talk to target
 */

void DeInit (void)
{
	if (Initted) bdm_deinit ();
	Initted = 0;
}

/* Init initializes parallel port to talk to target */

int Init (unsigned port, unsigned baud)
{
	DeInit ();
	if (!(ValidPorts () & (1 << (port - 1)))) return -1;

	bdm_init (0x378, baud);
	GetStatus ();
	Initted = 1;
	return 0;
}

/* bdm_error handles call to DriverError if error detected by bdm routines */

void bdm_error (int type)
{
	if ((type > BDM_FAULT_BERR) || (type < 0)) type = 0;
	DriverError (type, last_rw, last_addr);
}

static void bdm_clrerror (void)
{
	int count;
	long response;

	bdm_clk (BDM_RESYNC, CommandBitCount);
	bdm_clk (BDM_RESYNC, CommandBitCount);
	bdm_clk (BDM_RESYNC, CommandBitCount);
	bdm_clk (BDM_RESYNC, CommandBitCount);
	bdm_clk (BDM_RESYNC, CommandBitCount);
	while (!bdm_clk (0, 1)) ;
	while (bdm_clk (0, 1)) ;
	while (!bdm_clk (0, 1)) ;
	while (bdm_clk (0, 1)) ;
	bdm_clk (BDM_NOP, CommandBitCount - 2);
}

static LONG bdm_read (LONG addr, int pcount, unsigned ResponseSize,
	WORD *Where, WORD cmd, ...)
{
	va_list arg;
	WORD FirstArg;
	int err,count;
	LONG response,result;

	last_addr = addr;
	last_rw = 1;
	if (!Initted) bdm_error (BDM_FAULT_PORT);
	for (err = 3; err; err--)
	{
		result = 0;
		count = pcount;
		va_start (arg,cmd);
		response = bdm_clk (cmd, CommandBitCount);
		while (count--)
			if ((response = bdm_clk (va_arg(arg,WORD), CommandBitCount)) > BDM_NOTREADY)
			{
				if (response == BDM_BERR) bdm_error (BDM_FAULT_BERR);
				continue;
			}
		for (count = ResponseSize; count; count--)
		{
			while ((response = bdm_clk (BDM_NOP, CommandBitCount)) == BDM_NOTREADY) ;
			if (response < BDM_NOTREADY)
			{
				result <<= 16;
				result |= response;
				if (Where) *Where++ = response;
			}
			else
			{
				if (response == BDM_BERR) bdm_error (BDM_FAULT_BERR);
				break;
			}
		}
		if (response > BDM_NOTREADY)
		{
			if (response == BDM_BERR) bdm_error (BDM_FAULT_BERR);
			bdm_clrerror ();
			continue;
		}
		va_end (arg);
		break;
	}
	if (!err) bdm_error (BDM_FAULT_UNKNOWN);
	response = bdm_clk (BDM_NOP, CommandBitCount);
	if (response == BDM_BERR) bdm_error (BDM_FAULT_BERR);
	return result;
}

static void bdm_write (LONG addr, int pcount, WORD cmd, ...)
{
	va_list arg;
	int err,count;
	LONG response,result;

	last_addr = addr;
	last_rw = 0;
	if (!Initted) bdm_error (BDM_FAULT_PORT);
	for (err = 3; err; err--)
	{
		count = pcount;
		va_start (arg,cmd);
		response = bdm_clk (cmd, CommandBitCount);
		while (count--)
			if ((response = bdm_clk (va_arg(arg,WORD), CommandBitCount)) > BDM_NOTREADY)
			{
				if (response == BDM_BERR) bdm_error (BDM_FAULT_BERR);
				bdm_clrerror ();
				continue;
			}
		while ((response = bdm_clk (BDM_NOP, CommandBitCount)) == BDM_NOTREADY) ;
		if (response == BDM_CMDCMPLTE) break;
		else if (response == BDM_BERR) bdm_error (BDM_FAULT_BERR);

	}
	va_end (arg);
	if (!err) bdm_error (BDM_FAULT_UNKNOWN);
	response = bdm_clk (BDM_NOP, CommandBitCount);
	if (response == BDM_BERR) bdm_error (BDM_FAULT_BERR);
}

void RunChip (LONG where)
{
	StopChip ();
	if (where) PutReg (REG_PC, where);
	bdm_clk (BDM_GO, CommandBitCount);
	RegsValid = 0;
}

LONG GetByte (LONG x)
{
	int frozen,c;
	LONG result;

	frozen = StopChip ();
	if (fc == SupervisorData)
		result = bdm_read (x, 2, 1, 0,
			BDM_READ, (WORD) (x >> 16), (WORD) x);
	else
	{
		result = bdm_read (x, 2, 1, 0,
			BDM_RPMEM, (WORD) (x >> 16), (WORD) x & 0xfffe);
		if (x & 1) result &= 0xff;
		else result >>= 8;
	}
	if (frozen) RunChip (0L);
	return result;
}

void PutByte (LONG x, BYTE y)
{
	int frozen;
	unsigned stuff;

	frozen = StopChip ();
	if (fc == SupervisorData)
		bdm_write (x, 3, BDM_WRITE,
			(WORD) (x >> 16), (WORD) x, (WORD) y);
	else
	{
		if (x & 1) stuff = y;
		else
		{
			stuff = bdm_read (x, 2, 1, 0, BDM_RPMEM,
				(WORD) (x >> 16), (WORD) (x & 0xfffe));
			stuff &= (x & 1) ? 0xff00 : 0xff;
			stuff |= (x & 1) ? y : (WORD) y << 8;
			bdm_write (x, 3, BDM_WPMEM,
				(WORD) (x >> 16), (WORD) (x & 0xfffe), (WORD) stuff);
		}
	}
	if (frozen) RunChip (0L);
}

void FillByte (LONG x, BYTE y)
{
	PutByte (x, y);
}

LONG DumpByte (LONG x)
{
	return GetByte (x);
}

LONG GetWord (LONG x)
{
	int frozen,c,c1;
	LONG result;

	frozen = StopChip ();
	if (fc == SupervisorData)
		result = bdm_read (x, 2, 1, 0,
		BDM_READ, BDM_WORDSIZE | (WORD) (x >> 16), (WORD) x);
	else if (x & 1)
		result = ((WORD) GetByte (x) << 8) | GetByte (x + 1);
		else result = bdm_read (x, 2, 1, 0, BDM_RPMEM,
			(WORD) (x >> 16), (WORD) x);
	if (frozen) RunChip (0L);
	return result;
}

void PutWord (LONG x, WORD y)
{
	int frozen;

	frozen = StopChip ();
	if (fc == SupervisorData)
		bdm_write (x, 3, BDM_WRITE, BDM_WORDSIZE | (WORD) (x >> 16),
			(WORD) x, (WORD) y);
	else if (x & 1)
	{
		PutByte (x, y >> 8);
		PutByte (x + 1, y);
	}
	else bdm_write (x, 3, BDM_WPMEM, (WORD) (x >> 16), (WORD) x, (WORD) y);
	if (frozen) RunChip (0L);
}

void FillWord (LONG x, WORD y)
{
	PutWord (x, y);
}

LONG DumpWord (LONG x)
{
	return GetWord (x);
}

LONG GetLong (LONG x)
{
	int frozen;
	LONG result;

	frozen = StopChip ();
	if (fc == SupervisorData)
		result = bdm_read (x, 2, 2, 0, BDM_READ,
			BDM_LONGSIZE | (WORD) (x >> 16), (WORD) x);
	else if (x & 1)
		result = (LONG) GetByte (x) << 24
			| (LONG) GetWord (x + 1) << 8
			| GetByte (x + 3);
	else result = (LONG) GetWord (x) << 16 | GetWord (x + 2);
	if (frozen) RunChip (0L);
	return result;
}

void PutLong (LONG x, LONG y)
{
	int frozen;

	frozen = StopChip ();
	if (fc == SupervisorData)
		bdm_write (x, 3, BDM_WRITE, BDM_LONGSIZE | (WORD) (x >> 16),
			(WORD) x, (WORD) y);
	else if (x & 1)
	{
		PutByte (x, y >> 24);
		PutWord (x + 1, y >> 8);
		PutByte (x + 3, y);
	}
	else
	{
		PutWord (x, y >> 16);
		PutWord (x + 2, y);
	}
	if (frozen) RunChip (0L);
}

void FillLong (LONG x, LONG y)
{
	PutLong (x, y);
}

LONG DumpLong (LONG x)
{
	return GetLong (x);
}

static void ReadCPURegisters (void)
{
	bdm_read (REG_MINCPU, 1, REG_MAXCPU - REG_MINCPU + 1,
		&RegValues [REG_MINCPU], BDM_RREGM, (WORD) BDM_REGMASK);
	RegsValid |= CPURegsValid;
}

static void ReadDSPRegisters (void)
{
	bdm_read (REG_MINDSP, 0, REG_MAXDSP - REG_MINDSP + 1,
		&RegValues [REG_MINDSP], BDM_RDMAC);
	RegsValid |= DSPRegsValid;
}

static void ReadPCSPRegisters (void)
{
	bdm_read (REG_MINPCSP, 0, REG_MAXPCSP - REG_MINPCSP + 1,
		&RegValues [REG_MINPCSP], BDM_RPCSP);
	RegsValid |= PCSPRegsValid;
}

static void WriteDSPRegisters (void)
{
	bdm_write (0, REG_MAXDSP - REG_MINDSP + 1, BDM_WRMAC,
		RegValues [REG_H], RegValues [REG_I], RegValues [REG_AM0],
		RegValues [REG_AM1],RegValues [REG_AM2],RegValues [REG_XMYM]);
}

static void WritePCSPRegisters (void)
{
	bdm_write (0, REG_MAXPCSP - REG_MINPCSP + 1, BDM_WPCSP,
		RegValues [REG_PK],RegValues [REG_PC],
		RegValues [REG_SK],RegValues [REG_SP]);
}

LONG GetReg (unsigned which)
{
	int frozen;
	LONG result;

	frozen = StopChip ();

	if (which <= REG_MAXCPU)
	{
		if (!(RegsValid & CPURegsValid)) ReadCPURegisters ();
	}
	else if (which >= REG_MINDSP && which <= REG_MAXDSP)
	{
		if (!(RegsValid & DSPRegsValid)) ReadDSPRegisters ();
	}
	else if (which >= REG_MINPCSP && which <= REG_MAXPCSP)
	{
		if (!(RegsValid & PCSPRegsValid)) ReadPCSPRegisters ();
	}
	result = RegValues [which];
	if (frozen) RunChip (0L);
	return result;
}

void PutReg (unsigned which, LONG data)
{
	int frozen = StopChip ();

	if (which <= REG_MAXCPU)
	{
		RegValues [which] = data;
		bdm_write (0, 2, BDM_WREGM, (WORD) 1 << which, (WORD) data);
	}
	else if (which >= REG_MINDSP && which <= REG_MAXDSP)
	{
		if (!(RegsValid & DSPRegsValid)) ReadDSPRegisters ();
		RegValues [which] = data;
		WriteDSPRegisters ();
	}
	else if (which >= REG_MINPCSP && which <= REG_MAXPCSP)
	{
		if (!(RegsValid & PCSPRegsValid)) ReadPCSPRegisters ();
		RegValues [which] = data;
		WritePCSPRegisters ();
	}
	if (frozen) RunChip (0L);
}


/* end of bdmcpu16.c */
