/* bdm-pd.c - routines to talk to CPU16 or CPU32 target
 * via public-domain interface hardware on IBM PC parallel printer port
 * Copyright (C) 1992 by Scott Howard, all rights reserved
 * Permission is hereby granted to freely copy and use this code
 * or derivations thereof as long as no charge is made to anyone for its use
 * 
 * Adapted for Linux by D.Jeff Dionne, April 25 1995
 */

#include	"bdm.h"
#include	"bdmcalls.h"
#include	"bdmerror.h"
#include	"trgtstat.h"

#define	bdm_ctl         2               /* offset of control port from base */
#define	step_out        8               /* set low to gate IFETCH onto BKPT */
#define	dsi             4               /* data shift input - PC->MCU       */
#define	rst_out         2               /* set high to force reset on MCU   */
#define	reset           2               /* low when RESET asserted          */
#define	dsclk           1               /* data shift clock/breakpoint pin  */

#define	bdm_stat	1		/* offset of status port from base     */
#define	nc              0x80            /* not connected - low when unplugged  */
#define	pwr_dwn         0x40            /* power down - low when Vcc failed    */
#define	dso             0x20            /* data shift output - MCU->PC         */
#define	freeze          8               /* FREEZE asserted when MCU stopped    */

#define	nop_cmd         0               /* null (NOP) command                  */
#define	waitcnt         0xffff          /* no of loops to wait for response    */

extern void bdm_delay (unsigned);
extern void bdm_error (int);
static unsigned bdm_speed, bdm_port, old_ctl;
extern unsigned go_cmd, CommandBitCount;

/* bdm_deinit is called before program quits 
 * un-do any bad things which have been done to talk to target
 */

void bdm_deinit (void)
{
	outportb (bdm_port + bdm_ctl, old_ctl);
}

/* bdm_init initializes parallel port to talk to target */

void bdm_init (int port, int baud)
{
	RegsValid = 0;
	old_ctl = inportb (bdm_port + bdm_ctl);
	outportb (bdm_port+bdm_ctl,step_out);
	bdm_port = port;
	bdm_speed = baud;
}

void ResetChip (void)
{
	RegsValid = 0;
	outportb (bdm_port + bdm_ctl, rst_out + step_out);
	bdm_delay (waitcnt);
	outportb (bdm_port + bdm_ctl, step_out);
}

void RestartChip (void)
{
	unsigned LoopCount;

	RegsValid = 0;
	outportb (bdm_port + bdm_ctl, rst_out  | dsclk);
	bdm_delay (waitcnt);
	outportb (bdm_port + bdm_ctl, dsclk);
	bdm_delay (waitcnt);
	outportb (bdm_port + bdm_ctl, step_out | dsclk);
	for (LoopCount = 0xffff; LoopCount; LoopCount--)
		if (inportb (bdm_port + bdm_stat) & freeze) break;
	if (!LoopCount) bdm_error (BDM_FAULT_RESPONSE);
}

int StopChip (void)
{
	unsigned ctr;
	char frozen = 0;

	RegsValid = 0;
	if (inportb (bdm_port + bdm_stat) & freeze) return frozen;
	frozen = 1;
	outportb (bdm_port + bdm_ctl, dsclk + step_out);
	for (ctr = waitcnt; ctr; ctr--)
	{
		if (inportb (bdm_port + bdm_stat) & freeze) break;
		bdm_delay (1);
	}
	if (!ctr) bdm_error (BDM_FAULT_RESPONSE);
	return frozen;
}

/* bdm_clk sends <value> to MCU for <parameter> bits, returns MCU response */

LONG bdm_clk (WORD value, int count)
{
	LONG ShiftRegister = ((LONG) value) << (32 - count);

	unsigned char DataOut;

	unsigned stat = GetStatus ();

	if (stat & TARGETRESET)
		bdm_error (BDM_FAULT_RESET);
	if (stat & TARGETNC)
		bdm_error (BDM_FAULT_CABLE);
	if (stat & TARGETPOWER)
		bdm_error (BDM_FAULT_POWER);

	while (count--)
	{
		DataOut = (ShiftRegister & 0x80000000) ? dsi : 0;
		ShiftRegister <<= 1;
		if (!(inportb (bdm_port + bdm_stat) & dso))
			ShiftRegister |= 1;
		outportb (bdm_port + bdm_ctl, DataOut | step_out | dsclk);
		bdm_delay (bdm_speed + 1);
		outportb (bdm_port + bdm_ctl, DataOut | step_out);
		bdm_delay ((bdm_speed >> 1) + 1);
	}

	return ShiftRegister;
}

/* StepChip sends GO command word, then triggers breakpoint on first fetch        */

void StepChip (void)
{
#define	DataOut	(go_cmd & 1 ? dsi : 0)
	unsigned stat = GetStatus ();

	RegsValid = 0;
	if (stat & TARGETRESET)
		bdm_error (BDM_FAULT_RESET);
	if (stat & TARGETNC)
		bdm_error (BDM_FAULT_CABLE);
	if (stat & TARGETPOWER)
		bdm_error (BDM_FAULT_POWER);
	bdm_clk (go_cmd >> 1, CommandBitCount - 1);
	outportb (bdm_port + bdm_ctl, dsclk | step_out | DataOut);
	bdm_delay (bdm_speed + 1);
	disable ();
	outportb (bdm_port + bdm_ctl, dsclk | DataOut);
	bdm_delay (1);
	outportb (bdm_port + bdm_ctl, DataOut);
	enable ();
}

/* GetStatus returns status of MCU                                              */

unsigned GetStatus (void)
{
	BYTE temp = inportb (bdm_port + bdm_stat);

	if (!(temp & nc)) return TARGETNC;
	if (!(temp & pwr_dwn)) return TARGETPOWER;
	return 	(temp & freeze ? TARGETSTOPPED: 0)
		| (inportb (bdm_port + bdm_ctl) & reset ? TARGETRESET : 0);
}

/* GetStatusMask returns mask showing which stat bits are valid */

unsigned GetStatusMask (void)
{
	return TARGETRESET | TARGETSTOPPED | TARGETPOWER | TARGETNC;
}
