/*
svidvar.c Version 1.7.0 - Difference between 2 variances
datagen versus libc drand48
Copyright (C) 2005-2010  dondalah721@yahoo.com (Dondalah)

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to:

	Free Software Foundation, Inc.
	59 Temple Place - Suite 330
	Boston, MA  02111-1307, USA.
*/

/* This test is based on: */
/* Lincoln L. Chao */
/* Statistics for Management */
/* Palo Alto, CA: The Scientific Press, 1984 */
/* Chapter 10, Statistical Estimation */
/* Section 3, Interval Estimation About Proportions */

/* Lincoln L. Chao was at California State University, */
/* Long Beach, CA, while writing this book. */

/* The drand48 random number generator is taken from */
/* libc stdlib.h */
/* See the info discussion under libc, pseudo-random */
/* numbers, SVID Random */

/* svidvar compares the random number generator */
/* in datagen to the generator called drand48. */
/* A mean and variance is created with each */
/* generator, and the difference is compared */
/* to zero. */

/* drand48 corresponds to rndfrac in datagen. */
/* Therefore the parameters in svidvar do not */
/* offer a choice in technique for the datagen */
/* generator. */

/* Sample test: */

/* svidvar -c 95.0 -s 10000 -p 0.5 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include "rnd.h"

void putstx(pgm)
char *pgm;
   {
   fprintf(stderr,"Usage: %s options\n",
      pgm);
   fprintf(stderr,"Options:\n");
   fprintf(stderr,"-h = help\n");
   fprintf(stderr,"-c cc.c = confidence level\n");
   fprintf(stderr,"   where cc.c is 90.0, 95.0, "
      "98.0, or 99.0\n");
   fprintf(stderr,"   default = 95.0\n");
   fprintf(stderr,"-s zzzzzz = size of test\n");
   fprintf(stderr,"   where zzzzzz is 30 to 1000000000\n");
   fprintf(stderr,"   default = 10000\n");
   fprintf(stderr,"-p n.n = probability of success\n");
   fprintf(stderr,"   where n.n = 0.001 - 0.999\n");
   fprintf(stderr,"   default = 0.5\n");
   exit(1);
   } /* putstx */

int main(argc,argv)
int argc;
char **argv;
   {
   unsigned short int seed16v[3];
   unsigned short int *prevseed;
   int i;
   int size;
   int rslt;
   int obs_1;
   int obs_2;
   int tmp;
   unsigned char *sd1;
   char title_1[128];
   char title_2[128];
   char title_diff[128];
   double confidence;
   double zval;
   double tstprob;
   double dblobs_1;
   double dblobs_2;
   double prob_1;
   double prob_2;
   double q_1;
   double q_2;
   double stderr_1;
   double stderr_2;
   double quot_1;
   double quot_2;
   double lolmt_1;
   double lolmt_2;
   double hilmt_1;
   double hilmt_2;
   double stderr_diff;
   double popmean;
   double lolmt_diff;
   double hilmt_diff;
   double diff;
   double dblsz;
   double ran2(int *idum);
   if (argc == 1) putstx(*argv);
   size    = 10000;
   dblsz   = (double) size;
   confidence = 95.0;
   zval    = 1.96;
   tstprob = 0.5;
   i = 1;
   while (i < argc)
      {
      rslt = strcmp(*(argv+i),"-h");
      if (!rslt) putstx(*argv);
      rslt = strcmp(*(argv+i),"-s");
      if (!rslt)
	 {
	 i++;
	 if (i >= argc)
	    {
	    fprintf(stderr,"Missing size\n");
	    putstx(*argv);
	    } /* if out of arguments */
         size = atoi(*(argv+i));
         if (size < 30 || size > 1000000000)
            {
            fprintf(stderr,"Invalid size.\n");
            putstx(*argv);
            } /* bad size */
         dblsz = (double) size;
	 i++;
	 continue;
	 } /* if size */
      rslt = strcmp(*(argv+i),"-c");
      if (!rslt)
	 {
	 i++;
	 if (i >= argc)
	    {
	    fprintf(stderr,"Missing confidence level\n");
	    putstx(*argv);
	    } /* if out of arguments */
         confidence = atof(*(argv+i));
         if (confidence == 90.0)
	    zval = 1.645;
         else if (confidence == 95.0)
	    zval = 1.96;
         else if (confidence == 98.0)
	    zval = 2.326;
         else if (confidence == 99.0)
	    zval = 2.575;
	 else
            {
            fprintf(stderr,"Invalid confidence level.\n");
            putstx(*argv);
            } /* bad confidence */
	 i++;
	 continue;
	 } /* if confidence level */
      rslt = strcmp(*(argv+i),"-p");
      if (!rslt)
	 {
	 i++;
	 if (i >= argc)
	    {
	    fprintf(stderr,"Missing probability\n");
	    putstx(*argv);
	    } /* if out of arguments */
         tstprob = atof(*(argv+i));
         if (tstprob < 0.001 || tstprob > 0.999)
            {
            fprintf(stderr,"Invalid probability.\n");
            putstx(*argv);
            } /* bad tstprob */
	 i++;
	 continue;
	 } /* if probability */
      fprintf(stderr,"Invalid option %s.\n",
	 *(argv+i));
      putstx(*argv);
      } /* for each argument */
   popmean = tstprob * dblsz;
   if (popmean < 30.0)
      {
      fprintf(stderr,"Size is too small.\n");
      putstx(*argv);
      } /* if size too small */
   sd1 = (unsigned char *) rndinit();
   if (sd1 == NULL)
      {
      fprintf(stderr,"svidvar: out of memory "
	 "allocating sd one.\n");
      exit(1);
      } /* out of mem */
   /* Initialize drand48 global seed array. */
   /* drand48 has 2^48 initial seeds. */
   tmp = (int) rndpwr(16,sd1);
   seed16v[0] = (unsigned short int) tmp;
   tmp = (int) rndpwr(16,sd1);
   seed16v[1] = (unsigned short int) tmp;
   tmp = (int) rndpwr(16,sd1);
   seed16v[2] = (unsigned short int) tmp;
   prevseed = (unsigned short int *) seed48(seed16v);
   obs_1 = obs_2 = 0;
   i = size;
   while (i--)
      {
      if (rndfrac(sd1) < tstprob) obs_1++;
      if (drand48()    < tstprob) obs_2++;
      } /* generate observations */
   dblobs_1 = (double) obs_1;
   dblobs_2 = (double) obs_2;
   prob_1 = dblobs_1 / dblsz;
   prob_2 = dblobs_2 / dblsz;
   q_1    = 1.0 - prob_1;
   q_2    = 1.0 - prob_2;
   quot_1 = prob_1 * q_1 / dblsz;
   quot_2 = prob_2 * q_2 / dblsz;
   stderr_1 = sqrt(quot_1);
   stderr_2 = sqrt(quot_2);
   lolmt_1 = prob_1 - (zval * stderr_1);
   hilmt_1 = prob_1 + (zval * stderr_1);
   lolmt_2 = prob_2 - (zval * stderr_2);
   hilmt_2 = prob_2 + (zval * stderr_2);
   stderr_diff = sqrt(quot_1 + quot_2);
   diff = prob_1 - prob_2;
   lolmt_diff = diff - (zval * stderr_diff);
   hilmt_diff = diff + (zval * stderr_diff);
      strcpy(title_1,"Datagen    ");
      strcpy(title_2,"Drand48    ");
   strcpy(title_diff,"Difference ");

   if (lolmt_1 > tstprob)
      {
      printf("Test fails for Lower "
	 "Limit %s\n\n", title_1);
      }
   if (hilmt_1 < tstprob)
      {
      printf("Test fails for Upper "
	 "Limit %s\n\n", title_1);
      }
   if (lolmt_2 > tstprob)
      {
      printf("Test fails for Lower "
	 "Limit %s\n\n", title_2);
      }
   if (hilmt_2 < tstprob)
      {
      printf("Test fails for Upper "
	 "Limit %s\n\n", title_2);
      }
   if (lolmt_diff > 0.0)
      {
      printf("Test fails for Lower "
	 "Limit %s\n\n", title_diff);
      }
   if (hilmt_diff < 0.0)
      {
      printf("Test fails for Upper "
	 "Limit %s\n\n", title_diff);
      }

   printf("Difference between two mean proportions:\n\n");
   printf("    Probability "
      "    Confidence Level "
      "    Observations\n");
   printf("     %8.5f "
      "           %6.3f "
      "         %9.0f\n\n",
      tstprob, confidence, dblsz);
   printf("                Probability "
      "     Lower Limit "
      "     Upper Limit \n");

   printf("%s %16.11f %16.11f %16.11f\n",
      title_1, prob_1, lolmt_1, hilmt_1);
   printf("%s %16.11f %16.11f %16.11f\n",
      title_2, prob_2, lolmt_2, hilmt_2);

   printf("%s %16.11f %16.11f %16.11f\n",
      title_diff, diff, lolmt_diff, hilmt_diff);
   return(0);
   } /* main */
