/*
contin.c Version 1.7.0 - Continuous Distribution Test
Copyright (C) 2005-2010  dondalah721@yahoo.com (Dondalah)

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to:

	Free Software Foundation, Inc.
	59 Temple Place - Suite 330
	Boston, MA  02111-1307, USA.
*/

/* This test is described in:                     */
/* Donald E. Knuth, Stanford University           */
/* The Art of Computer Programming                */
/* Volume 2, Seminumerical Algorithms             */
/* Reading, MA: Addison-Wesley, 1981              */
/* Second Edition                                 */
/* Chapter 3  Random Numbers                      */
/* 3. Statistical Tests                           */
/* 2. Empirical Tests                             */
/* A. Equidistribution Test (Frequency Test)      */

/* Sample test: */

/* contin */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include "rnd.h"

/* F(0.05, 29, infinity)   */
#define F_VAL 1.47

typedef struct treefmt {
   struct treefmt *left;
   struct treefmt *right;
   double x;
   } treefmt;

void putstx(pgm)
char *pgm;
   {
   fprintf(stderr,"Usage: %s\n",
      pgm);
   exit(1);
   } /* putstx */

treefmt *inittree()
   {
   treefmt *node;
   node = (treefmt *) malloc(sizeof(treefmt) * 2048);
   if (node == NULL)
      {
      fprintf(stderr,"contin: out of memory "
	 "allocating observation tree.\n");
      exit(1);
      } /* out of mem */
   node->left  = NULL;
   node->right = NULL;
   node->x = -999999999.0;
   return(node);
   } /* inittree */

void isrt(treefmt *currnode, treefmt *newnode)
   {
   if (newnode->x < currnode->x)
      {
      if (currnode->left != NULL)
	 {
         isrt(currnode->left, newnode);
	 return;
	 } /* if right */
      else
	 {
	 currnode->left  = newnode;
	 return;
	 } /* insert at leaf node */
      } /* newnode->x < currnode->x */
   if (newnode->x > currnode->x)
      {
      if (currnode->right != NULL)
	 {
         isrt(currnode->right, newnode);
	 return;
	 } /* if right */
      else
	 {
	 currnode->right  = newnode;
	 return;
	 } /* insert at leaf node */
      } /* newnode->x > currnode->x */
   fprintf(stderr,"isrt: duplicate key %f\n",
      newnode->x);
   exit(1);
   } /* isrt */

int traverse(treefmt *node, double *lst, int ofst)
   {
   int newofst;
   double *p;
   newofst = ofst;
   if (node->left != NULL)
      {
      newofst = traverse(node->left, lst, newofst);
      } /* if left */
   p = (double *) lst + newofst;
   *p = node->x;
   newofst++;
   if (node->right != NULL)
      {
      newofst = traverse(node->right, lst, newofst);
      } /* if right */
   return(newofst);
   } /* traverse */

void cleantree(treefmt *node)
   {
   if (node->left != NULL)
      {
      cleantree(node->left);
      } /* if left */
   if (node->right != NULL)
      {
      cleantree(node->right);
      } /* if right */
   node->left  = NULL;
   node->right = NULL;
   node->x = -1.0;
   } /* cleantree */

int main(argc,argv)
int argc;
char **argv;
   {
   int i;
   int ofst;
   int ktst;
   int size;
   char plusmsg[64];
   char minusmsg[64];
   char chimsg[64];
   char f_msg[64];
   treefmt *obstree;
   treefmt *newnode;
   treefmt *kplustree;
   treefmt *newkpnode;
   treefmt *kminustree;
   treefmt *newkmnode;
   double dlta;
   double popx;
   double *p,*q;
   double dblsz;
   double rootsz;
   double thirty;
   double mindiff;
   double maxdiff;
   double minks;
   double maxks;
   double kplus;
   double kminus;
   double lolmtks;
   double hilmtks;
   double lolmtchi;
   double hilmtchi;
   double expect;
   double diff;
   double sumchi;
   double sumx;
   double sumxsqd;
   double glbsumx;
   double glbsumxsqd;
   double meanvar;
   double totmean;
   double var_b;
   double var_w;
   double f_lmt;
   double f_stat;
   double *obslst;
   double *kpluslst;
   double *kminuslst;
   double *lclchi;
   double *glbchi;
   double *chisqlst;
   double *chisqlstptr;
   double *mulst;
   double *mulstptr;
   double *varlst;
   double *varlstptr;
   double combo(int n, int r);
   void chirange(int degf, double *lorange,
      double *hirange);
   double ksrange(int degf, double *lorange,
      double *hirange);
   size  = 1000;
   thirty = 30.0;
   dblsz = (double) size;
   rootsz = sqrt(dblsz);
   expect = dblsz / thirty;
   sd = (unsigned char *) rndinit();
   if (sd == NULL)
      {
      fprintf(stderr,"contin: out of memory "
	 "allocating sd.\n");
      exit(1);
      } /* out of mem */
   obstree    = (treefmt *) inittree();
   kplustree  = (treefmt *) inittree();
   kminustree = (treefmt *) inittree();
   obslst = (double *) malloc(2048 * sizeof(double));
   if (obslst == NULL)
      {
      fprintf(stderr,"contin: out of memory "
	 "allocating observation list.\n");
      exit(1);
      } /* out of memory */
   kpluslst = (double *) malloc(1024 * sizeof(double));
   if (kpluslst == NULL)
      {
      fprintf(stderr,"contin: out of memory "
	 "allocating K+ list.\n");
      exit(1);
      } /* out of memory */
   kminuslst = (double *) malloc(1024 * sizeof(double));
   if (kminuslst == NULL)
      {
      fprintf(stderr,"contin: out of memory "
	 "allocating K- list.\n");
      exit(1);
      } /* out of memory */
   lclchi = (double *) malloc(1024 * sizeof(double));
   if (lclchi == NULL)
      {
      fprintf(stderr,"contin: out of memory "
	 "allocating local chi-square histogram.\n");
      exit(1);
      } /* out of memory */
   glbchi = (double *) malloc(1024 * sizeof(double));
   if (glbchi == NULL)
      {
      fprintf(stderr,"contin: out of memory "
	 "allocating global chi-square histogram.\n");
      exit(1);
      } /* out of memory */
   chisqlst = (double *) malloc(2048 * sizeof(double));
   if (chisqlst == NULL)
      {
      fprintf(stderr,"contin: out of memory "
	 "allocating chi-square list.\n");
      exit(1);
      } /* out of memory */
   mulst = (double *) malloc(1024 * sizeof(double));
   if (mulst == NULL)
      {
      fprintf(stderr,"contin: out of memory "
	 "allocating mean list.\n");
      exit(1);
      } /* out of memory */
   varlst = (double *) malloc(1024 * sizeof(double));
   if (varlst == NULL)
      {
      fprintf(stderr,"contin: out of memory "
	 "allocating variance list.\n");
      exit(1);
      } /* out of memory */
   newkpnode = (treefmt *) kplustree  + sizeof(treefmt);
   newkmnode = (treefmt *) kminustree + sizeof(treefmt);
   ksrange (30, &lolmtks,  &hilmtks);
   chirange(29, &lolmtchi, &hilmtchi);
   f_lmt  = F_VAL;     /* F(0.05, 29, infinity)  */
   printf("                    Lo "
      "                    Hi\n");
   printf("K+ K- Range  %15.6f %22.6f\n",
      lolmtks, hilmtks);
   printf("Chisq Range  %15.6f %22.6f\n",
      lolmtchi, hilmtchi);
   printf("f Statistic <=    %10.6f\n", f_lmt);
   printf("* = Failed\n");
   printf("\n");
   printf("             K+ "
      "          K- "
      "        Chi-square\n");
   p = (double *) glbchi;
   q = (double *) p + 1000;
   while (p < q) *p++ = 0.0;
   chisqlstptr = (double *) chisqlst;
   mulstptr = (double *) mulst;
   varlstptr = (double *) varlst;
   glbsumx = glbsumxsqd = 0.0;
   ktst = 0;
   while (ktst < 30)
      {
      ktst++;
      cleantree(obstree);
      newnode = (treefmt *) obstree + sizeof(treefmt);
      p = (double *) lclchi;
      q = (double *) p + 30;
      while (p < q) *p++ = 0.0;
      sumx = sumxsqd = 0.0;
      i = 1000;
      while (i--)
         {
         newnode->x     = rndfrac(sd);
	 /***********************************************************/
	 /* These distributions cause the test to fail. */
	 /* sine wave from 0 to 1 */
         /* newnode->x     = sin(rndfrac(sd) * M_PI_2); */
         /* newnode->x     = cos(rndfrac(sd) * M_PI_2); */
	 /* logarithmic curve from 0 to 1 */
         /* newnode->x     = log(rndfrac(sd) * (M_E-1.0) + 1.0); */
         /* newnode->x     = sqrt(rndfrac(sd)); */
	 /* exponential curve from 0 to 1 */
         /* newnode->x     = exp(-rndfrac(sd) * 2.0); */
         /* newnode->x     = rndfrac(sd) * rndfrac(sd); */
	 /***********************************************************/
         newnode->left  = NULL;
         newnode->right = NULL;
         isrt(obstree,newnode);
	 glbsumx    += newnode->x;
	 glbsumxsqd += (newnode->x * newnode->x);
	 sumx       += newnode->x;
	 sumxsqd    += (newnode->x * newnode->x);
	 ofst = (int) floor(newnode->x * thirty);
	 p = (double *) lclchi + ofst;
	 *p += 1.0;
	 p = (double *) glbchi + ofst;
	 *p += 1.0;
         newnode++;
         } /* for each observation */
      i = traverse(obstree->right,obslst,0);
      dlta = 1.0 / dblsz;
      popx = dlta;
      minks = -999999999.0;
      maxks = -999999999.0;
      p = (double *) obslst;
      q = (double *) p + size;
      while (p < q)
         {
         maxdiff = popx - *p;
         mindiff = *p - (popx - dlta);
         if (mindiff > minks) minks = mindiff;
         if (maxdiff > maxks) maxks = maxdiff;
         popx += dlta;
         p++;
         } /* for each observation */
      kplus  = rootsz * maxks;
      kminus = rootsz * minks;
      if (kplus < lolmtks || kplus > hilmtks)
         strcpy(plusmsg,"*");
      else
         strcpy(plusmsg," ");
      if (kminus < lolmtks || kminus > hilmtks)
         strcpy(minusmsg,"*");
      else
         strcpy(minusmsg," ");
      sumchi = 0.0;
      p = (double *) lclchi;
      q = (double *) p + 30;
      while (p < q)
	 {
	 diff = *p - expect;
	 sumchi += (diff * diff / expect);
	 p++;
	 } /* for each cell in histogram */
      if (sumchi < lolmtchi || sumchi > hilmtchi)
         strcpy(chimsg,"*");
      else
         strcpy(chimsg," ");
      printf("Test %3d %10.6f %s "
	 "%10.6f %s %12.6f %s\n",
         ktst, kplus, plusmsg, kminus, minusmsg,
	 sumchi, chimsg);
      newkpnode->x     = kplus;
      newkpnode->left  = NULL;
      newkpnode->right = NULL;
      isrt(kplustree,newkpnode);
      newkpnode++;
      newkmnode->x     = kminus;
      newkmnode->left  = NULL;
      newkmnode->right = NULL;
      isrt(kminustree,newkmnode);
      *chisqlstptr = sumchi;
      *mulstptr    = sumx / dblsz;
      *varlstptr   = (sumxsqd - (sumx * sumx / dblsz))
	 / (dblsz - 1.0);
      newkmnode++;
      chisqlstptr++;
      mulstptr++;
      varlstptr++;
      } /* for each test */
   printf("\n");
   i = traverse(kplustree->right,kpluslst,0);
   i = traverse(kminustree->right,kminuslst,0);
   /* the n=infinity range for the KS statistic */ 
   dlta = (1.2239 - .1601) / thirty;

   /* K+ */
   popx = dlta;
   popx = 0.1601;
   minks = -999999999.0;
   maxks = -999999999.0;
   p = (double *) kpluslst;
   q = (double *) p + 30;
   while (p < q)
      {
      maxdiff = popx - *p;
      mindiff = *p - (popx - dlta);
      if (mindiff > minks) minks = mindiff;
      if (maxdiff > maxks) maxks = maxdiff;
      popx += dlta;
      p++;
      } /* for each K+ test */
   kplus  = sqrt(30) * maxks;
   kminus = sqrt(30) * minks;
   if (kplus < lolmtks || kplus > hilmtks)
      strcpy(plusmsg,"Failed");
   else
      strcpy(plusmsg,"Passed");
   if (kminus < lolmtks || kminus > hilmtks)
      strcpy(minusmsg,"Failed");
   else
      strcpy(minusmsg,"Passed");
   printf("K+  %15.6f %s %15.6f %s\n",
      kplus, plusmsg, kminus, minusmsg);

   /* K- */
   popx = dlta;
   popx = 0.1601;
   minks = -999999999.0;
   maxks = -999999999.0;
   p = (double *) kminuslst;
   q = (double *) p + 30;
   while (p < q)
      {
      maxdiff = popx - *p;
      mindiff = *p - (popx - dlta);
      if (mindiff > minks) minks = mindiff;
      if (maxdiff > maxks) maxks = maxdiff;
      popx += dlta;
      p++;
      } /* for each K- test */
   kplus  = sqrt(30) * maxks;
   kminus = sqrt(30) * minks;
   if (kplus < lolmtks || kplus > hilmtks)
      strcpy(plusmsg,"Failed");
   else
      strcpy(plusmsg,"Passed");
   if (kminus < lolmtks || kminus > hilmtks)
      strcpy(minusmsg,"Failed");
   else
      strcpy(minusmsg,"Passed");
   printf("K-  %15.6f %s %15.6f %s\n",
      kplus, plusmsg, kminus, minusmsg);
   printf("\n");
   sumchi = 0.0;
   expect = 1000.0;
   p = (double *) glbchi;
   q = (double *) p + 30;
   while (p < q)
      {
      diff = *p - expect;
      sumchi += (diff * diff / expect);
      p++;
      } /* for each cell in histogram */
   if (sumchi < lolmtchi || sumchi > hilmtchi)
      strcpy(chimsg,"Failed");
   else
      strcpy(chimsg,"Passed");
   printf("Global Chi-square %10.6f %s\n", sumchi, chimsg);
   totmean = 0.0;
   mulstptr = (double *) mulst;
   i = 30;
   while (i--) totmean += *mulstptr++;
   totmean /= thirty;
   meanvar = 0.0;
   mulstptr = (double *) mulst;
   i = 30;
   while (i--)
      {
      diff = *mulstptr - totmean;
      meanvar += (diff * diff);
      } /* for each test mean */
   meanvar /= (thirty - 1.0);
   var_b = dblsz * meanvar;
   var_w = 0.0;
   varlstptr = (double *) varlst;
   i = 30;
   while (i--) var_w += *varlstptr++;
   var_w /= thirty;
   f_stat = var_b / var_w;
   if (f_stat > f_lmt) strcpy(f_msg,"Failed");
   else strcpy(f_msg,"Passed");
   printf("f Statistic       %10.6f %s\n", f_stat, f_msg);
   return(0);
   } /* main */
