/*
greg2jd.c Release 1.7.0  Gregorian to Julian Day Conversion
Copyright (C) 2004-2010   dondalah721@yahoo.com (Dondalah)

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to:

	Free Software Foundation, Inc.
	59 Temple Place - Suite 330
	Boston, MA  02111-1307, USA.
*/

#include <math.h>

/* This program is based on Peter Duffett-Smith */
/* Practical Astronomy with Your Calculator */
/* Third Edition */
/* Cambridge University Press */
/* ISBN 0 521 35629 6 hardback */
/* ISBN 0 521 35699 7 paperback */
/* Chapter 4: Julian Day Numbers */

/* In this implementation, */
/* dates prior to the Christian Era are true */
/* Julian dates.  Year 0 is treated as an error. */
/* Step 4 in the book has been modified */
/* to allow for true dates. */
/* December 31, -1 is immediatedly followed by */
/* January 1, 1. */

/* Oct 5, 1582 through Oct 14, 1582 are treated */
/* as errors.  October 4, 1582 is immediately */
/* followed by October 15, 1582. */
/* The return value is -1.0 if the date is */
/* invalid. */

/* Notice that the integer routine described */
/* in the book has been implemented as astrint() */
/* astrint(-3.914) = -3.0 */
/* astrint() gives the same result as the integer */
/* result from the modf() function. */

double greg2jd(int year,int month,
   int day,double hour)
   {
   int  tmpyear;
   int  tmpmonth;
   int  gregsw;
   int  flda;
   int  fldb;
   long fldc;
   int  fldd;
   double jd;
   double astrint(double num);
   if (year == 0)
      return(-1.0);
   if (month == 1 || month == 2)
      {
      tmpyear = year - 1;
      tmpmonth = month + 12;
      } /* adjust for Jan or Feb */
   else
      {
      tmpyear = year;
      tmpmonth = month;
      } /* else Mar through Dec */
   gregsw = 0;
   if (year > 1582) gregsw = 1;
   else if (year == 1582)
      {
      if (month > 10) gregsw = 1;
      else if (month == 10)
         {
         if (day >= 15) gregsw = 1;
         else if (day > 4 && day < 15)
            return(-1.0);
         } /* if Oct 1582 */
      } /* if 1582 */
   if (gregsw)
      {
      flda = tmpyear / 100;  /* century */
      fldb = 2 - flda + (flda / 4);
      } /* if Gregorian era */
   else fldb = 0;
   if (tmpyear < 0)
      {
      fldc = (long) astrint((365.25
         * (tmpyear + 1)) - 0.75);
      } /* if before Christian era */
   else
      {
      fldc = (long) astrint(365.25 * tmpyear);
      } /* else during Christian era */
   fldd = (int) astrint(30.6001 * (tmpmonth + 1.0));
   jd = (double) fldb + fldc + fldd
      + day + (hour / 24.0) + 1720994.5;
   return(jd);
   } /* greg2jd */
