/*
twobnm.c Version 1.4.0 - Binomial Test with
datagen and ran2
Copyright (C) 2005  dondalah@ripco.com (Dondalah)

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to:

	Free Software Foundation, Inc.
	59 Temple Place - Suite 330
	Boston, MA  02111-1307, USA.
*/

/* This test is based on: */
/* Lincoln L. Chao */
/* Statistics for Management */
/* Palo Alto, CA: The Scientific Press, 1984 */
/* Chapter 7, Discrete Probability Distributions */
/* Section 2, Binomial Distribution */

/* Lincoln L. Chao was at California State University, */
/* Long Beach, CA, while writing this book. */

/* The ran2 random number generator is taken from */
/* Numerical Recipes in C */
/* Chapter 7.1 */

/* twobnm compares the random number generator */
/* in datagen to the generator called ran2  */
/* for the binomial distribution. */
/* A mean and variance is created with each */
/* generator, and the difference is compared */
/* to zero. */

/* ran2 corresponds to rndfrac in datagen. */
/* It generates a random fraction between 0 and 1. */

/* Sample test: */

/* twobnm 100000 12 0.6 */

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include "rnd.h"

#define CONF 95.0
#define ZLMT 1.96
#define PCT "95.0% Confidence"

void putstx(pgm)
char *pgm;
   {
   fprintf(stderr,"Usage: %s size "
      "n_trials p_success\n",
      pgm);
   fprintf(stderr,"Where size is 1000 to 1000000000\n");
   fprintf(stderr,"Where n_trials is 2 to 30\n");
   fprintf(stderr,"Where p_success is 0.1 to 0.9\n");
   fprintf(stderr,"Example: %s 10000 12 0.6\n",
      pgm);
   fprintf(stderr,"n_trials is the number "
      "of simple random events for each binomial event\n");
   fprintf(stderr,"p_success is the probability "
      "of success for each simple random event\n");
   fprintf(stderr,"Size is more than 100 "
      "times larger than n_trials\n");
   fprintf(stderr,"Flipping a coin is an example "
      "of a simple random event with a\n");
   fprintf(stderr,"probability for success of 0.5\n");
   exit(1);
   } /* putstx */

int main(argc,argv)
int argc;
char **argv;
   {
   int i,j;
   int size,nobserv;
   int successes;
   int minone;
   int maxone;
   int mintwo;
   int maxtwo;
   int idum;
   int *wone;
   int *woneptr;
   int *woneend;
   int *wtwo;
   int *wtwoptr;
   int *wtwoend;
   char failmsg[64];
   char f_rslt[64];
   char critmsg[64];
   double psuccess;
   double qfail;
   double bcoeff;    /* binomial coefficient */
   double *p,*q;
   double *prob;
   double *probptr;
   double totprob;
   double success,fail;
   double mu;
   double stdev;
   double variance;
   double muone;
   double mutwo;
   double musq;
   double popmu;
   double popvar;
   double popstdev;
   double zscoreone;
   double zscoretwo;
   double est_xsq;
   double expect;
   double varone;
   double stdevone;
   double vartwo;
   double stdevtwo;
   double meandiff;
   double stdevdiff;
   double lolmt;
   double hilmt;
   double wval;
   double dblsuccess;
   double dbli;
   double dbln;
   double dblz;
   double diff;
   double sumchione;
   double sumchitwo;
   double dblsize;
   double sumxone,sumxonesqd;
   double sumxtwo,sumxtwosqd;
   double negtblv,postblv;
   double fstat;
   double fval;
   double combo(int n, int r);
   void chirange(int degf, double *lorange,
      double *hirange);
   double ran2(int *idum);
   double getftbl(int degf);
   if (argc != 4) putstx(*argv);
   size = atoi(*(argv+1));
   if (size < 1000 || size > 1000000000)
      {
      fprintf(stderr,"Invalid size.\n");
      putstx(*argv);
      } /* bad size */
   dblsize = (double) size;
   nobserv = atoi(*(argv+2));
   if (nobserv < 2)
      {
      fprintf(stderr,"Number of trials "
	 "is too small.\n");
      putstx(*argv);
      } /* not enough nobserv */
   if (nobserv > 30)
      {
      fprintf(stderr,"Number of trials "
	 "is too large.\n");
      putstx(*argv);
      } /* if nobserv too large */
   if (size < nobserv * 100)
      {
      fprintf(stderr,"Size is too small.\n");
      putstx(*argv);
      } /* if size too small */
   dbln = (double) nobserv;
   psuccess = (double) atof(*(argv+3));
   if (psuccess < 0.1)
      {
      fprintf(stderr,"Probability of success "
	 "is too small or negative.\n");
      putstx(*argv);
      } /* psuccess too small or negative */
   if (psuccess > 0.9)
      {
      fprintf(stderr,"Probability of success "
	 "is too large.\n");
      putstx(*argv);
      } /* if psuccess too large */
   qfail = 1.0 - psuccess;
   sd = (unsigned char *) rndinit();
   if (sd == NULL)
      {
      fprintf(stderr,"twobnm: out of memory "
	 "allocating sd.\n");
      exit(1);
      } /* out of mem */
   /* Allocate array for the */
   /* Bernoulli random variable W */
   wone = (int *) malloc((nobserv + 1) * sizeof(int) + 32);
   if (wone == NULL)
      {
      fprintf(stderr,"twobnm: out of memory "
	 "allocating datagen observation array.\n");
      exit(1);
      } /* out of memory */
   wtwo = (int *) malloc((nobserv + 1) * sizeof(int) + 32);
   if (wtwo == NULL)
      {
      fprintf(stderr,"twobnm: out of memory "
	 "allocating ran2 observation array.\n");
      exit(1);
      } /* out of memory */
   prob = (double *) malloc((nobserv + 1) * sizeof(double) + 32);
   if (prob == NULL)
      {
      fprintf(stderr,"twobnm: out of memory "
	 "allocating probability array.\n");
      exit(1);
      } /* out of memory */
   /* Initialize ran2 shuffle table. */
   /* ran2 has 2 billion initial seeds. */
   idum = (int) - rndpwr(31,sd);
   mu = dbln * psuccess;
   stdev = sqrt(mu * qfail);
   printf("Binomial Probability Distribution\n");
   printf("x = Number of successes "
      "after making %d trials\n",
      nobserv);
   printf(" x    P(x) "
      "                  "
      "Total P(x)             z\n");
   totprob = 0.0;
   p = (double *) prob;
   q = (double *) p + nobserv + 1;
   i = 0;
   while (p < q)
      {
      bcoeff  = combo(nobserv,i);
      success = pow(psuccess,(double) i);
      fail    = pow(qfail,(double) (nobserv - i));
      *p = bcoeff * success * fail;
      totprob += *p;
      dblz = ((double) i - mu) /stdev;
      printf("%2d %22.17f %22.17f %9.4f\n",
	 i, *p, totprob, dblz);
      i++;
      p++;
      } /* for each # of heads */
   printf("\n");
   mu = 0.0;
   p = (double *) prob;
   q = (double *) p + nobserv + 1;
   i = 0;
   while (p < q)
      {
      mu += ((double) i * *p++);
      i++;
      } /* for each # of trials */
   musq = mu * mu;
   est_xsq = 0.0;
   p = (double *) prob;
   q = (double *) p + nobserv + 1;
   i = 0;
   while (p < q)
      {
      est_xsq += ((double) (i * i) * *p++);
      i++;
      } /* for each # of trials */
   variance = est_xsq -  musq;
   stdev    = sqrt(variance);
   printf("             Mean       "
      "Variance   Stdev      Z Score "
      "     Min     Max\n");
   printf("Population %10.6f %10.6f %10.6f\n",
      mu, variance, stdev);
   popmu    = mu;
   popvar   = variance;
   popstdev = stdev;
   woneptr = (int *) wone;
   woneend = (int *) woneptr + nobserv + 1;
   while (woneptr < woneend) *woneptr++ = 0;
   wtwoptr = (int *) wtwo;
   wtwoend = (int *) wtwoptr + nobserv + 1;
   while (wtwoptr < wtwoend) *wtwoptr++ = 0;
   sumxone = sumxonesqd = 0.0;
   sumxtwo = sumxtwosqd = 0.0;
   minone =  999999999;
   maxone = -999999999;
   mintwo =  999999999;
   maxtwo = -999999999;
   i = size;
   while (i--)
      {
      successes = 0;
      j = nobserv;
      while (j--)
	 {
	 if (rndfrac(sd) < psuccess)
	    successes++;
	 } /* for each simple random event */
      woneptr = wone + successes;
      (*woneptr)++;
      dblsuccess = (double) successes;
      sumxone    += dblsuccess;
      sumxonesqd += (dblsuccess * dblsuccess);
      if (successes < minone) minone = successes;
      if (successes > maxone) maxone = successes;
      successes = 0;
      j = nobserv;
      while (j--)
	 {
	 if (ran2(&idum) < psuccess)
	    successes++;
	 } /* for each simple random event */
      wtwoptr = wtwo + successes;
      (*wtwoptr)++;
      dblsuccess = (double) successes;
      sumxtwo    += dblsuccess;
      sumxtwosqd += (dblsuccess * dblsuccess);
      if (successes < mintwo) mintwo = successes;
      if (successes > maxtwo) maxtwo = successes;
      } /* generate histogram */
   varone = (sumxonesqd - (sumxone * sumxone / dblsize))
      / (dblsize - 1.0);
   stdevone = sqrt(varone);
   sumchione = muone = 0.0;
   i = 0;
   woneptr = (int *) wone;
   woneend = (int *) woneptr + nobserv + 1;
   probptr = (double *) prob;
   while (woneptr < woneend)
      {
      wval = (double) *woneptr++;
      dbli = (double) i++;
      muone += (wval * dbli / dblsize);
      expect = (*probptr++) * dblsize;
      diff = wval - expect;
      sumchione += (diff * diff / expect);
      } /* sum chi-sq */
   zscoreone = (muone - popmu) / (popstdev / sqrt(dblsize));
   printf("Datagen    %10.6f %10.6f %10.6f %10.6f %7d %7d\n",
      muone, varone, sqrt(varone),
      zscoreone, minone, maxone);
   vartwo = (sumxtwosqd - (sumxtwo * sumxtwo / dblsize))
      / (dblsize - 1.0);
   stdevtwo = sqrt(vartwo);
   sumchitwo = mutwo = 0.0;
   i = 0;
   wtwoptr = (int *) wtwo;
   wtwoend = (int *) wtwoptr + nobserv + 1;
   probptr = (double *) prob;
   while (wtwoptr < wtwoend)
      {
      wval = (double) *wtwoptr++;
      dbli = (double) i++;
      mutwo += (wval * dbli / dblsize);
      expect = (*probptr++) * dblsize;
      diff = wval - expect;
      sumchitwo += (diff * diff / expect);
      } /* sum chi-sq */
   zscoretwo = (mutwo - popmu) / (popstdev / sqrt(dblsize));
   printf("Ran2       %10.6f %10.6f %10.6f %10.6f %7d %7d\n",
      mutwo, vartwo, sqrt(vartwo),
      zscoretwo, mintwo, maxtwo);
   printf("\n");
   meandiff = muone - mutwo;
   stdevdiff = sqrt((varone / dblsize) + (vartwo / dblsize));
   lolmt = meandiff - (ZLMT * stdevdiff);
   hilmt = meandiff + (ZLMT * stdevdiff);
   if (lolmt > 0.0 || hilmt < 0.0)
      strcpy(failmsg,"Test Failed");
   else
      strcpy(failmsg,"Test Passed");
   printf("             Mean       "
      "Low Lmt    High Lmt   %s\n", PCT);
   printf("Difference %10.6f %10.6f %10.6f   %s\n",
      meandiff, lolmt, hilmt, failmsg);
   printf("\n");
   printf("Datagen chi-square %12.6f\n", sumchione);
   printf("Ran2    chi-square %12.6f\n", sumchitwo);
   chirange(nobserv,&negtblv,&postblv);
   printf("Range at 95 percent: %f  to  %f\n",
      negtblv, postblv);
   printf("\n");
   if (varone > vartwo)
      {
      if (vartwo != 0.0)
         fstat = varone / vartwo;
      else
	 fstat = 1.0;
      } /* if var1 > var2 */
   else
      {
      if (varone != 0.0)
         fstat = vartwo / varone;
      else
	 fstat = 1.0;
      } /* if var2 > var1 */
   fval = getftbl(nobserv-1);
   if (fstat > fval)
      strcpy(f_rslt,"Test Failed");
   else
      strcpy(f_rslt,"Test Passed");
   strcpy(critmsg,"Critical Value    2 tail test, 95%");
   printf("     f Statistic   %s\n",
      critmsg);
   printf("%15.6f %15.6f      %s\n",
      fstat, fval, f_rslt);
   return(0);
   } /* main */
