/*
meandf.c Version 1.4.0 - Difference between 2 means
Copyright (C) 2005  dondalah@ripco.com (Dondalah)

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to:

	Free Software Foundation, Inc.
	59 Temple Place - Suite 330
	Boston, MA  02111-1307, USA.
*/

/* This test is based on: */
/* Lincoln L. Chao */
/* Statistics for Management */
/* Palo Alto, CA: The Scientific Press, 1984 */
/* Chapter 9, Statistical Estimation */
/* Section 3, Interval Estimation About Proportions */

/* Lincoln L. Chao was at California State University, */
/* Long Beach, CA, while writing this book. */

/* Sample test: */

/* meandf -c 95.0 -s 10000 -f -n 100 -p 0.5 */

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include "rnd.h"

void putstx(pgm)
char *pgm;
   {
   fprintf(stderr,"Usage: %s options\n",
      pgm);
   fprintf(stderr,"Options:\n");
   fprintf(stderr,"-f = generate fractions\n");
   fprintf(stderr,"-n ddddd = generate integers\n");
   fprintf(stderr,"   where ddddd = 2 - 1000000000\n");
   fprintf(stderr,"-b dd = generate dd bit numbers\n");
   fprintf(stderr,"   where dd = 1 - 30\n");
   fprintf(stderr,"-c cc.c = confidence level\n");
   fprintf(stderr,"   where cc.c is 90.0, 95.0, "
      "98.0, or 99.0\n");
   fprintf(stderr,"   default = 95.0\n");
   fprintf(stderr,"-s zzzzzz = size of test\n");
   fprintf(stderr,"   where zzzzzz is 30 to 1000000000\n");
   fprintf(stderr,"   default = 10000\n");
   fprintf(stderr,"-p n.n = probability of success\n");
   fprintf(stderr,"   where n.n = 0.001 - 0.999\n");
   fprintf(stderr,"   default = 0.5\n");
   fprintf(stderr,"Only two options (-f, -n, or -b) "
      "may be selected\n");
   exit(1);
   } /* putstx */

int main(argc,argv)
int argc;
char **argv;
   {
   int i;
   int size;
   int rslt;
   int bittop;
   int bitbot;
   int numtop;
   int numbot;
   int fracsw;
   int numsw;
   int bitsw;
   int totmean;
   int tstnum;
   int tstbit;
   int obs_1;
   int obs_2;
   int popnum;
   int popbit;
   unsigned char *sd1;
   unsigned char *sd2;
   unsigned char *sdptr1;
   unsigned char *sdend1;
   unsigned char *sdptr2;
   char title_1[128];
   char title_2[128];
   char title_diff[128];
   double confidence;
   double zval;
   double tstprob;
   double bitprob;
   double numprob;
   double dbltstnum;
   double dbltstbit;
   double dblobs_1;
   double dblobs_2;
   double prob_1;
   double prob_2;
   double q_1;
   double q_2;
   double stderr_1;
   double stderr_2;
   double quot_1;
   double quot_2;
   double lolmt_1;
   double lolmt_2;
   double hilmt_1;
   double hilmt_2;
   double stderr_diff;
   double popmean;
   double lolmt_diff;
   double hilmt_diff;
   double diff;
   double dblsz;
   fracsw = numsw = bitsw = totmean = 0;
   size    = 10000;
   dblsz   = (double) size;
   confidence = 95.0;
   zval    = 1.96;
   tstprob = 0.5;
   tstnum  = 1000000;
   tstbit  = 20;
   bitprob = 0.5;
   i = 1;
   while (i < argc)
      {
      rslt = strcmp(*(argv+i),"-f");
      if (!rslt)
	 {
	 fracsw = 1;
	 totmean++;
	 if (totmean > 2)
	    {
	    fprintf(stderr,"Too many options\n");
	    putstx(*argv);
	    } /* too many options */
	 i++;
	 continue;
	 } /* if fraction */
      rslt = strcmp(*(argv+i),"-n");
      if (!rslt)
	 {
	 numsw = 1;
	 totmean++;
	 if (totmean > 2)
	    {
	    fprintf(stderr,"Too many options\n");
	    putstx(*argv);
	    } /* too many options */
	 i++;
	 if (i >= argc)
	    {
	    fprintf(stderr,"Missing number "
	       "for option -n\n");
	    putstx(*argv);
	    } /* if out of arguments */
         tstnum = atoi(*(argv+i));
         if (tstnum < 2 || tstnum > 1000000000)
            {
            fprintf(stderr,"Invalid number %s.\n",
	       *(argv+i));
            putstx(*argv);
            } /* bad number */
	 i++;
	 continue;
	 } /* if integer */
      rslt = strcmp(*(argv+i),"-b");
      if (!rslt)
	 {
	 bitsw = 1;
	 totmean++;
	 if (totmean > 2)
	    {
	    fprintf(stderr,"Too many options\n");
	    putstx(*argv);
	    } /* too many options */
	 i++;
	 if (i >= argc)
	    {
	    fprintf(stderr,"Missing bits "
	       "for option -b\n");
	    putstx(*argv);
	    } /* if out of arguments */
         tstbit = atoi(*(argv+i));
         if (tstbit < 1 || tstbit > 30)
            {
            fprintf(stderr,"Invalid bits %s.\n",
	       *(argv+i));
            putstx(*argv);
            } /* bad bits */
	 i++;
	 continue;
	 } /* if bit string */
      rslt = strcmp(*(argv+i),"-s");
      if (!rslt)
	 {
	 i++;
	 if (i >= argc)
	    {
	    fprintf(stderr,"Missing size\n");
	    putstx(*argv);
	    } /* if out of arguments */
         size = atoi(*(argv+i));
         if (size < 30 || size > 1000000000)
            {
            fprintf(stderr,"Invalid size.\n");
            putstx(*argv);
            } /* bad size */
         dblsz = (double) size;
	 i++;
	 continue;
	 } /* if size */
      rslt = strcmp(*(argv+i),"-c");
      if (!rslt)
	 {
	 i++;
	 if (i >= argc)
	    {
	    fprintf(stderr,"Missing confidence level\n");
	    putstx(*argv);
	    } /* if out of arguments */
         confidence = atof(*(argv+i));
         if (confidence == 90.0)
	    zval = 1.645;
         else if (confidence == 95.0)
	    zval = 1.96;
         else if (confidence == 98.0)
	    zval = 2.326;
         else if (confidence == 99.0)
	    zval = 2.575;
	 else
            {
            fprintf(stderr,"Invalid confidence level.\n");
            putstx(*argv);
            } /* bad confidence */
	 i++;
	 continue;
	 } /* if confidence level */
      rslt = strcmp(*(argv+i),"-p");
      if (!rslt)
	 {
	 i++;
	 if (i >= argc)
	    {
	    fprintf(stderr,"Missing probability\n");
	    putstx(*argv);
	    } /* if out of arguments */
         tstprob = atof(*(argv+i));
         if (tstprob < 0.001 || tstprob > 0.999)
            {
            fprintf(stderr,"Invalid probability.\n");
            putstx(*argv);
            } /* bad tstprob */
	 i++;
	 continue;
	 } /* if probability */
      fprintf(stderr,"Invalid option %s.\n",
	 *(argv+i));
      putstx(*argv);
      } /* for each argument */
   if (totmean < 2)
      {
      fprintf(stderr,"Too few options\n");
      putstx(*argv);
      } /* too few options */
   if (numsw)
      {
      if (tstnum < 2)
         {
         fprintf(stderr,"Missing number "
	    "for option -n\n");
         putstx(*argv);
         } /* too few options */
      numbot = tstnum;
      numtop = (int) floor(((double) tstnum * tstprob) + 0.5);
      numprob = (double) numtop / (double) numbot;
      if (fabs(tstprob-numprob) > 0.00001)
         {
         fprintf(stderr,"Warning: probability "
            "is not n / %d\n", tstnum);
         fprintf(stderr,"where n is 1 to %d\n\n",
	    tstnum - 1);
         } /* probability not a rational number */
      } /* if -n option */
   if (bitsw)
      {
      if (tstbit < 1)
         {
         fprintf(stderr,"Missing number "
	    "for option -b\n");
         putstx(*argv);
         } /* too few options */
      bitbot = 1 << tstbit;
      i = 0;
      while (i < tstbit)
	 {
	 bittop = 1 << i;
         bitprob = (double) bittop / (double) bitbot;
         if (bitprob >= tstprob) break;
	 i++;
	 } /* for each bit fraction */
      if (tstprob != bitprob)
         {
         fprintf(stderr,"Warning: probability "
            "is not 1 / (2^b)\n");
         fprintf(stderr,"where b is 1 to %d\n\n",
	    tstbit);
         } /* probability not 1 / (2^n) */
      } /* if -b option */
   popmean = tstprob * dblsz;
   if (popmean < 30.0)
      {
      fprintf(stderr,"Size is too small.\n");
      putstx(*argv);
      } /* if size too small */
   sd1 = (unsigned char *) rndinit();
   if (sd1 == NULL)
      {
      fprintf(stderr,"meandf: out of memory "
	 "allocating sd one.\n");
      exit(1);
      } /* out of mem */
   sd2 = (unsigned char *) rndinit();
   if (sd2 == NULL)
      {
      fprintf(stderr,"meandf: out of memory "
	 "allocating sd two.\n");
      exit(1);
      } /* out of mem */
   /* Copy sd1 to sd2. */
   /* Both sd1 and sd2 are initialized */
   /* to the same value. */
   sdptr1 = (unsigned char *) sd1;
   sdend1 = (unsigned char *) sdptr1 + 16;
   sdptr2 = (unsigned char *) sd2;
   while (sdptr1 < sdend1)
      *sdptr2++ = *sdptr1++;
   *sdptr1 = '\0';
   *sdptr2 = '\0';
   obs_1 = obs_2 = 0;
   dbltstnum = (double) tstnum;
   dbltstbit = (double) (1 << tstbit);
   popnum = (int) floor((dbltstnum * tstprob) + 0.5);
   popbit = (int) floor((dbltstbit * tstprob) + 0.5);
   i = size;
   while (i--)
      {
      if (fracsw)
	 {
	 if (rndfrac(sd1) < tstprob) obs_1++;
	 if (numsw)
	    {
	    if (rndnum(tstnum,sd2) < popnum) obs_2++;
	    } /* if number */
	 else if (bitsw)
	    {
	    if (rndpwr(tstbit,sd2) < popbit) obs_2++;
	    } /* if bit */
	 else
	    {
	    fprintf(stderr,"meandf: logic error 001\n");
	    exit(1);
	    } /* else invalid option */
	 } /* if fraction */
      else
	 {
	 if (rndnum(tstnum,sd1) < popnum) obs_1++;
	 if (rndpwr(tstbit,sd2) < popbit) obs_2++;
	 } /* else number and bit */
      } /* generate observations */
   dblobs_1 = (double) obs_1;
   dblobs_2 = (double) obs_2;
   prob_1 = dblobs_1 / dblsz;
   prob_2 = dblobs_2 / dblsz;
   q_1    = 1.0 - prob_1;
   q_2    = 1.0 - prob_2;
   quot_1 = prob_1 * q_1 / dblsz;
   quot_2 = prob_2 * q_2 / dblsz;
   stderr_1 = sqrt(quot_1);
   stderr_2 = sqrt(quot_2);
   lolmt_1 = prob_1 - (zval * stderr_1);
   hilmt_1 = prob_1 + (zval * stderr_1);
   lolmt_2 = prob_2 - (zval * stderr_2);
   hilmt_2 = prob_2 + (zval * stderr_2);
   stderr_diff = sqrt(quot_1 + quot_2);
   diff = prob_1 - prob_2;
   lolmt_diff = diff - (zval * stderr_diff);
   hilmt_diff = diff + (zval * stderr_diff);
   if (fracsw)
      {
      strcpy(title_1,"Fraction   ");
      if (numsw)
         strcpy(title_2,"Number     ");
      else
         strcpy(title_2,"Bits       ");
      } /* if fraction */
   else
      {
      strcpy(title_1,"Number     ");
      strcpy(title_2,"Bits       ");
      } /* not fraction */
   strcpy(title_diff,"Difference ");

   if (lolmt_1 > tstprob)
      {
      printf("Test fails for Lower "
	 "Limit %s\n\n", title_1);
      }
   if (hilmt_1 < tstprob)
      {
      printf("Test fails for Upper "
	 "Limit %s\n\n", title_1);
      }
   if (lolmt_2 > tstprob)
      {
      printf("Test fails for Lower "
	 "Limit %s\n\n", title_2);
      }
   if (hilmt_2 < tstprob)
      {
      printf("Test fails for Upper "
	 "Limit %s\n\n", title_2);
      }
   if (lolmt_diff > 0.0)
      {
      printf("Test fails for Lower "
	 "Limit %s\n\n", title_diff);
      }
   if (hilmt_diff < 0.0)
      {
      printf("Test fails for Upper "
	 "Limit %s\n\n", title_diff);
      }

   printf("Difference between two mean proportions:\n\n");
   printf("    Probability "
      "    Confidence Level "
      "    Observations\n");
   printf("     %8.5f "
      "           %6.3f "
      "         %9.0f\n\n",
      tstprob, confidence, dblsz);
   printf("                Probability "
      "     Lower Limit "
      "     Upper Limit \n");

   printf("%s %16.11f %16.11f %16.11f\n",
      title_1, prob_1, lolmt_1, hilmt_1);
   printf("%s %16.11f %16.11f %16.11f\n",
      title_2, prob_2, lolmt_2, hilmt_2);

   printf("%s %16.11f %16.11f %16.11f\n",
      title_diff, diff, lolmt_diff, hilmt_diff);
   return(0);
   } /* main */
