/*
bnmerr.c Version 1.4.0 - Binomial based on error rate
Copyright (C) 2005  dondalah@ripco.com (Dondalah)

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to:

	Free Software Foundation, Inc.
	59 Temple Place - Suite 330
	Boston, MA  02111-1307, USA.
*/

/* This test is based on: */
/* Lincoln L. Chao */
/* Statistics for Management */
/* Palo Alto, CA: The Scientific Press, 1984 */
/* Chapter 7, Discrete Probability Distributions */
/* Section 2, Binomial Distribution */
/* Chapter 9, Statistical Estimation */
/* Section 4, Determination of Sample Size */

/* Lincoln L. Chao was at California State University, */
/* Long Beach, CA, while writing this book. */

/* Sample test: */

/* bnmerr 95.0 0.01 12 0.6 */

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include "rnd.h"

void putstx(pgm)
char *pgm;
   {
   fprintf(stderr,"Usage: %s confidence "
      "error_rate n_trials p_success\n",
      pgm);
   fprintf(stderr,"Where confidence is "
      "90.0, 95.0, 98.0, 99.0\n");
   fprintf(stderr,"      error_rate is 0.001 to 0.1\n");
   fprintf(stderr,"      n_trials is 2 to 30\n");
   fprintf(stderr,"      p_success is 0.1 to 0.9\n");
   fprintf(stderr,"Example: %s "
      "95.0 0.02 12 0.6\n",
      pgm);
   fprintf(stderr,"n_trials is the number "
      "of simple random events for each binomial event\n");
   fprintf(stderr,"p_success is the probability "
      "of success for each simple random event\n");
   exit(1);
   } /* putstx */

int main(argc,argv)
int argc;
char **argv;
   {
   int i,j;
   int size,nobserv;
   int successes;
   int min;
   int max;
   int *w;
   int *wptr;
   int *wend;
   double confidence;
   double errrate;
   double psuccess;
   double qfail;
   double bcoeff;    /* binomial coefficient */
   double *p,*q;
   double *prob;
   double *probptr;
   double totprob;
   double success,fail;
   double mu,musq;
   double est_xsq;
   double expect;
   double variance,stdev;
   double popmu;
   double popstdev;
   double dblsuccess;
   double dblz;
   double wval;
   double dbli;
   double diff;
   double zscore;
   double sumchi;
   double dblsize;
   double sumx,sumxsqd;
   double negtblv,postblv;
   int getn(double confidence, double errrate,
      double psuccess);
   double combo(int n, int r);
   void chirange(int degf, double *lorange,
      double *hirange);
   if (argc != 5) putstx(*argv);
   confidence = atof(*(argv+1));
   if (confidence == 90.0 || confidence == 95.0
      || confidence == 98.0 || confidence == 99.0);
   else
      {
      fprintf(stderr,"Invalid confidence level %s.\n",
	 *(argv+1));
      putstx(*argv);
      } /* bad confidence */
   errrate = atof(*(argv+2));
   if (errrate < 0.001 || errrate > 0.1)
      {
      fprintf(stderr,"Invalid error rate %s.\n",
	 *(argv+2));
      putstx(*argv);
      } /* bad error rate */
   nobserv = atoi(*(argv+3));
   if (nobserv < 2)
      {
      fprintf(stderr,"Number of observations "
	 "is too small.\n");
      putstx(*argv);
      } /* not enough nobserv */
   if (nobserv > 30)
      {
      fprintf(stderr,"Number of observations "
	 "is too large.\n");
      putstx(*argv);
      } /* if nobserv too large */
   psuccess = (double) atof(*(argv+4));
   if (psuccess < 0.1)
      {
      fprintf(stderr,"Probability of success "
	 "is too small or negative.\n");
      putstx(*argv);
      } /* psuccess too small or negative */
   if (psuccess > 0.9)
      {
      fprintf(stderr,"Probability of success "
	 "is too large.\n");
      putstx(*argv);
      } /* if psuccess too large */
   qfail = 1.0 - psuccess;
   size = getn(confidence, errrate, psuccess);
   dblsize = (double) size;
   sd = (unsigned char *) rndinit();
   if (sd == NULL)
      {
      fprintf(stderr,"bnmerr: out of memory "
	 "allocating sd.\n");
      exit(1);
      } /* out of mem */
   /* Allocate array for the */
   /* Bernoulli random variable W */
   w = (int *) malloc((nobserv + 1) * sizeof(int) + 32);
   if (w == NULL)
      {
      fprintf(stderr,"bnmerr: out of memory "
	 "allocating observation array.\n");
      exit(1);
      } /* out of memory */
   prob = (double *) malloc((nobserv + 1) * sizeof(double) + 32);
   if (prob == NULL)
      {
      fprintf(stderr,"bnmerr: out of memory "
	 "allocating probability array.\n");
      exit(1);
      } /* out of memory */
   mu = nobserv * psuccess;
   stdev = sqrt(mu * qfail);
   popmu = mu;
   popstdev = stdev;
   printf("Binomial Probability Distribution\n");
   printf("x = Number of successes "
      "after making %d trials\n",
      nobserv);
   printf(" x    P(x) "
      "                  "
      "Total P(x)             z\n");
   totprob = 0.0;
   p = (double *) prob;
   q = (double *) p + nobserv + 1;
   i = 0;
   while (p < q)
      {
      bcoeff  = combo(nobserv,i);
      success = pow(psuccess,(double) i);
      fail    = pow(qfail,(double) (nobserv - i));
      *p = bcoeff * success * fail;
      totprob += *p;
      dblz = ((double) i - mu) / stdev;
      printf("%2d %22.17f %22.17f %9.4f\n",
	 i, *p, totprob, dblz);
      i++;
      p++;
      } /* for each # of heads */
   printf("\n");
   mu = 0.0;
   p = (double *) prob;
   q = (double *) p + nobserv + 1;
   i = 0;
   while (p < q)
      {
      mu += ((double) i * *p++);
      i++;
      } /* for each # of observations */
   musq = mu * mu;
   est_xsq = 0.0;
   p = (double *) prob;
   q = (double *) p + nobserv + 1;
   i = 0;
   while (p < q)
      {
      est_xsq += ((double) (i * i) * *p++);
      i++;
      } /* for each # of observations */
   variance = est_xsq -  musq;
   stdev    = sqrt(variance);
   printf("  Confidence  Error Rate "
      "  Success Rate   Observations\n");
   printf("    %5.1f      %6.3f "
      "       %5.2f      %9d\n\n",
      confidence, errrate, psuccess, size);
   printf("             Mean       Variance   Stdev "
      "     Z Score     Min    Max\n");
   printf("Population %10.6f %10.6f %10.6f\n",
      mu, variance, stdev);
   wptr = (int *) w;
   wend = (int *) wptr + nobserv + 1;
   while (wptr < wend) *wptr++ = 0;
   min =  999999999;
   max = -999999999;
   sumx = sumxsqd = 0.0;
   i = size;
   while (i--)
      {
      successes = 0;
      j = nobserv;
      while (j--)
	 {
	 if (rndfrac(sd) < psuccess)
	    successes++;
	 } /* for each simple random event */
      wptr = w + successes;
      (*wptr)++;
      if (successes < min) min = successes;
      if (successes > max) max = successes;
      dblsuccess = (double) successes;
      sumx    += dblsuccess;
      sumxsqd += (dblsuccess * dblsuccess);
      } /* generate histogram */
   variance = (sumxsqd - (sumx * sumx / dblsize))
      / (dblsize - 1.0);
   stdev = sqrt(variance);
   sumchi = mu = 0.0;
   i = 0;
   wptr = (int *) w;
   wend = (int *) wptr + nobserv + 1;
   probptr = (double *) prob;
   while (wptr < wend)
      {
      wval = (double) *wptr++;
      dbli = (double) i++;
      mu += (wval * dbli / dblsize);
      expect = (*probptr++) * dblsize;
      diff = wval - expect;
      if (expect != 0.0)
         sumchi += (diff * diff / expect);
      else
	 {
	 fprintf(stderr,"bnmerr: logic error 001\n");
	 exit(1);
	 } /* divide by zero */
      } /* sum chi-sq */
   zscore = (mu - popmu) / (popstdev / sqrt(dblsize));
   printf("Observed   %10.6f %10.6f %10.6f %10.6f %6d %6d\n",
      mu, variance, sqrt(variance),
      zscore, min, max);
   printf("\n");
   printf("Chi-square %f\n", sumchi);
   chirange(nobserv,&negtblv,&postblv);
   printf("Range at 95 percent: %f  to  %f\n",
      negtblv, postblv);
   return(0);
   } /* main */
