# include <stdlib.h>
# include <stdio.h>
# include <string.h>

#define MAXWORDS 4096             /* distinct L_WORDs */

typedef struct {
        unsigned short  prev;
        unsigned short  next;
        unsigned short  prop;
        char           *word;
        } NODE;

static NODE *nodelist[MAXWORDS];  /* stash words - start at 1 */
static int   wordcount;


int          ukey(char *x, long n);
void         setup(void);
void         add_postops(int *ss, int sn);
void         add_inrels (int *ss, int sn);
void         add_prerels(int *ss, int sn);
void         add_pregens(int *ss, int sn);
void         add_inops  (int *ss, int sn, int p);
void         add_ignores(int *ss, int sn);
int          is_postop  (int x);
int          is_inrel   (int x);
int          is_prerel  (int x);
int          is_pregen  (int x);
int          is_inop    (int x);
int          is_ignore  (int x);

# define POSTOP         1
# define INREL          2
# define PREREL         4
# define INGEN          8
# define PREGEN        16
# define INOP0         32
# define INOP1         64
# define INOP2        128
# define INOP3        256
# define INOP4        512
# define INOP5       1024
# define INOP6       2048
# define INOP7       4096
# define IGNORE      8192

/* STANDARD PRELUDE 
 * %%inop \mapsto                                                  1
 * %%inop \upto                                                    2
 * %%inop + \cup \setminus \cat \uplus \uminus                     3
 * %%inop * \div \mod \cap \circ \comp \filter \extract \otimes    4
 * %%inop \oplus \bcount                                           5
 * %%inop \dres \rres \ndres \nrres                                6
 */
static char *inops[8][10] =
{
  {0,0,0,0,0,0,0,0,0,0},
  {"\\mapsto",0,0,0,0,0,0,0,0,0},
  {"\\upto",0,0,0,0,0,0,0,0,0},
  {"+","\\cup","\\setminus","\\cat","\\uplus","\\uminus",0,0,0,0},
  {"*","\\div","\\mod","\\cap","\\circ","\\comp","\\filter","\\extract","\\otimes",0},
  {"\\oplus","\\bcount",0,0,0,0,0,0,0,0},
  {"\\dres","\\rres","\\ndres","\\nrres",0,0,0,0,0,0},
  {0,0,0,0,0,0,0,0,0,0}
};


static void add(int t, int *ss, int sn)
/* add t-stuff from the list of keys in ss */
{
    while (sn-- > 0)
    {
        nodelist[*ss++]->prop |= t;
    }
}

void add_inops(int *ss, int sn, int p)
/* fill the inops table from the ss list of keys.  */
{
        switch(p)
        {
          case 0:
            add(INOP0,ss,sn);
            break;
          case 1:
            add(INOP1,ss,sn);
            break;
          case 2:
            add(INOP2,ss,sn);
            break;
          case 3:
            add(INOP3,ss,sn);
            break;
          case 4:
            add(INOP4,ss,sn);
            break;
          case 5:
            add(INOP5,ss,sn);
            break;
          case 6:
            add(INOP6,ss,sn);
            break;
          case 7:
            add(INOP7,ss,sn);
            break;
        }
          /* fail silently */
}

static int is(int t,int x)
/* look up key x in the ts table */
/* return 0 for fail */
{
    return nodelist[x]->prop & t;
}

int is_inop(int x)
{
    return is(INOP0|INOP1|INOP2|INOP3|INOP4|INOP5|INOP6|INOP7,x);
}

int level(int x)
/* look up x priority in the inops table */
/* return 0 for fail */
{

    if (is(INOP1,x))
      return 1;
    if (is(INOP2,x))
      return 2;
    if (is(INOP3,x))
      return 3;
    if (is(INOP4,x))
      return 4;
    if (is(INOP5,x))
      return 5;
    if (is(INOP6,x))
      return 6;
    if (is(INOP7,x))
      return 7;
    return 0;
}


void add_postops(int *ss, int sn)
/* fill the postops table from the ss temporary.  */
{
    add(POSTOP,ss,sn);
}

int is_postop(int x)
{
    return is(POSTOP,x);
}

void add_inrels(int *ss, int sn)
/* fill the inrels table from the ss temporary.  */
{
    add(INREL,ss,sn);
}

int is_inrel(int x)
{
    return is(INREL,x);
}

void add_prerels(int *ss, int sn)
/* fill the prerels table from the ss temporary.  */
{
    add(PREREL,ss,sn);
}

int is_prerel(int x)
{
    return is(PREREL,x);
}

void add_ingens(int *ss, int sn)
/* fill the ingens table from the ss temporary.  */
{
    add(INGEN,ss,sn);
}

int is_ingen(int x)
{
    return is(INGEN,x);
}

void add_pregens(int *ss, int sn)
/* fill the pregens table from the ss temporary.  */
{
    add(PREGEN,ss,sn);
}

int is_pregen(int x)
{
    return is(PREGEN,x);
}

void add_ignores(int *ss, int sn)
/* fill the ignores table from the ss temporary.  */
{
    add(IGNORE,ss,sn);
}

int is_ignore(int x)
{
    return is(IGNORE,x);
}

static int newkey(char *x, long n)
/* put a new string in the stash and give it a key */
/* negative returns are errors */
{ 
    char *y;
    NODE *z;
    /* the minimum position is 1 */

    if (wordcount >= MAXWORDS - 1)
        return -2;
    y = malloc((unsigned)n + 1);
    if (!y)  /* out of memory */
        return -1;
    z = (NODE *)malloc(sizeof(NODE));
    if (!z)  /* out of memory */
        return -1;
    strncpy(y,x,(unsigned int)n);
    y[n] = 0;
    z->word = y;
    z->prop = 0;
    z->prev = 0;
    z->next = 0;
    ++wordcount;
    nodelist[wordcount] = z;  /* store the pointer for comparisons */
    return wordcount;
}

static void addkey(int j, char *x, long n)
/* order a new string with key j */
{ 
    int i; /* the minimum position is 1 */
    int test;
    char *w;

    i = 1;

    while(i > 0)
    {
      w = nodelist[i]->word;
      if (w == NULL)    /* this is not possible! */
        return;
      test = strncmp(x,w,n);
      if (test < 0) /* x is smaller than w */
      {
        if (0 == nodelist[i]->prev)
        {
          nodelist[i]->prev = j;
          return;
        }
        i = nodelist[i]->prev;
        continue;
      }
      if (test > 0) /* x is bigger  than w */
      {
        if (0 == nodelist[i]->next)
        {
          nodelist[i]->next = j;
          return;
        }
        i = nodelist[i]->next;
        continue;
      }
      /* test == 0 */
      if (w[n])
      {
        if (0 == nodelist[i]->prev)
        {
          nodelist[i]->prev = j;
          return;
        }
        i = nodelist[i]->prev;
        continue;
      }
      /* equal */
      return;
    }
}

static int lookup(char *x, long n)
/* find a name in the list 
 * if not there return 0, else return its index
 */
{
    int i = 1;
    int test;
    char *w;

    if (wordcount < 1)
      return 0;

    while(i > 0)
    {
      w = nodelist[i]->word;
      /* nodelist[i] == NULL can only happen when nodelist is empty */
      test = strncmp(x,w,n);
      test = test<0?-1:test>0?1:0;
      switch (test)
      {
      case -1: /* x is smaller than w */
        i = nodelist[i]->prev;
        continue;
      case  1: /* x is bigger  than w */
        i = nodelist[i]->next;
        continue;
      case  0:
        if (w[n])     /* x is smaller than w */
        {
          i = nodelist[i]->prev;
          continue;
        }
        /* equal */
        return i;
      }
      /* this can't be reached */
    }
    return 0;
}


int ukey(char *x, long n)
/* look up str in notified word list and return unique key.
 * Add new key if necessary and return that. If we can't, return -1.
 */
{
    int i;

    i = lookup(x,n);
    if (i > 0)                         /* found */
        return i;
    i = newkey(x,n);                   /* new key */
    if (i > 0)
        addkey(i,x,n);                 /* order it */
    return i;
}


static void notify(int t, char **ss, int sn)
{
  int i; long n;

  while (sn-- > 0)
  {
    n = strlen(*ss);
    i = ukey(*ss,n);
    add(t,&i,1);
    ss++;
  }
}

void setup(void)
{
  notify(INOP0,inops[0],0);
  notify(INOP1,inops[1],1);
  notify(INOP2,inops[2],1);
  notify(INOP3,inops[3],6);
  notify(INOP4,inops[4],9);
  notify(INOP5,inops[5],2);
  notify(INOP6,inops[6],4);
  notify(INOP7,inops[7],0);
}

