#include <fstream.h>
#include <unistd.h>
#include <sys/stat.h>

#pragma implementation
#include "sources.h"

sourcefile::sourcefile(string name, int flags)
{
    filename = name;

    if (!(flags & reallyflag)) {
	lines = 0;
	return;
    }
    
    ifstream from(name.c_str());
    if (!from) {
	lines = 0;
	return;
    }

    if (flags & verboseflag) {
	cout << "Using source file " << name << '\n';
    }

    /* Count the number of lines in the sourcefile.
       Numlines is actually one larger than this. */
    numlines = 1;
    char* ch;
    while (from.gets(&ch)) {
	delete[] ch;
	numlines++;
    }

    lines = new int[numlines];
    memset(lines, -1, numlines * sizeof(int));
    /* The above line assumes that this will make -1 out
       of those int's.  */

    struct stat flstat;		// Need struct to avoid confusion with stat().
    if (stat(name.c_str(), &flstat)) {
	cerr << "Warning: could not stat " << name << '\n';
        _mtime = 0;
	return;
    }
    _mtime = flstat.st_mtime;

    lineswarned = false;
}

sourcefile::~sourcefile()
{
    delete[] lines;
}

int& sourcefile::operator[](unsigned int nr)
{
    if (!lines)
	return dummy;

    if (nr >= numlines) {
	if (!lineswarned) {
	    lineswarned = true;
	    cerr << "Warning: debug info indicates line " << nr << " in file " << filename
		<< "\nwhich seems to have only " << (numlines-1) << " lines\n";
	}
	return dummy;
    }

    if (lines[nr] == -1)
	lines[nr] = 0;
    return lines[nr];
}

void sourcefile::paste(string suffix) const
{
    if (!lines)
	return;

    // Should check length with pathconf(".", _PC_NAME_MAX)
    string outname(filename + suffix);
    unlink(outname.c_str());	// opening with ios::trunc doesn't seem to work ???
    ofstream to(outname.c_str());
    if (!to) {
	cerr << "Could not open " << outname << " to write to\n";
	return;
    }
    
    ifstream from(filename.c_str());
    if (!from) {
	cerr << "Could not open " << filename << " for input\n";
	return;
    }

    for (unsigned int i = 1; i < numlines; i++) {
	if (lines[i] == -1)
	    to << "     .";
	else if (lines[i] == 0)
	    to << "     -";
	else {
	    to.width(6);
	    to << lines[i];
	}
	char* ch;
	from.gets(&ch);
	to << '\t' << ch << '\n';
	delete[] ch;
    }
}

time_t sourcefile::mtime(void) const
{
    return lines ? _mtime : 0;
}

bool dir::less::operator()(const dir& x, const dir& y) const
{
    if (x.dev == y.dev) {
	return x.ino < y.ino;
    }
    return x.dev < y.dev;
}

dir::dir(const char* name) {
    dev = invalid;
    struct stat dirstat;	// Need struct to avoid confusion with stat().
    if (stat(name, &dirstat))
	return;
    if (!S_ISDIR(dirstat.st_mode))
	return;
    dev = dirstat.st_dev;
    ino = dirstat.st_ino;
}
