/* Main source file for bprof. */

#include <unistd.h>		// getopt()
#include <string>
#include <map>
#include <set>

#include "bmonout.h"
#include "execute.h"
#include "sources.h"

int main(int argc, char* argv[])
{
    string sffx(".bprof");	// Default suffix
    set<dir, dir::less> gooddirs;
    dir pwd(".");
    if (pwd.valid()) {
	gooddirs.insert(pwd);	// Always use pwd
    }
    int verbose = 0;		// Whether to print filenames

    int option;
    while ((option = getopt(argc, argv, "d:s:v")) != EOF) {
	switch (option) {
	  case 'd':
	    {
		dir extradir(optarg);
		if (extradir.valid()) {
		    gooddirs.insert(extradir);
		} else {
		    cerr << "Cannot find directory " << optarg << ", proceeding without\n";
		}
	    }
	    break;
	  case 's':
	    sffx = optarg;
	    break;
	  case 'v':
	    verbose = sourcefile::verboseflag;
	    break;
	  case '?':
	    exit(1);
	}
    }

    // First non-option argument is executable
    const char* filename;
    if (optind < argc) {
	filename = argv[optind++];
    } else {
	filename = "a.out";
    }
    executable aout(filename);
    
    // Other non-options args are bmon.out files
    int numbmon = argc - optind;
    bmonout** bmon;		// Array of bmonout pointers
    if (numbmon) {
	bmon = new bmonout*[numbmon];
	for (int i = 0; i < numbmon; i++) {
	    bmon[i] = new bmonout(argv[optind]);
	    if (bmon[i]->mtime() < aout.mtime()) {
		cerr << "Warning: executable is newer than " << argv[optind] << '\n';
	    }
	    optind++;
	}
    } else {
	numbmon = 1;
	bmon = new bmonout*[numbmon];
	bmon[0] = new bmonout("bmon.out");
	if (bmon[0]->mtime() < aout.mtime()) {
	    cerr << "Warning: executable is newer than bmon.out\n";
	}
    }

    // less<string> is the default third argument, but it seems that gcc 2.7 cannot yet handle this
    typedef map<string,sfpnt,less<string> > sourcemap;
    sourcemap source;

    // Inner loop should be over bmon files to speed things up,
    // although I tend to use only a single bmon file anyway.
    for (int i = 0; i < numbmon; i++) {
	int inum = bmon[i]->countnum();
	for (int j = 0; j < inum; j++) {
	    caddr_t eip = (*bmon[i])[j].pc;
	    const char* thisname;
	    unsigned int lineno;
	    if (!aout.find_line(eip, &thisname, &lineno)) {
		cerr << "Could not find line for eip\n";
		continue;
	    }
	    if (!thisname)
		continue;
	    string cname(thisname);
	    // If filename unknown, add to source set.
	    if (!source.count(cname)) {
		int lastslash = cname.rfind("/");
		string dirname = cname.substr(0, lastslash); // Final -1 ???
		dir filedir(dirname.c_str());
		int really = gooddirs.count(filedir) ? sourcefile::reallyflag : 0;
		sourcemap::value_type newkey(cname, new sourcefile(cname, really | verbose));
		source.insert(newkey);
		if (aout.mtime() < source[cname]->mtime())
		    cerr << "Warning: source file " << cname << " newer than executable\n";
	    }
	    sourcefile& sfile = *source[cname];
	    sfile[lineno] += (*bmon[i])[j].count;
	}
    }

    for (int i = 0; i < numbmon; i++) {
	delete bmon[i];
    }
    delete[] bmon;

    // Write the source files with data
    for (sourcemap::const_iterator p = source.begin(); p != source.end(); p++) {
	(*p).second->paste(sffx); // Why doesn't p->second work ???
    }

    return 0;
}
