/* This file gets linked with the application program to produce the
   timing data.  Before the program starts, it makes sure that every
   10 ms the routine profhandler() gets called.  This checks where the
   program was and increases a count of that address. */

/* This file must be linkable without C++ libraries, so use only C
   library routines */

#include <sys/time.h>		// itimer stuff
#include <signal.h>
#include <unistd.h>
#include <fcntl.h>
#include <cstdio>
#include <cstring>		// memset()
#include <cstdlib>		// getenv()

#include "bmon.h"
#include "arch.h"

/* I should have used the profil() routine to ease porting, but it
   cannot use a finer scale than one counter per two bytes. */

static caddr_t lowpc, highpc;	// First and (last+1) address of program text
static unsigned int* bbuf;	// Counters for all those addresses
static int bbufsize;		// Size of bbuf array

/* Call routines at start and end by making them constructor and
   destructor of a global object */

class _bmonclass {
    static void profhandler(BM_SIGARGS);
  public:
    _bmonclass();
    ~_bmonclass();
};

void _bmonclass::profhandler(BM_SIGARGS)
{
    int index = reinterpret_cast<caddr_t>(BM_EIP) - lowpc;
    if (index >= 0 && index < bbufsize)
	bbuf[index]++;
}

_bmonclass::_bmonclass()
{
    lowpc = BM_LOWPC;
    highpc = BM_HIGHPC;
    bbufsize = highpc - lowpc;
    bbuf = static_cast<typeof(bbuf)>(sbrk(bbufsize * sizeof(*bbuf)));
    if (bbuf == reinterpret_cast<typeof(bbuf)>(-1)) {
	perror("bmon.o");
	fflush(stderr);
	_exit(1);		// Avoid calling the destructor
    }
    // sbrk-ed memory is supposed to be zero, but I've seen otherwise.
    memset(bbuf, 0, bbufsize * sizeof(*bbuf));
   
    struct sigaction sigact;
    sigact.sa_handler = static_cast<void (*)(int)>(profhandler);
    sigset_t sigset;
    sigemptyset(&sigset);
    sigact.sa_mask = sigset;
    sigact.sa_flags = 0;
#ifdef SA_RESTART
    // This is somewhat redundant, because SIGVTALRM shouldn't arrive during system calls.
    sigact.sa_flags |= SA_RESTART;
#endif

    itimerval value;
    value.it_interval.tv_sec = 0;
    value.it_interval.tv_usec = 1; // Gets rounded up to 1e6/HZ
    value.it_value.tv_sec = 0;
    value.it_value.tv_usec = 1;

    sigaction(SIGVTALRM, &sigact, 0);
    setitimer(ITIMER_VIRTUAL, &value, 0);
}

static bool xwrite(int fd, void* address, size_t size)
{
    while (size) {
	ssize_t ret = write(fd, address, size);
	if (ret == -1) {
	    perror("Error writing bmon.out");
	    // Note that stdio cleanup comes _after_ global destructors.
	    return false;
	}
	address += ret;
	size -= ret;
    }
    return true;
}

static bool xlseek(int fd, off_t offset)
{
    return lseek(fd, offset, SEEK_SET) == offset;
}

_bmonclass::~_bmonclass()
{
    const char* dir;
    if ((dir = getenv("BPROFDIR"))) {
	if (chdir(dir)) {
	    perror("Could not chdir to BPROFDIR");
	    return;
	}
    }

    int fd = creat("bmon.out", 0666);
    if (fd == -1) {
	perror("bmon.out");
	return;
    }

    unsigned int items = 0;
    if (!xlseek(fd, sizeof(bmonheader))) {
	perror("bmon.out lseek 1");
	return;
    }
    for (int i = 0; i < bbufsize; i++) {
	if (bbuf[i]) {
	    bmonitem item;
	    item.pc = lowpc + i;
	    item.count = bbuf[i];
	    if (!xwrite(fd, &item, sizeof(item))) {
		return;
	    }
	    items++;
	}
    }

    bmonheader bmhead;
    bmhead.bmon_magic = BMON_MAGIC;
    bmhead.numofitems = items;
    if (!xlseek(fd, 0)) {
	perror("bmon.out lseek 2");
	return;
    }
    if (!xwrite(fd, &bmhead, sizeof(bmhead))) {
	return;
    }

    close(fd);
}

static _bmonclass bmonobject;	// Single instance of _bmonclass
