//______________________________________________________________________________

//	The Java Virtual Shelf
//______________________________________________________________________________

package org.demo.webwader.xmlrpc;

import org.ariane.tools.ToolBox;
import org.ariane.exec.ExecutionException;
import org.demo.webwader.Resources;
import org.demo.webwader.BadConfigurationException;
import org.demo.webwader.gui.GuiResources;
import org.demo.webwader.gui.Displayer;
import helma.xmlrpc.*;
import java.net.URL;
import java.net.MalformedURLException;
import java.util.Vector;

/**
 * A displayer based on the XmlRpcClientLite component.
 * <br>
 * It is the client part.
 * The XmlRpc API documentation is available
 * <a href="http://classic.helma.at/hannes/xmlrpc/">online</a>.
 *
 * @version $Id: XmlDisplayer.java,v 1.5 2001/06/07 19:40:52 lefevre Exp $
 * @author Jean-Paul Le Fvre
 * @see DisplayerServant
 */
//______________________________________________________________________________

public class XmlDisplayer extends XmlRpcClientLite implements Displayer {
  /**
   * @serial The unique instance.
   */
private  static XmlDisplayer displayer = null;
    /**
     * @serial The name of this tool.
     */
public static final String NAME = org.demo.webwader.gui.XmlDisplayer.NAME;
    /**
     * @serial The list of parameters passed to the server.
     */
private Vector param;

//______________________________________________________________________________
/**
 * Gets an unique instance of the displayer.
 * <br>
 * It is a singleton. If the object cannot be created a message is
 * printed and the instance stays null.
 * @return the unique instance.
 */
public static XmlDisplayer instance()
  {
      if(displayer == null) {
	  try {
	      displayer = new XmlDisplayer();
	  }
	  catch(MalformedURLException ex) {
	      ToolBox.warn("Can't create the XmlDisplayer", ex);
	  }
	  catch(BadConfigurationException ex) {
	      ToolBox.warn("Can't install the XML parser", ex);
	  }
      }

      return displayer;
  }
//______________________________________________________________________________
/**
 * Creates the XmlDisplayer.
 * <br>
 * It uses the URL : http://localhost:port to establish a connection
 * to the server. The port is found in the properties. The XML parser
 * is also defined in the properties.
 *
 * @throw MalformedURLException if the URL is invalid.
 * @see DisplayerServant#getPort()
 * @see DisplayerServant
 * @see #setDriver()
 */
private XmlDisplayer() throws MalformedURLException, BadConfigurationException
  {
      super("http://localhost:" + DisplayerServant.getPort());
      param = new Vector(1, 1);
      param.add(null); // Put at least one element at start because of set()
      setXmlDriver();
  }
//______________________________________________________________________________
/**
 * Installs the XML driver.
 * <br>
 * The driver is specified in the properties by
 * <code>org.xml.sax.parser</code>.
 * @throw BadConfigurationException if the driver is not found.
 * @see XmlRpc#setDriver
 */
static protected void setXmlDriver() throws BadConfigurationException
{
      String key = "org.xml.sax.parser";
      
      String val = Resources.instance().getProperties().getProperty(key,
					"org.openxml.parser.XMLParser");

      if(ToolBox.debug) {
	  ToolBox.warn("XmlDriver : " + val);
      }

      if(val == null)throw new BadConfigurationException("No value for " + key);

      try {
	  XmlRpc.setDriver(val);
      }
      catch(ClassNotFoundException ex) {
	  throw new BadConfigurationException(ex.getMessage());
      }
}
//______________________________________________________________________________
/**
 * Kills this displayer.
 * <br>
 * It stops the server.
 * @throw ExecutionException if the connection fails.
 */
final public void kill() throws ExecutionException
{
    try {
	param.set(0, "RIP");
	execute("Displayer.kill", param); // Must be called twice to exit loop.
	execute("Displayer.kill", param); // Throws an exception
    }
    catch(java.net.SocketException ex) {
	if(ToolBox.verbose) ToolBox.warn("The server is dead");
    }
    catch(Exception ex) {
	if(ToolBox.verbose) ToolBox.warn("Can't kill the server", ex);
	throw new ExecutionException(ex.getMessage());
    }
}
//______________________________________________________________________________
/**
 * Activates this displayer.
 * <br>
 * Simply displays the welcome page.
 * @throw ExecutionException if the connection fails.
 */
final public void activate() throws ExecutionException
{
    try {
	// Don't use displayPage() to catch exception.
	param.set(0, GuiResources.instance().getWelcomeLocation());
	execute("Displayer.displayPage", param);
    }
    catch(Exception ex) {
	if(ToolBox.verbose) ToolBox.warn("Can't execute displayPage", ex);
	throw new ExecutionException(ex.getMessage());
    }
}
//______________________________________________________________________________
/**
 * Displays the specified document.
 * <br>
 * The URL is loaded in the array of parameters.
 * The method name <code>Displayer.displayPage</code> is passed to
 * <code>execute</code>.
 *
 * @param url the url of the page.
 * @return true is the page has been displayed.
 * @see #execute
 */
final public boolean displayPage(URL url)
{
    if(url == null) return false;

    try {
	param.set(0, url.toExternalForm()); // Only string is supported.
	return ((Boolean)execute("Displayer.displayPage",param)).booleanValue();
    }
    catch(Exception ex) {
	ToolBox.warn("Can't display page " + url.toString(), ex);
	return false;
    }
}
//______________________________________________________________________________
/**
 * Returns the name of this tool.
 * @return the string.
 */
final public String getName()
{
    return NAME;
}
//______________________________________________________________________________
}


