//______________________________________________________________________________

//	Java Virtual Shelf
//______________________________________________________________________________

package org.demo.webwader.rbi;

import org.demo.webwader.VersionInfo;
import org.ariane.tools.TryProg;
import org.ariane.tools.ToolBox;
import org.ariane.tools.ResourcesException;
import java.io.*;
import java.rmi.*;
import java.rmi.registry.*;
import java.net.URL;
import java.net.MalformedURLException;

/**
 * The client used to test the displayer servant.
 * <P>
 * <Code>
 * Usage : java org.demo.webwader.rbi.ShowURL [-v|d|q][-V][-h]
 * </Code>
 * <P>
 * It parses the command line, it launches a browser and it starts a server.
 * A loop is started to enter data.
 *
 * @version $Id: ShowURL.java,v 1.5 2001/04/11 19:20:18 lefevre Exp $
 * @author  Jean-Paul Le Fvre
 */

public class ShowURL extends TryProg {
    /**
     * @serial The array of available displayers.
     */
private String[] DISPLAYERS;

//______________________________________________________________________________
/**
 * The main program.
 * @param args the command line.
 */
public static void main (String[] args)
  {
      ShowURL prog    = new ShowURL();

      if (! prog.parseArg(args)) {
	  System.err.println("Can't parse command line !");
	  exitWrong();
      }

      if (! prog.run()) {
	  System.err.println("Can't execute ShowURL !");
	  exitWrong();
      }

      System.exit(ToolBox.OK);
  }
//______________________________________________________________________________
/**
 * Creates the program.
 * <br>
 * It loads the properties used in the application : the default JVS
 * values, the general WebWader values and the user values.
 */
public ShowURL()
  {
      super("ShowURL");
      setResources();
  }
//______________________________________________________________________________
/**
 * Launches the program.
 * @return true if successful.
 */
public	boolean run()
  {
      if(ToolBox.debug) {
	  System.out.println(VersionInfo.instance().toString());
      }

      try {
	  DISPLAYERS = DisplayersBag.instance().getNamesList();
	  startLoop();
	  System.out.println("See you later !");
      }
      catch(ResourcesException ex) {
	  ToolBox.warn("Can't manage resources " + ex);
	  return false;
      }
      catch(NotBoundException ex) {
	  ToolBox.warn("Can't find the servant " + ex);
	  return false;
      }
      catch(RemoteException ex) {
	  ToolBox.warn("Can't create the connection " + ex);
	  return false;
      }
      catch(Exception ex) {
	  ToolBox.warn("Can't run ShowURL " + ex);
	  return false;
      }

      return true;
  }
//______________________________________________________________________________
/**
 * Manages the dialog with the servant.
 * <br>
 * @throw RemoteException if the server cannot be accessed.
 * @throw NotBoundException if the server cannot be accessed.
 * @throw MalformedURLException if the server has a wrong URL.
 */
private void startLoop() throws RemoteException, NotBoundException,
                         MalformedURLException
  {
      int port =
      resources.get(Browser.REGISTRY_PORT, Registry.REGISTRY_PORT);

      Registry registry = LocateRegistry.getRegistry(port);
      RemoteDisplayer
      displayer = (RemoteDisplayer)registry.lookup(Browser.SERVANT);
      ToolBox.warn("Server contacted");

      printOptions();

      boolean ok = true;
      BufferedReader d = new BufferedReader(new InputStreamReader(System.in));

      while(ok) {

	  System.out.print("Option : ");
	  String answer = null;
	  try {
	      answer = d.readLine();
	  }
	  catch(IOException ex) {
	      System.out.println("Can't read answer !");
	  }

	  /**
	   * First check for help or exit options.
	   */
	  if(answer.equalsIgnoreCase("H") || answer.equals("?")) {
	      printOptions();
	      continue;
	  }
	  else if(answer.equalsIgnoreCase("I") || answer.equalsIgnoreCase("D")){
	      System.out.println("Displayer : " + displayer.getName());
	      continue;
	  }
	  else if(answer.equalsIgnoreCase("Q") || answer.equalsIgnoreCase("X")){
	      return;
	  }

	  /**
	   * Try to see if it is a change displayer option.
	   * If so, process the request.
	   */
	  try {
	      int index = Integer.parseInt(answer);
	      if(ToolBox.debug) {
		  System.out.println("Requested displayer : "
				     + DISPLAYERS[index]);
	      }

	      if(displayer.changeDisplayer(DISPLAYERS[index])) {
		  System.out.println("Displayer changed"); 
	      }
	      else {
		  System.out.println("Displayer stays unchanged"); 
	      }
	      continue;
	  }
	  catch(NumberFormatException ex) {
	  }
	  catch(IndexOutOfBoundsException ex) {
	      ToolBox.warn("Invalid index", ex);
	      printOptions();
	      continue;
	  }
	  catch(RemoteException ex) {
	      ToolBox.warn("Can't change the displayer", ex);
	      continue;
	  }

	  /**
	   * The entered string is considered as an URL to display.
	   */
	  if(false) System.out.println("URL : " + answer);
	  try {
	      displayer.displayPage(new URL(answer));
	  }
	  catch(RemoteException ex) {
	      ToolBox.warn("Can't display " + answer, ex);
	  }
	  catch(MalformedURLException ex) {
	      ToolBox.warn("Invalid URL " + answer, ex);
	  }
      }  
  }
//______________________________________________________________________________
/**
 * Prints the list of options.
 */
private void printOptions()
  {
      System.out.println();
      System.out.println("To get help   : 'h' or '?'");
      System.out.println("To get info   : 'i' or 'd'");
      System.out.println("To exit enter : 'x' or 'q'");
      System.out.println("To display an URL give the location.");
      System.out.println("To change the displayer enter a number :");
      System.out.println();

      for(int i = 0; i < DISPLAYERS.length; i++) {
	  System.out.println(i + " : " + DISPLAYERS[i]);
      }

      System.out.println();
  }
//______________________________________________________________________________
/**
 * Installs the specific resources.
 * @see	org.demo.webwader.Resources
 */
protected void setResources()
  {
      resources = org.demo.webwader.Resources.instance();
  }
//______________________________________________________________________________
/**
 * Prints how to use the program.
 */
private	void printUsage()
  {
      System.out.println(
      "Usage   : java org.demo.webwader.rbi.ShowURL [options]");
      System.out.println("Options : [-v|-d|-q][-V][-h]");
  }
//______________________________________________________________________________
/**
 * Parses the command line.
 * <Br>
 * <Code>Usage   : -v|d|q -V -h</Code>
 * @return true if the command line is correct.
 */
public	boolean parseArg(String args[])
  {
    int i          = 0;
    int error      = 0;

    while(i < args.length && args[i].startsWith("-")) {

      String arg = args[i++];
      int j;

      for (j = 1; j < arg.length(); j++) {

	char flag = arg.charAt(j);

	switch (flag) {
	case 'V':
	  org.demo.webwader.VersionInfo.print();
	  return false;

	case 'v':
	  ToolBox.verbose = true;
	  ToolBox.debug   = false;
	  break;

	case 'd':
	  ToolBox.verbose = true;
	  ToolBox.debug   = true;
	  break;

	case 'q':
	  ToolBox.verbose = false;
	  ToolBox.debug   = false;
	  break;

	default	:
	  printUsage();
	  error++;
	  break;
	}
      }
    }

    return (error == 0);
  }
//______________________________________________________________________________
}
