//______________________________________________________________________________

//	The Java Virtual Shelf
//______________________________________________________________________________

package org.demo.webwader.rbi;

import java.awt.*;
import java.awt.event.*;
import javax.swing.*;
import javax.swing.border.Border;

import org.demo.webwader.gui.GuiResources;
import org.ariane.applet.Messenger;
import org.ariane.gui.TimerBar;
import org.ariane.tools.ToolBox;

/**
 * Manages the main frame and the menu bar of the Browser.
 * <Br>
 * The main frame is divided in four parts :
 * <Ul>
 * <Li>the top pane which displays a couple of nice icons.
 * <Li>the nave pane presenting controls for the navigation.
 * <Li>the main pane showing a html page.
 * <Li>the bottom pane used to give some feeback to the user.
 * </Ul>
 * @version $Id: MainFrame.java,v 1.8 2001/01/28 11:33:24 lefevre Exp $
 * @author Jean-Paul Le Fvre
 * @see Browser
 */
//______________________________________________________________________________

public	class MainFrame extends JFrame implements ActionListener {
    /**
     * The button Set Options label.
     */
private	static final String SET_OPTIONS = "SetOptions";
    /**
     * The button Quit label.
     */
private	static final String QUIT   = "Quit";
    /**
     * The button About label.
     */
private	static final String ABOUT  = "About Browser ...";
    /**
     * The button Manual label.
     */
private	static final String MANUAL = "Browser short help ...";
    /**
     * @serial The main container for the widgets.
     */
private	Container window;
    /**
     * @serial The object used to display transient messages.
     */
static private Messenger messenger;
    /**
     * @serial The object used to show work in progress.
     */
private TimerBar timerbar;
    /**
     * @serial The set of properties.
     */
private GuiResources resources;
  /**
   * @serial The unique instance.
   */
private static MainFrame main_frame = null;

//______________________________________________________________________________
/**
 * Gets an unique instance of the main frame.
 * <Br>
 * It is a singleton.
 * @return the unique instance.
 */
protected static MainFrame instance()
  {
    if(main_frame == null) main_frame = new MainFrame("Browser Harness");
    return main_frame;
  }
//______________________________________________________________________________
  /**
   * Displays a transient message at the bottom.
   * @parameter text the message to display.
   * @see Messenger#display
   */
static final protected void inform(String text)
{
    messenger.display(text);
}
//______________________________________________________________________________
  /**
   * Clears the message displayed at the bottom.
   * @see Messenger#erase
   */
static final protected void clearMessage()
{
    messenger.erase();
}
//______________________________________________________________________________
/**
 * Creates the MainFrame of the application.
 * @param name the name of the application.
 */
private MainFrame(String name)
  {
      super(name);
      ToolBox.dot();

      resources  = GuiResources.instance();

      ToolBox.dot();
      int width  = resources.get(Browser.APPLI + ".Width",  800);
      int height = resources.get(Browser.APPLI + ".Height", 700);
      setSize(width, height);

      JMenuBar menubar = buildMenuBar();
      setJMenuBar(menubar);

      window = getContentPane();
      window.setLayout(new BoxLayout(window, BoxLayout.Y_AXIS));

      window.setBackground(resources.getMainBackground());
      window.setForeground(resources.getMainForeground());
      window.setFont(resources.getHeaderFont());

      JComponent toppane = buildTopPane();
      window.add(toppane);
      toppane.setAlignmentX(LEFT_ALIGNMENT);
      ToolBox.dot();

      NavePane navepane = NavePane.instance();
      window.add(navepane);
      navepane.setAlignmentX(LEFT_ALIGNMENT);
      ToolBox.dot();

      BrowserPane browser = BrowserPane.instance();
      browser.setAlignmentX(LEFT_ALIGNMENT);
      window.add(browser);
      navepane.setBrowserPane(browser);
      ToolBox.dot();

      window.add(Box.createVerticalStrut(resources.getSpace()));

      JComponent bottompane = buildBottomPane();
      bottompane.setAlignmentX(LEFT_ALIGNMENT);
      window.add(bottompane);
      ToolBox.dot();

      setVisible(true);
      ToolBox.dot();

      System.out.println();

      inform("Welcome to this new session of Browser !");
  }
//______________________________________________________________________________
/**
 * Builds the top pane.
 * <Br>
 * It is a strip at the top of the viewer with icons and title.
 * The title is a gif image given by <code>WebWader.Browser.Title</code>.
 *
 * @return the built component.
 */
private JComponent buildTopPane()
  {
      JPanel pane    = new JPanel();
      pane.setLayout(new BoxLayout(pane, BoxLayout.X_AXIS));
      pane.add(Box.createRigidArea(new Dimension(resources.getSpace(), 50)));
      pane.setOpaque(false);

      Icon icon      = resources.getIcon("jvs.gif");
      JLabel label   = new JLabel(icon);
      label.setAlignmentX(Component.LEFT_ALIGNMENT);
      label.setBorder(resources.getLineBorder());
      pane.add(label);

      pane.add(Box.createHorizontalGlue());

      icon    = resources.getIcon(
		resources.get(Browser.APPLI + ".Title", "browser.gif"));
      label   = new JLabel(icon);
      label.setOpaque(false);
      label.setAlignmentX(Component.CENTER_ALIGNMENT);
      pane.add(label);
      pane.add(Box.createRigidArea(new Dimension(40*resources.getSpace(), 0)));

      return pane;
  }
//______________________________________________________________________________
/**
 * Builds the bottom pane.
 * <Br>
 * It is a strip at the bottom of the viewer.
 * It displays a message area.
 *
 * @return the built component.
 */
private JComponent buildBottomPane()
  {
      JPanel pane = new JPanel();
      pane.setLayout(new FlowLayout(FlowLayout.LEFT));
      pane.setOpaque(false);


      Component area = buildMessageArea();
      int h = area.getHeight();
      pane.add(area);

      pane.setMaximumSize(new Dimension(getWidth(), h));

      return pane;
  }
//______________________________________________________________________________
/**
 * Builds the message area.
 * <br>
 * Messages are transient.
 * @return the built component.
 * @see Messenger
 */
private JComponent buildMessageArea()
  {
      JLabel pane = new JLabel();

      pane.setBackground(resources.getMainBackground());
      pane.setForeground(resources.getMainForeground());
      pane.setFont(resources.getTextFont());
      ToolBox.dot();

      messenger = new Messenger(pane);
      messenger.setDisplayTime(
      resources.get("DisplayTime", 5));

      return pane;
  }
//______________________________________________________________________________
/**
 * Builds the menu bar.
 * @return the built component.
 */
private JMenuBar buildMenuBar()
  {
      JMenuBar menubar  = new JMenuBar();
      Color fg          = resources.getMainForeground();
      Color bg          = resources.getMainBackground();
      menubar.setBackground(bg);
      menubar.setForeground(fg);

      JMenu menu;
      JMenuItem item;

      menu   = new JMenu("File");
      menu.setToolTipText("To execute main operations");
      menu.setOpaque(false);
      menu.setForeground(fg);
      menubar.add(menu);

      item = new JMenuItem("Quit");
      item.setActionCommand(QUIT);
      setMenuItem(item);
      menu.addSeparator();
      menu.add(item);

      // The settings menu.

      menu = new JMenu("Settings");
      menu.setOpaque(false);
      menu.setForeground(fg);
      menu.setToolTipText("To change current configuration");
      menubar.add(menu);

      item = new JMenuItem("Set options ...");
      item.setActionCommand(SET_OPTIONS);
      setMenuItem(item);
      item.setToolTipText("To set various parameters");
      menu.add(item);

      // The help menu.

      menu   = new JMenu("Help");
      menu.setOpaque(false);
      menu.setForeground(fg);
      menubar.add(menu); //setHelpMenu
      menu.setToolTipText("To get information about WebWader");

      item = new JMenuItem(MANUAL);
      item.setToolTipText("Where to find documentation");
      menu.add(item);
      setMenuItem(item);

      menu.addSeparator();

      item = new JMenuItem(ABOUT);
      item.setToolTipText("To find out current version");
      menu.add(item);
      setMenuItem(item);

      return menubar;
  }
//______________________________________________________________________________
/**
 * Sets attributes to a menu item.
 * @param item the button to define.
 */
final private void setMenuItem(JMenuItem item)
  {
      item.addActionListener(this);
      item.setBackground(resources.getMainBackground());
      item.setForeground(resources.getMainForeground());
  }
//______________________________________________________________________________
/**
 * Converts MainFrame to String.
 * @return str	the string.
 */
public String toString()
  {
    return "MainFrame " + getName();
  }
//______________________________________________________________________________
/**
 * Carries out an operation when a menu item is selected.
 * <Br>
 * Depending on the command name stored in the event, a specific method
 * is called.
 *
 * @parameter ev the event generated.
 */
public void actionPerformed(ActionEvent ev)
{
    String cmd = ev.getActionCommand();

    if(cmd.equals(QUIT)) {
	processQuit();
    }
    else if(cmd.equals(SET_OPTIONS)) {
	processSetOptions();
    }
    else if(cmd.equals(MANUAL)) {
	processManual();
    }
    else if(cmd.equals(ABOUT)) {
	processAbout();
    }
    else {
	ToolBox.warn("Invalid command " + cmd);
    }
}
//______________________________________________________________________________
/**
 * Processes the Quit option.
 */
final private void processQuit()
{
    if(Confirmer.instance().ask("Really want to quit ?")) {
	ToolBox.warn("See you later");
	System.exit(0);
    }
}
//______________________________________________________________________________
/**
 * Processes the SetOptions option.
 * @see OptionPane#display
 */
final private void processSetOptions()
{
    OptionPane.instance().display();
}
//______________________________________________________________________________
/**
 * Processes the Manual option.
 */
final private void processManual()
{
    HelpWindow.instance().display();
}
//______________________________________________________________________________
/**
 * Processes the About option.
 */
final private void processAbout()
{
    VersionWindow.instance().display();
}
//______________________________________________________________________________
}

