//______________________________________________________________________________

//	The Java Virtual Shelf
//______________________________________________________________________________

package org.demo.webwader.rbi;

import org.ariane.tools.ToolBox;
import org.demo.webwader.gui.Displayer;
import org.demo.webwader.gui.AbstractDisplayersBag;
import java.lang.reflect.Method;
import java.lang.reflect.InvocationTargetException;

/**
 * The list of available displayers for the WebWader Browser harness.
 * <br>
 * It manages the collection of possible displayers and the current displayer.
 * It is strongly inspired from the object in package gui.
 * <p>
 * This file can be compiled even if some displayers are not available.
 * <p>
 * To add a new displayer :
 * <ul>
 * <li>Implement the code, update Makefile.am.
 * <li>Add an index definition.
 * <li>Increment the size of DISPLAYERS.
 * <li>Append the name in DISPLAYERS.
 * <li>Add a case in <code>DisplayersBag.loadInstances()<code>.
 * <li>Add a case in <code>DisplayersProbe.isAvailable()<code>.
 * <li>Add a choice in <code>DisplayersProbe.main()<code>.
 * <li>Add a choice in <code>webwader.properties<code>.
 * </ul>
 *
 * @version $Id: DisplayersBag.java,v 1.6 2001/01/20 09:32:42 lefevre Exp $
 * @author Jean-Paul Le Fvre
 * @see DisplayersProbe
 */
//______________________________________________________________________________

public class DisplayersBag extends AbstractDisplayersBag {
  /**
   * @serial The unique instance.
   */
private static DisplayersBag  bag = null;
    /**
     * @serial The index of the displayer SwingDisplayer.
     */
private static final int SWING = 0;
    /**
     * @serial The index of the displayer NetClue.
     */
private static final int CLUE = 1;
    /**
     * @serial The index of the displayer Horst.
     */
private static final int HORST = 2;
    /**
     * @serial The index of the displayer Ice.
     */
private static final int ICE = 3;
  /**
   * @serial The array of instances.
   */
protected Displayer[] instances = null;
    /**
     * @serial The name of the displayer NetClue.
     */
public static final String CLUE_DISPLAYER_NAME = "ClueDisplayer";
    /**
     * @serial The name of the displayer Ice.
     */
public static final String ICE_DISPLAYER_NAME = "IceDisplayer";
    /**
     * @serial The name of the displayer Horst.
     */
public static final String HORST_DISPLAYER_NAME = "HorstDisplayer";

//______________________________________________________________________________
/**
 * Creates the bag.
 * <br>
 * The names are defined in this package which should compile even
 * if the external classes are not available.
 */
private DisplayersBag()
  {
	DISPLAYERS        = new String[4];
	DISPLAYERS[SWING] = SwingDisplayer.NAME;
	DISPLAYERS[ICE]   = ICE_DISPLAYER_NAME;
	DISPLAYERS[HORST] = HORST_DISPLAYER_NAME;
	DISPLAYERS[CLUE]  = CLUE_DISPLAYER_NAME;
	/**
	 * Dymically load instances of external packages.
	 */
	loadInstances();
  }
//______________________________________________________________________________
/**
 * Gets an unique instance of the DisplayersBag.
 * <Br>
 * It is a singleton.
 * @return the unique instance.
 */
public static DisplayersBag instance()
  {
    if(bag == null) bag = new DisplayersBag();

    return bag;
  }
//______________________________________________________________________________
/**
 * Loads external viewers.
 * <br>
 * When it is  available a viewer is instanciated and the reference
 * is kept in an array.
 * @see #getInstance
 */
private void loadInstances()
  {
      instances = new Displayer[DISPLAYERS.length];
      int i     = 0;
      try {
	  /**
	   * This one is always available.
	   */
	  instances[i] = SwingDisplayer.instance();

	  i = ICE;
	  if(DisplayersProbe.isAvailable(DISPLAYERS[i])) {
	      instances[i] = getInstance("IceDisplayer");
	  }

	  i = CLUE;
	  if(DisplayersProbe.isAvailable(DISPLAYERS[i])) {
	      instances[i] = getInstance("ClueDisplayer");
	  }

	  i = HORST;
	  if(DisplayersProbe.isAvailable(DISPLAYERS[i])) {
	      instances[i] = getInstance("HorstDisplayer");
	  }

      }
      catch(Exception ex) {
	  ToolBox.warn("Can't load instance of " + DISPLAYERS[i], ex);
      }
  }
//______________________________________________________________________________
/**
 * Loads an external viewer.
 * @param name the name of the displayer
 * @return an instance of the viewer.
 * @see Class#forName
 * @see Method#invoke
 * @throw ClassNotFoundException if the class is not found.
 * @throw NoSuchMethodException is there is no instance() method.
 * @throw IllegalAccessException is invocation is denied.
 * @throw InvocationTargetException is invocation is denied.
 */
final private Displayer getInstance(String name)
    throws ClassNotFoundException, NoSuchMethodException,
           IllegalAccessException, InvocationTargetException
  {
      Class dsp   = Class.forName("org.demo.webwader.rbi." + name);
      Method ctor = dsp.getMethod("instance", null);

      return (Displayer)ctor.invoke(null, null);
  }
//______________________________________________________________________________
 /**
  * Prints the list of available displayers.
  * @see DisplayersProbe#isAvailable
  */
public void printAvailable()
  {
      final int l = 30;

      DisplayersProbe.isAvailable("IceDisplayer");

      for(int i = 0; i < DISPLAYERS.length; i++) {
	  String str = ToolBox.padString(DISPLAYERS[i] + " available", l);
	  System.out.println(str + ": " +
	  String.valueOf(DisplayersProbe.isAvailable(DISPLAYERS[i])));
      }
  }
//______________________________________________________________________________
/**
 * Sets the current tool used to display pages.
 * <Br>
 * If the displayer is not available no change occurs.
 * @param index the index of the new displayer.
 * @return true if the displayer is actually changed.
 * @see #loadInstances()
 */
final protected boolean setDisplayer(int index)
{
    Displayer dsp = instances[index];
    if(dsp == null) return false;

    displayer = dsp;
    return true;
}
//______________________________________________________________________________
}








