//______________________________________________________________________________

//	Java Virtual Shelf
//______________________________________________________________________________

package org.demo.webwader.rbi;

import org.demo.webwader.VersionInfo;
import org.ariane.tools.TryProg;
import org.ariane.tools.ToolBox;
import org.ariane.tools.ResourcesException;
import java.util.Locale;
import java.io.IOException;
import java.io.File;
import java.rmi.*;
import java.rmi.registry.*;
import java.net.MalformedURLException;

/**
 * The main program.
 * <P>
 * <code>
 * Usage : java org.demo.webwader.bri.Browser
 *         [-v|d|q][-V][-h] [url]
 * </code>
 * <P>
 * It parses the command line, it launches a browser and it starts a server.
 * <br>
 * To help debugging the server can be started with the following parameters
 * defined (See a RMI documentation) :
 * <code>
 * java -Djava.rmi.server.logCalls=true -Djava.rmi.dgc.leaseValue=10000
 * </code>
 *
 * @see MainFrame
 * @version $Id: Browser.java,v 1.10 2001/01/28 14:53:57 lefevre Exp $
 * @author  Jean-Paul Le Fvre
 */

public class Browser extends TryProg {
    /**
     * @serial The name of the application. Used to fetch properties.
     */
public static final String APPLI = "Browser";
    /**
     * @serial The name of the resource used to fetch the registry port.
     */
public static final String REGISTRY_PORT = ".Registry.Port";
    /**
     * The ident of the servant.
     */
public static final String SERVANT = "//localhost/RemoteDisplayer";
    /**
     * @serial the first URL to display.
     */
private String url = null;

//______________________________________________________________________________
    /**
     * The main program.
     * @param args the command line.
     */
public static void main (String[] args)
  {
      ToolBox.debug   = false;
      ToolBox.verbose = true;
      Browser prog    = new Browser();

      if (! prog.parseArg(args)) {
	  System.err.println("Can't parse command line !");
	  exitWrong();
      }

      if (! prog.run()) {
	  System.err.println("Can't execute Browser !");
	  exitWrong();
      }
  }
//______________________________________________________________________________
  /**
   * Creates the program.
   * <Br>
   * It loads the properties used in the application : the default JVS
   * values, the general WebWader values and the user values.
   */
public Browser()
  {
      super("Browser");
      setResources();
  }
//______________________________________________________________________________
/**
 * Launches the program : builds the main frame.
 * @return true if successful.
 * @see MainFrame
 */
public	boolean run()
  {
      if(ToolBox.verbose) {
	  String locale = Locale.getDefault().getDisplayName();
	  System.out.println("Creating Browser GUI " + locale + " !");
      }
      if(ToolBox.debug) {
	  System.out.println(VersionInfo.instance().toString());
	  DisplayersBag.instance().printAvailable();
      }

      ToolBox.dot();

    try {
	MainFrame.instance();

	NavePane pane = NavePane.instance();
	if(url != null)
	    pane.displayURL(url);
	else
	    pane.displayHomePage();

	startServant();
    }
    catch(ResourcesException ex) {
	ToolBox.warn("Can't manage resources " + ex);
	ToolBox.warn(
        "Try -Djvs.home=path_to_JVS or -cp path_to_JVS/lib/jvs.jar");
	return false;
    }
    catch(SecurityException ex) {
	ToolBox.warn("Can't grant permission to run " + ex);
	return false;
    }
    catch(RemoteException ex) {
	ToolBox.warn("Can't start the servant " + ex);
	return false;
    }
    catch(Exception ex) {
	ToolBox.warn("Can't run the Browser " + ex);
	if(ToolBox.debug) ex.printStackTrace();
	return false;
    }

    return true;
  }
//______________________________________________________________________________
/**
 * Makes the servant available on the net.
 * <br>
 * A registry is created on a port taken in the resources.
 * The port number used by the servant is also taken in the resources.
 * @throw RemoteException if the server cannot be created.
 * @throw MalformedURLException is the URL is invalid.
 * @see LocateRegistry
 */
private void startServant() throws RemoteException, MalformedURLException
{
    int port = resources.get(REGISTRY_PORT, Registry.REGISTRY_PORT);
    Registry registry = LocateRegistry.createRegistry(port);

    port = resources.get(Browser.APPLI + ".Port",  8334);
    DisplayerServant servant = new DisplayerServant(port);
    String ident = SERVANT;

    registry.rebind(ident, servant);

    StringBuffer buf = new StringBuffer("Displayer ");
    buf.append(ident).append('(').append(servant.getName()).append(')');
    buf.append(" bound on port ").append(port);

    ToolBox.warn(buf.toString());
}
//______________________________________________________________________________
/**
 * Installs the specific resources.
 * @see	org.demo.webwader.Resources
 */
protected void setResources()
  {
      resources = org.demo.webwader.Resources.instance();
  }
//______________________________________________________________________________
    /**
     * Prints how to use the program.
     */
private	void printUsage()
  {
      System.out.println(
      "Usage   : java org.demo.webwader.bri.Browser [options]");
      System.out.println(
      "Options : [-v|-d|-q][-V][-h] [url]");
  }
//______________________________________________________________________________
    /**
     * Parses the command line.
     * <Br>
     * <Code>Usage   : -v|d|q -V -h url</Code>
     * @return true if the command line is correct.
     */
public	boolean parseArg(String args[])
  {
    int i          = 0;
    int error      = 0;

    while(i < args.length && args[i].startsWith("-")) {

      String arg = args[i++];
      int j;

      for (j = 1; j < arg.length(); j++) {

	char flag = arg.charAt(j);

	switch (flag) {
	case 'V':
	  org.demo.webwader.VersionInfo.print();
	  break;

	case 'v':
	  ToolBox.verbose = true;
	  ToolBox.debug   = false;
	  break;

	case 'd':
	  ToolBox.verbose = true;
	  ToolBox.debug   = true;
	  break;

	case 'q':
	  ToolBox.verbose = false;
	  ToolBox.debug   = false;
	  break;

	default	:
	  printUsage();
	  error++;
	  break;
	}
      }
    }

    if(i < args.length) {
	url = args[i++];
    }

    return (error == 0);
  }
//______________________________________________________________________________
}





