//______________________________________________________________________________

//	The Java Virtual Shelf
//______________________________________________________________________________

package org.demo.webwader.gui;

import org.demo.webwader.Resources;
import org.ariane.exec.ExecutionException;
import org.ariane.tools.ToolBox;
import java.net.URL;
import java.lang.reflect.Method;
import java.lang.reflect.InvocationTargetException;

/**
 * A displayer used to communicate with a Browser using XmlRPC.
 * <br>
 * This file should compile without error even if the package xmlrpc
 * or xbrowser are not available. The implementation is loaded dynamically
 * when needed.
 *
 * @version $Id: XmlDisplayer.java,v 3.2 2001/06/08 16:11:21 lefevre Exp $
 * @author Jean-Paul Le Fvre
 * @see NavePane
 * @see org.demo.webwader.xmlrpc.XmlDisplayer
 */
//______________________________________________________________________________

public class XmlDisplayer extends AbstractDisplayer {
  /**
   * @serial The unique instance of this object.
   */
private  static XmlDisplayer instance = null;
  /**
   * @serial The current actual embedded displayer.
   */
private Displayer displayer = null;
    /**
     * @serial The name of this tool.
     */
public static final String NAME = "XBrowser";

//______________________________________________________________________________
/**
 * Gets an unique instance of the displayer.
 * <Br>
 * It is a singleton
 * @return the unique instance.
 */
protected static XmlDisplayer instance()
  {
    if(instance == null) instance = new XmlDisplayer();

    return instance;
  }
//______________________________________________________________________________
/**
 * Creates the displayer.
 */
private XmlDisplayer()
  {
      super();
  }
//______________________________________________________________________________
/**
 * Loads dynamically the xmlrpc viewer.
 * 
 * @return an instance of the viewer.
 * @see Class#forName
 * @see Method#invoke
 * @throw ClassNotFoundException if the class is not found.
 * @throw NoSuchMethodException is there is no instance() method.
 * @throw IllegalAccessException is invocation is denied.
 * @throw InvocationTargetException is invocation is denied.
 */
final private Displayer getInstance()
    throws ClassNotFoundException, NoSuchMethodException,
           IllegalAccessException, InvocationTargetException
  {
      Class dsp   = Class.forName("org.demo.webwader.xmlrpc.XmlDisplayer");
      Method ctor = dsp.getMethod("instance", null);

      return (Displayer)ctor.invoke(null, null);
  }
//______________________________________________________________________________
/**
 * Returns the name of the current remote displayer.
 * <br>
 * The derived displayers have their own local <code>getName()</code> methods.
 * This one is provided for testing. Names retrieved either locally or
 * remotely should be the same.
 * In case of problem null is returned.
 * @return the string or null is case of problem.
 * @see RemoteDisplayer#getName
 */
final public String getRemoteName()
    {
	try {
	    return displayer != null ? displayer.getName() : null;
	}
	catch(Exception ex) {
	    if(ToolBox.debug) ToolBox.warn("Can't get name of displayer", ex);
	    return null;
	}
    }
//______________________________________________________________________________
/**
 * Activates this displayer.
 */
final public void activate()
{
    try {
	if(displayer == null) {
	    displayer =  getInstance();
	}
	displayer.activate();
    }
    catch(Throwable ex) { // errors and execptions are possible.
	ToolBox.warn("Can't activate xml displayer " + ex.getMessage());
	ErrorWindow.instance().display(
	"Can't activate " + getName() + " !\n" +
	"Check if this displayer is available.");

	displayer = null;
	throw new ExecutionException("Remote displayer activation failed");
    }
}
//______________________________________________________________________________
/**
 * Displays the specified page in the browser.
 * <Br>
 * @param url the url of the page.
 * @return true is the page has been displayed.
 * @see #isToShow
 */
final public boolean displayPage(URL url)
{
    try {
	if(! isToShow(url)) return false;

	displayer.displayPage(url);
    }
    catch(Exception ex) {
	ToolBox.warn("Can't display page with " + getName(), ex);
	ErrorWindow.instance().display(
	"Can't display page with " + getName() + " !\n" +
	"Check if the connection is still alive.");
	return false;
    }

    return true;
}
//______________________________________________________________________________
/**
 * Returns the name of this tool.
 * @return the string.
 */
final public String getName()
{
    return NAME;
}
//______________________________________________________________________________
}








